/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a GameLift fleet of game hosting resources.
 * </p>
 * <p>
 * <b>Related actions</b>
 * </p>
 * <p>
 * <a>CreateFleet</a> | <a>DescribeFleetAttributes</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FleetAttributes implements SdkPojo, Serializable, ToCopyableBuilder<FleetAttributes.Builder, FleetAttributes> {
    private static final SdkField<String> FLEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetId").getter(getter(FleetAttributes::fleetId)).setter(setter(Builder::fleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetId").build()).build();

    private static final SdkField<String> FLEET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetArn").getter(getter(FleetAttributes::fleetArn)).setter(setter(Builder::fleetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetArn").build()).build();

    private static final SdkField<String> FLEET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetType").getter(getter(FleetAttributes::fleetTypeAsString)).setter(setter(Builder::fleetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetType").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(FleetAttributes::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(FleetAttributes::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(FleetAttributes::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(FleetAttributes::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> TERMINATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("TerminationTime").getter(getter(FleetAttributes::terminationTime))
            .setter(setter(Builder::terminationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TerminationTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(FleetAttributes::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> BUILD_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BuildId").getter(getter(FleetAttributes::buildId)).setter(setter(Builder::buildId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BuildId").build()).build();

    private static final SdkField<String> BUILD_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BuildArn").getter(getter(FleetAttributes::buildArn)).setter(setter(Builder::buildArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BuildArn").build()).build();

    private static final SdkField<String> SCRIPT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScriptId").getter(getter(FleetAttributes::scriptId)).setter(setter(Builder::scriptId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScriptId").build()).build();

    private static final SdkField<String> SCRIPT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScriptArn").getter(getter(FleetAttributes::scriptArn)).setter(setter(Builder::scriptArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScriptArn").build()).build();

    private static final SdkField<String> SERVER_LAUNCH_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerLaunchPath").getter(getter(FleetAttributes::serverLaunchPath))
            .setter(setter(Builder::serverLaunchPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerLaunchPath").build()).build();

    private static final SdkField<String> SERVER_LAUNCH_PARAMETERS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerLaunchParameters").getter(getter(FleetAttributes::serverLaunchParameters))
            .setter(setter(Builder::serverLaunchParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerLaunchParameters").build())
            .build();

    private static final SdkField<List<String>> LOG_PATHS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LogPaths")
            .getter(getter(FleetAttributes::logPaths))
            .setter(setter(Builder::logPaths))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogPaths").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NEW_GAME_SESSION_PROTECTION_POLICY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NewGameSessionProtectionPolicy")
            .getter(getter(FleetAttributes::newGameSessionProtectionPolicyAsString))
            .setter(setter(Builder::newGameSessionProtectionPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NewGameSessionProtectionPolicy")
                    .build()).build();

    private static final SdkField<String> OPERATING_SYSTEM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperatingSystem").getter(getter(FleetAttributes::operatingSystemAsString))
            .setter(setter(Builder::operatingSystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingSystem").build()).build();

    private static final SdkField<ResourceCreationLimitPolicy> RESOURCE_CREATION_LIMIT_POLICY_FIELD = SdkField
            .<ResourceCreationLimitPolicy> builder(MarshallingType.SDK_POJO)
            .memberName("ResourceCreationLimitPolicy")
            .getter(getter(FleetAttributes::resourceCreationLimitPolicy))
            .setter(setter(Builder::resourceCreationLimitPolicy))
            .constructor(ResourceCreationLimitPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceCreationLimitPolicy")
                    .build()).build();

    private static final SdkField<List<String>> METRIC_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("MetricGroups")
            .getter(getter(FleetAttributes::metricGroups))
            .setter(setter(Builder::metricGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> STOPPED_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("StoppedActions")
            .getter(getter(FleetAttributes::stoppedActionsAsStrings))
            .setter(setter(Builder::stoppedActionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StoppedActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INSTANCE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceRoleArn").getter(getter(FleetAttributes::instanceRoleArn))
            .setter(setter(Builder::instanceRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceRoleArn").build()).build();

    private static final SdkField<CertificateConfiguration> CERTIFICATE_CONFIGURATION_FIELD = SdkField
            .<CertificateConfiguration> builder(MarshallingType.SDK_POJO).memberName("CertificateConfiguration")
            .getter(getter(FleetAttributes::certificateConfiguration)).setter(setter(Builder::certificateConfiguration))
            .constructor(CertificateConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertificateConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FLEET_ID_FIELD,
            FLEET_ARN_FIELD, FLEET_TYPE_FIELD, INSTANCE_TYPE_FIELD, DESCRIPTION_FIELD, NAME_FIELD, CREATION_TIME_FIELD,
            TERMINATION_TIME_FIELD, STATUS_FIELD, BUILD_ID_FIELD, BUILD_ARN_FIELD, SCRIPT_ID_FIELD, SCRIPT_ARN_FIELD,
            SERVER_LAUNCH_PATH_FIELD, SERVER_LAUNCH_PARAMETERS_FIELD, LOG_PATHS_FIELD, NEW_GAME_SESSION_PROTECTION_POLICY_FIELD,
            OPERATING_SYSTEM_FIELD, RESOURCE_CREATION_LIMIT_POLICY_FIELD, METRIC_GROUPS_FIELD, STOPPED_ACTIONS_FIELD,
            INSTANCE_ROLE_ARN_FIELD, CERTIFICATE_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String fleetId;

    private final String fleetArn;

    private final String fleetType;

    private final String instanceType;

    private final String description;

    private final String name;

    private final Instant creationTime;

    private final Instant terminationTime;

    private final String status;

    private final String buildId;

    private final String buildArn;

    private final String scriptId;

    private final String scriptArn;

    private final String serverLaunchPath;

    private final String serverLaunchParameters;

    private final List<String> logPaths;

    private final String newGameSessionProtectionPolicy;

    private final String operatingSystem;

    private final ResourceCreationLimitPolicy resourceCreationLimitPolicy;

    private final List<String> metricGroups;

    private final List<String> stoppedActions;

    private final String instanceRoleArn;

    private final CertificateConfiguration certificateConfiguration;

    private FleetAttributes(BuilderImpl builder) {
        this.fleetId = builder.fleetId;
        this.fleetArn = builder.fleetArn;
        this.fleetType = builder.fleetType;
        this.instanceType = builder.instanceType;
        this.description = builder.description;
        this.name = builder.name;
        this.creationTime = builder.creationTime;
        this.terminationTime = builder.terminationTime;
        this.status = builder.status;
        this.buildId = builder.buildId;
        this.buildArn = builder.buildArn;
        this.scriptId = builder.scriptId;
        this.scriptArn = builder.scriptArn;
        this.serverLaunchPath = builder.serverLaunchPath;
        this.serverLaunchParameters = builder.serverLaunchParameters;
        this.logPaths = builder.logPaths;
        this.newGameSessionProtectionPolicy = builder.newGameSessionProtectionPolicy;
        this.operatingSystem = builder.operatingSystem;
        this.resourceCreationLimitPolicy = builder.resourceCreationLimitPolicy;
        this.metricGroups = builder.metricGroups;
        this.stoppedActions = builder.stoppedActions;
        this.instanceRoleArn = builder.instanceRoleArn;
        this.certificateConfiguration = builder.certificateConfiguration;
    }

    /**
     * <p>
     * A unique identifier for the fleet.
     * </p>
     * 
     * @return A unique identifier for the fleet.
     */
    public final String fleetId() {
        return fleetId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * that is assigned to a GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions.
     * Format is <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a
     * GameLift fleet ARN, the resource ID matches the <code>FleetId</code> value.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
     *         GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
     *         <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a
     *         GameLift fleet ARN, the resource ID matches the <code>FleetId</code> value.
     */
    public final String fleetArn() {
        return fleetArn;
    }

    /**
     * <p>
     * The kind of instances, On-Demand or Spot, that this fleet uses.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fleetType} will
     * return {@link FleetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fleetTypeAsString}.
     * </p>
     * 
     * @return The kind of instances, On-Demand or Spot, that this fleet uses.
     * @see FleetType
     */
    public final FleetType fleetType() {
        return FleetType.fromValue(fleetType);
    }

    /**
     * <p>
     * The kind of instances, On-Demand or Spot, that this fleet uses.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fleetType} will
     * return {@link FleetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fleetTypeAsString}.
     * </p>
     * 
     * @return The kind of instances, On-Demand or Spot, that this fleet uses.
     * @see FleetType
     */
    public final String fleetTypeAsString() {
        return fleetType;
    }

    /**
     * <p>
     * The EC2 instance type that determines the computing resources of each instance in the fleet. Instance type
     * defines the CPU, memory, storage, and networking capacity. See <a
     * href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a> for detailed descriptions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link EC2InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The EC2 instance type that determines the computing resources of each instance in the fleet. Instance
     *         type defines the CPU, memory, storage, and networking capacity. See <a
     *         href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a> for detailed descriptions.
     * @see EC2InstanceType
     */
    public final EC2InstanceType instanceType() {
        return EC2InstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The EC2 instance type that determines the computing resources of each instance in the fleet. Instance type
     * defines the CPU, memory, storage, and networking capacity. See <a
     * href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a> for detailed descriptions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link EC2InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The EC2 instance type that determines the computing resources of each instance in the fleet. Instance
     *         type defines the CPU, memory, storage, and networking capacity. See <a
     *         href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a> for detailed descriptions.
     * @see EC2InstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * A human-readable description of the fleet.
     * </p>
     * 
     * @return A human-readable description of the fleet.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A descriptive label that is associated with a fleet. Fleet names do not need to be unique.
     * </p>
     * 
     * @return A descriptive label that is associated with a fleet. Fleet names do not need to be unique.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was terminated. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was terminated. Format is a number expressed in Unix time
     *         as milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant terminationTime() {
        return terminationTime;
    }

    /**
     * <p>
     * Current status of the fleet. Possible fleet statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>NEW</b> -- A new fleet has been defined and desired instances is set to 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DOWNLOADING/VALIDATING/BUILDING/ACTIVATING</b> -- GameLift is setting up the new fleet, creating new instances
     * with the game build or Realtime script and starting server processes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- Hosts can now accept game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ERROR</b> -- An error occurred when downloading, validating, building, or activating the fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETING</b> -- Hosts are responding to a delete fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TERMINATED</b> -- The fleet no longer exists.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FleetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the fleet. Possible fleet statuses include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>NEW</b> -- A new fleet has been defined and desired instances is set to 1.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DOWNLOADING/VALIDATING/BUILDING/ACTIVATING</b> -- GameLift is setting up the new fleet, creating new
     *         instances with the game build or Realtime script and starting server processes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- Hosts can now accept game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ERROR</b> -- An error occurred when downloading, validating, building, or activating the fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETING</b> -- Hosts are responding to a delete fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TERMINATED</b> -- The fleet no longer exists.
     *         </p>
     *         </li>
     * @see FleetStatus
     */
    public final FleetStatus status() {
        return FleetStatus.fromValue(status);
    }

    /**
     * <p>
     * Current status of the fleet. Possible fleet statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>NEW</b> -- A new fleet has been defined and desired instances is set to 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DOWNLOADING/VALIDATING/BUILDING/ACTIVATING</b> -- GameLift is setting up the new fleet, creating new instances
     * with the game build or Realtime script and starting server processes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- Hosts can now accept game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ERROR</b> -- An error occurred when downloading, validating, building, or activating the fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETING</b> -- Hosts are responding to a delete fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TERMINATED</b> -- The fleet no longer exists.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FleetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the fleet. Possible fleet statuses include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>NEW</b> -- A new fleet has been defined and desired instances is set to 1.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DOWNLOADING/VALIDATING/BUILDING/ACTIVATING</b> -- GameLift is setting up the new fleet, creating new
     *         instances with the game build or Realtime script and starting server processes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- Hosts can now accept game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ERROR</b> -- An error occurred when downloading, validating, building, or activating the fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETING</b> -- Hosts are responding to a delete fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TERMINATED</b> -- The fleet no longer exists.
     *         </p>
     *         </li>
     * @see FleetStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A unique identifier for the build resource that is deployed on instances in this fleet.
     * </p>
     * 
     * @return A unique identifier for the build resource that is deployed on instances in this fleet.
     */
    public final String buildId() {
        return buildId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * associated with the GameLift build resource that is deployed on instances in this fleet. In a GameLift build ARN,
     * the resource ID matches the <code>BuildId</code> value.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
     *         GameLift build resource that is deployed on instances in this fleet. In a GameLift build ARN, the
     *         resource ID matches the <code>BuildId</code> value.
     */
    public final String buildArn() {
        return buildArn;
    }

    /**
     * <p>
     * A unique identifier for the Realtime script resource that is deployed on instances in this fleet.
     * </p>
     * 
     * @return A unique identifier for the Realtime script resource that is deployed on instances in this fleet.
     */
    public final String scriptId() {
        return scriptId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * associated with the GameLift script resource that is deployed on instances in this fleet. In a GameLift script
     * ARN, the resource ID matches the <code>ScriptId</code> value.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
     *         GameLift script resource that is deployed on instances in this fleet. In a GameLift script ARN, the
     *         resource ID matches the <code>ScriptId</code> value.
     */
    public final String scriptArn() {
        return scriptArn;
    }

    /**
     * <p>
     * <b>This parameter is no longer used.</b> Server launch paths are now defined using the fleet's
     * <a>RuntimeConfiguration</a> parameter. Requests that use this parameter instead continue to be valid.
     * </p>
     * 
     * @return <b>This parameter is no longer used.</b> Server launch paths are now defined using the fleet's
     *         <a>RuntimeConfiguration</a> parameter. Requests that use this parameter instead continue to be valid.
     */
    public final String serverLaunchPath() {
        return serverLaunchPath;
    }

    /**
     * <p>
     * <b>This parameter is no longer used.</b> Server launch parameters are now defined using the fleet's
     * <a>RuntimeConfiguration</a> parameter. Requests that use this parameter instead continue to be valid.
     * </p>
     * 
     * @return <b>This parameter is no longer used.</b> Server launch parameters are now defined using the fleet's
     *         <a>RuntimeConfiguration</a> parameter. Requests that use this parameter instead continue to be valid.
     */
    public final String serverLaunchParameters() {
        return serverLaunchParameters;
    }

    /**
     * Returns true if the LogPaths property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasLogPaths() {
        return logPaths != null && !(logPaths instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>This parameter is no longer used.</b> Game session log paths are now defined using the GameLift server API
     * <code>ProcessReady()</code> <code>logParameters</code>. See more information in the <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
     * >Server API Reference</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLogPaths()} to see if a value was sent in this field.
     * </p>
     * 
     * @return <b>This parameter is no longer used.</b> Game session log paths are now defined using the GameLift server
     *         API <code>ProcessReady()</code> <code>logParameters</code>. See more information in the <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
     *         >Server API Reference</a>.
     */
    public final List<String> logPaths() {
        return logPaths;
    }

    /**
     * <p>
     * The type of game session protection to set on all new instances that are started in the fleet.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated during
     * a scale-down event.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #newGameSessionProtectionPolicy} will return {@link ProtectionPolicy#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #newGameSessionProtectionPolicyAsString}.
     * </p>
     * 
     * @return The type of game session protection to set on all new instances that are started in the fleet.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated
     *         during a scale-down event.
     *         </p>
     *         </li>
     * @see ProtectionPolicy
     */
    public final ProtectionPolicy newGameSessionProtectionPolicy() {
        return ProtectionPolicy.fromValue(newGameSessionProtectionPolicy);
    }

    /**
     * <p>
     * The type of game session protection to set on all new instances that are started in the fleet.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated during
     * a scale-down event.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #newGameSessionProtectionPolicy} will return {@link ProtectionPolicy#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #newGameSessionProtectionPolicyAsString}.
     * </p>
     * 
     * @return The type of game session protection to set on all new instances that are started in the fleet.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated
     *         during a scale-down event.
     *         </p>
     *         </li>
     * @see ProtectionPolicy
     */
    public final String newGameSessionProtectionPolicyAsString() {
        return newGameSessionProtectionPolicy;
    }

    /**
     * <p>
     * The operating system of the fleet's computing resources. A fleet's operating system is determined by the OS of
     * the build or script that is deployed on this fleet.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return The operating system of the fleet's computing resources. A fleet's operating system is determined by the
     *         OS of the build or script that is deployed on this fleet.
     * @see OperatingSystem
     */
    public final OperatingSystem operatingSystem() {
        return OperatingSystem.fromValue(operatingSystem);
    }

    /**
     * <p>
     * The operating system of the fleet's computing resources. A fleet's operating system is determined by the OS of
     * the build or script that is deployed on this fleet.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return The operating system of the fleet's computing resources. A fleet's operating system is determined by the
     *         OS of the build or script that is deployed on this fleet.
     * @see OperatingSystem
     */
    public final String operatingSystemAsString() {
        return operatingSystem;
    }

    /**
     * <p>
     * The fleet policy that limits the number of game sessions an individual player can create over a span of time.
     * </p>
     * 
     * @return The fleet policy that limits the number of game sessions an individual player can create over a span of
     *         time.
     */
    public final ResourceCreationLimitPolicy resourceCreationLimitPolicy() {
        return resourceCreationLimitPolicy;
    }

    /**
     * Returns true if the MetricGroups property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasMetricGroups() {
        return metricGroups != null && !(metricGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view aggregated
     * metrics for fleets that are in a metric group. A fleet can be included in only one metric group at a time.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMetricGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view
     *         aggregated metrics for fleets that are in a metric group. A fleet can be included in only one metric
     *         group at a time.
     */
    public final List<String> metricGroups() {
        return metricGroups;
    }

    /**
     * <p>
     * A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet auto-scaling.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStoppedActions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet
     *         auto-scaling.
     */
    public final List<FleetAction> stoppedActions() {
        return FleetActionListCopier.copyStringToEnum(stoppedActions);
    }

    /**
     * Returns true if the StoppedActions property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasStoppedActions() {
        return stoppedActions != null && !(stoppedActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet auto-scaling.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStoppedActions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet
     *         auto-scaling.
     */
    public final List<String> stoppedActionsAsStrings() {
        return stoppedActions;
    }

    /**
     * <p>
     * A unique identifier for an AWS IAM role that manages access to your AWS services. With an instance role ARN set,
     * any application that runs on an instance in this fleet can assume the role, including install scripts, server
     * processes, and daemons (background processes). Create a role or look up a role's ARN by using the <a
     * href="https://console.aws.amazon.com/iam/">IAM dashboard</a> in the AWS Management Console. Learn more about
     * using on-box credentials for your game servers at <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html"> Access
     * external resources from a game server</a>.
     * </p>
     * 
     * @return A unique identifier for an AWS IAM role that manages access to your AWS services. With an instance role
     *         ARN set, any application that runs on an instance in this fleet can assume the role, including install
     *         scripts, server processes, and daemons (background processes). Create a role or look up a role's ARN by
     *         using the <a href="https://console.aws.amazon.com/iam/">IAM dashboard</a> in the AWS Management Console.
     *         Learn more about using on-box credentials for your game servers at <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html">
     *         Access external resources from a game server</a>.
     */
    public final String instanceRoleArn() {
        return instanceRoleArn;
    }

    /**
     * <p>
     * Indicates whether a TLS/SSL certificate was generated for the fleet.
     * </p>
     * 
     * @return Indicates whether a TLS/SSL certificate was generated for the fleet.
     */
    public final CertificateConfiguration certificateConfiguration() {
        return certificateConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fleetId());
        hashCode = 31 * hashCode + Objects.hashCode(fleetArn());
        hashCode = 31 * hashCode + Objects.hashCode(fleetTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(terminationTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(buildId());
        hashCode = 31 * hashCode + Objects.hashCode(buildArn());
        hashCode = 31 * hashCode + Objects.hashCode(scriptId());
        hashCode = 31 * hashCode + Objects.hashCode(scriptArn());
        hashCode = 31 * hashCode + Objects.hashCode(serverLaunchPath());
        hashCode = 31 * hashCode + Objects.hashCode(serverLaunchParameters());
        hashCode = 31 * hashCode + Objects.hashCode(hasLogPaths() ? logPaths() : null);
        hashCode = 31 * hashCode + Objects.hashCode(newGameSessionProtectionPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystemAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceCreationLimitPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricGroups() ? metricGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasStoppedActions() ? stoppedActionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(instanceRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(certificateConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FleetAttributes)) {
            return false;
        }
        FleetAttributes other = (FleetAttributes) obj;
        return Objects.equals(fleetId(), other.fleetId()) && Objects.equals(fleetArn(), other.fleetArn())
                && Objects.equals(fleetTypeAsString(), other.fleetTypeAsString())
                && Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(description(), other.description()) && Objects.equals(name(), other.name())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(terminationTime(), other.terminationTime())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(buildId(), other.buildId())
                && Objects.equals(buildArn(), other.buildArn()) && Objects.equals(scriptId(), other.scriptId())
                && Objects.equals(scriptArn(), other.scriptArn()) && Objects.equals(serverLaunchPath(), other.serverLaunchPath())
                && Objects.equals(serverLaunchParameters(), other.serverLaunchParameters())
                && hasLogPaths() == other.hasLogPaths() && Objects.equals(logPaths(), other.logPaths())
                && Objects.equals(newGameSessionProtectionPolicyAsString(), other.newGameSessionProtectionPolicyAsString())
                && Objects.equals(operatingSystemAsString(), other.operatingSystemAsString())
                && Objects.equals(resourceCreationLimitPolicy(), other.resourceCreationLimitPolicy())
                && hasMetricGroups() == other.hasMetricGroups() && Objects.equals(metricGroups(), other.metricGroups())
                && hasStoppedActions() == other.hasStoppedActions()
                && Objects.equals(stoppedActionsAsStrings(), other.stoppedActionsAsStrings())
                && Objects.equals(instanceRoleArn(), other.instanceRoleArn())
                && Objects.equals(certificateConfiguration(), other.certificateConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FleetAttributes").add("FleetId", fleetId()).add("FleetArn", fleetArn())
                .add("FleetType", fleetTypeAsString()).add("InstanceType", instanceTypeAsString())
                .add("Description", description()).add("Name", name()).add("CreationTime", creationTime())
                .add("TerminationTime", terminationTime()).add("Status", statusAsString()).add("BuildId", buildId())
                .add("BuildArn", buildArn()).add("ScriptId", scriptId()).add("ScriptArn", scriptArn())
                .add("ServerLaunchPath", serverLaunchPath()).add("ServerLaunchParameters", serverLaunchParameters())
                .add("LogPaths", hasLogPaths() ? logPaths() : null)
                .add("NewGameSessionProtectionPolicy", newGameSessionProtectionPolicyAsString())
                .add("OperatingSystem", operatingSystemAsString())
                .add("ResourceCreationLimitPolicy", resourceCreationLimitPolicy())
                .add("MetricGroups", hasMetricGroups() ? metricGroups() : null)
                .add("StoppedActions", hasStoppedActions() ? stoppedActionsAsStrings() : null)
                .add("InstanceRoleArn", instanceRoleArn()).add("CertificateConfiguration", certificateConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FleetId":
            return Optional.ofNullable(clazz.cast(fleetId()));
        case "FleetArn":
            return Optional.ofNullable(clazz.cast(fleetArn()));
        case "FleetType":
            return Optional.ofNullable(clazz.cast(fleetTypeAsString()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "TerminationTime":
            return Optional.ofNullable(clazz.cast(terminationTime()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "BuildId":
            return Optional.ofNullable(clazz.cast(buildId()));
        case "BuildArn":
            return Optional.ofNullable(clazz.cast(buildArn()));
        case "ScriptId":
            return Optional.ofNullable(clazz.cast(scriptId()));
        case "ScriptArn":
            return Optional.ofNullable(clazz.cast(scriptArn()));
        case "ServerLaunchPath":
            return Optional.ofNullable(clazz.cast(serverLaunchPath()));
        case "ServerLaunchParameters":
            return Optional.ofNullable(clazz.cast(serverLaunchParameters()));
        case "LogPaths":
            return Optional.ofNullable(clazz.cast(logPaths()));
        case "NewGameSessionProtectionPolicy":
            return Optional.ofNullable(clazz.cast(newGameSessionProtectionPolicyAsString()));
        case "OperatingSystem":
            return Optional.ofNullable(clazz.cast(operatingSystemAsString()));
        case "ResourceCreationLimitPolicy":
            return Optional.ofNullable(clazz.cast(resourceCreationLimitPolicy()));
        case "MetricGroups":
            return Optional.ofNullable(clazz.cast(metricGroups()));
        case "StoppedActions":
            return Optional.ofNullable(clazz.cast(stoppedActionsAsStrings()));
        case "InstanceRoleArn":
            return Optional.ofNullable(clazz.cast(instanceRoleArn()));
        case "CertificateConfiguration":
            return Optional.ofNullable(clazz.cast(certificateConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FleetAttributes, T> g) {
        return obj -> g.apply((FleetAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FleetAttributes> {
        /**
         * <p>
         * A unique identifier for the fleet.
         * </p>
         * 
         * @param fleetId
         *        A unique identifier for the fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetId(String fleetId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
         * GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
         * <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a GameLift
         * fleet ARN, the resource ID matches the <code>FleetId</code> value.
         * </p>
         * 
         * @param fleetArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        a GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
         *        <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a
         *        GameLift fleet ARN, the resource ID matches the <code>FleetId</code> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetArn(String fleetArn);

        /**
         * <p>
         * The kind of instances, On-Demand or Spot, that this fleet uses.
         * </p>
         * 
         * @param fleetType
         *        The kind of instances, On-Demand or Spot, that this fleet uses.
         * @see FleetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetType
         */
        Builder fleetType(String fleetType);

        /**
         * <p>
         * The kind of instances, On-Demand or Spot, that this fleet uses.
         * </p>
         * 
         * @param fleetType
         *        The kind of instances, On-Demand or Spot, that this fleet uses.
         * @see FleetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetType
         */
        Builder fleetType(FleetType fleetType);

        /**
         * <p>
         * The EC2 instance type that determines the computing resources of each instance in the fleet. Instance type
         * defines the CPU, memory, storage, and networking capacity. See <a
         * href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a> for detailed descriptions.
         * </p>
         * 
         * @param instanceType
         *        The EC2 instance type that determines the computing resources of each instance in the fleet. Instance
         *        type defines the CPU, memory, storage, and networking capacity. See <a
         *        href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a> for detailed
         *        descriptions.
         * @see EC2InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EC2InstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The EC2 instance type that determines the computing resources of each instance in the fleet. Instance type
         * defines the CPU, memory, storage, and networking capacity. See <a
         * href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a> for detailed descriptions.
         * </p>
         * 
         * @param instanceType
         *        The EC2 instance type that determines the computing resources of each instance in the fleet. Instance
         *        type defines the CPU, memory, storage, and networking capacity. See <a
         *        href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a> for detailed
         *        descriptions.
         * @see EC2InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EC2InstanceType
         */
        Builder instanceType(EC2InstanceType instanceType);

        /**
         * <p>
         * A human-readable description of the fleet.
         * </p>
         * 
         * @param description
         *        A human-readable description of the fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A descriptive label that is associated with a fleet. Fleet names do not need to be unique.
         * </p>
         * 
         * @param name
         *        A descriptive label that is associated with a fleet. Fleet names do not need to be unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param creationTime
         *        A time stamp indicating when this data object was created. Format is a number expressed in Unix time
         *        as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A time stamp indicating when this data object was terminated. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param terminationTime
         *        A time stamp indicating when this data object was terminated. Format is a number expressed in Unix
         *        time as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminationTime(Instant terminationTime);

        /**
         * <p>
         * Current status of the fleet. Possible fleet statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>NEW</b> -- A new fleet has been defined and desired instances is set to 1.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DOWNLOADING/VALIDATING/BUILDING/ACTIVATING</b> -- GameLift is setting up the new fleet, creating new
         * instances with the game build or Realtime script and starting server processes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- Hosts can now accept game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ERROR</b> -- An error occurred when downloading, validating, building, or activating the fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETING</b> -- Hosts are responding to a delete fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TERMINATED</b> -- The fleet no longer exists.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the fleet. Possible fleet statuses include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>NEW</b> -- A new fleet has been defined and desired instances is set to 1.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DOWNLOADING/VALIDATING/BUILDING/ACTIVATING</b> -- GameLift is setting up the new fleet, creating
         *        new instances with the game build or Realtime script and starting server processes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- Hosts can now accept game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ERROR</b> -- An error occurred when downloading, validating, building, or activating the fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETING</b> -- Hosts are responding to a delete fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TERMINATED</b> -- The fleet no longer exists.
         *        </p>
         *        </li>
         * @see FleetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the fleet. Possible fleet statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>NEW</b> -- A new fleet has been defined and desired instances is set to 1.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DOWNLOADING/VALIDATING/BUILDING/ACTIVATING</b> -- GameLift is setting up the new fleet, creating new
         * instances with the game build or Realtime script and starting server processes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- Hosts can now accept game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ERROR</b> -- An error occurred when downloading, validating, building, or activating the fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETING</b> -- Hosts are responding to a delete fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TERMINATED</b> -- The fleet no longer exists.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the fleet. Possible fleet statuses include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>NEW</b> -- A new fleet has been defined and desired instances is set to 1.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DOWNLOADING/VALIDATING/BUILDING/ACTIVATING</b> -- GameLift is setting up the new fleet, creating
         *        new instances with the game build or Realtime script and starting server processes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- Hosts can now accept game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ERROR</b> -- An error occurred when downloading, validating, building, or activating the fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETING</b> -- Hosts are responding to a delete fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TERMINATED</b> -- The fleet no longer exists.
         *        </p>
         *        </li>
         * @see FleetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetStatus
         */
        Builder status(FleetStatus status);

        /**
         * <p>
         * A unique identifier for the build resource that is deployed on instances in this fleet.
         * </p>
         * 
         * @param buildId
         *        A unique identifier for the build resource that is deployed on instances in this fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildId(String buildId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         * GameLift build resource that is deployed on instances in this fleet. In a GameLift build ARN, the resource ID
         * matches the <code>BuildId</code> value.
         * </p>
         * 
         * @param buildArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         *        GameLift build resource that is deployed on instances in this fleet. In a GameLift build ARN, the
         *        resource ID matches the <code>BuildId</code> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildArn(String buildArn);

        /**
         * <p>
         * A unique identifier for the Realtime script resource that is deployed on instances in this fleet.
         * </p>
         * 
         * @param scriptId
         *        A unique identifier for the Realtime script resource that is deployed on instances in this fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scriptId(String scriptId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         * GameLift script resource that is deployed on instances in this fleet. In a GameLift script ARN, the resource
         * ID matches the <code>ScriptId</code> value.
         * </p>
         * 
         * @param scriptArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         *        GameLift script resource that is deployed on instances in this fleet. In a GameLift script ARN, the
         *        resource ID matches the <code>ScriptId</code> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scriptArn(String scriptArn);

        /**
         * <p>
         * <b>This parameter is no longer used.</b> Server launch paths are now defined using the fleet's
         * <a>RuntimeConfiguration</a> parameter. Requests that use this parameter instead continue to be valid.
         * </p>
         * 
         * @param serverLaunchPath
         *        <b>This parameter is no longer used.</b> Server launch paths are now defined using the fleet's
         *        <a>RuntimeConfiguration</a> parameter. Requests that use this parameter instead continue to be valid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverLaunchPath(String serverLaunchPath);

        /**
         * <p>
         * <b>This parameter is no longer used.</b> Server launch parameters are now defined using the fleet's
         * <a>RuntimeConfiguration</a> parameter. Requests that use this parameter instead continue to be valid.
         * </p>
         * 
         * @param serverLaunchParameters
         *        <b>This parameter is no longer used.</b> Server launch parameters are now defined using the fleet's
         *        <a>RuntimeConfiguration</a> parameter. Requests that use this parameter instead continue to be valid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverLaunchParameters(String serverLaunchParameters);

        /**
         * <p>
         * <b>This parameter is no longer used.</b> Game session log paths are now defined using the GameLift server API
         * <code>ProcessReady()</code> <code>logParameters</code>. See more information in the <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
         * >Server API Reference</a>.
         * </p>
         * 
         * @param logPaths
         *        <b>This parameter is no longer used.</b> Game session log paths are now defined using the GameLift
         *        server API <code>ProcessReady()</code> <code>logParameters</code>. See more information in the <a
         *        href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
         *        >Server API Reference</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPaths(Collection<String> logPaths);

        /**
         * <p>
         * <b>This parameter is no longer used.</b> Game session log paths are now defined using the GameLift server API
         * <code>ProcessReady()</code> <code>logParameters</code>. See more information in the <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
         * >Server API Reference</a>.
         * </p>
         * 
         * @param logPaths
         *        <b>This parameter is no longer used.</b> Game session log paths are now defined using the GameLift
         *        server API <code>ProcessReady()</code> <code>logParameters</code>. See more information in the <a
         *        href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
         *        >Server API Reference</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPaths(String... logPaths);

        /**
         * <p>
         * The type of game session protection to set on all new instances that are started in the fleet.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated
         * during a scale-down event.
         * </p>
         * </li>
         * </ul>
         * 
         * @param newGameSessionProtectionPolicy
         *        The type of game session protection to set on all new instances that are started in the fleet.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be
         *        terminated during a scale-down event.
         *        </p>
         *        </li>
         * @see ProtectionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectionPolicy
         */
        Builder newGameSessionProtectionPolicy(String newGameSessionProtectionPolicy);

        /**
         * <p>
         * The type of game session protection to set on all new instances that are started in the fleet.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated
         * during a scale-down event.
         * </p>
         * </li>
         * </ul>
         * 
         * @param newGameSessionProtectionPolicy
         *        The type of game session protection to set on all new instances that are started in the fleet.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be
         *        terminated during a scale-down event.
         *        </p>
         *        </li>
         * @see ProtectionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectionPolicy
         */
        Builder newGameSessionProtectionPolicy(ProtectionPolicy newGameSessionProtectionPolicy);

        /**
         * <p>
         * The operating system of the fleet's computing resources. A fleet's operating system is determined by the OS
         * of the build or script that is deployed on this fleet.
         * </p>
         * 
         * @param operatingSystem
         *        The operating system of the fleet's computing resources. A fleet's operating system is determined by
         *        the OS of the build or script that is deployed on this fleet.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(String operatingSystem);

        /**
         * <p>
         * The operating system of the fleet's computing resources. A fleet's operating system is determined by the OS
         * of the build or script that is deployed on this fleet.
         * </p>
         * 
         * @param operatingSystem
         *        The operating system of the fleet's computing resources. A fleet's operating system is determined by
         *        the OS of the build or script that is deployed on this fleet.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(OperatingSystem operatingSystem);

        /**
         * <p>
         * The fleet policy that limits the number of game sessions an individual player can create over a span of time.
         * </p>
         * 
         * @param resourceCreationLimitPolicy
         *        The fleet policy that limits the number of game sessions an individual player can create over a span
         *        of time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceCreationLimitPolicy(ResourceCreationLimitPolicy resourceCreationLimitPolicy);

        /**
         * <p>
         * The fleet policy that limits the number of game sessions an individual player can create over a span of time.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceCreationLimitPolicy.Builder} avoiding
         * the need to create one manually via {@link ResourceCreationLimitPolicy#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceCreationLimitPolicy.Builder#build()} is called
         * immediately and its result is passed to {@link #resourceCreationLimitPolicy(ResourceCreationLimitPolicy)}.
         * 
         * @param resourceCreationLimitPolicy
         *        a consumer that will call methods on {@link ResourceCreationLimitPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceCreationLimitPolicy(ResourceCreationLimitPolicy)
         */
        default Builder resourceCreationLimitPolicy(Consumer<ResourceCreationLimitPolicy.Builder> resourceCreationLimitPolicy) {
            return resourceCreationLimitPolicy(ResourceCreationLimitPolicy.builder().applyMutation(resourceCreationLimitPolicy)
                    .build());
        }

        /**
         * <p>
         * Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view
         * aggregated metrics for fleets that are in a metric group. A fleet can be included in only one metric group at
         * a time.
         * </p>
         * 
         * @param metricGroups
         *        Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view
         *        aggregated metrics for fleets that are in a metric group. A fleet can be included in only one metric
         *        group at a time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricGroups(Collection<String> metricGroups);

        /**
         * <p>
         * Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view
         * aggregated metrics for fleets that are in a metric group. A fleet can be included in only one metric group at
         * a time.
         * </p>
         * 
         * @param metricGroups
         *        Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view
         *        aggregated metrics for fleets that are in a metric group. A fleet can be included in only one metric
         *        group at a time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricGroups(String... metricGroups);

        /**
         * <p>
         * A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet
         * auto-scaling.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet
         *        auto-scaling.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActionsWithStrings(Collection<String> stoppedActions);

        /**
         * <p>
         * A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet
         * auto-scaling.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet
         *        auto-scaling.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActionsWithStrings(String... stoppedActions);

        /**
         * <p>
         * A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet
         * auto-scaling.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet
         *        auto-scaling.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActions(Collection<FleetAction> stoppedActions);

        /**
         * <p>
         * A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet
         * auto-scaling.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet activity that has been suspended using <a>StopFleetActions</a>. This includes fleet
         *        auto-scaling.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActions(FleetAction... stoppedActions);

        /**
         * <p>
         * A unique identifier for an AWS IAM role that manages access to your AWS services. With an instance role ARN
         * set, any application that runs on an instance in this fleet can assume the role, including install scripts,
         * server processes, and daemons (background processes). Create a role or look up a role's ARN by using the <a
         * href="https://console.aws.amazon.com/iam/">IAM dashboard</a> in the AWS Management Console. Learn more about
         * using on-box credentials for your game servers at <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html"> Access
         * external resources from a game server</a>.
         * </p>
         * 
         * @param instanceRoleArn
         *        A unique identifier for an AWS IAM role that manages access to your AWS services. With an instance
         *        role ARN set, any application that runs on an instance in this fleet can assume the role, including
         *        install scripts, server processes, and daemons (background processes). Create a role or look up a
         *        role's ARN by using the <a href="https://console.aws.amazon.com/iam/">IAM dashboard</a> in the AWS
         *        Management Console. Learn more about using on-box credentials for your game servers at <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html">
         *        Access external resources from a game server</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceRoleArn(String instanceRoleArn);

        /**
         * <p>
         * Indicates whether a TLS/SSL certificate was generated for the fleet.
         * </p>
         * 
         * @param certificateConfiguration
         *        Indicates whether a TLS/SSL certificate was generated for the fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateConfiguration(CertificateConfiguration certificateConfiguration);

        /**
         * <p>
         * Indicates whether a TLS/SSL certificate was generated for the fleet.
         * </p>
         * This is a convenience that creates an instance of the {@link CertificateConfiguration.Builder} avoiding the
         * need to create one manually via {@link CertificateConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link CertificateConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #certificateConfiguration(CertificateConfiguration)}.
         * 
         * @param certificateConfiguration
         *        a consumer that will call methods on {@link CertificateConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #certificateConfiguration(CertificateConfiguration)
         */
        default Builder certificateConfiguration(Consumer<CertificateConfiguration.Builder> certificateConfiguration) {
            return certificateConfiguration(CertificateConfiguration.builder().applyMutation(certificateConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String fleetId;

        private String fleetArn;

        private String fleetType;

        private String instanceType;

        private String description;

        private String name;

        private Instant creationTime;

        private Instant terminationTime;

        private String status;

        private String buildId;

        private String buildArn;

        private String scriptId;

        private String scriptArn;

        private String serverLaunchPath;

        private String serverLaunchParameters;

        private List<String> logPaths = DefaultSdkAutoConstructList.getInstance();

        private String newGameSessionProtectionPolicy;

        private String operatingSystem;

        private ResourceCreationLimitPolicy resourceCreationLimitPolicy;

        private List<String> metricGroups = DefaultSdkAutoConstructList.getInstance();

        private List<String> stoppedActions = DefaultSdkAutoConstructList.getInstance();

        private String instanceRoleArn;

        private CertificateConfiguration certificateConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(FleetAttributes model) {
            fleetId(model.fleetId);
            fleetArn(model.fleetArn);
            fleetType(model.fleetType);
            instanceType(model.instanceType);
            description(model.description);
            name(model.name);
            creationTime(model.creationTime);
            terminationTime(model.terminationTime);
            status(model.status);
            buildId(model.buildId);
            buildArn(model.buildArn);
            scriptId(model.scriptId);
            scriptArn(model.scriptArn);
            serverLaunchPath(model.serverLaunchPath);
            serverLaunchParameters(model.serverLaunchParameters);
            logPaths(model.logPaths);
            newGameSessionProtectionPolicy(model.newGameSessionProtectionPolicy);
            operatingSystem(model.operatingSystem);
            resourceCreationLimitPolicy(model.resourceCreationLimitPolicy);
            metricGroups(model.metricGroups);
            stoppedActionsWithStrings(model.stoppedActions);
            instanceRoleArn(model.instanceRoleArn);
            certificateConfiguration(model.certificateConfiguration);
        }

        public final String getFleetId() {
            return fleetId;
        }

        @Override
        public final Builder fleetId(String fleetId) {
            this.fleetId = fleetId;
            return this;
        }

        public final void setFleetId(String fleetId) {
            this.fleetId = fleetId;
        }

        public final String getFleetArn() {
            return fleetArn;
        }

        @Override
        public final Builder fleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
            return this;
        }

        public final void setFleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
        }

        public final String getFleetType() {
            return fleetType;
        }

        @Override
        public final Builder fleetType(String fleetType) {
            this.fleetType = fleetType;
            return this;
        }

        @Override
        public final Builder fleetType(FleetType fleetType) {
            this.fleetType(fleetType == null ? null : fleetType.toString());
            return this;
        }

        public final void setFleetType(String fleetType) {
            this.fleetType = fleetType;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(EC2InstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getTerminationTime() {
            return terminationTime;
        }

        @Override
        public final Builder terminationTime(Instant terminationTime) {
            this.terminationTime = terminationTime;
            return this;
        }

        public final void setTerminationTime(Instant terminationTime) {
            this.terminationTime = terminationTime;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(FleetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getBuildId() {
            return buildId;
        }

        @Override
        public final Builder buildId(String buildId) {
            this.buildId = buildId;
            return this;
        }

        public final void setBuildId(String buildId) {
            this.buildId = buildId;
        }

        public final String getBuildArn() {
            return buildArn;
        }

        @Override
        public final Builder buildArn(String buildArn) {
            this.buildArn = buildArn;
            return this;
        }

        public final void setBuildArn(String buildArn) {
            this.buildArn = buildArn;
        }

        public final String getScriptId() {
            return scriptId;
        }

        @Override
        public final Builder scriptId(String scriptId) {
            this.scriptId = scriptId;
            return this;
        }

        public final void setScriptId(String scriptId) {
            this.scriptId = scriptId;
        }

        public final String getScriptArn() {
            return scriptArn;
        }

        @Override
        public final Builder scriptArn(String scriptArn) {
            this.scriptArn = scriptArn;
            return this;
        }

        public final void setScriptArn(String scriptArn) {
            this.scriptArn = scriptArn;
        }

        public final String getServerLaunchPath() {
            return serverLaunchPath;
        }

        @Override
        public final Builder serverLaunchPath(String serverLaunchPath) {
            this.serverLaunchPath = serverLaunchPath;
            return this;
        }

        public final void setServerLaunchPath(String serverLaunchPath) {
            this.serverLaunchPath = serverLaunchPath;
        }

        public final String getServerLaunchParameters() {
            return serverLaunchParameters;
        }

        @Override
        public final Builder serverLaunchParameters(String serverLaunchParameters) {
            this.serverLaunchParameters = serverLaunchParameters;
            return this;
        }

        public final void setServerLaunchParameters(String serverLaunchParameters) {
            this.serverLaunchParameters = serverLaunchParameters;
        }

        public final Collection<String> getLogPaths() {
            if (logPaths instanceof SdkAutoConstructList) {
                return null;
            }
            return logPaths;
        }

        @Override
        public final Builder logPaths(Collection<String> logPaths) {
            this.logPaths = StringListCopier.copy(logPaths);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logPaths(String... logPaths) {
            logPaths(Arrays.asList(logPaths));
            return this;
        }

        public final void setLogPaths(Collection<String> logPaths) {
            this.logPaths = StringListCopier.copy(logPaths);
        }

        public final String getNewGameSessionProtectionPolicy() {
            return newGameSessionProtectionPolicy;
        }

        @Override
        public final Builder newGameSessionProtectionPolicy(String newGameSessionProtectionPolicy) {
            this.newGameSessionProtectionPolicy = newGameSessionProtectionPolicy;
            return this;
        }

        @Override
        public final Builder newGameSessionProtectionPolicy(ProtectionPolicy newGameSessionProtectionPolicy) {
            this.newGameSessionProtectionPolicy(newGameSessionProtectionPolicy == null ? null : newGameSessionProtectionPolicy
                    .toString());
            return this;
        }

        public final void setNewGameSessionProtectionPolicy(String newGameSessionProtectionPolicy) {
            this.newGameSessionProtectionPolicy = newGameSessionProtectionPolicy;
        }

        public final String getOperatingSystem() {
            return operatingSystem;
        }

        @Override
        public final Builder operatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
            return this;
        }

        @Override
        public final Builder operatingSystem(OperatingSystem operatingSystem) {
            this.operatingSystem(operatingSystem == null ? null : operatingSystem.toString());
            return this;
        }

        public final void setOperatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
        }

        public final ResourceCreationLimitPolicy.Builder getResourceCreationLimitPolicy() {
            return resourceCreationLimitPolicy != null ? resourceCreationLimitPolicy.toBuilder() : null;
        }

        @Override
        public final Builder resourceCreationLimitPolicy(ResourceCreationLimitPolicy resourceCreationLimitPolicy) {
            this.resourceCreationLimitPolicy = resourceCreationLimitPolicy;
            return this;
        }

        public final void setResourceCreationLimitPolicy(ResourceCreationLimitPolicy.BuilderImpl resourceCreationLimitPolicy) {
            this.resourceCreationLimitPolicy = resourceCreationLimitPolicy != null ? resourceCreationLimitPolicy.build() : null;
        }

        public final Collection<String> getMetricGroups() {
            if (metricGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return metricGroups;
        }

        @Override
        public final Builder metricGroups(Collection<String> metricGroups) {
            this.metricGroups = MetricGroupListCopier.copy(metricGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricGroups(String... metricGroups) {
            metricGroups(Arrays.asList(metricGroups));
            return this;
        }

        public final void setMetricGroups(Collection<String> metricGroups) {
            this.metricGroups = MetricGroupListCopier.copy(metricGroups);
        }

        public final Collection<String> getStoppedActions() {
            if (stoppedActions instanceof SdkAutoConstructList) {
                return null;
            }
            return stoppedActions;
        }

        @Override
        public final Builder stoppedActionsWithStrings(Collection<String> stoppedActions) {
            this.stoppedActions = FleetActionListCopier.copy(stoppedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stoppedActionsWithStrings(String... stoppedActions) {
            stoppedActionsWithStrings(Arrays.asList(stoppedActions));
            return this;
        }

        @Override
        public final Builder stoppedActions(Collection<FleetAction> stoppedActions) {
            this.stoppedActions = FleetActionListCopier.copyEnumToString(stoppedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stoppedActions(FleetAction... stoppedActions) {
            stoppedActions(Arrays.asList(stoppedActions));
            return this;
        }

        public final void setStoppedActions(Collection<String> stoppedActions) {
            this.stoppedActions = FleetActionListCopier.copy(stoppedActions);
        }

        public final String getInstanceRoleArn() {
            return instanceRoleArn;
        }

        @Override
        public final Builder instanceRoleArn(String instanceRoleArn) {
            this.instanceRoleArn = instanceRoleArn;
            return this;
        }

        public final void setInstanceRoleArn(String instanceRoleArn) {
            this.instanceRoleArn = instanceRoleArn;
        }

        public final CertificateConfiguration.Builder getCertificateConfiguration() {
            return certificateConfiguration != null ? certificateConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder certificateConfiguration(CertificateConfiguration certificateConfiguration) {
            this.certificateConfiguration = certificateConfiguration;
            return this;
        }

        public final void setCertificateConfiguration(CertificateConfiguration.BuilderImpl certificateConfiguration) {
            this.certificateConfiguration = certificateConfiguration != null ? certificateConfiguration.build() : null;
        }

        @Override
        public FleetAttributes build() {
            return new FleetAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
