/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.forecast;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.forecast.internal.ForecastServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.forecast.model.CreateAutoPredictorRequest;
import software.amazon.awssdk.services.forecast.model.CreateAutoPredictorResponse;
import software.amazon.awssdk.services.forecast.model.CreateDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.CreateDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.model.CreateDatasetImportJobRequest;
import software.amazon.awssdk.services.forecast.model.CreateDatasetImportJobResponse;
import software.amazon.awssdk.services.forecast.model.CreateDatasetRequest;
import software.amazon.awssdk.services.forecast.model.CreateDatasetResponse;
import software.amazon.awssdk.services.forecast.model.CreateExplainabilityExportRequest;
import software.amazon.awssdk.services.forecast.model.CreateExplainabilityExportResponse;
import software.amazon.awssdk.services.forecast.model.CreateExplainabilityRequest;
import software.amazon.awssdk.services.forecast.model.CreateExplainabilityResponse;
import software.amazon.awssdk.services.forecast.model.CreateForecastExportJobRequest;
import software.amazon.awssdk.services.forecast.model.CreateForecastExportJobResponse;
import software.amazon.awssdk.services.forecast.model.CreateForecastRequest;
import software.amazon.awssdk.services.forecast.model.CreateForecastResponse;
import software.amazon.awssdk.services.forecast.model.CreateMonitorRequest;
import software.amazon.awssdk.services.forecast.model.CreateMonitorResponse;
import software.amazon.awssdk.services.forecast.model.CreatePredictorBacktestExportJobRequest;
import software.amazon.awssdk.services.forecast.model.CreatePredictorBacktestExportJobResponse;
import software.amazon.awssdk.services.forecast.model.CreatePredictorRequest;
import software.amazon.awssdk.services.forecast.model.CreatePredictorResponse;
import software.amazon.awssdk.services.forecast.model.CreateWhatIfAnalysisRequest;
import software.amazon.awssdk.services.forecast.model.CreateWhatIfAnalysisResponse;
import software.amazon.awssdk.services.forecast.model.CreateWhatIfForecastExportRequest;
import software.amazon.awssdk.services.forecast.model.CreateWhatIfForecastExportResponse;
import software.amazon.awssdk.services.forecast.model.CreateWhatIfForecastRequest;
import software.amazon.awssdk.services.forecast.model.CreateWhatIfForecastResponse;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetImportJobRequest;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetImportJobResponse;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetRequest;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetResponse;
import software.amazon.awssdk.services.forecast.model.DeleteExplainabilityExportRequest;
import software.amazon.awssdk.services.forecast.model.DeleteExplainabilityExportResponse;
import software.amazon.awssdk.services.forecast.model.DeleteExplainabilityRequest;
import software.amazon.awssdk.services.forecast.model.DeleteExplainabilityResponse;
import software.amazon.awssdk.services.forecast.model.DeleteForecastExportJobRequest;
import software.amazon.awssdk.services.forecast.model.DeleteForecastExportJobResponse;
import software.amazon.awssdk.services.forecast.model.DeleteForecastRequest;
import software.amazon.awssdk.services.forecast.model.DeleteForecastResponse;
import software.amazon.awssdk.services.forecast.model.DeleteMonitorRequest;
import software.amazon.awssdk.services.forecast.model.DeleteMonitorResponse;
import software.amazon.awssdk.services.forecast.model.DeletePredictorBacktestExportJobRequest;
import software.amazon.awssdk.services.forecast.model.DeletePredictorBacktestExportJobResponse;
import software.amazon.awssdk.services.forecast.model.DeletePredictorRequest;
import software.amazon.awssdk.services.forecast.model.DeletePredictorResponse;
import software.amazon.awssdk.services.forecast.model.DeleteResourceTreeRequest;
import software.amazon.awssdk.services.forecast.model.DeleteResourceTreeResponse;
import software.amazon.awssdk.services.forecast.model.DeleteWhatIfAnalysisRequest;
import software.amazon.awssdk.services.forecast.model.DeleteWhatIfAnalysisResponse;
import software.amazon.awssdk.services.forecast.model.DeleteWhatIfForecastExportRequest;
import software.amazon.awssdk.services.forecast.model.DeleteWhatIfForecastExportResponse;
import software.amazon.awssdk.services.forecast.model.DeleteWhatIfForecastRequest;
import software.amazon.awssdk.services.forecast.model.DeleteWhatIfForecastResponse;
import software.amazon.awssdk.services.forecast.model.DescribeAutoPredictorRequest;
import software.amazon.awssdk.services.forecast.model.DescribeAutoPredictorResponse;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetImportJobRequest;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetImportJobResponse;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetRequest;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetResponse;
import software.amazon.awssdk.services.forecast.model.DescribeExplainabilityExportRequest;
import software.amazon.awssdk.services.forecast.model.DescribeExplainabilityExportResponse;
import software.amazon.awssdk.services.forecast.model.DescribeExplainabilityRequest;
import software.amazon.awssdk.services.forecast.model.DescribeExplainabilityResponse;
import software.amazon.awssdk.services.forecast.model.DescribeForecastExportJobRequest;
import software.amazon.awssdk.services.forecast.model.DescribeForecastExportJobResponse;
import software.amazon.awssdk.services.forecast.model.DescribeForecastRequest;
import software.amazon.awssdk.services.forecast.model.DescribeForecastResponse;
import software.amazon.awssdk.services.forecast.model.DescribeMonitorRequest;
import software.amazon.awssdk.services.forecast.model.DescribeMonitorResponse;
import software.amazon.awssdk.services.forecast.model.DescribePredictorBacktestExportJobRequest;
import software.amazon.awssdk.services.forecast.model.DescribePredictorBacktestExportJobResponse;
import software.amazon.awssdk.services.forecast.model.DescribePredictorRequest;
import software.amazon.awssdk.services.forecast.model.DescribePredictorResponse;
import software.amazon.awssdk.services.forecast.model.DescribeWhatIfAnalysisRequest;
import software.amazon.awssdk.services.forecast.model.DescribeWhatIfAnalysisResponse;
import software.amazon.awssdk.services.forecast.model.DescribeWhatIfForecastExportRequest;
import software.amazon.awssdk.services.forecast.model.DescribeWhatIfForecastExportResponse;
import software.amazon.awssdk.services.forecast.model.DescribeWhatIfForecastRequest;
import software.amazon.awssdk.services.forecast.model.DescribeWhatIfForecastResponse;
import software.amazon.awssdk.services.forecast.model.ForecastException;
import software.amazon.awssdk.services.forecast.model.GetAccuracyMetricsRequest;
import software.amazon.awssdk.services.forecast.model.GetAccuracyMetricsResponse;
import software.amazon.awssdk.services.forecast.model.InvalidInputException;
import software.amazon.awssdk.services.forecast.model.InvalidNextTokenException;
import software.amazon.awssdk.services.forecast.model.LimitExceededException;
import software.amazon.awssdk.services.forecast.model.ListDatasetGroupsRequest;
import software.amazon.awssdk.services.forecast.model.ListDatasetGroupsResponse;
import software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsRequest;
import software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsResponse;
import software.amazon.awssdk.services.forecast.model.ListDatasetsRequest;
import software.amazon.awssdk.services.forecast.model.ListDatasetsResponse;
import software.amazon.awssdk.services.forecast.model.ListExplainabilitiesRequest;
import software.amazon.awssdk.services.forecast.model.ListExplainabilitiesResponse;
import software.amazon.awssdk.services.forecast.model.ListExplainabilityExportsRequest;
import software.amazon.awssdk.services.forecast.model.ListExplainabilityExportsResponse;
import software.amazon.awssdk.services.forecast.model.ListForecastExportJobsRequest;
import software.amazon.awssdk.services.forecast.model.ListForecastExportJobsResponse;
import software.amazon.awssdk.services.forecast.model.ListForecastsRequest;
import software.amazon.awssdk.services.forecast.model.ListForecastsResponse;
import software.amazon.awssdk.services.forecast.model.ListMonitorEvaluationsRequest;
import software.amazon.awssdk.services.forecast.model.ListMonitorEvaluationsResponse;
import software.amazon.awssdk.services.forecast.model.ListMonitorsRequest;
import software.amazon.awssdk.services.forecast.model.ListMonitorsResponse;
import software.amazon.awssdk.services.forecast.model.ListPredictorBacktestExportJobsRequest;
import software.amazon.awssdk.services.forecast.model.ListPredictorBacktestExportJobsResponse;
import software.amazon.awssdk.services.forecast.model.ListPredictorsRequest;
import software.amazon.awssdk.services.forecast.model.ListPredictorsResponse;
import software.amazon.awssdk.services.forecast.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.forecast.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.forecast.model.ListWhatIfAnalysesRequest;
import software.amazon.awssdk.services.forecast.model.ListWhatIfAnalysesResponse;
import software.amazon.awssdk.services.forecast.model.ListWhatIfForecastExportsRequest;
import software.amazon.awssdk.services.forecast.model.ListWhatIfForecastExportsResponse;
import software.amazon.awssdk.services.forecast.model.ListWhatIfForecastsRequest;
import software.amazon.awssdk.services.forecast.model.ListWhatIfForecastsResponse;
import software.amazon.awssdk.services.forecast.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.forecast.model.ResourceInUseException;
import software.amazon.awssdk.services.forecast.model.ResourceNotFoundException;
import software.amazon.awssdk.services.forecast.model.ResumeResourceRequest;
import software.amazon.awssdk.services.forecast.model.ResumeResourceResponse;
import software.amazon.awssdk.services.forecast.model.StopResourceRequest;
import software.amazon.awssdk.services.forecast.model.StopResourceResponse;
import software.amazon.awssdk.services.forecast.model.TagResourceRequest;
import software.amazon.awssdk.services.forecast.model.TagResourceResponse;
import software.amazon.awssdk.services.forecast.model.UntagResourceRequest;
import software.amazon.awssdk.services.forecast.model.UntagResourceResponse;
import software.amazon.awssdk.services.forecast.model.UpdateDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.UpdateDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.transform.CreateAutoPredictorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateDatasetGroupRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateDatasetImportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateDatasetRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateExplainabilityExportRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateExplainabilityRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateForecastExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateMonitorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreatePredictorBacktestExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreatePredictorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateWhatIfAnalysisRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateWhatIfForecastExportRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateWhatIfForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteDatasetGroupRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteDatasetImportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteDatasetRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteExplainabilityExportRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteExplainabilityRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteForecastExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteMonitorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeletePredictorBacktestExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeletePredictorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteResourceTreeRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteWhatIfAnalysisRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteWhatIfForecastExportRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteWhatIfForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeAutoPredictorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeDatasetGroupRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeDatasetImportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeDatasetRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeExplainabilityExportRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeExplainabilityRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeForecastExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeMonitorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribePredictorBacktestExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribePredictorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeWhatIfAnalysisRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeWhatIfForecastExportRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeWhatIfForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.GetAccuracyMetricsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListDatasetGroupsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListDatasetImportJobsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListDatasetsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListExplainabilitiesRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListExplainabilityExportsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListForecastExportJobsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListForecastsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListMonitorEvaluationsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListMonitorsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListPredictorBacktestExportJobsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListPredictorsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListWhatIfAnalysesRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListWhatIfForecastExportsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListWhatIfForecastsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ResumeResourceRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.StopResourceRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.UpdateDatasetGroupRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ForecastAsyncClient}.
 *
 * @see ForecastAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultForecastAsyncClient implements ForecastAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultForecastAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultForecastAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an Amazon Forecast predictor.
     * </p>
     * <p>
     * Amazon Forecast creates predictors with AutoPredictor, which involves applying the optimal combination of
     * algorithms to each time series in your datasets. You can use <a>CreateAutoPredictor</a> to create new predictors
     * or upgrade/retrain existing predictors.
     * </p>
     * <p>
     * <b>Creating new predictors</b>
     * </p>
     * <p>
     * The following parameters are required when creating a new predictor:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PredictorName</code> - A unique name for the predictor.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DatasetGroupArn</code> - The ARN of the dataset group used to train the predictor.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ForecastFrequency</code> - The granularity of your forecasts (hourly, daily, weekly, etc).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ForecastHorizon</code> - The number of time-steps that the model predicts. The forecast horizon is also
     * called the prediction length.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When creating a new predictor, do not specify a value for <code>ReferencePredictorArn</code>.
     * </p>
     * <p>
     * <b>Upgrading and retraining predictors</b>
     * </p>
     * <p>
     * The following parameters are required when retraining or upgrading a predictor:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PredictorName</code> - A unique name for the predictor.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReferencePredictorArn</code> - The ARN of the predictor to retrain or upgrade.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When upgrading or retraining a predictor, only specify values for the <code>ReferencePredictorArn</code> and
     * <code>PredictorName</code>.
     * </p>
     *
     * @param createAutoPredictorRequest
     * @return A Java Future containing the result of the CreateAutoPredictor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateAutoPredictor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateAutoPredictor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAutoPredictorResponse> createAutoPredictor(
            CreateAutoPredictorRequest createAutoPredictorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAutoPredictorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAutoPredictorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAutoPredictor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAutoPredictorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAutoPredictorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAutoPredictorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAutoPredictorRequest, CreateAutoPredictorResponse>()
                            .withOperationName("CreateAutoPredictor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAutoPredictorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAutoPredictorRequest));
            CompletableFuture<CreateAutoPredictorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon Forecast dataset. The information about the dataset that you provide helps Forecast understand
     * how to consume the data for model training. This includes the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i> <code>DataFrequency</code> </i> - How frequently your historical time-series data is collected.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i> <code>Domain</code> </i> and <i> <code>DatasetType</code> </i> - Each dataset has an associated dataset
     * domain and a type within the domain. Amazon Forecast provides a list of predefined domains and types within each
     * domain. For each unique dataset domain and type within the domain, Amazon Forecast requires your data to include
     * a minimum set of predefined fields.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i> <code>Schema</code> </i> - A schema specifies the fields in the dataset, including the field name and data
     * type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * After creating a dataset, you import your training data into it and add the dataset to a dataset group. You use
     * the dataset group to create a predictor. For more information, see <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/howitworks-datasets-groups.html">Importing datasets</a>.
     * </p>
     * <p>
     * To get a list of all your datasets, use the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_ListDatasets.html">ListDatasets</a> operation.
     * </p>
     * <p>
     * For example Forecast datasets, see the <a href="https://github.com/aws-samples/amazon-forecast-samples">Amazon
     * Forecast Sample GitHub repository</a>.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of a dataset must be <code>ACTIVE</code> before you can import training data. Use the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDataset.html">DescribeDataset</a> operation to
     * get the status.
     * </p>
     * </note>
     *
     * @param createDatasetRequest
     * @return A Java Future containing the result of the CreateDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatasetResponse> createDataset(CreateDatasetRequest createDatasetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDatasetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetRequest, CreateDatasetResponse>()
                            .withOperationName("CreateDataset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDatasetRequest));
            CompletableFuture<CreateDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a dataset group, which holds a collection of related datasets. You can add datasets to the dataset group
     * when you create the dataset group, or later by using the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_UpdateDatasetGroup.html">UpdateDatasetGroup</a>
     * operation.
     * </p>
     * <p>
     * After creating a dataset group and adding datasets, you use the dataset group when you create a predictor. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/howitworks-datasets-groups.html">Dataset groups</a>.
     * </p>
     * <p>
     * To get a list of all your datasets groups, use the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_ListDatasetGroups.html">ListDatasetGroups</a> operation.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of a dataset group must be <code>ACTIVE</code> before you can use the dataset group to
     * create a predictor. To get the status, use the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDatasetGroup.html">DescribeDatasetGroup</a>
     * operation.
     * </p>
     * </note>
     *
     * @param createDatasetGroupRequest
     * @return A Java Future containing the result of the CreateDatasetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateDatasetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatasetGroupResponse> createDatasetGroup(CreateDatasetGroupRequest createDatasetGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDatasetGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatasetGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDatasetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatasetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetGroupRequest, CreateDatasetGroupResponse>()
                            .withOperationName("CreateDatasetGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDatasetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDatasetGroupRequest));
            CompletableFuture<CreateDatasetGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports your training data to an Amazon Forecast dataset. You provide the location of your training data in an
     * Amazon Simple Storage Service (Amazon S3) bucket and the Amazon Resource Name (ARN) of the dataset that you want
     * to import the data to.
     * </p>
     * <p>
     * You must specify a <a href="https://docs.aws.amazon.com/forecast/latest/dg/API_DataSource.html">DataSource</a>
     * object that includes an Identity and Access Management (IAM) role that Amazon Forecast can assume to access the
     * data, as Amazon Forecast makes a copy of your data and processes it in an internal Amazon Web Services system.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/aws-forecast-iam-roles.html">Set up permissions</a>.
     * </p>
     * <p>
     * The training data must be in CSV or Parquet format. The delimiter must be a comma (,).
     * </p>
     * <p>
     * You can specify the path to a specific file, the S3 bucket, or to a folder in the S3 bucket. For the latter two
     * cases, Amazon Forecast imports all files up to the limit of 10,000 files.
     * </p>
     * <p>
     * Because dataset imports are not aggregated, your most recent dataset import is the one that is used when training
     * a predictor or generating a forecast. Make sure that your most recent dataset import contains all of the data you
     * want to model off of, and not just the new data collected since the previous import.
     * </p>
     * <p>
     * To get a list of all your dataset import jobs, filtered by specified criteria, use the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_ListDatasetImportJobs.html">ListDatasetImportJobs</a>
     * operation.
     * </p>
     *
     * @param createDatasetImportJobRequest
     * @return A Java Future containing the result of the CreateDatasetImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateDatasetImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateDatasetImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatasetImportJobResponse> createDatasetImportJob(
            CreateDatasetImportJobRequest createDatasetImportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDatasetImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatasetImportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatasetImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDatasetImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatasetImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetImportJobRequest, CreateDatasetImportJobResponse>()
                            .withOperationName("CreateDatasetImportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDatasetImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDatasetImportJobRequest));
            CompletableFuture<CreateDatasetImportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * Explainability is only available for Forecasts and Predictors generated from an AutoPredictor
     * (<a>CreateAutoPredictor</a>)
     * </p>
     * </note>
     * <p>
     * Creates an Amazon Forecast Explainability.
     * </p>
     * <p>
     * Explainability helps you better understand how the attributes in your datasets impact forecast. Amazon Forecast
     * uses a metric called Impact scores to quantify the relative impact of each attribute and determine whether they
     * increase or decrease forecast values.
     * </p>
     * <p>
     * To enable Forecast Explainability, your predictor must include at least one of the following: related time
     * series, item metadata, or additional datasets like Holidays and the Weather Index.
     * </p>
     * <p>
     * CreateExplainability accepts either a Predictor ARN or Forecast ARN. To receive aggregated Impact scores for all
     * time series and time points in your datasets, provide a Predictor ARN. To receive Impact scores for specific time
     * series and time points, provide a Forecast ARN.
     * </p>
     * <p>
     * <b>CreateExplainability with a Predictor ARN</b>
     * </p>
     * <note>
     * <p>
     * You can only have one Explainability resource per predictor. If you already enabled <code>ExplainPredictor</code>
     * in <a>CreateAutoPredictor</a>, that predictor already has an Explainability resource.
     * </p>
     * </note>
     * <p>
     * The following parameters are required when providing a Predictor ARN:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ExplainabilityName</code> - A unique name for the Explainability.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourceArn</code> - The Arn of the predictor.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TimePointGranularity</code> - Must be set to “ALL”.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TimeSeriesGranularity</code> - Must be set to “ALL”.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Do not specify a value for the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DataSource</code> - Only valid when TimeSeriesGranularity is “SPECIFIC”.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Schema</code> - Only valid when TimeSeriesGranularity is “SPECIFIC”.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StartDateTime</code> - Only valid when TimePointGranularity is “SPECIFIC”.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EndDateTime</code> - Only valid when TimePointGranularity is “SPECIFIC”.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>CreateExplainability with a Forecast ARN</b>
     * </p>
     * <note>
     * <p>
     * You can specify a maximum of 50 time series and 500 time points.
     * </p>
     * </note>
     * <p>
     * The following parameters are required when providing a Predictor ARN:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ExplainabilityName</code> - A unique name for the Explainability.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourceArn</code> - The Arn of the forecast.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TimePointGranularity</code> - Either “ALL” or “SPECIFIC”.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TimeSeriesGranularity</code> - Either “ALL” or “SPECIFIC”.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you set TimeSeriesGranularity to “SPECIFIC”, you must also provide the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DataSource</code> - The S3 location of the CSV file specifying your time series.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Schema</code> - The Schema defines the attributes and attribute types listed in the Data Source.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you set TimePointGranularity to “SPECIFIC”, you must also provide the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>StartDateTime</code> - The first timestamp in the range of time points.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EndDateTime</code> - The last timestamp in the range of time points.
     * </p>
     * </li>
     * </ul>
     *
     * @param createExplainabilityRequest
     * @return A Java Future containing the result of the CreateExplainability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateExplainability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateExplainability" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateExplainabilityResponse> createExplainability(
            CreateExplainabilityRequest createExplainabilityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createExplainabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExplainabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExplainability");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateExplainabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateExplainabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateExplainabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateExplainabilityRequest, CreateExplainabilityResponse>()
                            .withOperationName("CreateExplainability").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateExplainabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createExplainabilityRequest));
            CompletableFuture<CreateExplainabilityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports an Explainability resource created by the <a>CreateExplainability</a> operation. Exported files are
     * exported to an Amazon Simple Storage Service (Amazon S3) bucket.
     * </p>
     * <p>
     * You must specify a <a>DataDestination</a> object that includes an Amazon S3 bucket and an Identity and Access
     * Management (IAM) role that Amazon Forecast can assume to access the Amazon S3 bucket. For more information, see
     * <a>aws-forecast-iam-roles</a>.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the export job must be <code>ACTIVE</code> before you can access the export in your
     * Amazon S3 bucket. To get the status, use the <a>DescribeExplainabilityExport</a> operation.
     * </p>
     * </note>
     *
     * @param createExplainabilityExportRequest
     * @return A Java Future containing the result of the CreateExplainabilityExport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateExplainabilityExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateExplainabilityExport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateExplainabilityExportResponse> createExplainabilityExport(
            CreateExplainabilityExportRequest createExplainabilityExportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createExplainabilityExportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExplainabilityExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExplainabilityExport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateExplainabilityExportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateExplainabilityExportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateExplainabilityExportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateExplainabilityExportRequest, CreateExplainabilityExportResponse>()
                            .withOperationName("CreateExplainabilityExport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateExplainabilityExportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createExplainabilityExportRequest));
            CompletableFuture<CreateExplainabilityExportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a forecast for each item in the <code>TARGET_TIME_SERIES</code> dataset that was used to train the
     * predictor. This is known as inference. To retrieve the forecast for a single item at low latency, use the
     * operation. To export the complete forecast into your Amazon Simple Storage Service (Amazon S3) bucket, use the
     * <a>CreateForecastExportJob</a> operation.
     * </p>
     * <p>
     * The range of the forecast is determined by the <code>ForecastHorizon</code> value, which you specify in the
     * <a>CreatePredictor</a> request. When you query a forecast, you can request a specific date range within the
     * forecast.
     * </p>
     * <p>
     * To get a list of all your forecasts, use the <a>ListForecasts</a> operation.
     * </p>
     * <note>
     * <p>
     * The forecasts generated by Amazon Forecast are in the same time zone as the dataset that was used to create the
     * predictor.
     * </p>
     * </note>
     * <p>
     * For more information, see <a>howitworks-forecast</a>.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the forecast must be <code>ACTIVE</code> before you can query or export the forecast.
     * Use the <a>DescribeForecast</a> operation to get the status.
     * </p>
     * </note>
     * <p>
     * By default, a forecast includes predictions for every item (<code>item_id</code>) in the dataset group that was
     * used to train the predictor. However, you can use the <code>TimeSeriesSelector</code> object to generate a
     * forecast on a subset of time series. Forecast creation is skipped for any time series that you specify that are
     * not in the input dataset. The forecast export file will not contain these time series or their forecasted values.
     * </p>
     *
     * @param createForecastRequest
     * @return A Java Future containing the result of the CreateForecast operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateForecast
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateForecastResponse> createForecast(CreateForecastRequest createForecastRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createForecastRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createForecastRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateForecast");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateForecastResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateForecastRequest, CreateForecastResponse>()
                            .withOperationName("CreateForecast").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createForecastRequest));
            CompletableFuture<CreateForecastResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports a forecast created by the <a>CreateForecast</a> operation to your Amazon Simple Storage Service (Amazon
     * S3) bucket. The forecast file name will match the following conventions:
     * </p>
     * <p>
     * &lt;ForecastExportJobName&gt;_&lt;ExportTimestamp&gt;_&lt;PartNumber&gt;
     * </p>
     * <p>
     * where the &lt;ExportTimestamp&gt; component is in Java SimpleDateFormat (yyyy-MM-ddTHH-mm-ssZ).
     * </p>
     * <p>
     * You must specify a <a>DataDestination</a> object that includes an Identity and Access Management (IAM) role that
     * Amazon Forecast can assume to access the Amazon S3 bucket. For more information, see
     * <a>aws-forecast-iam-roles</a>.
     * </p>
     * <p>
     * For more information, see <a>howitworks-forecast</a>.
     * </p>
     * <p>
     * To get a list of all your forecast export jobs, use the <a>ListForecastExportJobs</a> operation.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the forecast export job must be <code>ACTIVE</code> before you can access the forecast
     * in your Amazon S3 bucket. To get the status, use the <a>DescribeForecastExportJob</a> operation.
     * </p>
     * </note>
     *
     * @param createForecastExportJobRequest
     * @return A Java Future containing the result of the CreateForecastExportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateForecastExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateForecastExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateForecastExportJobResponse> createForecastExportJob(
            CreateForecastExportJobRequest createForecastExportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createForecastExportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createForecastExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateForecastExportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateForecastExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateForecastExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateForecastExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateForecastExportJobRequest, CreateForecastExportJobResponse>()
                            .withOperationName("CreateForecastExportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateForecastExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createForecastExportJobRequest));
            CompletableFuture<CreateForecastExportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a predictor monitor resource for an existing auto predictor. Predictor monitoring allows you to see how
     * your predictor's performance changes over time. For more information, see <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/predictor-monitoring.html">Predictor Monitoring</a>.
     * </p>
     *
     * @param createMonitorRequest
     * @return A Java Future containing the result of the CreateMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMonitorResponse> createMonitor(CreateMonitorRequest createMonitorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMonitor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMonitorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMonitorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMonitorRequest, CreateMonitorResponse>()
                            .withOperationName("CreateMonitor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateMonitorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMonitorRequest));
            CompletableFuture<CreateMonitorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This operation creates a legacy predictor that does not include all the predictor functionalities provided by
     * Amazon Forecast. To create a predictor that is compatible with all aspects of Forecast, use
     * <a>CreateAutoPredictor</a>.
     * </p>
     * </note>
     * <p>
     * Creates an Amazon Forecast predictor.
     * </p>
     * <p>
     * In the request, provide a dataset group and either specify an algorithm or let Amazon Forecast choose an
     * algorithm for you using AutoML. If you specify an algorithm, you also can override algorithm-specific
     * hyperparameters.
     * </p>
     * <p>
     * Amazon Forecast uses the algorithm to train a predictor using the latest version of the datasets in the specified
     * dataset group. You can then generate a forecast using the <a>CreateForecast</a> operation.
     * </p>
     * <p>
     * To see the evaluation metrics, use the <a>GetAccuracyMetrics</a> operation.
     * </p>
     * <p>
     * You can specify a featurization configuration to fill and aggregate the data fields in the
     * <code>TARGET_TIME_SERIES</code> dataset to improve model training. For more information, see
     * <a>FeaturizationConfig</a>.
     * </p>
     * <p>
     * For RELATED_TIME_SERIES datasets, <code>CreatePredictor</code> verifies that the <code>DataFrequency</code>
     * specified when the dataset was created matches the <code>ForecastFrequency</code>. TARGET_TIME_SERIES datasets
     * don't have this restriction. Amazon Forecast also verifies the delimiter and timestamp format. For more
     * information, see <a>howitworks-datasets-groups</a>.
     * </p>
     * <p>
     * By default, predictors are trained and evaluated at the 0.1 (P10), 0.5 (P50), and 0.9 (P90) quantiles. You can
     * choose custom forecast types to train and evaluate your predictor by setting the <code>ForecastTypes</code>.
     * </p>
     * <p>
     * <b>AutoML</b>
     * </p>
     * <p>
     * If you want Amazon Forecast to evaluate each algorithm and choose the one that minimizes the
     * <code>objective function</code>, set <code>PerformAutoML</code> to <code>true</code>. The
     * <code>objective function</code> is defined as the mean of the weighted losses over the forecast types. By
     * default, these are the p10, p50, and p90 quantile losses. For more information, see <a>EvaluationResult</a>.
     * </p>
     * <p>
     * When AutoML is enabled, the following properties are disallowed:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AlgorithmArn</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HPOConfig</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PerformHPO</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TrainingParameters</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get a list of all of your predictors, use the <a>ListPredictors</a> operation.
     * </p>
     * <note>
     * <p>
     * Before you can use the predictor to create a forecast, the <code>Status</code> of the predictor must be
     * <code>ACTIVE</code>, signifying that training has completed. To get the status, use the <a>DescribePredictor</a>
     * operation.
     * </p>
     * </note>
     *
     * @param createPredictorRequest
     * @return A Java Future containing the result of the CreatePredictor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreatePredictor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreatePredictor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePredictorResponse> createPredictor(CreatePredictorRequest createPredictorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPredictorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPredictorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePredictor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePredictorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePredictorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePredictorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePredictorRequest, CreatePredictorResponse>()
                            .withOperationName("CreatePredictor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePredictorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPredictorRequest));
            CompletableFuture<CreatePredictorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports backtest forecasts and accuracy metrics generated by the <a>CreateAutoPredictor</a> or
     * <a>CreatePredictor</a> operations. Two folders containing CSV or Parquet files are exported to your specified S3
     * bucket.
     * </p>
     * <p>
     * The export file names will match the following conventions:
     * </p>
     * <p>
     * <code>&lt;ExportJobName&gt;_&lt;ExportTimestamp&gt;_&lt;PartNumber&gt;.csv</code>
     * </p>
     * <p>
     * The &lt;ExportTimestamp&gt; component is in Java SimpleDate format (yyyy-MM-ddTHH-mm-ssZ).
     * </p>
     * <p>
     * You must specify a <a>DataDestination</a> object that includes an Amazon S3 bucket and an Identity and Access
     * Management (IAM) role that Amazon Forecast can assume to access the Amazon S3 bucket. For more information, see
     * <a>aws-forecast-iam-roles</a>.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the export job must be <code>ACTIVE</code> before you can access the export in your
     * Amazon S3 bucket. To get the status, use the <a>DescribePredictorBacktestExportJob</a> operation.
     * </p>
     * </note>
     *
     * @param createPredictorBacktestExportJobRequest
     * @return A Java Future containing the result of the CreatePredictorBacktestExportJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreatePredictorBacktestExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreatePredictorBacktestExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePredictorBacktestExportJobResponse> createPredictorBacktestExportJob(
            CreatePredictorBacktestExportJobRequest createPredictorBacktestExportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPredictorBacktestExportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createPredictorBacktestExportJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePredictorBacktestExportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePredictorBacktestExportJobResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreatePredictorBacktestExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePredictorBacktestExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePredictorBacktestExportJobRequest, CreatePredictorBacktestExportJobResponse>()
                            .withOperationName("CreatePredictorBacktestExportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePredictorBacktestExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPredictorBacktestExportJobRequest));
            CompletableFuture<CreatePredictorBacktestExportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * What-if analysis is a scenario modeling technique where you make a hypothetical change to a time series and
     * compare the forecasts generated by these changes against the baseline, unchanged time series. It is important to
     * remember that the purpose of a what-if analysis is to understand how a forecast can change given different
     * modifications to the baseline time series.
     * </p>
     * <p>
     * For example, imagine you are a clothing retailer who is considering an end of season sale to clear space for new
     * styles. After creating a baseline forecast, you can use a what-if analysis to investigate how different sales
     * tactics might affect your goals.
     * </p>
     * <p>
     * You could create a scenario where everything is given a 25% markdown, and another where everything is given a
     * fixed dollar markdown. You could create a scenario where the sale lasts for one week and another where the sale
     * lasts for one month. With a what-if analysis, you can compare many different scenarios against each other.
     * </p>
     * <p>
     * Note that a what-if analysis is meant to display what the forecasting model has learned and how it will behave in
     * the scenarios that you are evaluating. Do not blindly use the results of the what-if analysis to make business
     * decisions. For instance, forecasts might not be accurate for novel scenarios where there is no reference
     * available to determine whether a forecast is good.
     * </p>
     * <p>
     * The <a>TimeSeriesSelector</a> object defines the items that you want in the what-if analysis.
     * </p>
     *
     * @param createWhatIfAnalysisRequest
     * @return A Java Future containing the result of the CreateWhatIfAnalysis operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateWhatIfAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateWhatIfAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWhatIfAnalysisResponse> createWhatIfAnalysis(
            CreateWhatIfAnalysisRequest createWhatIfAnalysisRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWhatIfAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWhatIfAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWhatIfAnalysis");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWhatIfAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWhatIfAnalysisResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWhatIfAnalysisResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWhatIfAnalysisRequest, CreateWhatIfAnalysisResponse>()
                            .withOperationName("CreateWhatIfAnalysis").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWhatIfAnalysisRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWhatIfAnalysisRequest));
            CompletableFuture<CreateWhatIfAnalysisResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A what-if forecast is a forecast that is created from a modified version of the baseline forecast. Each what-if
     * forecast incorporates either a replacement dataset or a set of transformations to the original dataset.
     * </p>
     *
     * @param createWhatIfForecastRequest
     * @return A Java Future containing the result of the CreateWhatIfForecast operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateWhatIfForecast
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateWhatIfForecast" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWhatIfForecastResponse> createWhatIfForecast(
            CreateWhatIfForecastRequest createWhatIfForecastRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWhatIfForecastRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWhatIfForecastRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWhatIfForecast");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWhatIfForecastResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWhatIfForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWhatIfForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWhatIfForecastRequest, CreateWhatIfForecastResponse>()
                            .withOperationName("CreateWhatIfForecast").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWhatIfForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWhatIfForecastRequest));
            CompletableFuture<CreateWhatIfForecastResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports a forecast created by the <a>CreateWhatIfForecast</a> operation to your Amazon Simple Storage Service
     * (Amazon S3) bucket. The forecast file name will match the following conventions:
     * </p>
     * <p>
     * <code>≈&lt;ForecastExportJobName&gt;_&lt;ExportTimestamp&gt;_&lt;PartNumber&gt;</code>
     * </p>
     * <p>
     * The &lt;ExportTimestamp&gt; component is in Java SimpleDateFormat (yyyy-MM-ddTHH-mm-ssZ).
     * </p>
     * <p>
     * You must specify a <a>DataDestination</a> object that includes an Identity and Access Management (IAM) role that
     * Amazon Forecast can assume to access the Amazon S3 bucket. For more information, see
     * <a>aws-forecast-iam-roles</a>.
     * </p>
     * <p>
     * For more information, see <a>howitworks-forecast</a>.
     * </p>
     * <p>
     * To get a list of all your what-if forecast export jobs, use the <a>ListWhatIfForecastExports</a> operation.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the forecast export job must be <code>ACTIVE</code> before you can access the forecast
     * in your Amazon S3 bucket. To get the status, use the <a>DescribeWhatIfForecastExport</a> operation.
     * </p>
     * </note>
     *
     * @param createWhatIfForecastExportRequest
     * @return A Java Future containing the result of the CreateWhatIfForecastExport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with this name. Try again with a different
     *         name.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateWhatIfForecastExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateWhatIfForecastExport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWhatIfForecastExportResponse> createWhatIfForecastExport(
            CreateWhatIfForecastExportRequest createWhatIfForecastExportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWhatIfForecastExportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWhatIfForecastExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWhatIfForecastExport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWhatIfForecastExportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWhatIfForecastExportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWhatIfForecastExportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWhatIfForecastExportRequest, CreateWhatIfForecastExportResponse>()
                            .withOperationName("CreateWhatIfForecastExport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWhatIfForecastExportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWhatIfForecastExportRequest));
            CompletableFuture<CreateWhatIfForecastExportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon Forecast dataset that was created using the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDataset.html">CreateDataset</a> operation. You can
     * only delete datasets that have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the status
     * use the <a href="https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDataset.html">DescribeDataset</a>
     * operation.
     * </p>
     * <note>
     * <p>
     * Forecast does not automatically update any dataset groups that contain the deleted dataset. In order to update
     * the dataset group, use the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_UpdateDatasetGroup.html">UpdateDatasetGroup</a>
     * operation, omitting the deleted dataset's ARN.
     * </p>
     * </note>
     *
     * @param deleteDatasetRequest
     * @return A Java Future containing the result of the DeleteDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetResponse> deleteDataset(DeleteDatasetRequest deleteDatasetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDatasetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetRequest, DeleteDatasetResponse>()
                            .withOperationName("DeleteDataset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDatasetRequest));
            CompletableFuture<DeleteDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a dataset group created using the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDatasetGroup.html">CreateDatasetGroup</a>
     * operation. You can only delete dataset groups that have a status of <code>ACTIVE</code>,
     * <code>CREATE_FAILED</code>, or <code>UPDATE_FAILED</code>. To get the status, use the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDatasetGroup.html">DescribeDatasetGroup</a>
     * operation.
     * </p>
     * <p>
     * This operation deletes only the dataset group, not the datasets in the group.
     * </p>
     *
     * @param deleteDatasetGroupRequest
     * @return A Java Future containing the result of the DeleteDatasetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteDatasetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetGroupResponse> deleteDatasetGroup(DeleteDatasetGroupRequest deleteDatasetGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDatasetGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatasetGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDatasetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetGroupRequest, DeleteDatasetGroupResponse>()
                            .withOperationName("DeleteDatasetGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDatasetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDatasetGroupRequest));
            CompletableFuture<DeleteDatasetGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a dataset import job created using the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDatasetImportJob.html">CreateDatasetImportJob</a>
     * operation. You can delete only dataset import jobs that have a status of <code>ACTIVE</code> or
     * <code>CREATE_FAILED</code>. To get the status, use the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDatasetImportJob.html"
     * >DescribeDatasetImportJob</a> operation.
     * </p>
     *
     * @param deleteDatasetImportJobRequest
     * @return A Java Future containing the result of the DeleteDatasetImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteDatasetImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteDatasetImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetImportJobResponse> deleteDatasetImportJob(
            DeleteDatasetImportJobRequest deleteDatasetImportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDatasetImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatasetImportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDatasetImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetImportJobRequest, DeleteDatasetImportJobResponse>()
                            .withOperationName("DeleteDatasetImportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDatasetImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDatasetImportJobRequest));
            CompletableFuture<DeleteDatasetImportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Explainability resource.
     * </p>
     * <p>
     * You can delete only predictor that have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the
     * status, use the <a>DescribeExplainability</a> operation.
     * </p>
     *
     * @param deleteExplainabilityRequest
     * @return A Java Future containing the result of the DeleteExplainability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteExplainability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteExplainability" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteExplainabilityResponse> deleteExplainability(
            DeleteExplainabilityRequest deleteExplainabilityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteExplainabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExplainabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExplainability");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteExplainabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteExplainabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteExplainabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteExplainabilityRequest, DeleteExplainabilityResponse>()
                            .withOperationName("DeleteExplainability").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteExplainabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteExplainabilityRequest));
            CompletableFuture<DeleteExplainabilityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Explainability export.
     * </p>
     *
     * @param deleteExplainabilityExportRequest
     * @return A Java Future containing the result of the DeleteExplainabilityExport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteExplainabilityExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteExplainabilityExport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteExplainabilityExportResponse> deleteExplainabilityExport(
            DeleteExplainabilityExportRequest deleteExplainabilityExportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteExplainabilityExportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExplainabilityExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExplainabilityExport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteExplainabilityExportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteExplainabilityExportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteExplainabilityExportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteExplainabilityExportRequest, DeleteExplainabilityExportResponse>()
                            .withOperationName("DeleteExplainabilityExport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteExplainabilityExportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteExplainabilityExportRequest));
            CompletableFuture<DeleteExplainabilityExportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a forecast created using the <a>CreateForecast</a> operation. You can delete only forecasts that have a
     * status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the status, use the <a>DescribeForecast</a>
     * operation.
     * </p>
     * <p>
     * You can't delete a forecast while it is being exported. After a forecast is deleted, you can no longer query the
     * forecast.
     * </p>
     *
     * @param deleteForecastRequest
     * @return A Java Future containing the result of the DeleteForecast operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteForecast
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteForecastResponse> deleteForecast(DeleteForecastRequest deleteForecastRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteForecastRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteForecastRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteForecast");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteForecastResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteForecastRequest, DeleteForecastResponse>()
                            .withOperationName("DeleteForecast").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteForecastRequest));
            CompletableFuture<DeleteForecastResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a forecast export job created using the <a>CreateForecastExportJob</a> operation. You can delete only
     * export jobs that have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the status, use the
     * <a>DescribeForecastExportJob</a> operation.
     * </p>
     *
     * @param deleteForecastExportJobRequest
     * @return A Java Future containing the result of the DeleteForecastExportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteForecastExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteForecastExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteForecastExportJobResponse> deleteForecastExportJob(
            DeleteForecastExportJobRequest deleteForecastExportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteForecastExportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteForecastExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteForecastExportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteForecastExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteForecastExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteForecastExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteForecastExportJobRequest, DeleteForecastExportJobResponse>()
                            .withOperationName("DeleteForecastExportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteForecastExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteForecastExportJobRequest));
            CompletableFuture<DeleteForecastExportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a monitor resource. You can only delete a monitor resource with a status of <code>ACTIVE</code>,
     * <code>ACTIVE_STOPPED</code>, <code>CREATE_FAILED</code>, or <code>CREATE_STOPPED</code>.
     * </p>
     *
     * @param deleteMonitorRequest
     * @return A Java Future containing the result of the DeleteMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMonitorResponse> deleteMonitor(DeleteMonitorRequest deleteMonitorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMonitor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMonitorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMonitorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMonitorRequest, DeleteMonitorResponse>()
                            .withOperationName("DeleteMonitor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMonitorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMonitorRequest));
            CompletableFuture<DeleteMonitorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a predictor created using the <a>DescribePredictor</a> or <a>CreatePredictor</a> operations. You can
     * delete only predictor that have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the status,
     * use the <a>DescribePredictor</a> operation.
     * </p>
     *
     * @param deletePredictorRequest
     * @return A Java Future containing the result of the DeletePredictor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeletePredictor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeletePredictor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePredictorResponse> deletePredictor(DeletePredictorRequest deletePredictorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePredictorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePredictorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePredictor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePredictorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePredictorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePredictorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePredictorRequest, DeletePredictorResponse>()
                            .withOperationName("DeletePredictor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePredictorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePredictorRequest));
            CompletableFuture<DeletePredictorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a predictor backtest export job.
     * </p>
     *
     * @param deletePredictorBacktestExportJobRequest
     * @return A Java Future containing the result of the DeletePredictorBacktestExportJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeletePredictorBacktestExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeletePredictorBacktestExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePredictorBacktestExportJobResponse> deletePredictorBacktestExportJob(
            DeletePredictorBacktestExportJobRequest deletePredictorBacktestExportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePredictorBacktestExportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deletePredictorBacktestExportJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePredictorBacktestExportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePredictorBacktestExportJobResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeletePredictorBacktestExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePredictorBacktestExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePredictorBacktestExportJobRequest, DeletePredictorBacktestExportJobResponse>()
                            .withOperationName("DeletePredictorBacktestExportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePredictorBacktestExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePredictorBacktestExportJobRequest));
            CompletableFuture<DeletePredictorBacktestExportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an entire resource tree. This operation will delete the parent resource and its child resources.
     * </p>
     * <p>
     * Child resources are resources that were created from another resource. For example, when a forecast is generated
     * from a predictor, the forecast is the child resource and the predictor is the parent resource.
     * </p>
     * <p>
     * Amazon Forecast resources possess the following parent-child resource hierarchies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Dataset</b>: dataset import jobs
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Dataset Group</b>: predictors, predictor backtest export jobs, forecasts, forecast export jobs
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Predictor</b>: predictor backtest export jobs, forecasts, forecast export jobs
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Forecast</b>: forecast export jobs
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * <code>DeleteResourceTree</code> will only delete Amazon Forecast resources, and will not delete datasets or
     * exported files stored in Amazon S3.
     * </p>
     * </note>
     *
     * @param deleteResourceTreeRequest
     * @return A Java Future containing the result of the DeleteResourceTree operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteResourceTree
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteResourceTree" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourceTreeResponse> deleteResourceTree(DeleteResourceTreeRequest deleteResourceTreeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourceTreeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceTreeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceTree");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourceTreeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourceTreeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourceTreeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourceTreeRequest, DeleteResourceTreeResponse>()
                            .withOperationName("DeleteResourceTree").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteResourceTreeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteResourceTreeRequest));
            CompletableFuture<DeleteResourceTreeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a what-if analysis created using the <a>CreateWhatIfAnalysis</a> operation. You can delete only what-if
     * analyses that have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the status, use the
     * <a>DescribeWhatIfAnalysis</a> operation.
     * </p>
     * <p>
     * You can't delete a what-if analysis while any of its forecasts are being exported.
     * </p>
     *
     * @param deleteWhatIfAnalysisRequest
     * @return A Java Future containing the result of the DeleteWhatIfAnalysis operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteWhatIfAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteWhatIfAnalysis" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWhatIfAnalysisResponse> deleteWhatIfAnalysis(
            DeleteWhatIfAnalysisRequest deleteWhatIfAnalysisRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWhatIfAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWhatIfAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWhatIfAnalysis");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWhatIfAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWhatIfAnalysisResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWhatIfAnalysisResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWhatIfAnalysisRequest, DeleteWhatIfAnalysisResponse>()
                            .withOperationName("DeleteWhatIfAnalysis").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWhatIfAnalysisRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWhatIfAnalysisRequest));
            CompletableFuture<DeleteWhatIfAnalysisResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a what-if forecast created using the <a>CreateWhatIfForecast</a> operation. You can delete only what-if
     * forecasts that have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the status, use the
     * <a>DescribeWhatIfForecast</a> operation.
     * </p>
     * <p>
     * You can't delete a what-if forecast while it is being exported. After a what-if forecast is deleted, you can no
     * longer query the what-if analysis.
     * </p>
     *
     * @param deleteWhatIfForecastRequest
     * @return A Java Future containing the result of the DeleteWhatIfForecast operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteWhatIfForecast
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteWhatIfForecast" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWhatIfForecastResponse> deleteWhatIfForecast(
            DeleteWhatIfForecastRequest deleteWhatIfForecastRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWhatIfForecastRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWhatIfForecastRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWhatIfForecast");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWhatIfForecastResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWhatIfForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWhatIfForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWhatIfForecastRequest, DeleteWhatIfForecastResponse>()
                            .withOperationName("DeleteWhatIfForecast").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWhatIfForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWhatIfForecastRequest));
            CompletableFuture<DeleteWhatIfForecastResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a what-if forecast export created using the <a>CreateWhatIfForecastExport</a> operation. You can delete
     * only what-if forecast exports that have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the
     * status, use the <a>DescribeWhatIfForecastExport</a> operation.
     * </p>
     *
     * @param deleteWhatIfForecastExportRequest
     * @return A Java Future containing the result of the DeleteWhatIfForecastExport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteWhatIfForecastExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteWhatIfForecastExport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWhatIfForecastExportResponse> deleteWhatIfForecastExport(
            DeleteWhatIfForecastExportRequest deleteWhatIfForecastExportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWhatIfForecastExportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWhatIfForecastExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWhatIfForecastExport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWhatIfForecastExportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWhatIfForecastExportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWhatIfForecastExportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWhatIfForecastExportRequest, DeleteWhatIfForecastExportResponse>()
                            .withOperationName("DeleteWhatIfForecastExport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWhatIfForecastExportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWhatIfForecastExportRequest));
            CompletableFuture<DeleteWhatIfForecastExportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a predictor created using the CreateAutoPredictor operation.
     * </p>
     *
     * @param describeAutoPredictorRequest
     * @return A Java Future containing the result of the DescribeAutoPredictor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeAutoPredictor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeAutoPredictor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAutoPredictorResponse> describeAutoPredictor(
            DescribeAutoPredictorRequest describeAutoPredictorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAutoPredictorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAutoPredictorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAutoPredictor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAutoPredictorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAutoPredictorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAutoPredictorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAutoPredictorRequest, DescribeAutoPredictorResponse>()
                            .withOperationName("DescribeAutoPredictor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAutoPredictorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAutoPredictorRequest));
            CompletableFuture<DescribeAutoPredictorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an Amazon Forecast dataset created using the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDataset.html">CreateDataset</a> operation.
     * </p>
     * <p>
     * In addition to listing the parameters specified in the <code>CreateDataset</code> request, this operation
     * includes the following dataset properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeDatasetRequest
     * @return A Java Future containing the result of the DescribeDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatasetResponse> describeDataset(DescribeDatasetRequest describeDatasetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDatasetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetRequest, DescribeDatasetResponse>()
                            .withOperationName("DescribeDataset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDatasetRequest));
            CompletableFuture<DescribeDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a dataset group created using the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDatasetGroup.html">CreateDatasetGroup</a>
     * operation.
     * </p>
     * <p>
     * In addition to listing the parameters provided in the <code>CreateDatasetGroup</code> request, this operation
     * includes the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DatasetArns</code> - The datasets belonging to the group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeDatasetGroupRequest
     * @return A Java Future containing the result of the DescribeDatasetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeDatasetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatasetGroupResponse> describeDatasetGroup(
            DescribeDatasetGroupRequest describeDatasetGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDatasetGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDatasetGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatasetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatasetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetGroupRequest, DescribeDatasetGroupResponse>()
                            .withOperationName("DescribeDatasetGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDatasetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDatasetGroupRequest));
            CompletableFuture<DescribeDatasetGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a dataset import job created using the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDatasetImportJob.html">CreateDatasetImportJob</a>
     * operation.
     * </p>
     * <p>
     * In addition to listing the parameters provided in the <code>CreateDatasetImportJob</code> request, this operation
     * includes the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DataSize</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FieldStatistics</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeDatasetImportJobRequest
     * @return A Java Future containing the result of the DescribeDatasetImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeDatasetImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeDatasetImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatasetImportJobResponse> describeDatasetImportJob(
            DescribeDatasetImportJobRequest describeDatasetImportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDatasetImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDatasetImportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatasetImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatasetImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatasetImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetImportJobRequest, DescribeDatasetImportJobResponse>()
                            .withOperationName("DescribeDatasetImportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDatasetImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDatasetImportJobRequest));
            CompletableFuture<DescribeDatasetImportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an Explainability resource created using the <a>CreateExplainability</a> operation.
     * </p>
     *
     * @param describeExplainabilityRequest
     * @return A Java Future containing the result of the DescribeExplainability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeExplainability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeExplainability"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeExplainabilityResponse> describeExplainability(
            DescribeExplainabilityRequest describeExplainabilityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeExplainabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeExplainabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeExplainability");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeExplainabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeExplainabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeExplainabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeExplainabilityRequest, DescribeExplainabilityResponse>()
                            .withOperationName("DescribeExplainability").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeExplainabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeExplainabilityRequest));
            CompletableFuture<DescribeExplainabilityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an Explainability export created using the <a>CreateExplainabilityExport</a> operation.
     * </p>
     *
     * @param describeExplainabilityExportRequest
     * @return A Java Future containing the result of the DescribeExplainabilityExport operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeExplainabilityExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeExplainabilityExport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeExplainabilityExportResponse> describeExplainabilityExport(
            DescribeExplainabilityExportRequest describeExplainabilityExportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeExplainabilityExportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeExplainabilityExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeExplainabilityExport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeExplainabilityExportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeExplainabilityExportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeExplainabilityExportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeExplainabilityExportRequest, DescribeExplainabilityExportResponse>()
                            .withOperationName("DescribeExplainabilityExport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeExplainabilityExportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeExplainabilityExportRequest));
            CompletableFuture<DescribeExplainabilityExportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a forecast created using the <a>CreateForecast</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided in the <code>CreateForecast</code> request, this operation lists
     * the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DatasetGroupArn</code> - The dataset group that provided the training data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeForecastRequest
     * @return A Java Future containing the result of the DescribeForecast operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeForecast
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeForecastResponse> describeForecast(DescribeForecastRequest describeForecastRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeForecastRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeForecastRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeForecast");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeForecastResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeForecastRequest, DescribeForecastResponse>()
                            .withOperationName("DescribeForecast").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeForecastRequest));
            CompletableFuture<DescribeForecastResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a forecast export job created using the <a>CreateForecastExportJob</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided by the user in the <code>CreateForecastExportJob</code> request,
     * this operation lists the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeForecastExportJobRequest
     * @return A Java Future containing the result of the DescribeForecastExportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeForecastExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeForecastExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeForecastExportJobResponse> describeForecastExportJob(
            DescribeForecastExportJobRequest describeForecastExportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeForecastExportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeForecastExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeForecastExportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeForecastExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeForecastExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeForecastExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeForecastExportJobRequest, DescribeForecastExportJobResponse>()
                            .withOperationName("DescribeForecastExportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeForecastExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeForecastExportJobRequest));
            CompletableFuture<DescribeForecastExportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a monitor resource. In addition to listing the properties provided in the <a>CreateMonitor</a> request,
     * this operation lists the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Baseline</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastEvaluationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastEvaluationState</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeMonitorRequest
     * @return A Java Future containing the result of the DescribeMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMonitorResponse> describeMonitor(DescribeMonitorRequest describeMonitorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeMonitorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMonitor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMonitorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMonitorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMonitorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMonitorRequest, DescribeMonitorResponse>()
                            .withOperationName("DescribeMonitor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeMonitorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeMonitorRequest));
            CompletableFuture<DescribeMonitorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This operation is only valid for legacy predictors created with CreatePredictor. If you are not using a legacy
     * predictor, use <a>DescribeAutoPredictor</a>.
     * </p>
     * </note>
     * <p>
     * Describes a predictor created using the <a>CreatePredictor</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided in the <code>CreatePredictor</code> request, this operation lists
     * the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DatasetImportJobArns</code> - The dataset import jobs used to import training data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutoMLAlgorithmArns</code> - If AutoML is performed, the algorithms that were evaluated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describePredictorRequest
     * @return A Java Future containing the result of the DescribePredictor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribePredictor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribePredictor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePredictorResponse> describePredictor(DescribePredictorRequest describePredictorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePredictorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePredictorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePredictor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePredictorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePredictorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePredictorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePredictorRequest, DescribePredictorResponse>()
                            .withOperationName("DescribePredictor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribePredictorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describePredictorRequest));
            CompletableFuture<DescribePredictorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a predictor backtest export job created using the <a>CreatePredictorBacktestExportJob</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided by the user in the <code>CreatePredictorBacktestExportJob</code>
     * request, this operation lists the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> (if an error occurred)
     * </p>
     * </li>
     * </ul>
     *
     * @param describePredictorBacktestExportJobRequest
     * @return A Java Future containing the result of the DescribePredictorBacktestExportJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribePredictorBacktestExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribePredictorBacktestExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePredictorBacktestExportJobResponse> describePredictorBacktestExportJob(
            DescribePredictorBacktestExportJobRequest describePredictorBacktestExportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePredictorBacktestExportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describePredictorBacktestExportJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePredictorBacktestExportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePredictorBacktestExportJobResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribePredictorBacktestExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePredictorBacktestExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePredictorBacktestExportJobRequest, DescribePredictorBacktestExportJobResponse>()
                            .withOperationName("DescribePredictorBacktestExportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribePredictorBacktestExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describePredictorBacktestExportJobRequest));
            CompletableFuture<DescribePredictorBacktestExportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the what-if analysis created using the <a>CreateWhatIfAnalysis</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided in the <code>CreateWhatIfAnalysis</code> request, this operation
     * lists the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeWhatIfAnalysisRequest
     * @return A Java Future containing the result of the DescribeWhatIfAnalysis operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeWhatIfAnalysis
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeWhatIfAnalysis"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWhatIfAnalysisResponse> describeWhatIfAnalysis(
            DescribeWhatIfAnalysisRequest describeWhatIfAnalysisRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeWhatIfAnalysisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWhatIfAnalysisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWhatIfAnalysis");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWhatIfAnalysisResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWhatIfAnalysisResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWhatIfAnalysisResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWhatIfAnalysisRequest, DescribeWhatIfAnalysisResponse>()
                            .withOperationName("DescribeWhatIfAnalysis").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeWhatIfAnalysisRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeWhatIfAnalysisRequest));
            CompletableFuture<DescribeWhatIfAnalysisResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the what-if forecast created using the <a>CreateWhatIfForecast</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided in the <code>CreateWhatIfForecast</code> request, this operation
     * lists the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeWhatIfForecastRequest
     * @return A Java Future containing the result of the DescribeWhatIfForecast operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeWhatIfForecast
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeWhatIfForecast"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWhatIfForecastResponse> describeWhatIfForecast(
            DescribeWhatIfForecastRequest describeWhatIfForecastRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeWhatIfForecastRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWhatIfForecastRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWhatIfForecast");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWhatIfForecastResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWhatIfForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWhatIfForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWhatIfForecastRequest, DescribeWhatIfForecastResponse>()
                            .withOperationName("DescribeWhatIfForecast").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeWhatIfForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeWhatIfForecastRequest));
            CompletableFuture<DescribeWhatIfForecastResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the what-if forecast export created using the <a>CreateWhatIfForecastExport</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided in the <code>CreateWhatIfForecastExport</code> request, this
     * operation lists the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeWhatIfForecastExportRequest
     * @return A Java Future containing the result of the DescribeWhatIfForecastExport operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeWhatIfForecastExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeWhatIfForecastExport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWhatIfForecastExportResponse> describeWhatIfForecastExport(
            DescribeWhatIfForecastExportRequest describeWhatIfForecastExportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeWhatIfForecastExportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWhatIfForecastExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWhatIfForecastExport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWhatIfForecastExportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWhatIfForecastExportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWhatIfForecastExportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWhatIfForecastExportRequest, DescribeWhatIfForecastExportResponse>()
                            .withOperationName("DescribeWhatIfForecastExport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeWhatIfForecastExportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeWhatIfForecastExportRequest));
            CompletableFuture<DescribeWhatIfForecastExportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides metrics on the accuracy of the models that were trained by the <a>CreatePredictor</a> operation. Use
     * metrics to see how well the model performed and to decide whether to use the predictor to generate a forecast.
     * For more information, see <a href="https://docs.aws.amazon.com/forecast/latest/dg/metrics.html">Predictor
     * Metrics</a>.
     * </p>
     * <p>
     * This operation generates metrics for each backtest window that was evaluated. The number of backtest windows (
     * <code>NumberOfBacktestWindows</code>) is specified using the <a>EvaluationParameters</a> object, which is
     * optionally included in the <code>CreatePredictor</code> request. If <code>NumberOfBacktestWindows</code> isn't
     * specified, the number defaults to one.
     * </p>
     * <p>
     * The parameters of the <code>filling</code> method determine which items contribute to the metrics. If you want
     * all items to contribute, specify <code>zero</code>. If you want only those items that have complete data in the
     * range being evaluated to contribute, specify <code>nan</code>. For more information, see
     * <a>FeaturizationMethod</a>.
     * </p>
     * <note>
     * <p>
     * Before you can get accuracy metrics, the <code>Status</code> of the predictor must be <code>ACTIVE</code>,
     * signifying that training has completed. To get the status, use the <a>DescribePredictor</a> operation.
     * </p>
     * </note>
     *
     * @param getAccuracyMetricsRequest
     * @return A Java Future containing the result of the GetAccuracyMetrics operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.GetAccuracyMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/GetAccuracyMetrics" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccuracyMetricsResponse> getAccuracyMetrics(GetAccuracyMetricsRequest getAccuracyMetricsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccuracyMetricsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccuracyMetricsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccuracyMetrics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccuracyMetricsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccuracyMetricsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccuracyMetricsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccuracyMetricsRequest, GetAccuracyMetricsResponse>()
                            .withOperationName("GetAccuracyMetrics").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAccuracyMetricsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccuracyMetricsRequest));
            CompletableFuture<GetAccuracyMetricsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of dataset groups created using the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDatasetGroup.html">CreateDatasetGroup</a>
     * operation. For each dataset group, this operation returns a summary of its properties, including its Amazon
     * Resource Name (ARN). You can retrieve the complete set of properties by using the dataset group ARN with the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDatasetGroup.html">DescribeDatasetGroup</a>
     * operation.
     * </p>
     *
     * @param listDatasetGroupsRequest
     * @return A Java Future containing the result of the ListDatasetGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListDatasetGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasetGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetGroupsResponse> listDatasetGroups(ListDatasetGroupsRequest listDatasetGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDatasetGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasetGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDatasetGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetGroupsRequest, ListDatasetGroupsResponse>()
                            .withOperationName("ListDatasetGroups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDatasetGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDatasetGroupsRequest));
            CompletableFuture<ListDatasetGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of dataset import jobs created using the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDatasetImportJob.html">CreateDatasetImportJob</a>
     * operation. For each import job, this operation returns a summary of its properties, including its Amazon Resource
     * Name (ARN). You can retrieve the complete set of properties by using the ARN with the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDatasetImportJob.html"
     * >DescribeDatasetImportJob</a> operation. You can filter the list by providing an array of <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_Filter.html">Filter</a> objects.
     * </p>
     *
     * @param listDatasetImportJobsRequest
     * @return A Java Future containing the result of the ListDatasetImportJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListDatasetImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasetImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetImportJobsResponse> listDatasetImportJobs(
            ListDatasetImportJobsRequest listDatasetImportJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDatasetImportJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetImportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasetImportJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetImportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDatasetImportJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetImportJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetImportJobsRequest, ListDatasetImportJobsResponse>()
                            .withOperationName("ListDatasetImportJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDatasetImportJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDatasetImportJobsRequest));
            CompletableFuture<ListDatasetImportJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of datasets created using the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDataset.html">CreateDataset</a> operation. For
     * each dataset, a summary of its properties, including its Amazon Resource Name (ARN), is returned. To retrieve the
     * complete set of properties, use the ARN with the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDataset.html">DescribeDataset</a> operation.
     * </p>
     *
     * @param listDatasetsRequest
     * @return A Java Future containing the result of the ListDatasets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListDatasets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetsResponse> listDatasets(ListDatasetsRequest listDatasetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDatasetsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDatasetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetsRequest, ListDatasetsResponse>()
                            .withOperationName("ListDatasets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDatasetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDatasetsRequest));
            CompletableFuture<ListDatasetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of Explainability resources created using the <a>CreateExplainability</a> operation. This
     * operation returns a summary for each Explainability. You can filter the list using an array of <a>Filter</a>
     * objects.
     * </p>
     * <p>
     * To retrieve the complete set of properties for a particular Explainability resource, use the ARN with the
     * <a>DescribeExplainability</a> operation.
     * </p>
     *
     * @param listExplainabilitiesRequest
     * @return A Java Future containing the result of the ListExplainabilities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListExplainabilities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListExplainabilities" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListExplainabilitiesResponse> listExplainabilities(
            ListExplainabilitiesRequest listExplainabilitiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExplainabilitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExplainabilitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExplainabilities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExplainabilitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListExplainabilitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExplainabilitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExplainabilitiesRequest, ListExplainabilitiesResponse>()
                            .withOperationName("ListExplainabilities").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListExplainabilitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listExplainabilitiesRequest));
            CompletableFuture<ListExplainabilitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of Explainability exports created using the <a>CreateExplainabilityExport</a> operation. This
     * operation returns a summary for each Explainability export. You can filter the list using an array of
     * <a>Filter</a> objects.
     * </p>
     * <p>
     * To retrieve the complete set of properties for a particular Explainability export, use the ARN with the
     * <a>DescribeExplainability</a> operation.
     * </p>
     *
     * @param listExplainabilityExportsRequest
     * @return A Java Future containing the result of the ListExplainabilityExports operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListExplainabilityExports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListExplainabilityExports"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListExplainabilityExportsResponse> listExplainabilityExports(
            ListExplainabilityExportsRequest listExplainabilityExportsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExplainabilityExportsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExplainabilityExportsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExplainabilityExports");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExplainabilityExportsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListExplainabilityExportsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExplainabilityExportsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExplainabilityExportsRequest, ListExplainabilityExportsResponse>()
                            .withOperationName("ListExplainabilityExports").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListExplainabilityExportsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listExplainabilityExportsRequest));
            CompletableFuture<ListExplainabilityExportsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of forecast export jobs created using the <a>CreateForecastExportJob</a> operation. For each
     * forecast export job, this operation returns a summary of its properties, including its Amazon Resource Name
     * (ARN). To retrieve the complete set of properties, use the ARN with the <a>DescribeForecastExportJob</a>
     * operation. You can filter the list using an array of <a>Filter</a> objects.
     * </p>
     *
     * @param listForecastExportJobsRequest
     * @return A Java Future containing the result of the ListForecastExportJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListForecastExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListForecastExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListForecastExportJobsResponse> listForecastExportJobs(
            ListForecastExportJobsRequest listForecastExportJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listForecastExportJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listForecastExportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListForecastExportJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListForecastExportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListForecastExportJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListForecastExportJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListForecastExportJobsRequest, ListForecastExportJobsResponse>()
                            .withOperationName("ListForecastExportJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListForecastExportJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listForecastExportJobsRequest));
            CompletableFuture<ListForecastExportJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of forecasts created using the <a>CreateForecast</a> operation. For each forecast, this operation
     * returns a summary of its properties, including its Amazon Resource Name (ARN). To retrieve the complete set of
     * properties, specify the ARN with the <a>DescribeForecast</a> operation. You can filter the list using an array of
     * <a>Filter</a> objects.
     * </p>
     *
     * @param listForecastsRequest
     * @return A Java Future containing the result of the ListForecasts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListForecasts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListForecasts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListForecastsResponse> listForecasts(ListForecastsRequest listForecastsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listForecastsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listForecastsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListForecasts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListForecastsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListForecastsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListForecastsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListForecastsRequest, ListForecastsResponse>()
                            .withOperationName("ListForecasts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListForecastsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listForecastsRequest));
            CompletableFuture<ListForecastsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the monitoring evaluation results and predictor events collected by the monitor resource during
     * different windows of time.
     * </p>
     * <p>
     * For information about monitoring see <a>predictor-monitoring</a>. For more information about retrieving
     * monitoring results see <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/predictor-monitoring-results.html">Viewing Monitoring
     * Results</a>.
     * </p>
     *
     * @param listMonitorEvaluationsRequest
     * @return A Java Future containing the result of the ListMonitorEvaluations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListMonitorEvaluations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListMonitorEvaluations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMonitorEvaluationsResponse> listMonitorEvaluations(
            ListMonitorEvaluationsRequest listMonitorEvaluationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMonitorEvaluationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMonitorEvaluationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMonitorEvaluations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMonitorEvaluationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMonitorEvaluationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMonitorEvaluationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMonitorEvaluationsRequest, ListMonitorEvaluationsResponse>()
                            .withOperationName("ListMonitorEvaluations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMonitorEvaluationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMonitorEvaluationsRequest));
            CompletableFuture<ListMonitorEvaluationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of monitors created with the <a>CreateMonitor</a> operation and <a>CreateAutoPredictor</a>
     * operation. For each monitor resource, this operation returns of a summary of its properties, including its Amazon
     * Resource Name (ARN). You can retrieve a complete set of properties of a monitor resource by specify the monitor's
     * ARN in the <a>DescribeMonitor</a> operation.
     * </p>
     *
     * @param listMonitorsRequest
     * @return A Java Future containing the result of the ListMonitors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMonitorsResponse> listMonitors(ListMonitorsRequest listMonitorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMonitorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMonitorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMonitors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMonitorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMonitorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMonitorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMonitorsRequest, ListMonitorsResponse>()
                            .withOperationName("ListMonitors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMonitorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMonitorsRequest));
            CompletableFuture<ListMonitorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of predictor backtest export jobs created using the <a>CreatePredictorBacktestExportJob</a>
     * operation. This operation returns a summary for each backtest export job. You can filter the list using an array
     * of <a>Filter</a> objects.
     * </p>
     * <p>
     * To retrieve the complete set of properties for a particular backtest export job, use the ARN with the
     * <a>DescribePredictorBacktestExportJob</a> operation.
     * </p>
     *
     * @param listPredictorBacktestExportJobsRequest
     * @return A Java Future containing the result of the ListPredictorBacktestExportJobs operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListPredictorBacktestExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListPredictorBacktestExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPredictorBacktestExportJobsResponse> listPredictorBacktestExportJobs(
            ListPredictorBacktestExportJobsRequest listPredictorBacktestExportJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPredictorBacktestExportJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listPredictorBacktestExportJobsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPredictorBacktestExportJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPredictorBacktestExportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPredictorBacktestExportJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPredictorBacktestExportJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPredictorBacktestExportJobsRequest, ListPredictorBacktestExportJobsResponse>()
                            .withOperationName("ListPredictorBacktestExportJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPredictorBacktestExportJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPredictorBacktestExportJobsRequest));
            CompletableFuture<ListPredictorBacktestExportJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of predictors created using the <a>CreateAutoPredictor</a> or <a>CreatePredictor</a> operations.
     * For each predictor, this operation returns a summary of its properties, including its Amazon Resource Name (ARN).
     * </p>
     * <p>
     * You can retrieve the complete set of properties by using the ARN with the <a>DescribeAutoPredictor</a> and
     * <a>DescribePredictor</a> operations. You can filter the list using an array of <a>Filter</a> objects.
     * </p>
     *
     * @param listPredictorsRequest
     * @return A Java Future containing the result of the ListPredictors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListPredictors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListPredictors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPredictorsResponse> listPredictors(ListPredictorsRequest listPredictorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPredictorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPredictorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPredictors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPredictorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPredictorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPredictorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPredictorsRequest, ListPredictorsResponse>()
                            .withOperationName("ListPredictors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPredictorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPredictorsRequest));
            CompletableFuture<ListPredictorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for an Amazon Forecast resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of what-if analyses created using the <a>CreateWhatIfAnalysis</a> operation. For each what-if
     * analysis, this operation returns a summary of its properties, including its Amazon Resource Name (ARN). You can
     * retrieve the complete set of properties by using the what-if analysis ARN with the <a>DescribeWhatIfAnalysis</a>
     * operation.
     * </p>
     *
     * @param listWhatIfAnalysesRequest
     * @return A Java Future containing the result of the ListWhatIfAnalyses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListWhatIfAnalyses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListWhatIfAnalyses" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWhatIfAnalysesResponse> listWhatIfAnalyses(ListWhatIfAnalysesRequest listWhatIfAnalysesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWhatIfAnalysesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWhatIfAnalysesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWhatIfAnalyses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWhatIfAnalysesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWhatIfAnalysesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWhatIfAnalysesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWhatIfAnalysesRequest, ListWhatIfAnalysesResponse>()
                            .withOperationName("ListWhatIfAnalyses").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWhatIfAnalysesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWhatIfAnalysesRequest));
            CompletableFuture<ListWhatIfAnalysesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of what-if forecast exports created using the <a>CreateWhatIfForecastExport</a> operation. For
     * each what-if forecast export, this operation returns a summary of its properties, including its Amazon Resource
     * Name (ARN). You can retrieve the complete set of properties by using the what-if forecast export ARN with the
     * <a>DescribeWhatIfForecastExport</a> operation.
     * </p>
     *
     * @param listWhatIfForecastExportsRequest
     * @return A Java Future containing the result of the ListWhatIfForecastExports operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListWhatIfForecastExports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListWhatIfForecastExports"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWhatIfForecastExportsResponse> listWhatIfForecastExports(
            ListWhatIfForecastExportsRequest listWhatIfForecastExportsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWhatIfForecastExportsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWhatIfForecastExportsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWhatIfForecastExports");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWhatIfForecastExportsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWhatIfForecastExportsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWhatIfForecastExportsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWhatIfForecastExportsRequest, ListWhatIfForecastExportsResponse>()
                            .withOperationName("ListWhatIfForecastExports").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWhatIfForecastExportsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWhatIfForecastExportsRequest));
            CompletableFuture<ListWhatIfForecastExportsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of what-if forecasts created using the <a>CreateWhatIfForecast</a> operation. For each what-if
     * forecast, this operation returns a summary of its properties, including its Amazon Resource Name (ARN). You can
     * retrieve the complete set of properties by using the what-if forecast ARN with the <a>DescribeWhatIfForecast</a>
     * operation.
     * </p>
     *
     * @param listWhatIfForecastsRequest
     * @return A Java Future containing the result of the ListWhatIfForecasts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListWhatIfForecasts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListWhatIfForecasts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWhatIfForecastsResponse> listWhatIfForecasts(
            ListWhatIfForecastsRequest listWhatIfForecastsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWhatIfForecastsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWhatIfForecastsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWhatIfForecasts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWhatIfForecastsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWhatIfForecastsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWhatIfForecastsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWhatIfForecastsRequest, ListWhatIfForecastsResponse>()
                            .withOperationName("ListWhatIfForecasts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWhatIfForecastsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWhatIfForecastsRequest));
            CompletableFuture<ListWhatIfForecastsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Resumes a stopped monitor resource.
     * </p>
     *
     * @param resumeResourceRequest
     * @return A Java Future containing the result of the ResumeResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ResumeResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ResumeResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ResumeResourceResponse> resumeResource(ResumeResourceRequest resumeResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resumeResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resumeResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResumeResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResumeResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ResumeResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResumeResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResumeResourceRequest, ResumeResourceResponse>()
                            .withOperationName("ResumeResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ResumeResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(resumeResourceRequest));
            CompletableFuture<ResumeResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a resource.
     * </p>
     * <p>
     * The resource undergoes the following states: <code>CREATE_STOPPING</code> and <code>CREATE_STOPPED</code>. You
     * cannot resume a resource once it has been stopped.
     * </p>
     * <p>
     * This operation can be applied to the following resources (and their corresponding child resources):
     * </p>
     * <ul>
     * <li>
     * <p>
     * Dataset Import Job
     * </p>
     * </li>
     * <li>
     * <p>
     * Predictor Job
     * </p>
     * </li>
     * <li>
     * <p>
     * Forecast Job
     * </p>
     * </li>
     * <li>
     * <p>
     * Forecast Export Job
     * </p>
     * </li>
     * <li>
     * <p>
     * Predictor Backtest Export Job
     * </p>
     * </li>
     * <li>
     * <p>
     * Explainability Job
     * </p>
     * </li>
     * <li>
     * <p>
     * Explainability Export Job
     * </p>
     * </li>
     * </ul>
     *
     * @param stopResourceRequest
     * @return A Java Future containing the result of the StopResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.StopResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/StopResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopResourceResponse> stopResource(StopResourceRequest stopResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopResourceRequest, StopResourceResponse>()
                            .withOperationName("StopResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopResourceRequest));
            CompletableFuture<StopResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource are not specified in the request parameters, they are not changed. When a resource is deleted, the tags
     * associated with that resource are also deleted.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>LimitExceededException The limit on the number of resources per account has been exceeded.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Replaces the datasets in a dataset group with the specified datasets.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the dataset group must be <code>ACTIVE</code> before you can use the dataset group to
     * create a predictor. Use the <a
     * href="https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDatasetGroup.html">DescribeDatasetGroup</a>
     * operation to get the status.
     * </p>
     * </note>
     *
     * @param updateDatasetGroupRequest
     * @return A Java Future containing the result of the UpdateDatasetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.UpdateDatasetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/UpdateDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDatasetGroupResponse> updateDatasetGroup(UpdateDatasetGroupRequest updateDatasetGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDatasetGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDatasetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDatasetGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDatasetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDatasetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDatasetGroupRequest, UpdateDatasetGroupResponse>()
                            .withOperationName("UpdateDatasetGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDatasetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDatasetGroupRequest));
            CompletableFuture<UpdateDatasetGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ForecastServiceClientConfiguration serviceClientConfiguration() {
        return new ForecastServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ForecastException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ForecastServiceClientConfigurationBuilder serviceConfigBuilder = new ForecastServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
