/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.forecast;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.forecast.model.CreateDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.CreateDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.model.CreateDatasetImportJobRequest;
import software.amazon.awssdk.services.forecast.model.CreateDatasetImportJobResponse;
import software.amazon.awssdk.services.forecast.model.CreateDatasetRequest;
import software.amazon.awssdk.services.forecast.model.CreateDatasetResponse;
import software.amazon.awssdk.services.forecast.model.CreateForecastExportJobRequest;
import software.amazon.awssdk.services.forecast.model.CreateForecastExportJobResponse;
import software.amazon.awssdk.services.forecast.model.CreateForecastRequest;
import software.amazon.awssdk.services.forecast.model.CreateForecastResponse;
import software.amazon.awssdk.services.forecast.model.CreatePredictorBacktestExportJobRequest;
import software.amazon.awssdk.services.forecast.model.CreatePredictorBacktestExportJobResponse;
import software.amazon.awssdk.services.forecast.model.CreatePredictorRequest;
import software.amazon.awssdk.services.forecast.model.CreatePredictorResponse;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetImportJobRequest;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetImportJobResponse;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetRequest;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetResponse;
import software.amazon.awssdk.services.forecast.model.DeleteForecastExportJobRequest;
import software.amazon.awssdk.services.forecast.model.DeleteForecastExportJobResponse;
import software.amazon.awssdk.services.forecast.model.DeleteForecastRequest;
import software.amazon.awssdk.services.forecast.model.DeleteForecastResponse;
import software.amazon.awssdk.services.forecast.model.DeletePredictorBacktestExportJobRequest;
import software.amazon.awssdk.services.forecast.model.DeletePredictorBacktestExportJobResponse;
import software.amazon.awssdk.services.forecast.model.DeletePredictorRequest;
import software.amazon.awssdk.services.forecast.model.DeletePredictorResponse;
import software.amazon.awssdk.services.forecast.model.DeleteResourceTreeRequest;
import software.amazon.awssdk.services.forecast.model.DeleteResourceTreeResponse;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetImportJobRequest;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetImportJobResponse;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetRequest;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetResponse;
import software.amazon.awssdk.services.forecast.model.DescribeForecastExportJobRequest;
import software.amazon.awssdk.services.forecast.model.DescribeForecastExportJobResponse;
import software.amazon.awssdk.services.forecast.model.DescribeForecastRequest;
import software.amazon.awssdk.services.forecast.model.DescribeForecastResponse;
import software.amazon.awssdk.services.forecast.model.DescribePredictorBacktestExportJobRequest;
import software.amazon.awssdk.services.forecast.model.DescribePredictorBacktestExportJobResponse;
import software.amazon.awssdk.services.forecast.model.DescribePredictorRequest;
import software.amazon.awssdk.services.forecast.model.DescribePredictorResponse;
import software.amazon.awssdk.services.forecast.model.ForecastException;
import software.amazon.awssdk.services.forecast.model.ForecastRequest;
import software.amazon.awssdk.services.forecast.model.GetAccuracyMetricsRequest;
import software.amazon.awssdk.services.forecast.model.GetAccuracyMetricsResponse;
import software.amazon.awssdk.services.forecast.model.InvalidInputException;
import software.amazon.awssdk.services.forecast.model.InvalidNextTokenException;
import software.amazon.awssdk.services.forecast.model.LimitExceededException;
import software.amazon.awssdk.services.forecast.model.ListDatasetGroupsRequest;
import software.amazon.awssdk.services.forecast.model.ListDatasetGroupsResponse;
import software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsRequest;
import software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsResponse;
import software.amazon.awssdk.services.forecast.model.ListDatasetsRequest;
import software.amazon.awssdk.services.forecast.model.ListDatasetsResponse;
import software.amazon.awssdk.services.forecast.model.ListForecastExportJobsRequest;
import software.amazon.awssdk.services.forecast.model.ListForecastExportJobsResponse;
import software.amazon.awssdk.services.forecast.model.ListForecastsRequest;
import software.amazon.awssdk.services.forecast.model.ListForecastsResponse;
import software.amazon.awssdk.services.forecast.model.ListPredictorBacktestExportJobsRequest;
import software.amazon.awssdk.services.forecast.model.ListPredictorBacktestExportJobsResponse;
import software.amazon.awssdk.services.forecast.model.ListPredictorsRequest;
import software.amazon.awssdk.services.forecast.model.ListPredictorsResponse;
import software.amazon.awssdk.services.forecast.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.forecast.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.forecast.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.forecast.model.ResourceInUseException;
import software.amazon.awssdk.services.forecast.model.ResourceNotFoundException;
import software.amazon.awssdk.services.forecast.model.StopResourceRequest;
import software.amazon.awssdk.services.forecast.model.StopResourceResponse;
import software.amazon.awssdk.services.forecast.model.TagResourceRequest;
import software.amazon.awssdk.services.forecast.model.TagResourceResponse;
import software.amazon.awssdk.services.forecast.model.UntagResourceRequest;
import software.amazon.awssdk.services.forecast.model.UntagResourceResponse;
import software.amazon.awssdk.services.forecast.model.UpdateDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.UpdateDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.paginators.ListDatasetGroupsIterable;
import software.amazon.awssdk.services.forecast.paginators.ListDatasetImportJobsIterable;
import software.amazon.awssdk.services.forecast.paginators.ListDatasetsIterable;
import software.amazon.awssdk.services.forecast.paginators.ListForecastExportJobsIterable;
import software.amazon.awssdk.services.forecast.paginators.ListForecastsIterable;
import software.amazon.awssdk.services.forecast.paginators.ListPredictorBacktestExportJobsIterable;
import software.amazon.awssdk.services.forecast.paginators.ListPredictorsIterable;
import software.amazon.awssdk.services.forecast.transform.CreateDatasetGroupRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateDatasetImportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateDatasetRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateForecastExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreatePredictorBacktestExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreatePredictorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteDatasetGroupRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteDatasetImportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteDatasetRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteForecastExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeletePredictorBacktestExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeletePredictorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteResourceTreeRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeDatasetGroupRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeDatasetImportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeDatasetRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeForecastExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribePredictorBacktestExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribePredictorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.GetAccuracyMetricsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListDatasetGroupsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListDatasetImportJobsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListDatasetsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListForecastExportJobsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListForecastsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListPredictorBacktestExportJobsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListPredictorsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.StopResourceRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.UpdateDatasetGroupRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ForecastClient}.
 *
 * @see ForecastClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultForecastClient implements ForecastClient {
    private static final Logger log = Logger.loggerFor(DefaultForecastClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultForecastClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an Amazon Forecast dataset. The information about the dataset that you provide helps Forecast understand
     * how to consume the data for model training. This includes the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i> <code>DataFrequency</code> </i> - How frequently your historical time-series data is collected.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i> <code>Domain</code> </i> and <i> <code>DatasetType</code> </i> - Each dataset has an associated dataset
     * domain and a type within the domain. Amazon Forecast provides a list of predefined domains and types within each
     * domain. For each unique dataset domain and type within the domain, Amazon Forecast requires your data to include
     * a minimum set of predefined fields.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i> <code>Schema</code> </i> - A schema specifies the fields in the dataset, including the field name and data
     * type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * After creating a dataset, you import your training data into it and add the dataset to a dataset group. You use
     * the dataset group to create a predictor. For more information, see <a>howitworks-datasets-groups</a>.
     * </p>
     * <p>
     * To get a list of all your datasets, use the <a>ListDatasets</a> operation.
     * </p>
     * <p>
     * For example Forecast datasets, see the <a href="https://github.com/aws-samples/amazon-forecast-samples">Amazon
     * Forecast Sample GitHub repository</a>.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of a dataset must be <code>ACTIVE</code> before you can import training data. Use the
     * <a>DescribeDataset</a> operation to get the status.
     * </p>
     * </note>
     *
     * @param createDatasetRequest
     * @return Result of the CreateDataset operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceAlreadyExistsException
     *         There is already a resource with this name. Try again with a different name.
     * @throws LimitExceededException
     *         The limit on the number of resources per account has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.CreateDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateDatasetResponse createDataset(CreateDatasetRequest createDatasetRequest) throws InvalidInputException,
            ResourceAlreadyExistsException, LimitExceededException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataset");

            return clientHandler.execute(new ClientExecutionParams<CreateDatasetRequest, CreateDatasetResponse>()
                    .withOperationName("CreateDataset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDatasetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDatasetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a dataset group, which holds a collection of related datasets. You can add datasets to the dataset group
     * when you create the dataset group, or later by using the <a>UpdateDatasetGroup</a> operation.
     * </p>
     * <p>
     * After creating a dataset group and adding datasets, you use the dataset group when you create a predictor. For
     * more information, see <a>howitworks-datasets-groups</a>.
     * </p>
     * <p>
     * To get a list of all your datasets groups, use the <a>ListDatasetGroups</a> operation.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of a dataset group must be <code>ACTIVE</code> before you can use the dataset group to
     * create a predictor. To get the status, use the <a>DescribeDatasetGroup</a> operation.
     * </p>
     * </note>
     *
     * @param createDatasetGroupRequest
     * @return Result of the CreateDatasetGroup operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceAlreadyExistsException
     *         There is already a resource with this name. Try again with a different name.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws LimitExceededException
     *         The limit on the number of resources per account has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.CreateDatasetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateDatasetGroupResponse createDatasetGroup(CreateDatasetGroupRequest createDatasetGroupRequest)
            throws InvalidInputException, ResourceAlreadyExistsException, ResourceNotFoundException, ResourceInUseException,
            LimitExceededException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDatasetGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatasetGroup");

            return clientHandler.execute(new ClientExecutionParams<CreateDatasetGroupRequest, CreateDatasetGroupResponse>()
                    .withOperationName("CreateDatasetGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDatasetGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDatasetGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Imports your training data to an Amazon Forecast dataset. You provide the location of your training data in an
     * Amazon Simple Storage Service (Amazon S3) bucket and the Amazon Resource Name (ARN) of the dataset that you want
     * to import the data to.
     * </p>
     * <p>
     * You must specify a <a>DataSource</a> object that includes an AWS Identity and Access Management (IAM) role that
     * Amazon Forecast can assume to access the data, as Amazon Forecast makes a copy of your data and processes it in
     * an internal AWS system. For more information, see <a>aws-forecast-iam-roles</a>.
     * </p>
     * <p>
     * The training data must be in CSV format. The delimiter must be a comma (,).
     * </p>
     * <p>
     * You can specify the path to a specific CSV file, the S3 bucket, or to a folder in the S3 bucket. For the latter
     * two cases, Amazon Forecast imports all files up to the limit of 10,000 files.
     * </p>
     * <p>
     * Because dataset imports are not aggregated, your most recent dataset import is the one that is used when training
     * a predictor or generating a forecast. Make sure that your most recent dataset import contains all of the data you
     * want to model off of, and not just the new data collected since the previous import.
     * </p>
     * <p>
     * To get a list of all your dataset import jobs, filtered by specified criteria, use the
     * <a>ListDatasetImportJobs</a> operation.
     * </p>
     *
     * @param createDatasetImportJobRequest
     * @return Result of the CreateDatasetImportJob operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceAlreadyExistsException
     *         There is already a resource with this name. Try again with a different name.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws LimitExceededException
     *         The limit on the number of resources per account has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.CreateDatasetImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateDatasetImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDatasetImportJobResponse createDatasetImportJob(CreateDatasetImportJobRequest createDatasetImportJobRequest)
            throws InvalidInputException, ResourceAlreadyExistsException, ResourceNotFoundException, ResourceInUseException,
            LimitExceededException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDatasetImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDatasetImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatasetImportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetImportJobRequest, CreateDatasetImportJobResponse>()
                            .withOperationName("CreateDatasetImportJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createDatasetImportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDatasetImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a forecast for each item in the <code>TARGET_TIME_SERIES</code> dataset that was used to train the
     * predictor. This is known as inference. To retrieve the forecast for a single item at low latency, use the
     * operation. To export the complete forecast into your Amazon Simple Storage Service (Amazon S3) bucket, use the
     * <a>CreateForecastExportJob</a> operation.
     * </p>
     * <p>
     * The range of the forecast is determined by the <code>ForecastHorizon</code> value, which you specify in the
     * <a>CreatePredictor</a> request. When you query a forecast, you can request a specific date range within the
     * forecast.
     * </p>
     * <p>
     * To get a list of all your forecasts, use the <a>ListForecasts</a> operation.
     * </p>
     * <note>
     * <p>
     * The forecasts generated by Amazon Forecast are in the same time zone as the dataset that was used to create the
     * predictor.
     * </p>
     * </note>
     * <p>
     * For more information, see <a>howitworks-forecast</a>.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the forecast must be <code>ACTIVE</code> before you can query or export the forecast.
     * Use the <a>DescribeForecast</a> operation to get the status.
     * </p>
     * </note>
     *
     * @param createForecastRequest
     * @return Result of the CreateForecast operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceAlreadyExistsException
     *         There is already a resource with this name. Try again with a different name.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws LimitExceededException
     *         The limit on the number of resources per account has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.CreateForecast
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateForecastResponse createForecast(CreateForecastRequest createForecastRequest) throws InvalidInputException,
            ResourceAlreadyExistsException, ResourceNotFoundException, ResourceInUseException, LimitExceededException,
            AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateForecastResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateForecastResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createForecastRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateForecast");

            return clientHandler.execute(new ClientExecutionParams<CreateForecastRequest, CreateForecastResponse>()
                    .withOperationName("CreateForecast").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createForecastRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateForecastRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Exports a forecast created by the <a>CreateForecast</a> operation to your Amazon Simple Storage Service (Amazon
     * S3) bucket. The forecast file name will match the following conventions:
     * </p>
     * <p>
     * &lt;ForecastExportJobName&gt;_&lt;ExportTimestamp&gt;_&lt;PartNumber&gt;
     * </p>
     * <p>
     * where the &lt;ExportTimestamp&gt; component is in Java SimpleDateFormat (yyyy-MM-ddTHH-mm-ssZ).
     * </p>
     * <p>
     * You must specify a <a>DataDestination</a> object that includes an AWS Identity and Access Management (IAM) role
     * that Amazon Forecast can assume to access the Amazon S3 bucket. For more information, see
     * <a>aws-forecast-iam-roles</a>.
     * </p>
     * <p>
     * For more information, see <a>howitworks-forecast</a>.
     * </p>
     * <p>
     * To get a list of all your forecast export jobs, use the <a>ListForecastExportJobs</a> operation.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the forecast export job must be <code>ACTIVE</code> before you can access the forecast
     * in your Amazon S3 bucket. To get the status, use the <a>DescribeForecastExportJob</a> operation.
     * </p>
     * </note>
     *
     * @param createForecastExportJobRequest
     * @return Result of the CreateForecastExportJob operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceAlreadyExistsException
     *         There is already a resource with this name. Try again with a different name.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws LimitExceededException
     *         The limit on the number of resources per account has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.CreateForecastExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateForecastExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateForecastExportJobResponse createForecastExportJob(CreateForecastExportJobRequest createForecastExportJobRequest)
            throws InvalidInputException, ResourceAlreadyExistsException, ResourceNotFoundException, ResourceInUseException,
            LimitExceededException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateForecastExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateForecastExportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createForecastExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateForecastExportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateForecastExportJobRequest, CreateForecastExportJobResponse>()
                            .withOperationName("CreateForecastExportJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createForecastExportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateForecastExportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon Forecast predictor.
     * </p>
     * <p>
     * In the request, provide a dataset group and either specify an algorithm or let Amazon Forecast choose an
     * algorithm for you using AutoML. If you specify an algorithm, you also can override algorithm-specific
     * hyperparameters.
     * </p>
     * <p>
     * Amazon Forecast uses the algorithm to train a predictor using the latest version of the datasets in the specified
     * dataset group. You can then generate a forecast using the <a>CreateForecast</a> operation.
     * </p>
     * <p>
     * To see the evaluation metrics, use the <a>GetAccuracyMetrics</a> operation.
     * </p>
     * <p>
     * You can specify a featurization configuration to fill and aggregate the data fields in the
     * <code>TARGET_TIME_SERIES</code> dataset to improve model training. For more information, see
     * <a>FeaturizationConfig</a>.
     * </p>
     * <p>
     * For RELATED_TIME_SERIES datasets, <code>CreatePredictor</code> verifies that the <code>DataFrequency</code>
     * specified when the dataset was created matches the <code>ForecastFrequency</code>. TARGET_TIME_SERIES datasets
     * don't have this restriction. Amazon Forecast also verifies the delimiter and timestamp format. For more
     * information, see <a>howitworks-datasets-groups</a>.
     * </p>
     * <p>
     * By default, predictors are trained and evaluated at the 0.1 (P10), 0.5 (P50), and 0.9 (P90) quantiles. You can
     * choose custom forecast types to train and evaluate your predictor by setting the <code>ForecastTypes</code>.
     * </p>
     * <p>
     * <b>AutoML</b>
     * </p>
     * <p>
     * If you want Amazon Forecast to evaluate each algorithm and choose the one that minimizes the
     * <code>objective function</code>, set <code>PerformAutoML</code> to <code>true</code>. The
     * <code>objective function</code> is defined as the mean of the weighted losses over the forecast types. By
     * default, these are the p10, p50, and p90 quantile losses. For more information, see <a>EvaluationResult</a>.
     * </p>
     * <p>
     * When AutoML is enabled, the following properties are disallowed:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AlgorithmArn</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HPOConfig</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PerformHPO</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TrainingParameters</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get a list of all of your predictors, use the <a>ListPredictors</a> operation.
     * </p>
     * <note>
     * <p>
     * Before you can use the predictor to create a forecast, the <code>Status</code> of the predictor must be
     * <code>ACTIVE</code>, signifying that training has completed. To get the status, use the <a>DescribePredictor</a>
     * operation.
     * </p>
     * </note>
     *
     * @param createPredictorRequest
     * @return Result of the CreatePredictor operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceAlreadyExistsException
     *         There is already a resource with this name. Try again with a different name.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws LimitExceededException
     *         The limit on the number of resources per account has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.CreatePredictor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreatePredictor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreatePredictorResponse createPredictor(CreatePredictorRequest createPredictorRequest) throws InvalidInputException,
            ResourceAlreadyExistsException, ResourceNotFoundException, ResourceInUseException, LimitExceededException,
            AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePredictorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreatePredictorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPredictorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePredictor");

            return clientHandler.execute(new ClientExecutionParams<CreatePredictorRequest, CreatePredictorResponse>()
                    .withOperationName("CreatePredictor").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createPredictorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreatePredictorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Exports backtest forecasts and accuracy metrics generated by the <a>CreatePredictor</a> operation. Two folders
     * containing CSV files are exported to your specified S3 bucket.
     * </p>
     * <p>
     * The export file names will match the following conventions:
     * </p>
     * <p>
     * <code>&lt;ExportJobName&gt;_&lt;ExportTimestamp&gt;_&lt;PartNumber&gt;.csv</code>
     * </p>
     * <p>
     * The &lt;ExportTimestamp&gt; component is in Java SimpleDate format (yyyy-MM-ddTHH-mm-ssZ).
     * </p>
     * <p>
     * You must specify a <a>DataDestination</a> object that includes an Amazon S3 bucket and an AWS Identity and Access
     * Management (IAM) role that Amazon Forecast can assume to access the Amazon S3 bucket. For more information, see
     * <a>aws-forecast-iam-roles</a>.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the export job must be <code>ACTIVE</code> before you can access the export in your
     * Amazon S3 bucket. To get the status, use the <a>DescribePredictorBacktestExportJob</a> operation.
     * </p>
     * </note>
     *
     * @param createPredictorBacktestExportJobRequest
     * @return Result of the CreatePredictorBacktestExportJob operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceAlreadyExistsException
     *         There is already a resource with this name. Try again with a different name.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws LimitExceededException
     *         The limit on the number of resources per account has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.CreatePredictorBacktestExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreatePredictorBacktestExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePredictorBacktestExportJobResponse createPredictorBacktestExportJob(
            CreatePredictorBacktestExportJobRequest createPredictorBacktestExportJobRequest) throws InvalidInputException,
            ResourceAlreadyExistsException, ResourceNotFoundException, ResourceInUseException, LimitExceededException,
            AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePredictorBacktestExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePredictorBacktestExportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createPredictorBacktestExportJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePredictorBacktestExportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePredictorBacktestExportJobRequest, CreatePredictorBacktestExportJobResponse>()
                            .withOperationName("CreatePredictorBacktestExportJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createPredictorBacktestExportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreatePredictorBacktestExportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon Forecast dataset that was created using the <a>CreateDataset</a> operation. You can only delete
     * datasets that have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the status use the
     * <a>DescribeDataset</a> operation.
     * </p>
     * <note>
     * <p>
     * Forecast does not automatically update any dataset groups that contain the deleted dataset. In order to update
     * the dataset group, use the operation, omitting the deleted dataset's ARN.
     * </p>
     * </note>
     *
     * @param deleteDatasetRequest
     * @return Result of the DeleteDataset operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DeleteDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteDatasetResponse deleteDataset(DeleteDatasetRequest deleteDatasetRequest) throws InvalidInputException,
            ResourceNotFoundException, ResourceInUseException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataset");

            return clientHandler.execute(new ClientExecutionParams<DeleteDatasetRequest, DeleteDatasetResponse>()
                    .withOperationName("DeleteDataset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDatasetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDatasetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a dataset group created using the <a>CreateDatasetGroup</a> operation. You can only delete dataset groups
     * that have a status of <code>ACTIVE</code>, <code>CREATE_FAILED</code>, or <code>UPDATE_FAILED</code>. To get the
     * status, use the <a>DescribeDatasetGroup</a> operation.
     * </p>
     * <p>
     * This operation deletes only the dataset group, not the datasets in the group.
     * </p>
     *
     * @param deleteDatasetGroupRequest
     * @return Result of the DeleteDatasetGroup operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DeleteDatasetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteDatasetGroupResponse deleteDatasetGroup(DeleteDatasetGroupRequest deleteDatasetGroupRequest)
            throws InvalidInputException, ResourceNotFoundException, ResourceInUseException, AwsServiceException,
            SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDatasetGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatasetGroup");

            return clientHandler.execute(new ClientExecutionParams<DeleteDatasetGroupRequest, DeleteDatasetGroupResponse>()
                    .withOperationName("DeleteDatasetGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDatasetGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDatasetGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a dataset import job created using the <a>CreateDatasetImportJob</a> operation. You can delete only
     * dataset import jobs that have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the status,
     * use the <a>DescribeDatasetImportJob</a> operation.
     * </p>
     *
     * @param deleteDatasetImportJobRequest
     * @return Result of the DeleteDatasetImportJob operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DeleteDatasetImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteDatasetImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDatasetImportJobResponse deleteDatasetImportJob(DeleteDatasetImportJobRequest deleteDatasetImportJobRequest)
            throws InvalidInputException, ResourceNotFoundException, ResourceInUseException, AwsServiceException,
            SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDatasetImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDatasetImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatasetImportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetImportJobRequest, DeleteDatasetImportJobResponse>()
                            .withOperationName("DeleteDatasetImportJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteDatasetImportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteDatasetImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a forecast created using the <a>CreateForecast</a> operation. You can delete only forecasts that have a
     * status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the status, use the <a>DescribeForecast</a>
     * operation.
     * </p>
     * <p>
     * You can't delete a forecast while it is being exported. After a forecast is deleted, you can no longer query the
     * forecast.
     * </p>
     *
     * @param deleteForecastRequest
     * @return Result of the DeleteForecast operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DeleteForecast
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteForecastResponse deleteForecast(DeleteForecastRequest deleteForecastRequest) throws InvalidInputException,
            ResourceNotFoundException, ResourceInUseException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteForecastResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteForecastResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteForecastRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteForecast");

            return clientHandler.execute(new ClientExecutionParams<DeleteForecastRequest, DeleteForecastResponse>()
                    .withOperationName("DeleteForecast").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteForecastRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteForecastRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a forecast export job created using the <a>CreateForecastExportJob</a> operation. You can delete only
     * export jobs that have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the status, use the
     * <a>DescribeForecastExportJob</a> operation.
     * </p>
     *
     * @param deleteForecastExportJobRequest
     * @return Result of the DeleteForecastExportJob operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DeleteForecastExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteForecastExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteForecastExportJobResponse deleteForecastExportJob(DeleteForecastExportJobRequest deleteForecastExportJobRequest)
            throws InvalidInputException, ResourceNotFoundException, ResourceInUseException, AwsServiceException,
            SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteForecastExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteForecastExportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteForecastExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteForecastExportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteForecastExportJobRequest, DeleteForecastExportJobResponse>()
                            .withOperationName("DeleteForecastExportJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteForecastExportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteForecastExportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a predictor created using the <a>CreatePredictor</a> operation. You can delete only predictor that have a
     * status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. To get the status, use the <a>DescribePredictor</a>
     * operation.
     * </p>
     *
     * @param deletePredictorRequest
     * @return Result of the DeletePredictor operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DeletePredictor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeletePredictor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePredictorResponse deletePredictor(DeletePredictorRequest deletePredictorRequest) throws InvalidInputException,
            ResourceNotFoundException, ResourceInUseException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePredictorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePredictorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePredictorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePredictor");

            return clientHandler.execute(new ClientExecutionParams<DeletePredictorRequest, DeletePredictorResponse>()
                    .withOperationName("DeletePredictor").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deletePredictorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePredictorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a predictor backtest export job.
     * </p>
     *
     * @param deletePredictorBacktestExportJobRequest
     * @return Result of the DeletePredictorBacktestExportJob operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DeletePredictorBacktestExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeletePredictorBacktestExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeletePredictorBacktestExportJobResponse deletePredictorBacktestExportJob(
            DeletePredictorBacktestExportJobRequest deletePredictorBacktestExportJobRequest) throws InvalidInputException,
            ResourceNotFoundException, ResourceInUseException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePredictorBacktestExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePredictorBacktestExportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deletePredictorBacktestExportJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePredictorBacktestExportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DeletePredictorBacktestExportJobRequest, DeletePredictorBacktestExportJobResponse>()
                            .withOperationName("DeletePredictorBacktestExportJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deletePredictorBacktestExportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeletePredictorBacktestExportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an entire resource tree. This operation will delete the parent resource and its child resources.
     * </p>
     * <p>
     * Child resources are resources that were created from another resource. For example, when a forecast is generated
     * from a predictor, the forecast is the child resource and the predictor is the parent resource.
     * </p>
     * <p>
     * Amazon Forecast resources possess the following parent-child resource hierarchies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Dataset</b>: dataset import jobs
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Dataset Group</b>: predictors, predictor backtest export jobs, forecasts, forecast export jobs
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Predictor</b>: predictor backtest export jobs, forecasts, forecast export jobs
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Forecast</b>: forecast export jobs
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * <code>DeleteResourceTree</code> will only delete Amazon Forecast resources, and will not delete datasets or
     * exported files stored in Amazon S3.
     * </p>
     * </note>
     *
     * @param deleteResourceTreeRequest
     * @return Result of the DeleteResourceTree operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DeleteResourceTree
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteResourceTree" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteResourceTreeResponse deleteResourceTree(DeleteResourceTreeRequest deleteResourceTreeRequest)
            throws InvalidInputException, ResourceNotFoundException, ResourceInUseException, AwsServiceException,
            SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourceTreeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourceTreeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceTreeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceTree");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourceTreeRequest, DeleteResourceTreeResponse>()
                    .withOperationName("DeleteResourceTree").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteResourceTreeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourceTreeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an Amazon Forecast dataset created using the <a>CreateDataset</a> operation.
     * </p>
     * <p>
     * In addition to listing the parameters specified in the <code>CreateDataset</code> request, this operation
     * includes the following dataset properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeDatasetRequest
     * @return Result of the DescribeDataset operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DescribeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeDatasetResponse describeDataset(DescribeDatasetRequest describeDatasetRequest) throws InvalidInputException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataset");

            return clientHandler.execute(new ClientExecutionParams<DescribeDatasetRequest, DescribeDatasetResponse>()
                    .withOperationName("DescribeDataset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeDatasetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDatasetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a dataset group created using the <a>CreateDatasetGroup</a> operation.
     * </p>
     * <p>
     * In addition to listing the parameters provided in the <code>CreateDatasetGroup</code> request, this operation
     * includes the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DatasetArns</code> - The datasets belonging to the group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeDatasetGroupRequest
     * @return Result of the DescribeDatasetGroup operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DescribeDatasetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeDatasetGroupResponse describeDatasetGroup(DescribeDatasetGroupRequest describeDatasetGroupRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDatasetGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDatasetGroup");

            return clientHandler.execute(new ClientExecutionParams<DescribeDatasetGroupRequest, DescribeDatasetGroupResponse>()
                    .withOperationName("DescribeDatasetGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeDatasetGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDatasetGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a dataset import job created using the <a>CreateDatasetImportJob</a> operation.
     * </p>
     * <p>
     * In addition to listing the parameters provided in the <code>CreateDatasetImportJob</code> request, this operation
     * includes the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DataSize</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FieldStatistics</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeDatasetImportJobRequest
     * @return Result of the DescribeDatasetImportJob operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DescribeDatasetImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeDatasetImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDatasetImportJobResponse describeDatasetImportJob(
            DescribeDatasetImportJobRequest describeDatasetImportJobRequest) throws InvalidInputException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDatasetImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDatasetImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDatasetImportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetImportJobRequest, DescribeDatasetImportJobResponse>()
                            .withOperationName("DescribeDatasetImportJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeDatasetImportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeDatasetImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a forecast created using the <a>CreateForecast</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided in the <code>CreateForecast</code> request, this operation lists
     * the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DatasetGroupArn</code> - The dataset group that provided the training data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeForecastRequest
     * @return Result of the DescribeForecast operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DescribeForecast
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeForecastResponse describeForecast(DescribeForecastRequest describeForecastRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeForecastResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeForecastResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeForecastRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeForecast");

            return clientHandler.execute(new ClientExecutionParams<DescribeForecastRequest, DescribeForecastResponse>()
                    .withOperationName("DescribeForecast").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeForecastRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeForecastRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a forecast export job created using the <a>CreateForecastExportJob</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided by the user in the <code>CreateForecastExportJob</code> request,
     * this operation lists the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeForecastExportJobRequest
     * @return Result of the DescribeForecastExportJob operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DescribeForecastExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeForecastExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeForecastExportJobResponse describeForecastExportJob(
            DescribeForecastExportJobRequest describeForecastExportJobRequest) throws InvalidInputException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeForecastExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeForecastExportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeForecastExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeForecastExportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeForecastExportJobRequest, DescribeForecastExportJobResponse>()
                            .withOperationName("DescribeForecastExportJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeForecastExportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeForecastExportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a predictor created using the <a>CreatePredictor</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided in the <code>CreatePredictor</code> request, this operation lists
     * the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DatasetImportJobArns</code> - The dataset import jobs used to import training data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutoMLAlgorithmArns</code> - If AutoML is performed, the algorithms that were evaluated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describePredictorRequest
     * @return Result of the DescribePredictor operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DescribePredictor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribePredictor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribePredictorResponse describePredictor(DescribePredictorRequest describePredictorRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePredictorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribePredictorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePredictorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePredictor");

            return clientHandler.execute(new ClientExecutionParams<DescribePredictorRequest, DescribePredictorResponse>()
                    .withOperationName("DescribePredictor").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describePredictorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribePredictorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a predictor backtest export job created using the <a>CreatePredictorBacktestExportJob</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided by the user in the <code>CreatePredictorBacktestExportJob</code>
     * request, this operation lists the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> (if an error occurred)
     * </p>
     * </li>
     * </ul>
     *
     * @param describePredictorBacktestExportJobRequest
     * @return Result of the DescribePredictorBacktestExportJob operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.DescribePredictorBacktestExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribePredictorBacktestExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribePredictorBacktestExportJobResponse describePredictorBacktestExportJob(
            DescribePredictorBacktestExportJobRequest describePredictorBacktestExportJobRequest) throws InvalidInputException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePredictorBacktestExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribePredictorBacktestExportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describePredictorBacktestExportJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePredictorBacktestExportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribePredictorBacktestExportJobRequest, DescribePredictorBacktestExportJobResponse>()
                            .withOperationName("DescribePredictorBacktestExportJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describePredictorBacktestExportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribePredictorBacktestExportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides metrics on the accuracy of the models that were trained by the <a>CreatePredictor</a> operation. Use
     * metrics to see how well the model performed and to decide whether to use the predictor to generate a forecast.
     * For more information, see <a href="https://docs.aws.amazon.com/forecast/latest/dg/metrics.html">Predictor
     * Metrics</a>.
     * </p>
     * <p>
     * This operation generates metrics for each backtest window that was evaluated. The number of backtest windows (
     * <code>NumberOfBacktestWindows</code>) is specified using the <a>EvaluationParameters</a> object, which is
     * optionally included in the <code>CreatePredictor</code> request. If <code>NumberOfBacktestWindows</code> isn't
     * specified, the number defaults to one.
     * </p>
     * <p>
     * The parameters of the <code>filling</code> method determine which items contribute to the metrics. If you want
     * all items to contribute, specify <code>zero</code>. If you want only those items that have complete data in the
     * range being evaluated to contribute, specify <code>nan</code>. For more information, see
     * <a>FeaturizationMethod</a>.
     * </p>
     * <note>
     * <p>
     * Before you can get accuracy metrics, the <code>Status</code> of the predictor must be <code>ACTIVE</code>,
     * signifying that training has completed. To get the status, use the <a>DescribePredictor</a> operation.
     * </p>
     * </note>
     *
     * @param getAccuracyMetricsRequest
     * @return Result of the GetAccuracyMetrics operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.GetAccuracyMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/GetAccuracyMetrics" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetAccuracyMetricsResponse getAccuracyMetrics(GetAccuracyMetricsRequest getAccuracyMetricsRequest)
            throws InvalidInputException, ResourceNotFoundException, ResourceInUseException, AwsServiceException,
            SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAccuracyMetricsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAccuracyMetricsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccuracyMetricsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccuracyMetrics");

            return clientHandler.execute(new ClientExecutionParams<GetAccuracyMetricsRequest, GetAccuracyMetricsResponse>()
                    .withOperationName("GetAccuracyMetrics").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAccuracyMetricsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAccuracyMetricsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of dataset groups created using the <a>CreateDatasetGroup</a> operation. For each dataset group,
     * this operation returns a summary of its properties, including its Amazon Resource Name (ARN). You can retrieve
     * the complete set of properties by using the dataset group ARN with the <a>DescribeDatasetGroup</a> operation.
     * </p>
     *
     * @param listDatasetGroupsRequest
     * @return Result of the ListDatasetGroups operation returned by the service.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListDatasetGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasetGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDatasetGroupsResponse listDatasetGroups(ListDatasetGroupsRequest listDatasetGroupsRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDatasetGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDatasetGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasetGroups");

            return clientHandler.execute(new ClientExecutionParams<ListDatasetGroupsRequest, ListDatasetGroupsResponse>()
                    .withOperationName("ListDatasetGroups").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDatasetGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDatasetGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of dataset groups created using the <a>CreateDatasetGroup</a> operation. For each dataset group,
     * this operation returns a summary of its properties, including its Amazon Resource Name (ARN). You can retrieve
     * the complete set of properties by using the dataset group ARN with the <a>DescribeDatasetGroup</a> operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatasetGroups(software.amazon.awssdk.services.forecast.model.ListDatasetGroupsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetGroupsIterable responses = client.listDatasetGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.forecast.paginators.ListDatasetGroupsIterable responses = client
     *             .listDatasetGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.forecast.model.ListDatasetGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetGroupsIterable responses = client.listDatasetGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatasetGroups(software.amazon.awssdk.services.forecast.model.ListDatasetGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDatasetGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListDatasetGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasetGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDatasetGroupsIterable listDatasetGroupsPaginator(ListDatasetGroupsRequest listDatasetGroupsRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, ForecastException {
        return new ListDatasetGroupsIterable(this, applyPaginatorUserAgent(listDatasetGroupsRequest));
    }

    /**
     * <p>
     * Returns a list of dataset import jobs created using the <a>CreateDatasetImportJob</a> operation. For each import
     * job, this operation returns a summary of its properties, including its Amazon Resource Name (ARN). You can
     * retrieve the complete set of properties by using the ARN with the <a>DescribeDatasetImportJob</a> operation. You
     * can filter the list by providing an array of <a>Filter</a> objects.
     * </p>
     *
     * @param listDatasetImportJobsRequest
     * @return Result of the ListDatasetImportJobs operation returned by the service.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListDatasetImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasetImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDatasetImportJobsResponse listDatasetImportJobs(ListDatasetImportJobsRequest listDatasetImportJobsRequest)
            throws InvalidNextTokenException, InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDatasetImportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDatasetImportJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetImportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasetImportJobs");

            return clientHandler.execute(new ClientExecutionParams<ListDatasetImportJobsRequest, ListDatasetImportJobsResponse>()
                    .withOperationName("ListDatasetImportJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDatasetImportJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDatasetImportJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of dataset import jobs created using the <a>CreateDatasetImportJob</a> operation. For each import
     * job, this operation returns a summary of its properties, including its Amazon Resource Name (ARN). You can
     * retrieve the complete set of properties by using the ARN with the <a>DescribeDatasetImportJob</a> operation. You
     * can filter the list by providing an array of <a>Filter</a> objects.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatasetImportJobs(software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetImportJobsIterable responses = client.listDatasetImportJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.forecast.paginators.ListDatasetImportJobsIterable responses = client
     *             .listDatasetImportJobsPaginator(request);
     *     for (software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetImportJobsIterable responses = client.listDatasetImportJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatasetImportJobs(software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDatasetImportJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListDatasetImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasetImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDatasetImportJobsIterable listDatasetImportJobsPaginator(ListDatasetImportJobsRequest listDatasetImportJobsRequest)
            throws InvalidNextTokenException, InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        return new ListDatasetImportJobsIterable(this, applyPaginatorUserAgent(listDatasetImportJobsRequest));
    }

    /**
     * <p>
     * Returns a list of datasets created using the <a>CreateDataset</a> operation. For each dataset, a summary of its
     * properties, including its Amazon Resource Name (ARN), is returned. To retrieve the complete set of properties,
     * use the ARN with the <a>DescribeDataset</a> operation.
     * </p>
     *
     * @param listDatasetsRequest
     * @return Result of the ListDatasets operation returned by the service.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListDatasets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDatasetsResponse listDatasets(ListDatasetsRequest listDatasetsRequest) throws InvalidNextTokenException,
            AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDatasetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDatasetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasets");

            return clientHandler.execute(new ClientExecutionParams<ListDatasetsRequest, ListDatasetsResponse>()
                    .withOperationName("ListDatasets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDatasetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDatasetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of datasets created using the <a>CreateDataset</a> operation. For each dataset, a summary of its
     * properties, including its Amazon Resource Name (ARN), is returned. To retrieve the complete set of properties,
     * use the ARN with the <a>DescribeDataset</a> operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listDatasets(software.amazon.awssdk.services.forecast.model.ListDatasetsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetsIterable responses = client.listDatasetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.forecast.paginators.ListDatasetsIterable responses = client.listDatasetsPaginator(request);
     *     for (software.amazon.awssdk.services.forecast.model.ListDatasetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetsIterable responses = client.listDatasetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatasets(software.amazon.awssdk.services.forecast.model.ListDatasetsRequest)} operation.</b>
     * </p>
     *
     * @param listDatasetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListDatasets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDatasetsIterable listDatasetsPaginator(ListDatasetsRequest listDatasetsRequest) throws InvalidNextTokenException,
            AwsServiceException, SdkClientException, ForecastException {
        return new ListDatasetsIterable(this, applyPaginatorUserAgent(listDatasetsRequest));
    }

    /**
     * <p>
     * Returns a list of forecast export jobs created using the <a>CreateForecastExportJob</a> operation. For each
     * forecast export job, this operation returns a summary of its properties, including its Amazon Resource Name
     * (ARN). To retrieve the complete set of properties, use the ARN with the <a>DescribeForecastExportJob</a>
     * operation. You can filter the list using an array of <a>Filter</a> objects.
     * </p>
     *
     * @param listForecastExportJobsRequest
     * @return Result of the ListForecastExportJobs operation returned by the service.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListForecastExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListForecastExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListForecastExportJobsResponse listForecastExportJobs(ListForecastExportJobsRequest listForecastExportJobsRequest)
            throws InvalidNextTokenException, InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListForecastExportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListForecastExportJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listForecastExportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListForecastExportJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListForecastExportJobsRequest, ListForecastExportJobsResponse>()
                            .withOperationName("ListForecastExportJobs").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listForecastExportJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListForecastExportJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of forecast export jobs created using the <a>CreateForecastExportJob</a> operation. For each
     * forecast export job, this operation returns a summary of its properties, including its Amazon Resource Name
     * (ARN). To retrieve the complete set of properties, use the ARN with the <a>DescribeForecastExportJob</a>
     * operation. You can filter the list using an array of <a>Filter</a> objects.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listForecastExportJobs(software.amazon.awssdk.services.forecast.model.ListForecastExportJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListForecastExportJobsIterable responses = client.listForecastExportJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.forecast.paginators.ListForecastExportJobsIterable responses = client
     *             .listForecastExportJobsPaginator(request);
     *     for (software.amazon.awssdk.services.forecast.model.ListForecastExportJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListForecastExportJobsIterable responses = client.listForecastExportJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listForecastExportJobs(software.amazon.awssdk.services.forecast.model.ListForecastExportJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listForecastExportJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListForecastExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListForecastExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListForecastExportJobsIterable listForecastExportJobsPaginator(
            ListForecastExportJobsRequest listForecastExportJobsRequest) throws InvalidNextTokenException, InvalidInputException,
            AwsServiceException, SdkClientException, ForecastException {
        return new ListForecastExportJobsIterable(this, applyPaginatorUserAgent(listForecastExportJobsRequest));
    }

    /**
     * <p>
     * Returns a list of forecasts created using the <a>CreateForecast</a> operation. For each forecast, this operation
     * returns a summary of its properties, including its Amazon Resource Name (ARN). To retrieve the complete set of
     * properties, specify the ARN with the <a>DescribeForecast</a> operation. You can filter the list using an array of
     * <a>Filter</a> objects.
     * </p>
     *
     * @param listForecastsRequest
     * @return Result of the ListForecasts operation returned by the service.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListForecasts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListForecasts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListForecastsResponse listForecasts(ListForecastsRequest listForecastsRequest) throws InvalidNextTokenException,
            InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListForecastsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListForecastsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listForecastsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListForecasts");

            return clientHandler.execute(new ClientExecutionParams<ListForecastsRequest, ListForecastsResponse>()
                    .withOperationName("ListForecasts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listForecastsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListForecastsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of forecasts created using the <a>CreateForecast</a> operation. For each forecast, this operation
     * returns a summary of its properties, including its Amazon Resource Name (ARN). To retrieve the complete set of
     * properties, specify the ARN with the <a>DescribeForecast</a> operation. You can filter the list using an array of
     * <a>Filter</a> objects.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listForecasts(software.amazon.awssdk.services.forecast.model.ListForecastsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListForecastsIterable responses = client.listForecastsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.forecast.paginators.ListForecastsIterable responses = client.listForecastsPaginator(request);
     *     for (software.amazon.awssdk.services.forecast.model.ListForecastsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListForecastsIterable responses = client.listForecastsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listForecasts(software.amazon.awssdk.services.forecast.model.ListForecastsRequest)} operation.</b>
     * </p>
     *
     * @param listForecastsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListForecasts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListForecasts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListForecastsIterable listForecastsPaginator(ListForecastsRequest listForecastsRequest)
            throws InvalidNextTokenException, InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        return new ListForecastsIterable(this, applyPaginatorUserAgent(listForecastsRequest));
    }

    /**
     * <p>
     * Returns a list of predictor backtest export jobs created using the <a>CreatePredictorBacktestExportJob</a>
     * operation. This operation returns a summary for each backtest export job. You can filter the list using an array
     * of <a>Filter</a> objects.
     * </p>
     * <p>
     * To retrieve the complete set of properties for a particular backtest export job, use the ARN with the
     * <a>DescribePredictorBacktestExportJob</a> operation.
     * </p>
     *
     * @param listPredictorBacktestExportJobsRequest
     * @return Result of the ListPredictorBacktestExportJobs operation returned by the service.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListPredictorBacktestExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListPredictorBacktestExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPredictorBacktestExportJobsResponse listPredictorBacktestExportJobs(
            ListPredictorBacktestExportJobsRequest listPredictorBacktestExportJobsRequest) throws InvalidNextTokenException,
            InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPredictorBacktestExportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPredictorBacktestExportJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listPredictorBacktestExportJobsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPredictorBacktestExportJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListPredictorBacktestExportJobsRequest, ListPredictorBacktestExportJobsResponse>()
                            .withOperationName("ListPredictorBacktestExportJobs").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listPredictorBacktestExportJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListPredictorBacktestExportJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of predictor backtest export jobs created using the <a>CreatePredictorBacktestExportJob</a>
     * operation. This operation returns a summary for each backtest export job. You can filter the list using an array
     * of <a>Filter</a> objects.
     * </p>
     * <p>
     * To retrieve the complete set of properties for a particular backtest export job, use the ARN with the
     * <a>DescribePredictorBacktestExportJob</a> operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPredictorBacktestExportJobs(software.amazon.awssdk.services.forecast.model.ListPredictorBacktestExportJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListPredictorBacktestExportJobsIterable responses = client.listPredictorBacktestExportJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.forecast.paginators.ListPredictorBacktestExportJobsIterable responses = client
     *             .listPredictorBacktestExportJobsPaginator(request);
     *     for (software.amazon.awssdk.services.forecast.model.ListPredictorBacktestExportJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListPredictorBacktestExportJobsIterable responses = client.listPredictorBacktestExportJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPredictorBacktestExportJobs(software.amazon.awssdk.services.forecast.model.ListPredictorBacktestExportJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPredictorBacktestExportJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListPredictorBacktestExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListPredictorBacktestExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPredictorBacktestExportJobsIterable listPredictorBacktestExportJobsPaginator(
            ListPredictorBacktestExportJobsRequest listPredictorBacktestExportJobsRequest) throws InvalidNextTokenException,
            InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        return new ListPredictorBacktestExportJobsIterable(this, applyPaginatorUserAgent(listPredictorBacktestExportJobsRequest));
    }

    /**
     * <p>
     * Returns a list of predictors created using the <a>CreatePredictor</a> operation. For each predictor, this
     * operation returns a summary of its properties, including its Amazon Resource Name (ARN). You can retrieve the
     * complete set of properties by using the ARN with the <a>DescribePredictor</a> operation. You can filter the list
     * using an array of <a>Filter</a> objects.
     * </p>
     *
     * @param listPredictorsRequest
     * @return Result of the ListPredictors operation returned by the service.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListPredictors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListPredictors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPredictorsResponse listPredictors(ListPredictorsRequest listPredictorsRequest) throws InvalidNextTokenException,
            InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPredictorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPredictorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPredictorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPredictors");

            return clientHandler.execute(new ClientExecutionParams<ListPredictorsRequest, ListPredictorsResponse>()
                    .withOperationName("ListPredictors").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPredictorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPredictorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of predictors created using the <a>CreatePredictor</a> operation. For each predictor, this
     * operation returns a summary of its properties, including its Amazon Resource Name (ARN). You can retrieve the
     * complete set of properties by using the ARN with the <a>DescribePredictor</a> operation. You can filter the list
     * using an array of <a>Filter</a> objects.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPredictors(software.amazon.awssdk.services.forecast.model.ListPredictorsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListPredictorsIterable responses = client.listPredictorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.forecast.paginators.ListPredictorsIterable responses = client
     *             .listPredictorsPaginator(request);
     *     for (software.amazon.awssdk.services.forecast.model.ListPredictorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListPredictorsIterable responses = client.listPredictorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPredictors(software.amazon.awssdk.services.forecast.model.ListPredictorsRequest)} operation.</b>
     * </p>
     *
     * @param listPredictorsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The token is not valid. Tokens expire after 24 hours.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListPredictors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListPredictors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPredictorsIterable listPredictorsPaginator(ListPredictorsRequest listPredictorsRequest)
            throws InvalidNextTokenException, InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        return new ListPredictorsIterable(this, applyPaginatorUserAgent(listPredictorsRequest));
    }

    /**
     * <p>
     * Lists the tags for an Amazon Forecast resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a resource.
     * </p>
     * <p>
     * The resource undergoes the following states: <code>CREATE_STOPPING</code> and <code>CREATE_STOPPED</code>. You
     * cannot resume a resource once it has been stopped.
     * </p>
     * <p>
     * This operation can be applied to the following resources (and their corresponding child resources):
     * </p>
     * <ul>
     * <li>
     * <p>
     * Dataset Import Job
     * </p>
     * </li>
     * <li>
     * <p>
     * Predictor Job
     * </p>
     * </li>
     * <li>
     * <p>
     * Forecast Job
     * </p>
     * </li>
     * <li>
     * <p>
     * Forecast Export Job
     * </p>
     * </li>
     * <li>
     * <p>
     * Predictor Backtest Export Job
     * </p>
     * </li>
     * </ul>
     *
     * @param stopResourceRequest
     * @return Result of the StopResource operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws LimitExceededException
     *         The limit on the number of resources per account has been exceeded.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.StopResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/StopResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopResourceResponse stopResource(StopResourceRequest stopResourceRequest) throws InvalidInputException,
            LimitExceededException, ResourceNotFoundException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopResource");

            return clientHandler.execute(new ClientExecutionParams<StopResourceRequest, StopResourceResponse>()
                    .withOperationName("StopResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource are not specified in the request parameters, they are not changed. When a resource is deleted, the tags
     * associated with that resource are also deleted.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws LimitExceededException
     *         The limit on the number of resources per account has been exceeded.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            LimitExceededException, InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InvalidInputException, AwsServiceException, SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Replaces the datasets in a dataset group with the specified datasets.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the dataset group must be <code>ACTIVE</code> before you can use the dataset group to
     * create a predictor. Use the <a>DescribeDatasetGroup</a> operation to get the status.
     * </p>
     * </note>
     *
     * @param updateDatasetGroupRequest
     * @return Result of the UpdateDatasetGroup operation returned by the service.
     * @throws InvalidInputException
     *         We can't process the request because it includes an invalid value or a value that exceeds the valid
     *         range.
     * @throws ResourceNotFoundException
     *         We can't find a resource with that Amazon Resource Name (ARN). Check the ARN and try again.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ForecastException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ForecastClient.UpdateDatasetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/UpdateDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateDatasetGroupResponse updateDatasetGroup(UpdateDatasetGroupRequest updateDatasetGroupRequest)
            throws InvalidInputException, ResourceNotFoundException, ResourceInUseException, AwsServiceException,
            SdkClientException, ForecastException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDatasetGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDatasetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "forecast");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDatasetGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdateDatasetGroupRequest, UpdateDatasetGroupResponse>()
                    .withOperationName("UpdateDatasetGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateDatasetGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDatasetGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ForecastException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ForecastRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
