/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.forecast.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The results of evaluating an algorithm. Returned as part of the <a>GetAccuracyMetrics</a> response.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EvaluationResult implements SdkPojo, Serializable,
        ToCopyableBuilder<EvaluationResult.Builder, EvaluationResult> {
    private static final SdkField<String> ALGORITHM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlgorithmArn").getter(getter(EvaluationResult::algorithmArn)).setter(setter(Builder::algorithmArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlgorithmArn").build()).build();

    private static final SdkField<List<WindowSummary>> TEST_WINDOWS_FIELD = SdkField
            .<List<WindowSummary>> builder(MarshallingType.LIST)
            .memberName("TestWindows")
            .getter(getter(EvaluationResult::testWindows))
            .setter(setter(Builder::testWindows))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TestWindows").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<WindowSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(WindowSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALGORITHM_ARN_FIELD,
            TEST_WINDOWS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String algorithmArn;

    private final List<WindowSummary> testWindows;

    private EvaluationResult(BuilderImpl builder) {
        this.algorithmArn = builder.algorithmArn;
        this.testWindows = builder.testWindows;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the algorithm that was evaluated.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the algorithm that was evaluated.
     */
    public String algorithmArn() {
        return algorithmArn;
    }

    /**
     * Returns true if the TestWindows property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTestWindows() {
        return testWindows != null && !(testWindows instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The array of test windows used for evaluating the algorithm. The <code>NumberOfBacktestWindows</code> from the
     * <a>EvaluationParameters</a> object determines the number of windows in the array.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTestWindows()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The array of test windows used for evaluating the algorithm. The <code>NumberOfBacktestWindows</code>
     *         from the <a>EvaluationParameters</a> object determines the number of windows in the array.
     */
    public List<WindowSummary> testWindows() {
        return testWindows;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(algorithmArn());
        hashCode = 31 * hashCode + Objects.hashCode(testWindows());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EvaluationResult)) {
            return false;
        }
        EvaluationResult other = (EvaluationResult) obj;
        return Objects.equals(algorithmArn(), other.algorithmArn()) && Objects.equals(testWindows(), other.testWindows());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("EvaluationResult").add("AlgorithmArn", algorithmArn()).add("TestWindows", testWindows()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AlgorithmArn":
            return Optional.ofNullable(clazz.cast(algorithmArn()));
        case "TestWindows":
            return Optional.ofNullable(clazz.cast(testWindows()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EvaluationResult, T> g) {
        return obj -> g.apply((EvaluationResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EvaluationResult> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the algorithm that was evaluated.
         * </p>
         * 
         * @param algorithmArn
         *        The Amazon Resource Name (ARN) of the algorithm that was evaluated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmArn(String algorithmArn);

        /**
         * <p>
         * The array of test windows used for evaluating the algorithm. The <code>NumberOfBacktestWindows</code> from
         * the <a>EvaluationParameters</a> object determines the number of windows in the array.
         * </p>
         * 
         * @param testWindows
         *        The array of test windows used for evaluating the algorithm. The <code>NumberOfBacktestWindows</code>
         *        from the <a>EvaluationParameters</a> object determines the number of windows in the array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder testWindows(Collection<WindowSummary> testWindows);

        /**
         * <p>
         * The array of test windows used for evaluating the algorithm. The <code>NumberOfBacktestWindows</code> from
         * the <a>EvaluationParameters</a> object determines the number of windows in the array.
         * </p>
         * 
         * @param testWindows
         *        The array of test windows used for evaluating the algorithm. The <code>NumberOfBacktestWindows</code>
         *        from the <a>EvaluationParameters</a> object determines the number of windows in the array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder testWindows(WindowSummary... testWindows);

        /**
         * <p>
         * The array of test windows used for evaluating the algorithm. The <code>NumberOfBacktestWindows</code> from
         * the <a>EvaluationParameters</a> object determines the number of windows in the array.
         * </p>
         * This is a convenience that creates an instance of the {@link List<WindowSummary>.Builder} avoiding the need
         * to create one manually via {@link List<WindowSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<WindowSummary>.Builder#build()} is called immediately and
         * its result is passed to {@link #testWindows(List<WindowSummary>)}.
         * 
         * @param testWindows
         *        a consumer that will call methods on {@link List<WindowSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #testWindows(List<WindowSummary>)
         */
        Builder testWindows(Consumer<WindowSummary.Builder>... testWindows);
    }

    static final class BuilderImpl implements Builder {
        private String algorithmArn;

        private List<WindowSummary> testWindows = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EvaluationResult model) {
            algorithmArn(model.algorithmArn);
            testWindows(model.testWindows);
        }

        public final String getAlgorithmArn() {
            return algorithmArn;
        }

        @Override
        public final Builder algorithmArn(String algorithmArn) {
            this.algorithmArn = algorithmArn;
            return this;
        }

        public final void setAlgorithmArn(String algorithmArn) {
            this.algorithmArn = algorithmArn;
        }

        public final Collection<WindowSummary.Builder> getTestWindows() {
            return testWindows != null ? testWindows.stream().map(WindowSummary::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder testWindows(Collection<WindowSummary> testWindows) {
            this.testWindows = TestWindowsCopier.copy(testWindows);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder testWindows(WindowSummary... testWindows) {
            testWindows(Arrays.asList(testWindows));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder testWindows(Consumer<WindowSummary.Builder>... testWindows) {
            testWindows(Stream.of(testWindows).map(c -> WindowSummary.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setTestWindows(Collection<WindowSummary.BuilderImpl> testWindows) {
            this.testWindows = TestWindowsCopier.copyFromBuilder(testWindows);
        }

        @Override
        public EvaluationResult build() {
            return new EvaluationResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
