/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fms.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Detailed information about an entry violation in a network ACL. The violation is against the network ACL
 * specification inside the Firewall Manager network ACL policy. This data object is part of
 * <code>InvalidNetworkAclEntriesViolation</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EntryViolation implements SdkPojo, Serializable, ToCopyableBuilder<EntryViolation.Builder, EntryViolation> {
    private static final SdkField<EntryDescription> EXPECTED_ENTRY_FIELD = SdkField
            .<EntryDescription> builder(MarshallingType.SDK_POJO).memberName("ExpectedEntry")
            .getter(getter(EntryViolation::expectedEntry)).setter(setter(Builder::expectedEntry))
            .constructor(EntryDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpectedEntry").build()).build();

    private static final SdkField<String> EXPECTED_EVALUATION_ORDER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExpectedEvaluationOrder").getter(getter(EntryViolation::expectedEvaluationOrder))
            .setter(setter(Builder::expectedEvaluationOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpectedEvaluationOrder").build())
            .build();

    private static final SdkField<String> ACTUAL_EVALUATION_ORDER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActualEvaluationOrder").getter(getter(EntryViolation::actualEvaluationOrder))
            .setter(setter(Builder::actualEvaluationOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActualEvaluationOrder").build())
            .build();

    private static final SdkField<EntryDescription> ENTRY_AT_EXPECTED_EVALUATION_ORDER_FIELD = SdkField
            .<EntryDescription> builder(MarshallingType.SDK_POJO)
            .memberName("EntryAtExpectedEvaluationOrder")
            .getter(getter(EntryViolation::entryAtExpectedEvaluationOrder))
            .setter(setter(Builder::entryAtExpectedEvaluationOrder))
            .constructor(EntryDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntryAtExpectedEvaluationOrder")
                    .build()).build();

    private static final SdkField<List<EntryDescription>> ENTRIES_WITH_CONFLICTS_FIELD = SdkField
            .<List<EntryDescription>> builder(MarshallingType.LIST)
            .memberName("EntriesWithConflicts")
            .getter(getter(EntryViolation::entriesWithConflicts))
            .setter(setter(Builder::entriesWithConflicts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntriesWithConflicts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EntryDescription> builder(MarshallingType.SDK_POJO)
                                            .constructor(EntryDescription::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ENTRY_VIOLATION_REASONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EntryViolationReasons")
            .getter(getter(EntryViolation::entryViolationReasonsAsStrings))
            .setter(setter(Builder::entryViolationReasonsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntryViolationReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPECTED_ENTRY_FIELD,
            EXPECTED_EVALUATION_ORDER_FIELD, ACTUAL_EVALUATION_ORDER_FIELD, ENTRY_AT_EXPECTED_EVALUATION_ORDER_FIELD,
            ENTRIES_WITH_CONFLICTS_FIELD, ENTRY_VIOLATION_REASONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final EntryDescription expectedEntry;

    private final String expectedEvaluationOrder;

    private final String actualEvaluationOrder;

    private final EntryDescription entryAtExpectedEvaluationOrder;

    private final List<EntryDescription> entriesWithConflicts;

    private final List<String> entryViolationReasons;

    private EntryViolation(BuilderImpl builder) {
        this.expectedEntry = builder.expectedEntry;
        this.expectedEvaluationOrder = builder.expectedEvaluationOrder;
        this.actualEvaluationOrder = builder.actualEvaluationOrder;
        this.entryAtExpectedEvaluationOrder = builder.entryAtExpectedEvaluationOrder;
        this.entriesWithConflicts = builder.entriesWithConflicts;
        this.entryViolationReasons = builder.entryViolationReasons;
    }

    /**
     * <p>
     * The Firewall Manager-managed network ACL entry that is involved in the entry violation.
     * </p>
     * 
     * @return The Firewall Manager-managed network ACL entry that is involved in the entry violation.
     */
    public final EntryDescription expectedEntry() {
        return expectedEntry;
    }

    /**
     * <p>
     * The evaluation location within the ordered list of entries where the <code>ExpectedEntry</code> should be,
     * according to the network ACL policy specifications.
     * </p>
     * 
     * @return The evaluation location within the ordered list of entries where the <code>ExpectedEntry</code> should
     *         be, according to the network ACL policy specifications.
     */
    public final String expectedEvaluationOrder() {
        return expectedEvaluationOrder;
    }

    /**
     * <p>
     * The evaluation location within the ordered list of entries where the <code>ExpectedEntry</code> is currently
     * located.
     * </p>
     * 
     * @return The evaluation location within the ordered list of entries where the <code>ExpectedEntry</code> is
     *         currently located.
     */
    public final String actualEvaluationOrder() {
        return actualEvaluationOrder;
    }

    /**
     * <p>
     * The entry that's currently in the <code>ExpectedEvaluationOrder</code> location, in place of the expected entry.
     * </p>
     * 
     * @return The entry that's currently in the <code>ExpectedEvaluationOrder</code> location, in place of the expected
     *         entry.
     */
    public final EntryDescription entryAtExpectedEvaluationOrder() {
        return entryAtExpectedEvaluationOrder;
    }

    /**
     * For responses, this returns true if the service returned a value for the EntriesWithConflicts property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEntriesWithConflicts() {
        return entriesWithConflicts != null && !(entriesWithConflicts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of entries that are in conflict with <code>ExpectedEntry</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntriesWithConflicts} method.
     * </p>
     * 
     * @return The list of entries that are in conflict with <code>ExpectedEntry</code>.
     */
    public final List<EntryDescription> entriesWithConflicts() {
        return entriesWithConflicts;
    }

    /**
     * <p>
     * Descriptions of the violations that Firewall Manager found for these entries.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntryViolationReasons} method.
     * </p>
     * 
     * @return Descriptions of the violations that Firewall Manager found for these entries.
     */
    public final List<EntryViolationReason> entryViolationReasons() {
        return EntryViolationReasonsCopier.copyStringToEnum(entryViolationReasons);
    }

    /**
     * For responses, this returns true if the service returned a value for the EntryViolationReasons property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasEntryViolationReasons() {
        return entryViolationReasons != null && !(entryViolationReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Descriptions of the violations that Firewall Manager found for these entries.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntryViolationReasons} method.
     * </p>
     * 
     * @return Descriptions of the violations that Firewall Manager found for these entries.
     */
    public final List<String> entryViolationReasonsAsStrings() {
        return entryViolationReasons;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(expectedEntry());
        hashCode = 31 * hashCode + Objects.hashCode(expectedEvaluationOrder());
        hashCode = 31 * hashCode + Objects.hashCode(actualEvaluationOrder());
        hashCode = 31 * hashCode + Objects.hashCode(entryAtExpectedEvaluationOrder());
        hashCode = 31 * hashCode + Objects.hashCode(hasEntriesWithConflicts() ? entriesWithConflicts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEntryViolationReasons() ? entryViolationReasonsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EntryViolation)) {
            return false;
        }
        EntryViolation other = (EntryViolation) obj;
        return Objects.equals(expectedEntry(), other.expectedEntry())
                && Objects.equals(expectedEvaluationOrder(), other.expectedEvaluationOrder())
                && Objects.equals(actualEvaluationOrder(), other.actualEvaluationOrder())
                && Objects.equals(entryAtExpectedEvaluationOrder(), other.entryAtExpectedEvaluationOrder())
                && hasEntriesWithConflicts() == other.hasEntriesWithConflicts()
                && Objects.equals(entriesWithConflicts(), other.entriesWithConflicts())
                && hasEntryViolationReasons() == other.hasEntryViolationReasons()
                && Objects.equals(entryViolationReasonsAsStrings(), other.entryViolationReasonsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EntryViolation").add("ExpectedEntry", expectedEntry())
                .add("ExpectedEvaluationOrder", expectedEvaluationOrder()).add("ActualEvaluationOrder", actualEvaluationOrder())
                .add("EntryAtExpectedEvaluationOrder", entryAtExpectedEvaluationOrder())
                .add("EntriesWithConflicts", hasEntriesWithConflicts() ? entriesWithConflicts() : null)
                .add("EntryViolationReasons", hasEntryViolationReasons() ? entryViolationReasonsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ExpectedEntry":
            return Optional.ofNullable(clazz.cast(expectedEntry()));
        case "ExpectedEvaluationOrder":
            return Optional.ofNullable(clazz.cast(expectedEvaluationOrder()));
        case "ActualEvaluationOrder":
            return Optional.ofNullable(clazz.cast(actualEvaluationOrder()));
        case "EntryAtExpectedEvaluationOrder":
            return Optional.ofNullable(clazz.cast(entryAtExpectedEvaluationOrder()));
        case "EntriesWithConflicts":
            return Optional.ofNullable(clazz.cast(entriesWithConflicts()));
        case "EntryViolationReasons":
            return Optional.ofNullable(clazz.cast(entryViolationReasonsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ExpectedEntry", EXPECTED_ENTRY_FIELD);
        map.put("ExpectedEvaluationOrder", EXPECTED_EVALUATION_ORDER_FIELD);
        map.put("ActualEvaluationOrder", ACTUAL_EVALUATION_ORDER_FIELD);
        map.put("EntryAtExpectedEvaluationOrder", ENTRY_AT_EXPECTED_EVALUATION_ORDER_FIELD);
        map.put("EntriesWithConflicts", ENTRIES_WITH_CONFLICTS_FIELD);
        map.put("EntryViolationReasons", ENTRY_VIOLATION_REASONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EntryViolation, T> g) {
        return obj -> g.apply((EntryViolation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EntryViolation> {
        /**
         * <p>
         * The Firewall Manager-managed network ACL entry that is involved in the entry violation.
         * </p>
         * 
         * @param expectedEntry
         *        The Firewall Manager-managed network ACL entry that is involved in the entry violation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expectedEntry(EntryDescription expectedEntry);

        /**
         * <p>
         * The Firewall Manager-managed network ACL entry that is involved in the entry violation.
         * </p>
         * This is a convenience method that creates an instance of the {@link EntryDescription.Builder} avoiding the
         * need to create one manually via {@link EntryDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EntryDescription.Builder#build()} is called immediately and its
         * result is passed to {@link #expectedEntry(EntryDescription)}.
         * 
         * @param expectedEntry
         *        a consumer that will call methods on {@link EntryDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #expectedEntry(EntryDescription)
         */
        default Builder expectedEntry(Consumer<EntryDescription.Builder> expectedEntry) {
            return expectedEntry(EntryDescription.builder().applyMutation(expectedEntry).build());
        }

        /**
         * <p>
         * The evaluation location within the ordered list of entries where the <code>ExpectedEntry</code> should be,
         * according to the network ACL policy specifications.
         * </p>
         * 
         * @param expectedEvaluationOrder
         *        The evaluation location within the ordered list of entries where the <code>ExpectedEntry</code> should
         *        be, according to the network ACL policy specifications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expectedEvaluationOrder(String expectedEvaluationOrder);

        /**
         * <p>
         * The evaluation location within the ordered list of entries where the <code>ExpectedEntry</code> is currently
         * located.
         * </p>
         * 
         * @param actualEvaluationOrder
         *        The evaluation location within the ordered list of entries where the <code>ExpectedEntry</code> is
         *        currently located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actualEvaluationOrder(String actualEvaluationOrder);

        /**
         * <p>
         * The entry that's currently in the <code>ExpectedEvaluationOrder</code> location, in place of the expected
         * entry.
         * </p>
         * 
         * @param entryAtExpectedEvaluationOrder
         *        The entry that's currently in the <code>ExpectedEvaluationOrder</code> location, in place of the
         *        expected entry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryAtExpectedEvaluationOrder(EntryDescription entryAtExpectedEvaluationOrder);

        /**
         * <p>
         * The entry that's currently in the <code>ExpectedEvaluationOrder</code> location, in place of the expected
         * entry.
         * </p>
         * This is a convenience method that creates an instance of the {@link EntryDescription.Builder} avoiding the
         * need to create one manually via {@link EntryDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EntryDescription.Builder#build()} is called immediately and its
         * result is passed to {@link #entryAtExpectedEvaluationOrder(EntryDescription)}.
         * 
         * @param entryAtExpectedEvaluationOrder
         *        a consumer that will call methods on {@link EntryDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #entryAtExpectedEvaluationOrder(EntryDescription)
         */
        default Builder entryAtExpectedEvaluationOrder(Consumer<EntryDescription.Builder> entryAtExpectedEvaluationOrder) {
            return entryAtExpectedEvaluationOrder(EntryDescription.builder().applyMutation(entryAtExpectedEvaluationOrder)
                    .build());
        }

        /**
         * <p>
         * The list of entries that are in conflict with <code>ExpectedEntry</code>.
         * </p>
         * 
         * @param entriesWithConflicts
         *        The list of entries that are in conflict with <code>ExpectedEntry</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entriesWithConflicts(Collection<EntryDescription> entriesWithConflicts);

        /**
         * <p>
         * The list of entries that are in conflict with <code>ExpectedEntry</code>.
         * </p>
         * 
         * @param entriesWithConflicts
         *        The list of entries that are in conflict with <code>ExpectedEntry</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entriesWithConflicts(EntryDescription... entriesWithConflicts);

        /**
         * <p>
         * The list of entries that are in conflict with <code>ExpectedEntry</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fms.model.EntryDescription.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.fms.model.EntryDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.fms.model.EntryDescription.Builder#build()} is called immediately and
         * its result is passed to {@link #entriesWithConflicts(List<EntryDescription>)}.
         * 
         * @param entriesWithConflicts
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.fms.model.EntryDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #entriesWithConflicts(java.util.Collection<EntryDescription>)
         */
        Builder entriesWithConflicts(Consumer<EntryDescription.Builder>... entriesWithConflicts);

        /**
         * <p>
         * Descriptions of the violations that Firewall Manager found for these entries.
         * </p>
         * 
         * @param entryViolationReasons
         *        Descriptions of the violations that Firewall Manager found for these entries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryViolationReasonsWithStrings(Collection<String> entryViolationReasons);

        /**
         * <p>
         * Descriptions of the violations that Firewall Manager found for these entries.
         * </p>
         * 
         * @param entryViolationReasons
         *        Descriptions of the violations that Firewall Manager found for these entries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryViolationReasonsWithStrings(String... entryViolationReasons);

        /**
         * <p>
         * Descriptions of the violations that Firewall Manager found for these entries.
         * </p>
         * 
         * @param entryViolationReasons
         *        Descriptions of the violations that Firewall Manager found for these entries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryViolationReasons(Collection<EntryViolationReason> entryViolationReasons);

        /**
         * <p>
         * Descriptions of the violations that Firewall Manager found for these entries.
         * </p>
         * 
         * @param entryViolationReasons
         *        Descriptions of the violations that Firewall Manager found for these entries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryViolationReasons(EntryViolationReason... entryViolationReasons);
    }

    static final class BuilderImpl implements Builder {
        private EntryDescription expectedEntry;

        private String expectedEvaluationOrder;

        private String actualEvaluationOrder;

        private EntryDescription entryAtExpectedEvaluationOrder;

        private List<EntryDescription> entriesWithConflicts = DefaultSdkAutoConstructList.getInstance();

        private List<String> entryViolationReasons = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EntryViolation model) {
            expectedEntry(model.expectedEntry);
            expectedEvaluationOrder(model.expectedEvaluationOrder);
            actualEvaluationOrder(model.actualEvaluationOrder);
            entryAtExpectedEvaluationOrder(model.entryAtExpectedEvaluationOrder);
            entriesWithConflicts(model.entriesWithConflicts);
            entryViolationReasonsWithStrings(model.entryViolationReasons);
        }

        public final EntryDescription.Builder getExpectedEntry() {
            return expectedEntry != null ? expectedEntry.toBuilder() : null;
        }

        public final void setExpectedEntry(EntryDescription.BuilderImpl expectedEntry) {
            this.expectedEntry = expectedEntry != null ? expectedEntry.build() : null;
        }

        @Override
        public final Builder expectedEntry(EntryDescription expectedEntry) {
            this.expectedEntry = expectedEntry;
            return this;
        }

        public final String getExpectedEvaluationOrder() {
            return expectedEvaluationOrder;
        }

        public final void setExpectedEvaluationOrder(String expectedEvaluationOrder) {
            this.expectedEvaluationOrder = expectedEvaluationOrder;
        }

        @Override
        public final Builder expectedEvaluationOrder(String expectedEvaluationOrder) {
            this.expectedEvaluationOrder = expectedEvaluationOrder;
            return this;
        }

        public final String getActualEvaluationOrder() {
            return actualEvaluationOrder;
        }

        public final void setActualEvaluationOrder(String actualEvaluationOrder) {
            this.actualEvaluationOrder = actualEvaluationOrder;
        }

        @Override
        public final Builder actualEvaluationOrder(String actualEvaluationOrder) {
            this.actualEvaluationOrder = actualEvaluationOrder;
            return this;
        }

        public final EntryDescription.Builder getEntryAtExpectedEvaluationOrder() {
            return entryAtExpectedEvaluationOrder != null ? entryAtExpectedEvaluationOrder.toBuilder() : null;
        }

        public final void setEntryAtExpectedEvaluationOrder(EntryDescription.BuilderImpl entryAtExpectedEvaluationOrder) {
            this.entryAtExpectedEvaluationOrder = entryAtExpectedEvaluationOrder != null ? entryAtExpectedEvaluationOrder.build()
                    : null;
        }

        @Override
        public final Builder entryAtExpectedEvaluationOrder(EntryDescription entryAtExpectedEvaluationOrder) {
            this.entryAtExpectedEvaluationOrder = entryAtExpectedEvaluationOrder;
            return this;
        }

        public final List<EntryDescription.Builder> getEntriesWithConflicts() {
            List<EntryDescription.Builder> result = EntriesWithConflictsCopier.copyToBuilder(this.entriesWithConflicts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEntriesWithConflicts(Collection<EntryDescription.BuilderImpl> entriesWithConflicts) {
            this.entriesWithConflicts = EntriesWithConflictsCopier.copyFromBuilder(entriesWithConflicts);
        }

        @Override
        public final Builder entriesWithConflicts(Collection<EntryDescription> entriesWithConflicts) {
            this.entriesWithConflicts = EntriesWithConflictsCopier.copy(entriesWithConflicts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entriesWithConflicts(EntryDescription... entriesWithConflicts) {
            entriesWithConflicts(Arrays.asList(entriesWithConflicts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entriesWithConflicts(Consumer<EntryDescription.Builder>... entriesWithConflicts) {
            entriesWithConflicts(Stream.of(entriesWithConflicts).map(c -> EntryDescription.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getEntryViolationReasons() {
            if (entryViolationReasons instanceof SdkAutoConstructList) {
                return null;
            }
            return entryViolationReasons;
        }

        public final void setEntryViolationReasons(Collection<String> entryViolationReasons) {
            this.entryViolationReasons = EntryViolationReasonsCopier.copy(entryViolationReasons);
        }

        @Override
        public final Builder entryViolationReasonsWithStrings(Collection<String> entryViolationReasons) {
            this.entryViolationReasons = EntryViolationReasonsCopier.copy(entryViolationReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entryViolationReasonsWithStrings(String... entryViolationReasons) {
            entryViolationReasonsWithStrings(Arrays.asList(entryViolationReasons));
            return this;
        }

        @Override
        public final Builder entryViolationReasons(Collection<EntryViolationReason> entryViolationReasons) {
            this.entryViolationReasons = EntryViolationReasonsCopier.copyEnumToString(entryViolationReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entryViolationReasons(EntryViolationReason... entryViolationReasons) {
            entryViolationReasons(Arrays.asList(entryViolationReasons));
            return this;
        }

        @Override
        public EntryViolation build() {
            return new EntryViolation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
