/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fms.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details of the resource that is not protected by the policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComplianceViolator implements SdkPojo, Serializable,
        ToCopyableBuilder<ComplianceViolator.Builder, ComplianceViolator> {
    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceId").getter(getter(ComplianceViolator::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<String> VIOLATION_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ViolationReason").getter(getter(ComplianceViolator::violationReasonAsString))
            .setter(setter(Builder::violationReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ViolationReason").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(ComplianceViolator::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<Map<String, String>> METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Metadata")
            .getter(getter(ComplianceViolator::metadata))
            .setter(setter(Builder::metadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_ID_FIELD,
            VIOLATION_REASON_FIELD, RESOURCE_TYPE_FIELD, METADATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String resourceId;

    private final String violationReason;

    private final String resourceType;

    private final Map<String, String> metadata;

    private ComplianceViolator(BuilderImpl builder) {
        this.resourceId = builder.resourceId;
        this.violationReason = builder.violationReason;
        this.resourceType = builder.resourceType;
        this.metadata = builder.metadata;
    }

    /**
     * <p>
     * The resource ID.
     * </p>
     * 
     * @return The resource ID.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The reason that the resource is not protected by the policy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #violationReason}
     * will return {@link ViolationReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #violationReasonAsString}.
     * </p>
     * 
     * @return The reason that the resource is not protected by the policy.
     * @see ViolationReason
     */
    public final ViolationReason violationReason() {
        return ViolationReason.fromValue(violationReason);
    }

    /**
     * <p>
     * The reason that the resource is not protected by the policy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #violationReason}
     * will return {@link ViolationReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #violationReasonAsString}.
     * </p>
     * 
     * @return The reason that the resource is not protected by the policy.
     * @see ViolationReason
     */
    public final String violationReasonAsString() {
        return violationReason;
    }

    /**
     * <p>
     * The resource type. This is in the format shown in the <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">Amazon
     * Web Services Resource Types Reference</a>. For example: <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>,
     * <code>AWS::CloudFront::Distribution</code>, or <code>AWS::NetworkFirewall::FirewallPolicy</code>.
     * </p>
     * 
     * @return The resource type. This is in the format shown in the <a
     *         href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
     *         >Amazon Web Services Resource Types Reference</a>. For example:
     *         <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>, <code>AWS::CloudFront::Distribution</code>, or
     *         <code>AWS::NetworkFirewall::FirewallPolicy</code>.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Metadata property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMetadata() {
        return metadata != null && !(metadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Metadata about the resource that doesn't comply with the policy scope.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetadata} method.
     * </p>
     * 
     * @return Metadata about the resource that doesn't comply with the policy scope.
     */
    public final Map<String, String> metadata() {
        return metadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(violationReasonAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetadata() ? metadata() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComplianceViolator)) {
            return false;
        }
        ComplianceViolator other = (ComplianceViolator) obj;
        return Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(violationReasonAsString(), other.violationReasonAsString())
                && Objects.equals(resourceType(), other.resourceType()) && hasMetadata() == other.hasMetadata()
                && Objects.equals(metadata(), other.metadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ComplianceViolator").add("ResourceId", resourceId())
                .add("ViolationReason", violationReasonAsString()).add("ResourceType", resourceType())
                .add("Metadata", hasMetadata() ? metadata() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "ViolationReason":
            return Optional.ofNullable(clazz.cast(violationReasonAsString()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "Metadata":
            return Optional.ofNullable(clazz.cast(metadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ResourceId", RESOURCE_ID_FIELD);
        map.put("ViolationReason", VIOLATION_REASON_FIELD);
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("Metadata", METADATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ComplianceViolator, T> g) {
        return obj -> g.apply((ComplianceViolator) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComplianceViolator> {
        /**
         * <p>
         * The resource ID.
         * </p>
         * 
         * @param resourceId
         *        The resource ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The reason that the resource is not protected by the policy.
         * </p>
         * 
         * @param violationReason
         *        The reason that the resource is not protected by the policy.
         * @see ViolationReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ViolationReason
         */
        Builder violationReason(String violationReason);

        /**
         * <p>
         * The reason that the resource is not protected by the policy.
         * </p>
         * 
         * @param violationReason
         *        The reason that the resource is not protected by the policy.
         * @see ViolationReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ViolationReason
         */
        Builder violationReason(ViolationReason violationReason);

        /**
         * <p>
         * The resource type. This is in the format shown in the <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         * >Amazon Web Services Resource Types Reference</a>. For example:
         * <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>, <code>AWS::CloudFront::Distribution</code>, or
         * <code>AWS::NetworkFirewall::FirewallPolicy</code>.
         * </p>
         * 
         * @param resourceType
         *        The resource type. This is in the format shown in the <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         *        >Amazon Web Services Resource Types Reference</a>. For example:
         *        <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>, <code>AWS::CloudFront::Distribution</code>, or
         *        <code>AWS::NetworkFirewall::FirewallPolicy</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * Metadata about the resource that doesn't comply with the policy scope.
         * </p>
         * 
         * @param metadata
         *        Metadata about the resource that doesn't comply with the policy scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metadata(Map<String, String> metadata);
    }

    static final class BuilderImpl implements Builder {
        private String resourceId;

        private String violationReason;

        private String resourceType;

        private Map<String, String> metadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ComplianceViolator model) {
            resourceId(model.resourceId);
            violationReason(model.violationReason);
            resourceType(model.resourceType);
            metadata(model.metadata);
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getViolationReason() {
            return violationReason;
        }

        public final void setViolationReason(String violationReason) {
            this.violationReason = violationReason;
        }

        @Override
        public final Builder violationReason(String violationReason) {
            this.violationReason = violationReason;
            return this;
        }

        @Override
        public final Builder violationReason(ViolationReason violationReason) {
            this.violationReason(violationReason == null ? null : violationReason.toString());
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final Map<String, String> getMetadata() {
            if (metadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return metadata;
        }

        public final void setMetadata(Map<String, String> metadata) {
            this.metadata = ComplianceViolatorMetadataCopier.copy(metadata);
        }

        @Override
        public final Builder metadata(Map<String, String> metadata) {
            this.metadata = ComplianceViolatorMetadataCopier.copy(metadata);
            return this;
        }

        @Override
        public ComplianceViolator build() {
            return new ComplianceViolator(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
