/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchevents.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutEventsRequestEntryMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an event to be submitted.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutEventsRequestEntry implements StructuredPojo,
        ToCopyableBuilder<PutEventsRequestEntry.Builder, PutEventsRequestEntry> {
    private final Instant time;

    private final String source;

    private final List<String> resources;

    private final String detailType;

    private final String detail;

    private PutEventsRequestEntry(BuilderImpl builder) {
        this.time = builder.time;
        this.source = builder.source;
        this.resources = builder.resources;
        this.detailType = builder.detailType;
        this.detail = builder.detail;
    }

    /**
     * <p>
     * The timestamp of the event, per <a href="https://www.rfc-editor.org/rfc/rfc3339.txt">RFC3339</a>. If no timestamp
     * is provided, the timestamp of the <a>PutEvents</a> call is used.
     * </p>
     * 
     * @return The timestamp of the event, per <a href="https://www.rfc-editor.org/rfc/rfc3339.txt">RFC3339</a>. If no
     *         timestamp is provided, the timestamp of the <a>PutEvents</a> call is used.
     */
    public Instant time() {
        return time;
    }

    /**
     * <p>
     * The source of the event.
     * </p>
     * 
     * @return The source of the event.
     */
    public String source() {
        return source;
    }

    /**
     * <p>
     * AWS resources, identified by Amazon Resource Name (ARN), which the event primarily concerns. Any number,
     * including zero, may be present.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return AWS resources, identified by Amazon Resource Name (ARN), which the event primarily concerns. Any number,
     *         including zero, may be present.
     */
    public List<String> resources() {
        return resources;
    }

    /**
     * <p>
     * Free-form string used to decide what fields to expect in the event detail.
     * </p>
     * 
     * @return Free-form string used to decide what fields to expect in the event detail.
     */
    public String detailType() {
        return detailType;
    }

    /**
     * <p>
     * In the JSON sense, an object containing fields, which may also contain nested subobjects. No constraints are
     * imposed on its contents.
     * </p>
     * 
     * @return In the JSON sense, an object containing fields, which may also contain nested subobjects. No constraints
     *         are imposed on its contents.
     */
    public String detail() {
        return detail;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(time());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(resources());
        hashCode = 31 * hashCode + Objects.hashCode(detailType());
        hashCode = 31 * hashCode + Objects.hashCode(detail());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutEventsRequestEntry)) {
            return false;
        }
        PutEventsRequestEntry other = (PutEventsRequestEntry) obj;
        return Objects.equals(time(), other.time()) && Objects.equals(source(), other.source())
                && Objects.equals(resources(), other.resources()) && Objects.equals(detailType(), other.detailType())
                && Objects.equals(detail(), other.detail());
    }

    @Override
    public String toString() {
        return ToString.builder("PutEventsRequestEntry").add("Time", time()).add("Source", source())
                .add("Resources", resources()).add("DetailType", detailType()).add("Detail", detail()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Time":
            return Optional.ofNullable(clazz.cast(time()));
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "DetailType":
            return Optional.ofNullable(clazz.cast(detailType()));
        case "Detail":
            return Optional.ofNullable(clazz.cast(detail()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        PutEventsRequestEntryMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, PutEventsRequestEntry> {
        /**
         * <p>
         * The timestamp of the event, per <a href="https://www.rfc-editor.org/rfc/rfc3339.txt">RFC3339</a>. If no
         * timestamp is provided, the timestamp of the <a>PutEvents</a> call is used.
         * </p>
         * 
         * @param time
         *        The timestamp of the event, per <a href="https://www.rfc-editor.org/rfc/rfc3339.txt">RFC3339</a>. If
         *        no timestamp is provided, the timestamp of the <a>PutEvents</a> call is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder time(Instant time);

        /**
         * <p>
         * The source of the event.
         * </p>
         * 
         * @param source
         *        The source of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * AWS resources, identified by Amazon Resource Name (ARN), which the event primarily concerns. Any number,
         * including zero, may be present.
         * </p>
         * 
         * @param resources
         *        AWS resources, identified by Amazon Resource Name (ARN), which the event primarily concerns. Any
         *        number, including zero, may be present.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<String> resources);

        /**
         * <p>
         * AWS resources, identified by Amazon Resource Name (ARN), which the event primarily concerns. Any number,
         * including zero, may be present.
         * </p>
         * 
         * @param resources
         *        AWS resources, identified by Amazon Resource Name (ARN), which the event primarily concerns. Any
         *        number, including zero, may be present.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(String... resources);

        /**
         * <p>
         * Free-form string used to decide what fields to expect in the event detail.
         * </p>
         * 
         * @param detailType
         *        Free-form string used to decide what fields to expect in the event detail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detailType(String detailType);

        /**
         * <p>
         * In the JSON sense, an object containing fields, which may also contain nested subobjects. No constraints are
         * imposed on its contents.
         * </p>
         * 
         * @param detail
         *        In the JSON sense, an object containing fields, which may also contain nested subobjects. No
         *        constraints are imposed on its contents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detail(String detail);
    }

    static final class BuilderImpl implements Builder {
        private Instant time;

        private String source;

        private List<String> resources = DefaultSdkAutoConstructList.getInstance();

        private String detailType;

        private String detail;

        private BuilderImpl() {
        }

        private BuilderImpl(PutEventsRequestEntry model) {
            time(model.time);
            source(model.source);
            resources(model.resources);
            detailType(model.detailType);
            detail(model.detail);
        }

        public final Instant getTime() {
            return time;
        }

        @Override
        public final Builder time(Instant time) {
            this.time = time;
            return this;
        }

        public final void setTime(Instant time) {
            this.time = time;
        }

        public final String getSource() {
            return source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        public final Collection<String> getResources() {
            return resources;
        }

        @Override
        public final Builder resources(Collection<String> resources) {
            this.resources = EventResourceListCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(String... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        public final void setResources(Collection<String> resources) {
            this.resources = EventResourceListCopier.copy(resources);
        }

        public final String getDetailType() {
            return detailType;
        }

        @Override
        public final Builder detailType(String detailType) {
            this.detailType = detailType;
            return this;
        }

        public final void setDetailType(String detailType) {
            this.detailType = detailType;
        }

        public final String getDetail() {
            return detail;
        }

        @Override
        public final Builder detail(String detail) {
            this.detail = detail;
            return this;
        }

        public final void setDetail(String detail) {
            this.detail = detail;
        }

        @Override
        public PutEventsRequestEntry build() {
            return new PutEventsRequestEntry(this);
        }
    }
}
