/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsErrorDetails;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * At least one of your specified cluster subnets is in an Availability Zone that does not support Amazon EKS. The
 * exception output specifies the supported Availability Zones for your account, from which you can choose subnets for
 * your cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UnsupportedAvailabilityZoneException extends EksException implements
        ToCopyableBuilder<UnsupportedAvailabilityZoneException.Builder, UnsupportedAvailabilityZoneException> {
    private static final SdkField<String> CLUSTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UnsupportedAvailabilityZoneException::clusterName)).setter(setter(Builder::clusterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clusterName").build()).build();

    private static final SdkField<List<String>> VALID_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(UnsupportedAvailabilityZoneException::validZones))
            .setter(setter(Builder::validZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("validZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLUSTER_NAME_FIELD,
            VALID_ZONES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String clusterName;

    private final List<String> validZones;

    private UnsupportedAvailabilityZoneException(BuilderImpl builder) {
        super(builder);
        this.clusterName = builder.clusterName;
        this.validZones = builder.validZones;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    /**
     * <p>
     * The Amazon EKS cluster associated with the exception.
     * </p>
     * 
     * @return The Amazon EKS cluster associated with the exception.
     */
    public String clusterName() {
        return clusterName;
    }

    /**
     * <p>
     * The supported Availability Zones for your account. Choose subnets in these Availability Zones for your cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The supported Availability Zones for your account. Choose subnets in these Availability Zones for your
     *         cluster.
     */
    public List<String> validZones() {
        return validZones;
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UnsupportedAvailabilityZoneException, T> g) {
        return obj -> g.apply((UnsupportedAvailabilityZoneException) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UnsupportedAvailabilityZoneException>,
            EksException.Builder {
        /**
         * <p>
         * The Amazon EKS cluster associated with the exception.
         * </p>
         * 
         * @param clusterName
         *        The Amazon EKS cluster associated with the exception.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterName(String clusterName);

        /**
         * <p>
         * The supported Availability Zones for your account. Choose subnets in these Availability Zones for your
         * cluster.
         * </p>
         * 
         * @param validZones
         *        The supported Availability Zones for your account. Choose subnets in these Availability Zones for your
         *        cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validZones(Collection<String> validZones);

        /**
         * <p>
         * The supported Availability Zones for your account. Choose subnets in these Availability Zones for your
         * cluster.
         * </p>
         * 
         * @param validZones
         *        The supported Availability Zones for your account. Choose subnets in these Availability Zones for your
         *        cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validZones(String... validZones);

        @Override
        Builder awsErrorDetails(AwsErrorDetails awsErrorDetails);

        @Override
        Builder message(String message);

        @Override
        Builder requestId(String requestId);

        @Override
        Builder statusCode(int statusCode);

        @Override
        Builder cause(Throwable cause);
    }

    static final class BuilderImpl extends EksException.BuilderImpl implements Builder {
        private String clusterName;

        private List<String> validZones = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UnsupportedAvailabilityZoneException model) {
            super(model);
            clusterName(model.clusterName);
            validZones(model.validZones);
        }

        public final String getClusterName() {
            return clusterName;
        }

        @Override
        public final Builder clusterName(String clusterName) {
            this.clusterName = clusterName;
            return this;
        }

        public final void setClusterName(String clusterName) {
            this.clusterName = clusterName;
        }

        public final Collection<String> getValidZones() {
            return validZones;
        }

        @Override
        public final Builder validZones(Collection<String> validZones) {
            this.validZones = StringListCopier.copy(validZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validZones(String... validZones) {
            validZones(Arrays.asList(validZones));
            return this;
        }

        public final void setValidZones(Collection<String> validZones) {
            this.validZones = StringListCopier.copy(validZones);
        }

        @Override
        public BuilderImpl awsErrorDetails(AwsErrorDetails awsErrorDetails) {
            this.awsErrorDetails = awsErrorDetails;
            return this;
        }

        @Override
        public BuilderImpl message(String message) {
            this.message = message;
            return this;
        }

        @Override
        public BuilderImpl requestId(String requestId) {
            this.requestId = requestId;
            return this;
        }

        @Override
        public BuilderImpl statusCode(int statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        @Override
        public BuilderImpl cause(Throwable cause) {
            this.cause = cause;
            return this;
        }

        @Override
        public UnsupportedAvailabilityZoneException build() {
            return new UnsupportedAvailabilityZoneException(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
