/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.eks.EksClient;
import software.amazon.awssdk.services.eks.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.eks.model.DescribeAddonRequest;
import software.amazon.awssdk.services.eks.model.DescribeAddonResponse;
import software.amazon.awssdk.services.eks.model.DescribeClusterRequest;
import software.amazon.awssdk.services.eks.model.DescribeClusterResponse;
import software.amazon.awssdk.services.eks.model.DescribeFargateProfileRequest;
import software.amazon.awssdk.services.eks.model.DescribeFargateProfileResponse;
import software.amazon.awssdk.services.eks.model.DescribeNodegroupRequest;
import software.amazon.awssdk.services.eks.model.DescribeNodegroupResponse;
import software.amazon.awssdk.services.eks.model.EksRequest;
import software.amazon.awssdk.services.eks.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultEksWaiter implements EksWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final EksClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeClusterResponse> clusterActiveWaiter;

    private final Waiter<DescribeClusterResponse> clusterDeletedWaiter;

    private final Waiter<DescribeNodegroupResponse> nodegroupActiveWaiter;

    private final Waiter<DescribeNodegroupResponse> nodegroupDeletedWaiter;

    private final Waiter<DescribeAddonResponse> addonActiveWaiter;

    private final Waiter<DescribeAddonResponse> addonDeletedWaiter;

    private final Waiter<DescribeFargateProfileResponse> fargateProfileActiveWaiter;

    private final Waiter<DescribeFargateProfileResponse> fargateProfileDeletedWaiter;

    private DefaultEksWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = EksClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.clusterActiveWaiter = Waiter.builder(DescribeClusterResponse.class).acceptors(clusterActiveWaiterAcceptors())
                .overrideConfiguration(clusterActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.clusterDeletedWaiter = Waiter.builder(DescribeClusterResponse.class).acceptors(clusterDeletedWaiterAcceptors())
                .overrideConfiguration(clusterDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.nodegroupActiveWaiter = Waiter.builder(DescribeNodegroupResponse.class).acceptors(nodegroupActiveWaiterAcceptors())
                .overrideConfiguration(nodegroupActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.nodegroupDeletedWaiter = Waiter.builder(DescribeNodegroupResponse.class)
                .acceptors(nodegroupDeletedWaiterAcceptors())
                .overrideConfiguration(nodegroupDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.addonActiveWaiter = Waiter.builder(DescribeAddonResponse.class).acceptors(addonActiveWaiterAcceptors())
                .overrideConfiguration(addonActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.addonDeletedWaiter = Waiter.builder(DescribeAddonResponse.class).acceptors(addonDeletedWaiterAcceptors())
                .overrideConfiguration(addonDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.fargateProfileActiveWaiter = Waiter.builder(DescribeFargateProfileResponse.class)
                .acceptors(fargateProfileActiveWaiterAcceptors())
                .overrideConfiguration(fargateProfileActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.fargateProfileDeletedWaiter = Waiter.builder(DescribeFargateProfileResponse.class)
                .acceptors(fargateProfileDeletedWaiterAcceptors())
                .overrideConfiguration(fargateProfileDeletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeAddonResponse> waitUntilAddonActive(DescribeAddonRequest describeAddonRequest) {
        return addonActiveWaiter.run(() -> client.describeAddon(applyWaitersUserAgent(describeAddonRequest)));
    }

    @Override
    public WaiterResponse<DescribeAddonResponse> waitUntilAddonActive(DescribeAddonRequest describeAddonRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return addonActiveWaiter.run(() -> client.describeAddon(applyWaitersUserAgent(describeAddonRequest)),
                addonActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeAddonResponse> waitUntilAddonDeleted(DescribeAddonRequest describeAddonRequest) {
        return addonDeletedWaiter.run(() -> client.describeAddon(applyWaitersUserAgent(describeAddonRequest)));
    }

    @Override
    public WaiterResponse<DescribeAddonResponse> waitUntilAddonDeleted(DescribeAddonRequest describeAddonRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return addonDeletedWaiter.run(() -> client.describeAddon(applyWaitersUserAgent(describeAddonRequest)),
                addonDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterActive(DescribeClusterRequest describeClusterRequest) {
        return clusterActiveWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterActive(DescribeClusterRequest describeClusterRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return clusterActiveWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterDeleted(DescribeClusterRequest describeClusterRequest) {
        return clusterDeletedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterDeleted(DescribeClusterRequest describeClusterRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return clusterDeletedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeFargateProfileResponse> waitUntilFargateProfileActive(
            DescribeFargateProfileRequest describeFargateProfileRequest) {
        return fargateProfileActiveWaiter.run(() -> client
                .describeFargateProfile(applyWaitersUserAgent(describeFargateProfileRequest)));
    }

    @Override
    public WaiterResponse<DescribeFargateProfileResponse> waitUntilFargateProfileActive(
            DescribeFargateProfileRequest describeFargateProfileRequest, WaiterOverrideConfiguration overrideConfig) {
        return fargateProfileActiveWaiter.run(
                () -> client.describeFargateProfile(applyWaitersUserAgent(describeFargateProfileRequest)),
                fargateProfileActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeFargateProfileResponse> waitUntilFargateProfileDeleted(
            DescribeFargateProfileRequest describeFargateProfileRequest) {
        return fargateProfileDeletedWaiter.run(() -> client
                .describeFargateProfile(applyWaitersUserAgent(describeFargateProfileRequest)));
    }

    @Override
    public WaiterResponse<DescribeFargateProfileResponse> waitUntilFargateProfileDeleted(
            DescribeFargateProfileRequest describeFargateProfileRequest, WaiterOverrideConfiguration overrideConfig) {
        return fargateProfileDeletedWaiter.run(
                () -> client.describeFargateProfile(applyWaitersUserAgent(describeFargateProfileRequest)),
                fargateProfileDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeNodegroupResponse> waitUntilNodegroupActive(DescribeNodegroupRequest describeNodegroupRequest) {
        return nodegroupActiveWaiter.run(() -> client.describeNodegroup(applyWaitersUserAgent(describeNodegroupRequest)));
    }

    @Override
    public WaiterResponse<DescribeNodegroupResponse> waitUntilNodegroupActive(DescribeNodegroupRequest describeNodegroupRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return nodegroupActiveWaiter.run(() -> client.describeNodegroup(applyWaitersUserAgent(describeNodegroupRequest)),
                nodegroupActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeNodegroupResponse> waitUntilNodegroupDeleted(DescribeNodegroupRequest describeNodegroupRequest) {
        return nodegroupDeletedWaiter.run(() -> client.describeNodegroup(applyWaitersUserAgent(describeNodegroupRequest)));
    }

    @Override
    public WaiterResponse<DescribeNodegroupResponse> waitUntilNodegroupDeleted(DescribeNodegroupRequest describeNodegroupRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return nodegroupDeletedWaiter.run(() -> client.describeNodegroup(applyWaitersUserAgent(describeNodegroupRequest)),
                nodegroupDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("cluster").field("status").value(), "DELETING");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (cluster.status=DELETING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("cluster").field("status").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (cluster.status=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("cluster").field("status").value(), "ACTIVE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("cluster").field("status").value(), "ACTIVE");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (cluster.status=ACTIVE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("cluster").field("status").value(), "CREATING");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (cluster.status=CREATING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("cluster").field("status").value(), "PENDING");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (cluster.status=PENDING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNodegroupResponse>> nodegroupActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNodegroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("nodegroup").field("status").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (nodegroup.status=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("nodegroup").field("status").value(), "ACTIVE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNodegroupResponse>> nodegroupDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNodegroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("nodegroup").field("status").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (nodegroup.status=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAddonResponse>> addonActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAddonResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("addon").field("status").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (addon.status=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("addon").field("status").value(), "DEGRADED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (addon.status=DEGRADED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("addon").field("status").value(), "ACTIVE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAddonResponse>> addonDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAddonResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("addon").field("status").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (addon.status=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeFargateProfileResponse>> fargateProfileActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeFargateProfileResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("fargateProfile").field("status").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (fargateProfile.status=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("fargateProfile").field("status").value(), "ACTIVE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeFargateProfileResponse>> fargateProfileDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeFargateProfileResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("fargateProfile").field("status").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (fargateProfile.status=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration clusterActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration clusterDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration nodegroupActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(80);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration nodegroupDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration addonActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration addonDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration fargateProfileActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration fargateProfileDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static EksWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends EksRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements EksWaiter.Builder {
        private EksClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public EksWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public EksWaiter.Builder client(EksClient client) {
            this.client = client;
            return this;
        }

        public EksWaiter build() {
            return new DefaultEksWaiter(this);
        }
    }
}
