/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing an OpenID Connect (OIDC) configuration. Before associating an OIDC identity provider to your
 * cluster, review the considerations in <a
 * href="https://docs.aws.amazon.com/eks/latest/userguide/authenticate-oidc-identity-provider.html">Authenticating users
 * for your cluster from an OpenID Connect identity provider</a> in the <i>Amazon EKS User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OidcIdentityProviderConfigRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<OidcIdentityProviderConfigRequest.Builder, OidcIdentityProviderConfigRequest> {
    private static final SdkField<String> IDENTITY_PROVIDER_CONFIG_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("identityProviderConfigName")
            .getter(getter(OidcIdentityProviderConfigRequest::identityProviderConfigName))
            .setter(setter(Builder::identityProviderConfigName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identityProviderConfigName").build())
            .build();

    private static final SdkField<String> ISSUER_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("issuerUrl").getter(getter(OidcIdentityProviderConfigRequest::issuerUrl))
            .setter(setter(Builder::issuerUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("issuerUrl").build()).build();

    private static final SdkField<String> CLIENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clientId").getter(getter(OidcIdentityProviderConfigRequest::clientId)).setter(setter(Builder::clientId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientId").build()).build();

    private static final SdkField<String> USERNAME_CLAIM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("usernameClaim").getter(getter(OidcIdentityProviderConfigRequest::usernameClaim))
            .setter(setter(Builder::usernameClaim))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("usernameClaim").build()).build();

    private static final SdkField<String> USERNAME_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("usernamePrefix").getter(getter(OidcIdentityProviderConfigRequest::usernamePrefix))
            .setter(setter(Builder::usernamePrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("usernamePrefix").build()).build();

    private static final SdkField<String> GROUPS_CLAIM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("groupsClaim").getter(getter(OidcIdentityProviderConfigRequest::groupsClaim))
            .setter(setter(Builder::groupsClaim))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("groupsClaim").build()).build();

    private static final SdkField<String> GROUPS_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("groupsPrefix").getter(getter(OidcIdentityProviderConfigRequest::groupsPrefix))
            .setter(setter(Builder::groupsPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("groupsPrefix").build()).build();

    private static final SdkField<Map<String, String>> REQUIRED_CLAIMS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("requiredClaims")
            .getter(getter(OidcIdentityProviderConfigRequest::requiredClaims))
            .setter(setter(Builder::requiredClaims))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requiredClaims").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            IDENTITY_PROVIDER_CONFIG_NAME_FIELD, ISSUER_URL_FIELD, CLIENT_ID_FIELD, USERNAME_CLAIM_FIELD, USERNAME_PREFIX_FIELD,
            GROUPS_CLAIM_FIELD, GROUPS_PREFIX_FIELD, REQUIRED_CLAIMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String identityProviderConfigName;

    private final String issuerUrl;

    private final String clientId;

    private final String usernameClaim;

    private final String usernamePrefix;

    private final String groupsClaim;

    private final String groupsPrefix;

    private final Map<String, String> requiredClaims;

    private OidcIdentityProviderConfigRequest(BuilderImpl builder) {
        this.identityProviderConfigName = builder.identityProviderConfigName;
        this.issuerUrl = builder.issuerUrl;
        this.clientId = builder.clientId;
        this.usernameClaim = builder.usernameClaim;
        this.usernamePrefix = builder.usernamePrefix;
        this.groupsClaim = builder.groupsClaim;
        this.groupsPrefix = builder.groupsPrefix;
        this.requiredClaims = builder.requiredClaims;
    }

    /**
     * <p>
     * The name of the OIDC provider configuration.
     * </p>
     * 
     * @return The name of the OIDC provider configuration.
     */
    public final String identityProviderConfigName() {
        return identityProviderConfigName;
    }

    /**
     * <p>
     * The URL of the OpenID identity provider that allows the API server to discover public signing keys for verifying
     * tokens. The URL must begin with <code>https://</code> and should correspond to the <code>iss</code> claim in the
     * provider's OIDC ID tokens. Per the OIDC standard, path components are allowed but query parameters are not.
     * Typically the URL consists of only a hostname, like <code>https://server.example.org</code> or
     * <code>https://example.com</code>. This URL should point to the level below
     * <code>.well-known/openid-configuration</code> and must be publicly accessible over the internet.
     * </p>
     * 
     * @return The URL of the OpenID identity provider that allows the API server to discover public signing keys for
     *         verifying tokens. The URL must begin with <code>https://</code> and should correspond to the
     *         <code>iss</code> claim in the provider's OIDC ID tokens. Per the OIDC standard, path components are
     *         allowed but query parameters are not. Typically the URL consists of only a hostname, like
     *         <code>https://server.example.org</code> or <code>https://example.com</code>. This URL should point to the
     *         level below <code>.well-known/openid-configuration</code> and must be publicly accessible over the
     *         internet.
     */
    public final String issuerUrl() {
        return issuerUrl;
    }

    /**
     * <p>
     * This is also known as <i>audience</i>. The ID for the client application that makes authentication requests to
     * the OpenID identity provider.
     * </p>
     * 
     * @return This is also known as <i>audience</i>. The ID for the client application that makes authentication
     *         requests to the OpenID identity provider.
     */
    public final String clientId() {
        return clientId;
    }

    /**
     * <p>
     * The JSON Web Token (JWT) claim to use as the username. The default is <code>sub</code>, which is expected to be a
     * unique identifier of the end user. You can choose other claims, such as <code>email</code> or <code>name</code>,
     * depending on the OpenID identity provider. Claims other than <code>email</code> are prefixed with the issuer URL
     * to prevent naming clashes with other plug-ins.
     * </p>
     * 
     * @return The JSON Web Token (JWT) claim to use as the username. The default is <code>sub</code>, which is expected
     *         to be a unique identifier of the end user. You can choose other claims, such as <code>email</code> or
     *         <code>name</code>, depending on the OpenID identity provider. Claims other than <code>email</code> are
     *         prefixed with the issuer URL to prevent naming clashes with other plug-ins.
     */
    public final String usernameClaim() {
        return usernameClaim;
    }

    /**
     * <p>
     * The prefix that is prepended to username claims to prevent clashes with existing names. If you do not provide
     * this field, and <code>username</code> is a value other than <code>email</code>, the prefix defaults to
     * <code>issuerurl#</code>. You can use the value <code>-</code> to disable all prefixing.
     * </p>
     * 
     * @return The prefix that is prepended to username claims to prevent clashes with existing names. If you do not
     *         provide this field, and <code>username</code> is a value other than <code>email</code>, the prefix
     *         defaults to <code>issuerurl#</code>. You can use the value <code>-</code> to disable all prefixing.
     */
    public final String usernamePrefix() {
        return usernamePrefix;
    }

    /**
     * <p>
     * The JWT claim that the provider uses to return your groups.
     * </p>
     * 
     * @return The JWT claim that the provider uses to return your groups.
     */
    public final String groupsClaim() {
        return groupsClaim;
    }

    /**
     * <p>
     * The prefix that is prepended to group claims to prevent clashes with existing names (such as <code>system:</code>
     * groups). For example, the value<code> oidc:</code> will create group names like <code>oidc:engineering</code> and
     * <code>oidc:infra</code>.
     * </p>
     * 
     * @return The prefix that is prepended to group claims to prevent clashes with existing names (such as
     *         <code>system:</code> groups). For example, the value<code> oidc:</code> will create group names like
     *         <code>oidc:engineering</code> and <code>oidc:infra</code>.
     */
    public final String groupsPrefix() {
        return groupsPrefix;
    }

    /**
     * For responses, this returns true if the service returned a value for the RequiredClaims property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRequiredClaims() {
        return requiredClaims != null && !(requiredClaims instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The key value pairs that describe required claims in the identity token. If set, each claim is verified to be
     * present in the token with a matching value. For the maximum number of claims that you can require, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/service-quotas.html">Amazon EKS service quotas</a> in the
     * <i>Amazon EKS User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequiredClaims} method.
     * </p>
     * 
     * @return The key value pairs that describe required claims in the identity token. If set, each claim is verified
     *         to be present in the token with a matching value. For the maximum number of claims that you can require,
     *         see <a href="https://docs.aws.amazon.com/eks/latest/userguide/service-quotas.html">Amazon EKS service
     *         quotas</a> in the <i>Amazon EKS User Guide</i>.
     */
    public final Map<String, String> requiredClaims() {
        return requiredClaims;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(identityProviderConfigName());
        hashCode = 31 * hashCode + Objects.hashCode(issuerUrl());
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(usernameClaim());
        hashCode = 31 * hashCode + Objects.hashCode(usernamePrefix());
        hashCode = 31 * hashCode + Objects.hashCode(groupsClaim());
        hashCode = 31 * hashCode + Objects.hashCode(groupsPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(hasRequiredClaims() ? requiredClaims() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OidcIdentityProviderConfigRequest)) {
            return false;
        }
        OidcIdentityProviderConfigRequest other = (OidcIdentityProviderConfigRequest) obj;
        return Objects.equals(identityProviderConfigName(), other.identityProviderConfigName())
                && Objects.equals(issuerUrl(), other.issuerUrl()) && Objects.equals(clientId(), other.clientId())
                && Objects.equals(usernameClaim(), other.usernameClaim())
                && Objects.equals(usernamePrefix(), other.usernamePrefix()) && Objects.equals(groupsClaim(), other.groupsClaim())
                && Objects.equals(groupsPrefix(), other.groupsPrefix()) && hasRequiredClaims() == other.hasRequiredClaims()
                && Objects.equals(requiredClaims(), other.requiredClaims());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OidcIdentityProviderConfigRequest")
                .add("IdentityProviderConfigName", identityProviderConfigName()).add("IssuerUrl", issuerUrl())
                .add("ClientId", clientId()).add("UsernameClaim", usernameClaim()).add("UsernamePrefix", usernamePrefix())
                .add("GroupsClaim", groupsClaim()).add("GroupsPrefix", groupsPrefix())
                .add("RequiredClaims", hasRequiredClaims() ? requiredClaims() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "identityProviderConfigName":
            return Optional.ofNullable(clazz.cast(identityProviderConfigName()));
        case "issuerUrl":
            return Optional.ofNullable(clazz.cast(issuerUrl()));
        case "clientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "usernameClaim":
            return Optional.ofNullable(clazz.cast(usernameClaim()));
        case "usernamePrefix":
            return Optional.ofNullable(clazz.cast(usernamePrefix()));
        case "groupsClaim":
            return Optional.ofNullable(clazz.cast(groupsClaim()));
        case "groupsPrefix":
            return Optional.ofNullable(clazz.cast(groupsPrefix()));
        case "requiredClaims":
            return Optional.ofNullable(clazz.cast(requiredClaims()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OidcIdentityProviderConfigRequest, T> g) {
        return obj -> g.apply((OidcIdentityProviderConfigRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OidcIdentityProviderConfigRequest> {
        /**
         * <p>
         * The name of the OIDC provider configuration.
         * </p>
         * 
         * @param identityProviderConfigName
         *        The name of the OIDC provider configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identityProviderConfigName(String identityProviderConfigName);

        /**
         * <p>
         * The URL of the OpenID identity provider that allows the API server to discover public signing keys for
         * verifying tokens. The URL must begin with <code>https://</code> and should correspond to the <code>iss</code>
         * claim in the provider's OIDC ID tokens. Per the OIDC standard, path components are allowed but query
         * parameters are not. Typically the URL consists of only a hostname, like
         * <code>https://server.example.org</code> or <code>https://example.com</code>. This URL should point to the
         * level below <code>.well-known/openid-configuration</code> and must be publicly accessible over the internet.
         * </p>
         * 
         * @param issuerUrl
         *        The URL of the OpenID identity provider that allows the API server to discover public signing keys for
         *        verifying tokens. The URL must begin with <code>https://</code> and should correspond to the
         *        <code>iss</code> claim in the provider's OIDC ID tokens. Per the OIDC standard, path components are
         *        allowed but query parameters are not. Typically the URL consists of only a hostname, like
         *        <code>https://server.example.org</code> or <code>https://example.com</code>. This URL should point to
         *        the level below <code>.well-known/openid-configuration</code> and must be publicly accessible over the
         *        internet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issuerUrl(String issuerUrl);

        /**
         * <p>
         * This is also known as <i>audience</i>. The ID for the client application that makes authentication requests
         * to the OpenID identity provider.
         * </p>
         * 
         * @param clientId
         *        This is also known as <i>audience</i>. The ID for the client application that makes authentication
         *        requests to the OpenID identity provider.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * The JSON Web Token (JWT) claim to use as the username. The default is <code>sub</code>, which is expected to
         * be a unique identifier of the end user. You can choose other claims, such as <code>email</code> or
         * <code>name</code>, depending on the OpenID identity provider. Claims other than <code>email</code> are
         * prefixed with the issuer URL to prevent naming clashes with other plug-ins.
         * </p>
         * 
         * @param usernameClaim
         *        The JSON Web Token (JWT) claim to use as the username. The default is <code>sub</code>, which is
         *        expected to be a unique identifier of the end user. You can choose other claims, such as
         *        <code>email</code> or <code>name</code>, depending on the OpenID identity provider. Claims other than
         *        <code>email</code> are prefixed with the issuer URL to prevent naming clashes with other plug-ins.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usernameClaim(String usernameClaim);

        /**
         * <p>
         * The prefix that is prepended to username claims to prevent clashes with existing names. If you do not provide
         * this field, and <code>username</code> is a value other than <code>email</code>, the prefix defaults to
         * <code>issuerurl#</code>. You can use the value <code>-</code> to disable all prefixing.
         * </p>
         * 
         * @param usernamePrefix
         *        The prefix that is prepended to username claims to prevent clashes with existing names. If you do not
         *        provide this field, and <code>username</code> is a value other than <code>email</code>, the prefix
         *        defaults to <code>issuerurl#</code>. You can use the value <code>-</code> to disable all prefixing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usernamePrefix(String usernamePrefix);

        /**
         * <p>
         * The JWT claim that the provider uses to return your groups.
         * </p>
         * 
         * @param groupsClaim
         *        The JWT claim that the provider uses to return your groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupsClaim(String groupsClaim);

        /**
         * <p>
         * The prefix that is prepended to group claims to prevent clashes with existing names (such as
         * <code>system:</code> groups). For example, the value<code> oidc:</code> will create group names like
         * <code>oidc:engineering</code> and <code>oidc:infra</code>.
         * </p>
         * 
         * @param groupsPrefix
         *        The prefix that is prepended to group claims to prevent clashes with existing names (such as
         *        <code>system:</code> groups). For example, the value<code> oidc:</code> will create group names like
         *        <code>oidc:engineering</code> and <code>oidc:infra</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupsPrefix(String groupsPrefix);

        /**
         * <p>
         * The key value pairs that describe required claims in the identity token. If set, each claim is verified to be
         * present in the token with a matching value. For the maximum number of claims that you can require, see <a
         * href="https://docs.aws.amazon.com/eks/latest/userguide/service-quotas.html">Amazon EKS service quotas</a> in
         * the <i>Amazon EKS User Guide</i>.
         * </p>
         * 
         * @param requiredClaims
         *        The key value pairs that describe required claims in the identity token. If set, each claim is
         *        verified to be present in the token with a matching value. For the maximum number of claims that you
         *        can require, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/service-quotas.html">Amazon
         *        EKS service quotas</a> in the <i>Amazon EKS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requiredClaims(Map<String, String> requiredClaims);
    }

    static final class BuilderImpl implements Builder {
        private String identityProviderConfigName;

        private String issuerUrl;

        private String clientId;

        private String usernameClaim;

        private String usernamePrefix;

        private String groupsClaim;

        private String groupsPrefix;

        private Map<String, String> requiredClaims = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(OidcIdentityProviderConfigRequest model) {
            identityProviderConfigName(model.identityProviderConfigName);
            issuerUrl(model.issuerUrl);
            clientId(model.clientId);
            usernameClaim(model.usernameClaim);
            usernamePrefix(model.usernamePrefix);
            groupsClaim(model.groupsClaim);
            groupsPrefix(model.groupsPrefix);
            requiredClaims(model.requiredClaims);
        }

        public final String getIdentityProviderConfigName() {
            return identityProviderConfigName;
        }

        public final void setIdentityProviderConfigName(String identityProviderConfigName) {
            this.identityProviderConfigName = identityProviderConfigName;
        }

        @Override
        @Transient
        public final Builder identityProviderConfigName(String identityProviderConfigName) {
            this.identityProviderConfigName = identityProviderConfigName;
            return this;
        }

        public final String getIssuerUrl() {
            return issuerUrl;
        }

        public final void setIssuerUrl(String issuerUrl) {
            this.issuerUrl = issuerUrl;
        }

        @Override
        @Transient
        public final Builder issuerUrl(String issuerUrl) {
            this.issuerUrl = issuerUrl;
            return this;
        }

        public final String getClientId() {
            return clientId;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        @Override
        @Transient
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final String getUsernameClaim() {
            return usernameClaim;
        }

        public final void setUsernameClaim(String usernameClaim) {
            this.usernameClaim = usernameClaim;
        }

        @Override
        @Transient
        public final Builder usernameClaim(String usernameClaim) {
            this.usernameClaim = usernameClaim;
            return this;
        }

        public final String getUsernamePrefix() {
            return usernamePrefix;
        }

        public final void setUsernamePrefix(String usernamePrefix) {
            this.usernamePrefix = usernamePrefix;
        }

        @Override
        @Transient
        public final Builder usernamePrefix(String usernamePrefix) {
            this.usernamePrefix = usernamePrefix;
            return this;
        }

        public final String getGroupsClaim() {
            return groupsClaim;
        }

        public final void setGroupsClaim(String groupsClaim) {
            this.groupsClaim = groupsClaim;
        }

        @Override
        @Transient
        public final Builder groupsClaim(String groupsClaim) {
            this.groupsClaim = groupsClaim;
            return this;
        }

        public final String getGroupsPrefix() {
            return groupsPrefix;
        }

        public final void setGroupsPrefix(String groupsPrefix) {
            this.groupsPrefix = groupsPrefix;
        }

        @Override
        @Transient
        public final Builder groupsPrefix(String groupsPrefix) {
            this.groupsPrefix = groupsPrefix;
            return this;
        }

        public final Map<String, String> getRequiredClaims() {
            if (requiredClaims instanceof SdkAutoConstructMap) {
                return null;
            }
            return requiredClaims;
        }

        public final void setRequiredClaims(Map<String, String> requiredClaims) {
            this.requiredClaims = _requiredClaimsMapCopier.copy(requiredClaims);
        }

        @Override
        @Transient
        public final Builder requiredClaims(Map<String, String> requiredClaims) {
            this.requiredClaims = _requiredClaimsMapCopier.copy(requiredClaims);
            return this;
        }

        @Override
        public OidcIdentityProviderConfigRequest build() {
            return new OidcIdentityProviderConfigRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
