/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an add-on version.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AddonVersionInfo implements SdkPojo, Serializable,
        ToCopyableBuilder<AddonVersionInfo.Builder, AddonVersionInfo> {
    private static final SdkField<String> ADDON_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("addonVersion").getter(getter(AddonVersionInfo::addonVersion)).setter(setter(Builder::addonVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("addonVersion").build()).build();

    private static final SdkField<List<String>> ARCHITECTURE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("architecture")
            .getter(getter(AddonVersionInfo::architecture))
            .setter(setter(Builder::architecture))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("architecture").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Compatibility>> COMPATIBILITIES_FIELD = SdkField
            .<List<Compatibility>> builder(MarshallingType.LIST)
            .memberName("compatibilities")
            .getter(getter(AddonVersionInfo::compatibilities))
            .setter(setter(Builder::compatibilities))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("compatibilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Compatibility> builder(MarshallingType.SDK_POJO)
                                            .constructor(Compatibility::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ADDON_VERSION_FIELD,
            ARCHITECTURE_FIELD, COMPATIBILITIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String addonVersion;

    private final List<String> architecture;

    private final List<Compatibility> compatibilities;

    private AddonVersionInfo(BuilderImpl builder) {
        this.addonVersion = builder.addonVersion;
        this.architecture = builder.architecture;
        this.compatibilities = builder.compatibilities;
    }

    /**
     * <p>
     * The version of the add-on.
     * </p>
     * 
     * @return The version of the add-on.
     */
    public final String addonVersion() {
        return addonVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the Architecture property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasArchitecture() {
        return architecture != null && !(architecture instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The architectures that the version supports.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArchitecture} method.
     * </p>
     * 
     * @return The architectures that the version supports.
     */
    public final List<String> architecture() {
        return architecture;
    }

    /**
     * For responses, this returns true if the service returned a value for the Compatibilities property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCompatibilities() {
        return compatibilities != null && !(compatibilities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An object that represents the compatibilities of a version.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCompatibilities} method.
     * </p>
     * 
     * @return An object that represents the compatibilities of a version.
     */
    public final List<Compatibility> compatibilities() {
        return compatibilities;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(addonVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasArchitecture() ? architecture() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCompatibilities() ? compatibilities() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AddonVersionInfo)) {
            return false;
        }
        AddonVersionInfo other = (AddonVersionInfo) obj;
        return Objects.equals(addonVersion(), other.addonVersion()) && hasArchitecture() == other.hasArchitecture()
                && Objects.equals(architecture(), other.architecture()) && hasCompatibilities() == other.hasCompatibilities()
                && Objects.equals(compatibilities(), other.compatibilities());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AddonVersionInfo").add("AddonVersion", addonVersion())
                .add("Architecture", hasArchitecture() ? architecture() : null)
                .add("Compatibilities", hasCompatibilities() ? compatibilities() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "addonVersion":
            return Optional.ofNullable(clazz.cast(addonVersion()));
        case "architecture":
            return Optional.ofNullable(clazz.cast(architecture()));
        case "compatibilities":
            return Optional.ofNullable(clazz.cast(compatibilities()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AddonVersionInfo, T> g) {
        return obj -> g.apply((AddonVersionInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AddonVersionInfo> {
        /**
         * <p>
         * The version of the add-on.
         * </p>
         * 
         * @param addonVersion
         *        The version of the add-on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addonVersion(String addonVersion);

        /**
         * <p>
         * The architectures that the version supports.
         * </p>
         * 
         * @param architecture
         *        The architectures that the version supports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder architecture(Collection<String> architecture);

        /**
         * <p>
         * The architectures that the version supports.
         * </p>
         * 
         * @param architecture
         *        The architectures that the version supports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder architecture(String... architecture);

        /**
         * <p>
         * An object that represents the compatibilities of a version.
         * </p>
         * 
         * @param compatibilities
         *        An object that represents the compatibilities of a version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compatibilities(Collection<Compatibility> compatibilities);

        /**
         * <p>
         * An object that represents the compatibilities of a version.
         * </p>
         * 
         * @param compatibilities
         *        An object that represents the compatibilities of a version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compatibilities(Compatibility... compatibilities);

        /**
         * <p>
         * An object that represents the compatibilities of a version.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Compatibility>.Builder} avoiding the
         * need to create one manually via {@link List<Compatibility>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Compatibility>.Builder#build()} is called immediately and
         * its result is passed to {@link #compatibilities(List<Compatibility>)}.
         * 
         * @param compatibilities
         *        a consumer that will call methods on {@link List<Compatibility>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #compatibilities(List<Compatibility>)
         */
        Builder compatibilities(Consumer<Compatibility.Builder>... compatibilities);
    }

    static final class BuilderImpl implements Builder {
        private String addonVersion;

        private List<String> architecture = DefaultSdkAutoConstructList.getInstance();

        private List<Compatibility> compatibilities = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AddonVersionInfo model) {
            addonVersion(model.addonVersion);
            architecture(model.architecture);
            compatibilities(model.compatibilities);
        }

        public final String getAddonVersion() {
            return addonVersion;
        }

        public final void setAddonVersion(String addonVersion) {
            this.addonVersion = addonVersion;
        }

        @Override
        public final Builder addonVersion(String addonVersion) {
            this.addonVersion = addonVersion;
            return this;
        }

        public final Collection<String> getArchitecture() {
            if (architecture instanceof SdkAutoConstructList) {
                return null;
            }
            return architecture;
        }

        public final void setArchitecture(Collection<String> architecture) {
            this.architecture = StringListCopier.copy(architecture);
        }

        @Override
        public final Builder architecture(Collection<String> architecture) {
            this.architecture = StringListCopier.copy(architecture);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder architecture(String... architecture) {
            architecture(Arrays.asList(architecture));
            return this;
        }

        public final List<Compatibility.Builder> getCompatibilities() {
            List<Compatibility.Builder> result = CompatibilitiesCopier.copyToBuilder(this.compatibilities);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCompatibilities(Collection<Compatibility.BuilderImpl> compatibilities) {
            this.compatibilities = CompatibilitiesCopier.copyFromBuilder(compatibilities);
        }

        @Override
        public final Builder compatibilities(Collection<Compatibility> compatibilities) {
            this.compatibilities = CompatibilitiesCopier.copy(compatibilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder compatibilities(Compatibility... compatibilities) {
            compatibilities(Arrays.asList(compatibilities));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder compatibilities(Consumer<Compatibility.Builder>... compatibilities) {
            compatibilities(Stream.of(compatibilities).map(c -> Compatibility.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public AddonVersionInfo build() {
            return new AddonVersionInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
