/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing the remote access configuration for the managed node group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RemoteAccessConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<RemoteAccessConfig.Builder, RemoteAccessConfig> {
    private static final SdkField<String> EC2_SSH_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ec2SshKey").getter(getter(RemoteAccessConfig::ec2SshKey)).setter(setter(Builder::ec2SshKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ec2SshKey").build()).build();

    private static final SdkField<List<String>> SOURCE_SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("sourceSecurityGroups")
            .getter(getter(RemoteAccessConfig::sourceSecurityGroups))
            .setter(setter(Builder::sourceSecurityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceSecurityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EC2_SSH_KEY_FIELD,
            SOURCE_SECURITY_GROUPS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ec2SshKey;

    private final List<String> sourceSecurityGroups;

    private RemoteAccessConfig(BuilderImpl builder) {
        this.ec2SshKey = builder.ec2SshKey;
        this.sourceSecurityGroups = builder.sourceSecurityGroups;
    }

    /**
     * <p>
     * The Amazon EC2 SSH key that provides access for SSH communication with the nodes in the managed node group. For
     * more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html">Amazon EC2
     * key pairs and Linux instances</a> in the <i>Amazon Elastic Compute Cloud User Guide for Linux Instances</i>.
     * </p>
     * 
     * @return The Amazon EC2 SSH key that provides access for SSH communication with the nodes in the managed node
     *         group. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html">Amazon EC2 key pairs and
     *         Linux instances</a> in the <i>Amazon Elastic Compute Cloud User Guide for Linux Instances</i>.
     */
    public final String ec2SshKey() {
        return ec2SshKey;
    }

    /**
     * Returns true if the SourceSecurityGroups property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasSourceSecurityGroups() {
        return sourceSecurityGroups != null && !(sourceSecurityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The security groups that are allowed SSH access (port 22) to the nodes. If you specify an Amazon EC2 SSH key but
     * do not specify a source security group when you create a managed node group, then port 22 on the nodes is opened
     * to the internet (0.0.0.0/0). For more information, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html">Security Groups for Your VPC</a>
     * in the <i>Amazon Virtual Private Cloud User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSourceSecurityGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The security groups that are allowed SSH access (port 22) to the nodes. If you specify an Amazon EC2 SSH
     *         key but do not specify a source security group when you create a managed node group, then port 22 on the
     *         nodes is opened to the internet (0.0.0.0/0). For more information, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html">Security Groups for Your
     *         VPC</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
     */
    public final List<String> sourceSecurityGroups() {
        return sourceSecurityGroups;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ec2SshKey());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceSecurityGroups() ? sourceSecurityGroups() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RemoteAccessConfig)) {
            return false;
        }
        RemoteAccessConfig other = (RemoteAccessConfig) obj;
        return Objects.equals(ec2SshKey(), other.ec2SshKey()) && hasSourceSecurityGroups() == other.hasSourceSecurityGroups()
                && Objects.equals(sourceSecurityGroups(), other.sourceSecurityGroups());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RemoteAccessConfig").add("Ec2SshKey", ec2SshKey())
                .add("SourceSecurityGroups", hasSourceSecurityGroups() ? sourceSecurityGroups() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ec2SshKey":
            return Optional.ofNullable(clazz.cast(ec2SshKey()));
        case "sourceSecurityGroups":
            return Optional.ofNullable(clazz.cast(sourceSecurityGroups()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RemoteAccessConfig, T> g) {
        return obj -> g.apply((RemoteAccessConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RemoteAccessConfig> {
        /**
         * <p>
         * The Amazon EC2 SSH key that provides access for SSH communication with the nodes in the managed node group.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html">Amazon EC2 key pairs and Linux
         * instances</a> in the <i>Amazon Elastic Compute Cloud User Guide for Linux Instances</i>.
         * </p>
         * 
         * @param ec2SshKey
         *        The Amazon EC2 SSH key that provides access for SSH communication with the nodes in the managed node
         *        group. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html">Amazon EC2 key pairs and
         *        Linux instances</a> in the <i>Amazon Elastic Compute Cloud User Guide for Linux Instances</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2SshKey(String ec2SshKey);

        /**
         * <p>
         * The security groups that are allowed SSH access (port 22) to the nodes. If you specify an Amazon EC2 SSH key
         * but do not specify a source security group when you create a managed node group, then port 22 on the nodes is
         * opened to the internet (0.0.0.0/0). For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html">Security Groups for Your
         * VPC</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * </p>
         * 
         * @param sourceSecurityGroups
         *        The security groups that are allowed SSH access (port 22) to the nodes. If you specify an Amazon EC2
         *        SSH key but do not specify a source security group when you create a managed node group, then port 22
         *        on the nodes is opened to the internet (0.0.0.0/0). For more information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html">Security Groups for
         *        Your VPC</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceSecurityGroups(Collection<String> sourceSecurityGroups);

        /**
         * <p>
         * The security groups that are allowed SSH access (port 22) to the nodes. If you specify an Amazon EC2 SSH key
         * but do not specify a source security group when you create a managed node group, then port 22 on the nodes is
         * opened to the internet (0.0.0.0/0). For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html">Security Groups for Your
         * VPC</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * </p>
         * 
         * @param sourceSecurityGroups
         *        The security groups that are allowed SSH access (port 22) to the nodes. If you specify an Amazon EC2
         *        SSH key but do not specify a source security group when you create a managed node group, then port 22
         *        on the nodes is opened to the internet (0.0.0.0/0). For more information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html">Security Groups for
         *        Your VPC</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceSecurityGroups(String... sourceSecurityGroups);
    }

    static final class BuilderImpl implements Builder {
        private String ec2SshKey;

        private List<String> sourceSecurityGroups = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RemoteAccessConfig model) {
            ec2SshKey(model.ec2SshKey);
            sourceSecurityGroups(model.sourceSecurityGroups);
        }

        public final String getEc2SshKey() {
            return ec2SshKey;
        }

        @Override
        public final Builder ec2SshKey(String ec2SshKey) {
            this.ec2SshKey = ec2SshKey;
            return this;
        }

        public final void setEc2SshKey(String ec2SshKey) {
            this.ec2SshKey = ec2SshKey;
        }

        public final Collection<String> getSourceSecurityGroups() {
            if (sourceSecurityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return sourceSecurityGroups;
        }

        @Override
        public final Builder sourceSecurityGroups(Collection<String> sourceSecurityGroups) {
            this.sourceSecurityGroups = StringListCopier.copy(sourceSecurityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceSecurityGroups(String... sourceSecurityGroups) {
            sourceSecurityGroups(Arrays.asList(sourceSecurityGroups));
            return this;
        }

        public final void setSourceSecurityGroups(Collection<String> sourceSecurityGroups) {
            this.sourceSecurityGroups = StringListCopier.copy(sourceSecurityGroups);
        }

        @Override
        public RemoteAccessConfig build() {
            return new RemoteAccessConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
