/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing the resources associated with the node group, such as Auto Scaling groups and security groups
 * for remote access.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NodegroupResources implements SdkPojo, Serializable,
        ToCopyableBuilder<NodegroupResources.Builder, NodegroupResources> {
    private static final SdkField<List<AutoScalingGroup>> AUTO_SCALING_GROUPS_FIELD = SdkField
            .<List<AutoScalingGroup>> builder(MarshallingType.LIST)
            .getter(getter(NodegroupResources::autoScalingGroups))
            .setter(setter(Builder::autoScalingGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoScalingGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AutoScalingGroup> builder(MarshallingType.SDK_POJO)
                                            .constructor(AutoScalingGroup::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> REMOTE_ACCESS_SECURITY_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(NodegroupResources::remoteAccessSecurityGroup)).setter(setter(Builder::remoteAccessSecurityGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("remoteAccessSecurityGroup").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTO_SCALING_GROUPS_FIELD,
            REMOTE_ACCESS_SECURITY_GROUP_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<AutoScalingGroup> autoScalingGroups;

    private final String remoteAccessSecurityGroup;

    private NodegroupResources(BuilderImpl builder) {
        this.autoScalingGroups = builder.autoScalingGroups;
        this.remoteAccessSecurityGroup = builder.remoteAccessSecurityGroup;
    }

    /**
     * Returns true if the AutoScalingGroups property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAutoScalingGroups() {
        return autoScalingGroups != null && !(autoScalingGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Auto Scaling groups associated with the node group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAutoScalingGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Auto Scaling groups associated with the node group.
     */
    public List<AutoScalingGroup> autoScalingGroups() {
        return autoScalingGroups;
    }

    /**
     * <p>
     * The remote access security group associated with the node group. This security group controls SSH access to the
     * worker nodes.
     * </p>
     * 
     * @return The remote access security group associated with the node group. This security group controls SSH access
     *         to the worker nodes.
     */
    public String remoteAccessSecurityGroup() {
        return remoteAccessSecurityGroup;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingGroups());
        hashCode = 31 * hashCode + Objects.hashCode(remoteAccessSecurityGroup());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NodegroupResources)) {
            return false;
        }
        NodegroupResources other = (NodegroupResources) obj;
        return Objects.equals(autoScalingGroups(), other.autoScalingGroups())
                && Objects.equals(remoteAccessSecurityGroup(), other.remoteAccessSecurityGroup());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("NodegroupResources").add("AutoScalingGroups", autoScalingGroups())
                .add("RemoteAccessSecurityGroup", remoteAccessSecurityGroup()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "autoScalingGroups":
            return Optional.ofNullable(clazz.cast(autoScalingGroups()));
        case "remoteAccessSecurityGroup":
            return Optional.ofNullable(clazz.cast(remoteAccessSecurityGroup()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NodegroupResources, T> g) {
        return obj -> g.apply((NodegroupResources) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NodegroupResources> {
        /**
         * <p>
         * The Auto Scaling groups associated with the node group.
         * </p>
         * 
         * @param autoScalingGroups
         *        The Auto Scaling groups associated with the node group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroups(Collection<AutoScalingGroup> autoScalingGroups);

        /**
         * <p>
         * The Auto Scaling groups associated with the node group.
         * </p>
         * 
         * @param autoScalingGroups
         *        The Auto Scaling groups associated with the node group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroups(AutoScalingGroup... autoScalingGroups);

        /**
         * <p>
         * The Auto Scaling groups associated with the node group.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AutoScalingGroup>.Builder} avoiding the
         * need to create one manually via {@link List<AutoScalingGroup>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AutoScalingGroup>.Builder#build()} is called immediately and
         * its result is passed to {@link #autoScalingGroups(List<AutoScalingGroup>)}.
         * 
         * @param autoScalingGroups
         *        a consumer that will call methods on {@link List<AutoScalingGroup>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoScalingGroups(List<AutoScalingGroup>)
         */
        Builder autoScalingGroups(Consumer<AutoScalingGroup.Builder>... autoScalingGroups);

        /**
         * <p>
         * The remote access security group associated with the node group. This security group controls SSH access to
         * the worker nodes.
         * </p>
         * 
         * @param remoteAccessSecurityGroup
         *        The remote access security group associated with the node group. This security group controls SSH
         *        access to the worker nodes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remoteAccessSecurityGroup(String remoteAccessSecurityGroup);
    }

    static final class BuilderImpl implements Builder {
        private List<AutoScalingGroup> autoScalingGroups = DefaultSdkAutoConstructList.getInstance();

        private String remoteAccessSecurityGroup;

        private BuilderImpl() {
        }

        private BuilderImpl(NodegroupResources model) {
            autoScalingGroups(model.autoScalingGroups);
            remoteAccessSecurityGroup(model.remoteAccessSecurityGroup);
        }

        public final Collection<AutoScalingGroup.Builder> getAutoScalingGroups() {
            return autoScalingGroups != null ? autoScalingGroups.stream().map(AutoScalingGroup::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder autoScalingGroups(Collection<AutoScalingGroup> autoScalingGroups) {
            this.autoScalingGroups = AutoScalingGroupListCopier.copy(autoScalingGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder autoScalingGroups(AutoScalingGroup... autoScalingGroups) {
            autoScalingGroups(Arrays.asList(autoScalingGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder autoScalingGroups(Consumer<AutoScalingGroup.Builder>... autoScalingGroups) {
            autoScalingGroups(Stream.of(autoScalingGroups).map(c -> AutoScalingGroup.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAutoScalingGroups(Collection<AutoScalingGroup.BuilderImpl> autoScalingGroups) {
            this.autoScalingGroups = AutoScalingGroupListCopier.copyFromBuilder(autoScalingGroups);
        }

        public final String getRemoteAccessSecurityGroup() {
            return remoteAccessSecurityGroup;
        }

        @Override
        public final Builder remoteAccessSecurityGroup(String remoteAccessSecurityGroup) {
            this.remoteAccessSecurityGroup = remoteAccessSecurityGroup;
            return this;
        }

        public final void setRemoteAccessSecurityGroup(String remoteAccessSecurityGroup) {
            this.remoteAccessSecurityGroup = remoteAccessSecurityGroup;
        }

        @Override
        public NodegroupResources build() {
            return new NodegroupResources(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
