/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The load balancer configuration to use with a service or task set.
 * </p>
 * <p>
 * When you add, update, or remove a load balancer configuration, Amazon ECS starts a new deployment with the updated
 * Elastic Load Balancing configuration. This causes tasks to register to and deregister from load balancers.
 * </p>
 * <p>
 * We recommend that you verify this on a test environment before you update the Elastic Load Balancing configuration.
 * </p>
 * <p>
 * A service-linked role is required for services that use multiple target groups. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using-service-linked-roles.html">Using
 * service-linked roles</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LoadBalancer implements SdkPojo, Serializable, ToCopyableBuilder<LoadBalancer.Builder, LoadBalancer> {
    private static final SdkField<String> TARGET_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("targetGroupArn").getter(getter(LoadBalancer::targetGroupArn)).setter(setter(Builder::targetGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetGroupArn").build()).build();

    private static final SdkField<String> LOAD_BALANCER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("loadBalancerName").getter(getter(LoadBalancer::loadBalancerName))
            .setter(setter(Builder::loadBalancerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("loadBalancerName").build()).build();

    private static final SdkField<String> CONTAINER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("containerName").getter(getter(LoadBalancer::containerName)).setter(setter(Builder::containerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerName").build()).build();

    private static final SdkField<Integer> CONTAINER_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("containerPort").getter(getter(LoadBalancer::containerPort)).setter(setter(Builder::containerPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerPort").build()).build();

    private static final SdkField<AdvancedConfiguration> ADVANCED_CONFIGURATION_FIELD = SdkField
            .<AdvancedConfiguration> builder(MarshallingType.SDK_POJO).memberName("advancedConfiguration")
            .getter(getter(LoadBalancer::advancedConfiguration)).setter(setter(Builder::advancedConfiguration))
            .constructor(AdvancedConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("advancedConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TARGET_GROUP_ARN_FIELD,
            LOAD_BALANCER_NAME_FIELD, CONTAINER_NAME_FIELD, CONTAINER_PORT_FIELD, ADVANCED_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String targetGroupArn;

    private final String loadBalancerName;

    private final String containerName;

    private final Integer containerPort;

    private final AdvancedConfiguration advancedConfiguration;

    private LoadBalancer(BuilderImpl builder) {
        this.targetGroupArn = builder.targetGroupArn;
        this.loadBalancerName = builder.loadBalancerName;
        this.containerName = builder.containerName;
        this.containerPort = builder.containerPort;
        this.advancedConfiguration = builder.advancedConfiguration;
    }

    /**
     * <p>
     * The full Amazon Resource Name (ARN) of the Elastic Load Balancing target group or groups associated with a
     * service or task set.
     * </p>
     * <p>
     * A target group ARN is only specified when using an Application Load Balancer or Network Load Balancer.
     * </p>
     * <p>
     * For services using the <code>ECS</code> deployment controller, you can specify one or multiple target groups. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/register-multiple-targetgroups.html"
     * >Registering multiple target groups with a service</a> in the <i>Amazon Elastic Container Service Developer
     * Guide</i>.
     * </p>
     * <p>
     * For services using the <code>CODE_DEPLOY</code> deployment controller, you're required to define two target
     * groups for the load balancer. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-bluegreen.html">Blue/green
     * deployment with CodeDeploy</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * If your service's task definition uses the <code>awsvpc</code> network mode, you must choose <code>ip</code> as
     * the target type, not <code>instance</code>. Do this when creating your target groups because tasks that use the
     * <code>awsvpc</code> network mode are associated with an elastic network interface, not an Amazon EC2 instance.
     * This network mode is required for the Fargate launch type.
     * </p>
     * </important>
     * 
     * @return The full Amazon Resource Name (ARN) of the Elastic Load Balancing target group or groups associated with
     *         a service or task set.</p>
     *         <p>
     *         A target group ARN is only specified when using an Application Load Balancer or Network Load Balancer.
     *         </p>
     *         <p>
     *         For services using the <code>ECS</code> deployment controller, you can specify one or multiple target
     *         groups. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/register-multiple-targetgroups.html"
     *         >Registering multiple target groups with a service</a> in the <i>Amazon Elastic Container Service
     *         Developer Guide</i>.
     *         </p>
     *         <p>
     *         For services using the <code>CODE_DEPLOY</code> deployment controller, you're required to define two
     *         target groups for the load balancer. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-bluegreen.html"
     *         >Blue/green deployment with CodeDeploy</a> in the <i>Amazon Elastic Container Service Developer
     *         Guide</i>.
     *         </p>
     *         <important>
     *         <p>
     *         If your service's task definition uses the <code>awsvpc</code> network mode, you must choose
     *         <code>ip</code> as the target type, not <code>instance</code>. Do this when creating your target groups
     *         because tasks that use the <code>awsvpc</code> network mode are associated with an elastic network
     *         interface, not an Amazon EC2 instance. This network mode is required for the Fargate launch type.
     *         </p>
     */
    public final String targetGroupArn() {
        return targetGroupArn;
    }

    /**
     * <p>
     * The name of the load balancer to associate with the Amazon ECS service or task set.
     * </p>
     * <p>
     * If you are using an Application Load Balancer or a Network Load Balancer the load balancer name parameter should
     * be omitted.
     * </p>
     * 
     * @return The name of the load balancer to associate with the Amazon ECS service or task set.</p>
     *         <p>
     *         If you are using an Application Load Balancer or a Network Load Balancer the load balancer name parameter
     *         should be omitted.
     */
    public final String loadBalancerName() {
        return loadBalancerName;
    }

    /**
     * <p>
     * The name of the container (as it appears in a container definition) to associate with the load balancer.
     * </p>
     * <p>
     * You need to specify the container name when configuring the target group for an Amazon ECS load balancer.
     * </p>
     * 
     * @return The name of the container (as it appears in a container definition) to associate with the load
     *         balancer.</p>
     *         <p>
     *         You need to specify the container name when configuring the target group for an Amazon ECS load balancer.
     */
    public final String containerName() {
        return containerName;
    }

    /**
     * <p>
     * The port on the container to associate with the load balancer. This port must correspond to a
     * <code>containerPort</code> in the task definition the tasks in the service are using. For tasks that use the EC2
     * launch type, the container instance they're launched on must allow ingress traffic on the <code>hostPort</code>
     * of the port mapping.
     * </p>
     * 
     * @return The port on the container to associate with the load balancer. This port must correspond to a
     *         <code>containerPort</code> in the task definition the tasks in the service are using. For tasks that use
     *         the EC2 launch type, the container instance they're launched on must allow ingress traffic on the
     *         <code>hostPort</code> of the port mapping.
     */
    public final Integer containerPort() {
        return containerPort;
    }

    /**
     * <p>
     * The advanced settings for the load balancer used in blue/green deployments. Specify the alternate target group,
     * listener rules, and IAM role required for traffic shifting during blue/green deployments.
     * </p>
     * 
     * @return The advanced settings for the load balancer used in blue/green deployments. Specify the alternate target
     *         group, listener rules, and IAM role required for traffic shifting during blue/green deployments.
     */
    public final AdvancedConfiguration advancedConfiguration() {
        return advancedConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(targetGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(loadBalancerName());
        hashCode = 31 * hashCode + Objects.hashCode(containerName());
        hashCode = 31 * hashCode + Objects.hashCode(containerPort());
        hashCode = 31 * hashCode + Objects.hashCode(advancedConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LoadBalancer)) {
            return false;
        }
        LoadBalancer other = (LoadBalancer) obj;
        return Objects.equals(targetGroupArn(), other.targetGroupArn())
                && Objects.equals(loadBalancerName(), other.loadBalancerName())
                && Objects.equals(containerName(), other.containerName())
                && Objects.equals(containerPort(), other.containerPort())
                && Objects.equals(advancedConfiguration(), other.advancedConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LoadBalancer").add("TargetGroupArn", targetGroupArn())
                .add("LoadBalancerName", loadBalancerName()).add("ContainerName", containerName())
                .add("ContainerPort", containerPort()).add("AdvancedConfiguration", advancedConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "targetGroupArn":
            return Optional.ofNullable(clazz.cast(targetGroupArn()));
        case "loadBalancerName":
            return Optional.ofNullable(clazz.cast(loadBalancerName()));
        case "containerName":
            return Optional.ofNullable(clazz.cast(containerName()));
        case "containerPort":
            return Optional.ofNullable(clazz.cast(containerPort()));
        case "advancedConfiguration":
            return Optional.ofNullable(clazz.cast(advancedConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("targetGroupArn", TARGET_GROUP_ARN_FIELD);
        map.put("loadBalancerName", LOAD_BALANCER_NAME_FIELD);
        map.put("containerName", CONTAINER_NAME_FIELD);
        map.put("containerPort", CONTAINER_PORT_FIELD);
        map.put("advancedConfiguration", ADVANCED_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LoadBalancer, T> g) {
        return obj -> g.apply((LoadBalancer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LoadBalancer> {
        /**
         * <p>
         * The full Amazon Resource Name (ARN) of the Elastic Load Balancing target group or groups associated with a
         * service or task set.
         * </p>
         * <p>
         * A target group ARN is only specified when using an Application Load Balancer or Network Load Balancer.
         * </p>
         * <p>
         * For services using the <code>ECS</code> deployment controller, you can specify one or multiple target groups.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/register-multiple-targetgroups.html"
         * >Registering multiple target groups with a service</a> in the <i>Amazon Elastic Container Service Developer
         * Guide</i>.
         * </p>
         * <p>
         * For services using the <code>CODE_DEPLOY</code> deployment controller, you're required to define two target
         * groups for the load balancer. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-bluegreen.html">Blue/green
         * deployment with CodeDeploy</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <important>
         * <p>
         * If your service's task definition uses the <code>awsvpc</code> network mode, you must choose <code>ip</code>
         * as the target type, not <code>instance</code>. Do this when creating your target groups because tasks that
         * use the <code>awsvpc</code> network mode are associated with an elastic network interface, not an Amazon EC2
         * instance. This network mode is required for the Fargate launch type.
         * </p>
         * </important>
         * 
         * @param targetGroupArn
         *        The full Amazon Resource Name (ARN) of the Elastic Load Balancing target group or groups associated
         *        with a service or task set.</p>
         *        <p>
         *        A target group ARN is only specified when using an Application Load Balancer or Network Load Balancer.
         *        </p>
         *        <p>
         *        For services using the <code>ECS</code> deployment controller, you can specify one or multiple target
         *        groups. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/register-multiple-targetgroups.html"
         *        >Registering multiple target groups with a service</a> in the <i>Amazon Elastic Container Service
         *        Developer Guide</i>.
         *        </p>
         *        <p>
         *        For services using the <code>CODE_DEPLOY</code> deployment controller, you're required to define two
         *        target groups for the load balancer. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-bluegreen.html"
         *        >Blue/green deployment with CodeDeploy</a> in the <i>Amazon Elastic Container Service Developer
         *        Guide</i>.
         *        </p>
         *        <important>
         *        <p>
         *        If your service's task definition uses the <code>awsvpc</code> network mode, you must choose
         *        <code>ip</code> as the target type, not <code>instance</code>. Do this when creating your target
         *        groups because tasks that use the <code>awsvpc</code> network mode are associated with an elastic
         *        network interface, not an Amazon EC2 instance. This network mode is required for the Fargate launch
         *        type.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetGroupArn(String targetGroupArn);

        /**
         * <p>
         * The name of the load balancer to associate with the Amazon ECS service or task set.
         * </p>
         * <p>
         * If you are using an Application Load Balancer or a Network Load Balancer the load balancer name parameter
         * should be omitted.
         * </p>
         * 
         * @param loadBalancerName
         *        The name of the load balancer to associate with the Amazon ECS service or task set.</p>
         *        <p>
         *        If you are using an Application Load Balancer or a Network Load Balancer the load balancer name
         *        parameter should be omitted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancerName(String loadBalancerName);

        /**
         * <p>
         * The name of the container (as it appears in a container definition) to associate with the load balancer.
         * </p>
         * <p>
         * You need to specify the container name when configuring the target group for an Amazon ECS load balancer.
         * </p>
         * 
         * @param containerName
         *        The name of the container (as it appears in a container definition) to associate with the load
         *        balancer.</p>
         *        <p>
         *        You need to specify the container name when configuring the target group for an Amazon ECS load
         *        balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerName(String containerName);

        /**
         * <p>
         * The port on the container to associate with the load balancer. This port must correspond to a
         * <code>containerPort</code> in the task definition the tasks in the service are using. For tasks that use the
         * EC2 launch type, the container instance they're launched on must allow ingress traffic on the
         * <code>hostPort</code> of the port mapping.
         * </p>
         * 
         * @param containerPort
         *        The port on the container to associate with the load balancer. This port must correspond to a
         *        <code>containerPort</code> in the task definition the tasks in the service are using. For tasks that
         *        use the EC2 launch type, the container instance they're launched on must allow ingress traffic on the
         *        <code>hostPort</code> of the port mapping.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerPort(Integer containerPort);

        /**
         * <p>
         * The advanced settings for the load balancer used in blue/green deployments. Specify the alternate target
         * group, listener rules, and IAM role required for traffic shifting during blue/green deployments.
         * </p>
         * 
         * @param advancedConfiguration
         *        The advanced settings for the load balancer used in blue/green deployments. Specify the alternate
         *        target group, listener rules, and IAM role required for traffic shifting during blue/green
         *        deployments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder advancedConfiguration(AdvancedConfiguration advancedConfiguration);

        /**
         * <p>
         * The advanced settings for the load balancer used in blue/green deployments. Specify the alternate target
         * group, listener rules, and IAM role required for traffic shifting during blue/green deployments.
         * </p>
         * This is a convenience method that creates an instance of the {@link AdvancedConfiguration.Builder} avoiding
         * the need to create one manually via {@link AdvancedConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AdvancedConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #advancedConfiguration(AdvancedConfiguration)}.
         * 
         * @param advancedConfiguration
         *        a consumer that will call methods on {@link AdvancedConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #advancedConfiguration(AdvancedConfiguration)
         */
        default Builder advancedConfiguration(Consumer<AdvancedConfiguration.Builder> advancedConfiguration) {
            return advancedConfiguration(AdvancedConfiguration.builder().applyMutation(advancedConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String targetGroupArn;

        private String loadBalancerName;

        private String containerName;

        private Integer containerPort;

        private AdvancedConfiguration advancedConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(LoadBalancer model) {
            targetGroupArn(model.targetGroupArn);
            loadBalancerName(model.loadBalancerName);
            containerName(model.containerName);
            containerPort(model.containerPort);
            advancedConfiguration(model.advancedConfiguration);
        }

        public final String getTargetGroupArn() {
            return targetGroupArn;
        }

        public final void setTargetGroupArn(String targetGroupArn) {
            this.targetGroupArn = targetGroupArn;
        }

        @Override
        public final Builder targetGroupArn(String targetGroupArn) {
            this.targetGroupArn = targetGroupArn;
            return this;
        }

        public final String getLoadBalancerName() {
            return loadBalancerName;
        }

        public final void setLoadBalancerName(String loadBalancerName) {
            this.loadBalancerName = loadBalancerName;
        }

        @Override
        public final Builder loadBalancerName(String loadBalancerName) {
            this.loadBalancerName = loadBalancerName;
            return this;
        }

        public final String getContainerName() {
            return containerName;
        }

        public final void setContainerName(String containerName) {
            this.containerName = containerName;
        }

        @Override
        public final Builder containerName(String containerName) {
            this.containerName = containerName;
            return this;
        }

        public final Integer getContainerPort() {
            return containerPort;
        }

        public final void setContainerPort(Integer containerPort) {
            this.containerPort = containerPort;
        }

        @Override
        public final Builder containerPort(Integer containerPort) {
            this.containerPort = containerPort;
            return this;
        }

        public final AdvancedConfiguration.Builder getAdvancedConfiguration() {
            return advancedConfiguration != null ? advancedConfiguration.toBuilder() : null;
        }

        public final void setAdvancedConfiguration(AdvancedConfiguration.BuilderImpl advancedConfiguration) {
            this.advancedConfiguration = advancedConfiguration != null ? advancedConfiguration.build() : null;
        }

        @Override
        public final Builder advancedConfiguration(AdvancedConfiguration advancedConfiguration) {
            this.advancedConfiguration = advancedConfiguration;
            return this;
        }

        @Override
        public LoadBalancer build() {
            return new LoadBalancer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
