/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a volume.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Volume implements SdkPojo, Serializable, ToCopyableBuilder<Volume.Builder, Volume> {
    private static final SdkField<List<VolumeAttachment>> ATTACHMENTS_FIELD = SdkField
            .<List<VolumeAttachment>> builder(MarshallingType.LIST)
            .getter(getter(Volume::attachments))
            .setter(setter(Builder::attachments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttachmentSet")
                    .unmarshallLocationName("attachmentSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<VolumeAttachment> builder(MarshallingType.SDK_POJO)
                                            .constructor(VolumeAttachment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Volume::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone")
                    .unmarshallLocationName("availabilityZone").build()).build();

    private static final SdkField<Instant> CREATE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Volume::createTime))
            .setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTime")
                    .unmarshallLocationName("createTime").build()).build();

    private static final SdkField<Boolean> ENCRYPTED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(Volume::encrypted))
            .setter(setter(Builder::encrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Encrypted")
                    .unmarshallLocationName("encrypted").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Volume::kmsKeyId))
            .setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId")
                    .unmarshallLocationName("kmsKeyId").build()).build();

    private static final SdkField<Integer> SIZE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Volume::size))
            .setter(setter(Builder::size))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Size")
                    .unmarshallLocationName("size").build()).build();

    private static final SdkField<String> SNAPSHOT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Volume::snapshotId))
            .setter(setter(Builder::snapshotId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotId")
                    .unmarshallLocationName("snapshotId").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Volume::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status")
                    .unmarshallLocationName("status").build()).build();

    private static final SdkField<String> VOLUME_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Volume::volumeId))
            .setter(setter(Builder::volumeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeId")
                    .unmarshallLocationName("volumeId").build()).build();

    private static final SdkField<Integer> IOPS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Volume::iops))
            .setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Iops")
                    .unmarshallLocationName("iops").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(Volume::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> VOLUME_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Volume::volumeTypeAsString))
            .setter(setter(Builder::volumeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeType")
                    .unmarshallLocationName("volumeType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ATTACHMENTS_FIELD,
            AVAILABILITY_ZONE_FIELD, CREATE_TIME_FIELD, ENCRYPTED_FIELD, KMS_KEY_ID_FIELD, SIZE_FIELD, SNAPSHOT_ID_FIELD,
            STATE_FIELD, VOLUME_ID_FIELD, IOPS_FIELD, TAGS_FIELD, VOLUME_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<VolumeAttachment> attachments;

    private final String availabilityZone;

    private final Instant createTime;

    private final Boolean encrypted;

    private final String kmsKeyId;

    private final Integer size;

    private final String snapshotId;

    private final String state;

    private final String volumeId;

    private final Integer iops;

    private final List<Tag> tags;

    private final String volumeType;

    private Volume(BuilderImpl builder) {
        this.attachments = builder.attachments;
        this.availabilityZone = builder.availabilityZone;
        this.createTime = builder.createTime;
        this.encrypted = builder.encrypted;
        this.kmsKeyId = builder.kmsKeyId;
        this.size = builder.size;
        this.snapshotId = builder.snapshotId;
        this.state = builder.state;
        this.volumeId = builder.volumeId;
        this.iops = builder.iops;
        this.tags = builder.tags;
        this.volumeType = builder.volumeType;
    }

    /**
     * <p>
     * Information about the volume attachments.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Information about the volume attachments.
     */
    public List<VolumeAttachment> attachments() {
        return attachments;
    }

    /**
     * <p>
     * The Availability Zone for the volume.
     * </p>
     * 
     * @return The Availability Zone for the volume.
     */
    public String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * The time stamp when volume creation was initiated.
     * </p>
     * 
     * @return The time stamp when volume creation was initiated.
     */
    public Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * Indicates whether the volume is encrypted.
     * </p>
     * 
     * @return Indicates whether the volume is encrypted.
     */
    public Boolean encrypted() {
        return encrypted;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the AWS Key Management Service (AWS KMS) customer master key (CMK) that was
     * used to protect the volume encryption key for the volume.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the AWS Key Management Service (AWS KMS) customer master key (CMK) that
     *         was used to protect the volume encryption key for the volume.
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The size of the volume, in GiBs.
     * </p>
     * 
     * @return The size of the volume, in GiBs.
     */
    public Integer size() {
        return size;
    }

    /**
     * <p>
     * The snapshot from which the volume was created, if applicable.
     * </p>
     * 
     * @return The snapshot from which the volume was created, if applicable.
     */
    public String snapshotId() {
        return snapshotId;
    }

    /**
     * <p>
     * The volume state.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link VolumeState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The volume state.
     * @see VolumeState
     */
    public VolumeState state() {
        return VolumeState.fromValue(state);
    }

    /**
     * <p>
     * The volume state.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link VolumeState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The volume state.
     * @see VolumeState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The ID of the volume.
     * </p>
     * 
     * @return The ID of the volume.
     */
    public String volumeId() {
        return volumeId;
    }

    /**
     * <p>
     * The number of I/O operations per second (IOPS) that the volume supports. For Provisioned IOPS SSD volumes, this
     * represents the number of IOPS that are provisioned for the volume. For General Purpose SSD volumes, this
     * represents the baseline performance of the volume and the rate at which the volume accumulates I/O credits for
     * bursting. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume Types</a> in the
     * <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * <p>
     * Constraints: Range is 100-16,000 IOPS for <code>gp2</code> volumes and 100 to 64,000IOPS for <code>io1</code>
     * volumes, in most Regions. The maximum IOPS for <code>io1</code> of 64,000 is guaranteed only on <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Nitro-based
     * instances</a>. Other instance families guarantee performance up to 32,000 IOPS.
     * </p>
     * <p>
     * Condition: This parameter is required for requests to create <code>io1</code> volumes; it is not used in requests
     * to create <code>gp2</code>, <code>st1</code>, <code>sc1</code>, or <code>standard</code> volumes.
     * </p>
     * 
     * @return The number of I/O operations per second (IOPS) that the volume supports. For Provisioned IOPS SSD
     *         volumes, this represents the number of IOPS that are provisioned for the volume. For General Purpose SSD
     *         volumes, this represents the baseline performance of the volume and the rate at which the volume
     *         accumulates I/O credits for bursting. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume
     *         Types</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.</p>
     *         <p>
     *         Constraints: Range is 100-16,000 IOPS for <code>gp2</code> volumes and 100 to 64,000IOPS for
     *         <code>io1</code> volumes, in most Regions. The maximum IOPS for <code>io1</code> of 64,000 is guaranteed
     *         only on <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances"
     *         >Nitro-based instances</a>. Other instance families guarantee performance up to 32,000 IOPS.
     *         </p>
     *         <p>
     *         Condition: This parameter is required for requests to create <code>io1</code> volumes; it is not used in
     *         requests to create <code>gp2</code>, <code>st1</code>, <code>sc1</code>, or <code>standard</code>
     *         volumes.
     */
    public Integer iops() {
        return iops;
    }

    /**
     * <p>
     * Any tags assigned to the volume.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Any tags assigned to the volume.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned IOPS SSD,
     * <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or <code>standard</code> for
     * Magnetic volumes.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned
     *         IOPS SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or
     *         <code>standard</code> for Magnetic volumes.
     * @see VolumeType
     */
    public VolumeType volumeType() {
        return VolumeType.fromValue(volumeType);
    }

    /**
     * <p>
     * The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned IOPS SSD,
     * <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or <code>standard</code> for
     * Magnetic volumes.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned
     *         IOPS SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or
     *         <code>standard</code> for Magnetic volumes.
     * @see VolumeType
     */
    public String volumeTypeAsString() {
        return volumeType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(attachments());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(encrypted());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(size());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(volumeId());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(volumeTypeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Volume)) {
            return false;
        }
        Volume other = (Volume) obj;
        return Objects.equals(attachments(), other.attachments()) && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(createTime(), other.createTime()) && Objects.equals(encrypted(), other.encrypted())
                && Objects.equals(kmsKeyId(), other.kmsKeyId()) && Objects.equals(size(), other.size())
                && Objects.equals(snapshotId(), other.snapshotId()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(volumeId(), other.volumeId()) && Objects.equals(iops(), other.iops())
                && Objects.equals(tags(), other.tags()) && Objects.equals(volumeTypeAsString(), other.volumeTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Volume").add("Attachments", attachments()).add("AvailabilityZone", availabilityZone())
                .add("CreateTime", createTime()).add("Encrypted", encrypted()).add("KmsKeyId", kmsKeyId()).add("Size", size())
                .add("SnapshotId", snapshotId()).add("State", stateAsString()).add("VolumeId", volumeId()).add("Iops", iops())
                .add("Tags", tags()).add("VolumeType", volumeTypeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Attachments":
            return Optional.ofNullable(clazz.cast(attachments()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "CreateTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "Encrypted":
            return Optional.ofNullable(clazz.cast(encrypted()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "Size":
            return Optional.ofNullable(clazz.cast(size()));
        case "SnapshotId":
            return Optional.ofNullable(clazz.cast(snapshotId()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "VolumeId":
            return Optional.ofNullable(clazz.cast(volumeId()));
        case "Iops":
            return Optional.ofNullable(clazz.cast(iops()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "VolumeType":
            return Optional.ofNullable(clazz.cast(volumeTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Volume, T> g) {
        return obj -> g.apply((Volume) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Volume> {
        /**
         * <p>
         * Information about the volume attachments.
         * </p>
         * 
         * @param attachments
         *        Information about the volume attachments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachments(Collection<VolumeAttachment> attachments);

        /**
         * <p>
         * Information about the volume attachments.
         * </p>
         * 
         * @param attachments
         *        Information about the volume attachments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachments(VolumeAttachment... attachments);

        /**
         * <p>
         * Information about the volume attachments.
         * </p>
         * This is a convenience that creates an instance of the {@link List<VolumeAttachment>.Builder} avoiding the
         * need to create one manually via {@link List<VolumeAttachment>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<VolumeAttachment>.Builder#build()} is called immediately and
         * its result is passed to {@link #attachments(List<VolumeAttachment>)}.
         * 
         * @param attachments
         *        a consumer that will call methods on {@link List<VolumeAttachment>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachments(List<VolumeAttachment>)
         */
        Builder attachments(Consumer<VolumeAttachment.Builder>... attachments);

        /**
         * <p>
         * The Availability Zone for the volume.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone for the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * The time stamp when volume creation was initiated.
         * </p>
         * 
         * @param createTime
         *        The time stamp when volume creation was initiated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * Indicates whether the volume is encrypted.
         * </p>
         * 
         * @param encrypted
         *        Indicates whether the volume is encrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encrypted(Boolean encrypted);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the AWS Key Management Service (AWS KMS) customer master key (CMK) that was
         * used to protect the volume encryption key for the volume.
         * </p>
         * 
         * @param kmsKeyId
         *        The Amazon Resource Name (ARN) of the AWS Key Management Service (AWS KMS) customer master key (CMK)
         *        that was used to protect the volume encryption key for the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The size of the volume, in GiBs.
         * </p>
         * 
         * @param size
         *        The size of the volume, in GiBs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder size(Integer size);

        /**
         * <p>
         * The snapshot from which the volume was created, if applicable.
         * </p>
         * 
         * @param snapshotId
         *        The snapshot from which the volume was created, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotId(String snapshotId);

        /**
         * <p>
         * The volume state.
         * </p>
         * 
         * @param state
         *        The volume state.
         * @see VolumeState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeState
         */
        Builder state(String state);

        /**
         * <p>
         * The volume state.
         * </p>
         * 
         * @param state
         *        The volume state.
         * @see VolumeState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeState
         */
        Builder state(VolumeState state);

        /**
         * <p>
         * The ID of the volume.
         * </p>
         * 
         * @param volumeId
         *        The ID of the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeId(String volumeId);

        /**
         * <p>
         * The number of I/O operations per second (IOPS) that the volume supports. For Provisioned IOPS SSD volumes,
         * this represents the number of IOPS that are provisioned for the volume. For General Purpose SSD volumes, this
         * represents the baseline performance of the volume and the rate at which the volume accumulates I/O credits
         * for bursting. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume Types</a> in
         * the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * <p>
         * Constraints: Range is 100-16,000 IOPS for <code>gp2</code> volumes and 100 to 64,000IOPS for <code>io1</code>
         * volumes, in most Regions. The maximum IOPS for <code>io1</code> of 64,000 is guaranteed only on <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances"
         * >Nitro-based instances</a>. Other instance families guarantee performance up to 32,000 IOPS.
         * </p>
         * <p>
         * Condition: This parameter is required for requests to create <code>io1</code> volumes; it is not used in
         * requests to create <code>gp2</code>, <code>st1</code>, <code>sc1</code>, or <code>standard</code> volumes.
         * </p>
         * 
         * @param iops
         *        The number of I/O operations per second (IOPS) that the volume supports. For Provisioned IOPS SSD
         *        volumes, this represents the number of IOPS that are provisioned for the volume. For General Purpose
         *        SSD volumes, this represents the baseline performance of the volume and the rate at which the volume
         *        accumulates I/O credits for bursting. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume
         *        Types</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.</p>
         *        <p>
         *        Constraints: Range is 100-16,000 IOPS for <code>gp2</code> volumes and 100 to 64,000IOPS for
         *        <code>io1</code> volumes, in most Regions. The maximum IOPS for <code>io1</code> of 64,000 is
         *        guaranteed only on <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances"
         *        >Nitro-based instances</a>. Other instance families guarantee performance up to 32,000 IOPS.
         *        </p>
         *        <p>
         *        Condition: This parameter is required for requests to create <code>io1</code> volumes; it is not used
         *        in requests to create <code>gp2</code>, <code>st1</code>, <code>sc1</code>, or <code>standard</code>
         *        volumes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);

        /**
         * <p>
         * Any tags assigned to the volume.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the volume.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the volume.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned IOPS
         * SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or <code>standard</code>
         * for Magnetic volumes.
         * </p>
         * 
         * @param volumeType
         *        The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for
         *        Provisioned IOPS SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or
         *        <code>standard</code> for Magnetic volumes.
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(String volumeType);

        /**
         * <p>
         * The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned IOPS
         * SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or <code>standard</code>
         * for Magnetic volumes.
         * </p>
         * 
         * @param volumeType
         *        The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for
         *        Provisioned IOPS SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or
         *        <code>standard</code> for Magnetic volumes.
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(VolumeType volumeType);
    }

    static final class BuilderImpl implements Builder {
        private List<VolumeAttachment> attachments = DefaultSdkAutoConstructList.getInstance();

        private String availabilityZone;

        private Instant createTime;

        private Boolean encrypted;

        private String kmsKeyId;

        private Integer size;

        private String snapshotId;

        private String state;

        private String volumeId;

        private Integer iops;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String volumeType;

        private BuilderImpl() {
        }

        private BuilderImpl(Volume model) {
            attachments(model.attachments);
            availabilityZone(model.availabilityZone);
            createTime(model.createTime);
            encrypted(model.encrypted);
            kmsKeyId(model.kmsKeyId);
            size(model.size);
            snapshotId(model.snapshotId);
            state(model.state);
            volumeId(model.volumeId);
            iops(model.iops);
            tags(model.tags);
            volumeType(model.volumeType);
        }

        public final Collection<VolumeAttachment.Builder> getAttachments() {
            return attachments != null ? attachments.stream().map(VolumeAttachment::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder attachments(Collection<VolumeAttachment> attachments) {
            this.attachments = VolumeAttachmentListCopier.copy(attachments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachments(VolumeAttachment... attachments) {
            attachments(Arrays.asList(attachments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachments(Consumer<VolumeAttachment.Builder>... attachments) {
            attachments(Stream.of(attachments).map(c -> VolumeAttachment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAttachments(Collection<VolumeAttachment.BuilderImpl> attachments) {
            this.attachments = VolumeAttachmentListCopier.copyFromBuilder(attachments);
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        public final Boolean getEncrypted() {
            return encrypted;
        }

        @Override
        public final Builder encrypted(Boolean encrypted) {
            this.encrypted = encrypted;
            return this;
        }

        public final void setEncrypted(Boolean encrypted) {
            this.encrypted = encrypted;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final Integer getSize() {
            return size;
        }

        @Override
        public final Builder size(Integer size) {
            this.size = size;
            return this;
        }

        public final void setSize(Integer size) {
            this.size = size;
        }

        public final String getSnapshotId() {
            return snapshotId;
        }

        @Override
        public final Builder snapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
            return this;
        }

        public final void setSnapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
        }

        public final String getStateAsString() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(VolumeState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getVolumeId() {
            return volumeId;
        }

        @Override
        public final Builder volumeId(String volumeId) {
            this.volumeId = volumeId;
            return this;
        }

        public final void setVolumeId(String volumeId) {
            this.volumeId = volumeId;
        }

        public final Integer getIops() {
            return iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final String getVolumeTypeAsString() {
            return volumeType;
        }

        @Override
        public final Builder volumeType(String volumeType) {
            this.volumeType = volumeType;
            return this;
        }

        @Override
        public final Builder volumeType(VolumeType volumeType) {
            this.volumeType(volumeType == null ? null : volumeType.toString());
            return this;
        }

        public final void setVolumeType(String volumeType) {
            this.volumeType = volumeType;
        }

        @Override
        public Volume build() {
            return new Volume(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
