/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import org.reactivestreams.Subscriber;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.core.pagination.async.AsyncPageFetcher;
import software.amazon.awssdk.core.pagination.async.EmptySubscription;
import software.amazon.awssdk.core.pagination.async.PaginatedItemsPublisher;
import software.amazon.awssdk.core.pagination.async.ResponsesSubscription;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.ec2.Ec2AsyncClient;
import software.amazon.awssdk.services.ec2.model.DescribeInstancesRequest;
import software.amazon.awssdk.services.ec2.model.DescribeInstancesResponse;
import software.amazon.awssdk.services.ec2.model.Reservation;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.ec2.Ec2AsyncClient#describeInstancesPaginator(software.amazon.awssdk.services.ec2.model.DescribeInstancesRequest)}
 * operation which is a paginated operation. This class is a type of {@link org.reactivestreams.Publisher} which can be
 * used to provide a sequence of {@link software.amazon.awssdk.services.ec2.model.DescribeInstancesResponse} response
 * pages as per demand from the subscriber.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. If there are errors in your request, you will see the failures
 * only after you start streaming the data. The subscribe method should be called as a request to start streaming data.
 * For more info, see {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the
 * subscribe method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data
 * from the starting request.
 * </p>
 *
 * <p>
 * The following are few ways to use the response class:
 * </p>
 * 1) Using the subscribe helper method
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ec2.paginators.DescribeInstancesPublisher publisher = client.describeInstancesPaginator(request);
 * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
 * future.get();
 * }
 * </pre>
 *
 * 2) Using a custom subscriber
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ec2.paginators.DescribeInstancesPublisher publisher = client.describeInstancesPaginator(request);
 * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ec2.model.DescribeInstancesResponse>() {
 * 
 * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
 * 
 * 
 * public void onNext(software.amazon.awssdk.services.ec2.model.DescribeInstancesResponse response) { //... };
 * });}
 * </pre>
 * 
 * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #describeInstances(software.amazon.awssdk.services.ec2.model.DescribeInstancesRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DescribeInstancesPublisher implements SdkPublisher<DescribeInstancesResponse> {
    private final Ec2AsyncClient client;

    private final DescribeInstancesRequest firstRequest;

    private final AsyncPageFetcher nextPageFetcher;

    private boolean isLastPage;

    public DescribeInstancesPublisher(Ec2AsyncClient client, DescribeInstancesRequest firstRequest) {
        this(client, firstRequest, false);
    }

    private DescribeInstancesPublisher(Ec2AsyncClient client, DescribeInstancesRequest firstRequest, boolean isLastPage) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.isLastPage = isLastPage;
        this.nextPageFetcher = new DescribeInstancesResponseFetcher();
    }

    @Override
    public void subscribe(Subscriber<? super DescribeInstancesResponse> subscriber) {
        subscriber.onSubscribe(ResponsesSubscription.builder().subscriber(subscriber).nextPageFetcher(nextPageFetcher).build());
    }

    /**
     * Returns a publisher that can be used to get a stream of data. You need to subscribe to the publisher to request
     * the stream of data. The publisher has a helper forEach method that takes in a {@link java.util.function.Consumer}
     * and then applies that consumer to each response returned by the service.
     */
    public final SdkPublisher<Reservation> reservations() {
        Function<DescribeInstancesResponse, Iterator<Reservation>> getIterator = response -> {
            if (response != null && response.reservations() != null) {
                return response.reservations().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsPublisher.builder().nextPageFetcher(new DescribeInstancesResponseFetcher())
                .iteratorFunction(getIterator).isLastPage(isLastPage).build();
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link DescribeInstancesPublisher} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    private final DescribeInstancesPublisher resume(DescribeInstancesResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new DescribeInstancesPublisher(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken())
                    .build());
        }
        return new DescribeInstancesPublisher(client, firstRequest, true) {
            @Override
            public void subscribe(Subscriber<? super DescribeInstancesResponse> subscriber) {
                subscriber.onSubscribe(new EmptySubscription(subscriber));
            }
        };
    }

    private class DescribeInstancesResponseFetcher implements AsyncPageFetcher<DescribeInstancesResponse> {
        @Override
        public boolean hasNextPage(final DescribeInstancesResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public CompletableFuture<DescribeInstancesResponse> nextPage(final DescribeInstancesResponse previousPage) {
            if (previousPage == null) {
                return client.describeInstances(firstRequest);
            }
            return client.describeInstances(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
