/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the route propagation configuration between a route server and a route table.
 * </p>
 * <p>
 * When enabled, route server propagation installs the routes in the FIB on the route table you've specified. Route
 * server supports IPv4 and IPv6 route propagation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RouteServerPropagation implements SdkPojo, Serializable,
        ToCopyableBuilder<RouteServerPropagation.Builder, RouteServerPropagation> {
    private static final SdkField<String> ROUTE_SERVER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RouteServerId")
            .getter(getter(RouteServerPropagation::routeServerId))
            .setter(setter(Builder::routeServerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteServerId")
                    .unmarshallLocationName("routeServerId").build()).build();

    private static final SdkField<String> ROUTE_TABLE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RouteTableId")
            .getter(getter(RouteServerPropagation::routeTableId))
            .setter(setter(Builder::routeTableId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteTableId")
                    .unmarshallLocationName("routeTableId").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(RouteServerPropagation::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ROUTE_SERVER_ID_FIELD,
            ROUTE_TABLE_ID_FIELD, STATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String routeServerId;

    private final String routeTableId;

    private final String state;

    private RouteServerPropagation(BuilderImpl builder) {
        this.routeServerId = builder.routeServerId;
        this.routeTableId = builder.routeTableId;
        this.state = builder.state;
    }

    /**
     * <p>
     * The ID of the route server configured for route propagation.
     * </p>
     * 
     * @return The ID of the route server configured for route propagation.
     */
    public final String routeServerId() {
        return routeServerId;
    }

    /**
     * <p>
     * The ID of the route table configured for route server propagation.
     * </p>
     * 
     * @return The ID of the route table configured for route server propagation.
     */
    public final String routeTableId() {
        return routeTableId;
    }

    /**
     * <p>
     * The current state of route propagation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouteServerPropagationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of route propagation.
     * @see RouteServerPropagationState
     */
    public final RouteServerPropagationState state() {
        return RouteServerPropagationState.fromValue(state);
    }

    /**
     * <p>
     * The current state of route propagation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouteServerPropagationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of route propagation.
     * @see RouteServerPropagationState
     */
    public final String stateAsString() {
        return state;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(routeServerId());
        hashCode = 31 * hashCode + Objects.hashCode(routeTableId());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RouteServerPropagation)) {
            return false;
        }
        RouteServerPropagation other = (RouteServerPropagation) obj;
        return Objects.equals(routeServerId(), other.routeServerId()) && Objects.equals(routeTableId(), other.routeTableId())
                && Objects.equals(stateAsString(), other.stateAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RouteServerPropagation").add("RouteServerId", routeServerId())
                .add("RouteTableId", routeTableId()).add("State", stateAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RouteServerId":
            return Optional.ofNullable(clazz.cast(routeServerId()));
        case "RouteTableId":
            return Optional.ofNullable(clazz.cast(routeTableId()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RouteServerId", ROUTE_SERVER_ID_FIELD);
        map.put("RouteTableId", ROUTE_TABLE_ID_FIELD);
        map.put("State", STATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RouteServerPropagation, T> g) {
        return obj -> g.apply((RouteServerPropagation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RouteServerPropagation> {
        /**
         * <p>
         * The ID of the route server configured for route propagation.
         * </p>
         * 
         * @param routeServerId
         *        The ID of the route server configured for route propagation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeServerId(String routeServerId);

        /**
         * <p>
         * The ID of the route table configured for route server propagation.
         * </p>
         * 
         * @param routeTableId
         *        The ID of the route table configured for route server propagation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeTableId(String routeTableId);

        /**
         * <p>
         * The current state of route propagation.
         * </p>
         * 
         * @param state
         *        The current state of route propagation.
         * @see RouteServerPropagationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerPropagationState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of route propagation.
         * </p>
         * 
         * @param state
         *        The current state of route propagation.
         * @see RouteServerPropagationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerPropagationState
         */
        Builder state(RouteServerPropagationState state);
    }

    static final class BuilderImpl implements Builder {
        private String routeServerId;

        private String routeTableId;

        private String state;

        private BuilderImpl() {
        }

        private BuilderImpl(RouteServerPropagation model) {
            routeServerId(model.routeServerId);
            routeTableId(model.routeTableId);
            state(model.state);
        }

        public final String getRouteServerId() {
            return routeServerId;
        }

        public final void setRouteServerId(String routeServerId) {
            this.routeServerId = routeServerId;
        }

        @Override
        public final Builder routeServerId(String routeServerId) {
            this.routeServerId = routeServerId;
            return this;
        }

        public final String getRouteTableId() {
            return routeTableId;
        }

        public final void setRouteTableId(String routeTableId) {
            this.routeTableId = routeTableId;
        }

        @Override
        public final Builder routeTableId(String routeTableId) {
            this.routeTableId = routeTableId;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(RouteServerPropagationState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        @Override
        public RouteServerPropagation build() {
            return new RouteServerPropagation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
