/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A resource type to check for image references. Associated options can also be specified if the resource type is an
 * EC2 instance or launch template.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceTypeRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceTypeRequest.Builder, ResourceTypeRequest> {
    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceType")
            .getter(getter(ResourceTypeRequest::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType")
                    .unmarshallLocationName("ResourceType").build()).build();

    private static final SdkField<List<ResourceTypeOption>> RESOURCE_TYPE_OPTIONS_FIELD = SdkField
            .<List<ResourceTypeOption>> builder(MarshallingType.LIST)
            .memberName("ResourceTypeOptions")
            .getter(getter(ResourceTypeRequest::resourceTypeOptions))
            .setter(setter(Builder::resourceTypeOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTypeOption")
                    .unmarshallLocationName("ResourceTypeOption").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceTypeOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceTypeOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPE_FIELD,
            RESOURCE_TYPE_OPTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String resourceType;

    private final List<ResourceTypeOption> resourceTypeOptions;

    private ResourceTypeRequest(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.resourceTypeOptions = builder.resourceTypeOptions;
    }

    /**
     * <p>
     * The resource type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ImageReferenceResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type.
     * @see ImageReferenceResourceType
     */
    public final ImageReferenceResourceType resourceType() {
        return ImageReferenceResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The resource type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ImageReferenceResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type.
     * @see ImageReferenceResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTypeOptions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTypeOptions() {
        return resourceTypeOptions != null && !(resourceTypeOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The options that affect the scope of the response. Valid only when <code>ResourceType</code> is
     * <code>ec2:Instance</code> or <code>ec2:LaunchTemplate</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypeOptions} method.
     * </p>
     * 
     * @return The options that affect the scope of the response. Valid only when <code>ResourceType</code> is
     *         <code>ec2:Instance</code> or <code>ec2:LaunchTemplate</code>.
     */
    public final List<ResourceTypeOption> resourceTypeOptions() {
        return resourceTypeOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypeOptions() ? resourceTypeOptions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceTypeRequest)) {
            return false;
        }
        ResourceTypeRequest other = (ResourceTypeRequest) obj;
        return Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && hasResourceTypeOptions() == other.hasResourceTypeOptions()
                && Objects.equals(resourceTypeOptions(), other.resourceTypeOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceTypeRequest").add("ResourceType", resourceTypeAsString())
                .add("ResourceTypeOptions", hasResourceTypeOptions() ? resourceTypeOptions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "ResourceTypeOptions":
            return Optional.ofNullable(clazz.cast(resourceTypeOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("ResourceTypeOption", RESOURCE_TYPE_OPTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ResourceTypeRequest, T> g) {
        return obj -> g.apply((ResourceTypeRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceTypeRequest> {
        /**
         * <p>
         * The resource type.
         * </p>
         * 
         * @param resourceType
         *        The resource type.
         * @see ImageReferenceResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageReferenceResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The resource type.
         * </p>
         * 
         * @param resourceType
         *        The resource type.
         * @see ImageReferenceResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageReferenceResourceType
         */
        Builder resourceType(ImageReferenceResourceType resourceType);

        /**
         * <p>
         * The options that affect the scope of the response. Valid only when <code>ResourceType</code> is
         * <code>ec2:Instance</code> or <code>ec2:LaunchTemplate</code>.
         * </p>
         * 
         * @param resourceTypeOptions
         *        The options that affect the scope of the response. Valid only when <code>ResourceType</code> is
         *        <code>ec2:Instance</code> or <code>ec2:LaunchTemplate</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypeOptions(Collection<ResourceTypeOption> resourceTypeOptions);

        /**
         * <p>
         * The options that affect the scope of the response. Valid only when <code>ResourceType</code> is
         * <code>ec2:Instance</code> or <code>ec2:LaunchTemplate</code>.
         * </p>
         * 
         * @param resourceTypeOptions
         *        The options that affect the scope of the response. Valid only when <code>ResourceType</code> is
         *        <code>ec2:Instance</code> or <code>ec2:LaunchTemplate</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypeOptions(ResourceTypeOption... resourceTypeOptions);

        /**
         * <p>
         * The options that affect the scope of the response. Valid only when <code>ResourceType</code> is
         * <code>ec2:Instance</code> or <code>ec2:LaunchTemplate</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.ResourceTypeOption.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.ResourceTypeOption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.ResourceTypeOption.Builder#build()} is called immediately
         * and its result is passed to {@link #resourceTypeOptions(List<ResourceTypeOption>)}.
         * 
         * @param resourceTypeOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.ResourceTypeOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceTypeOptions(java.util.Collection<ResourceTypeOption>)
         */
        Builder resourceTypeOptions(Consumer<ResourceTypeOption.Builder>... resourceTypeOptions);
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private List<ResourceTypeOption> resourceTypeOptions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceTypeRequest model) {
            resourceType(model.resourceType);
            resourceTypeOptions(model.resourceTypeOptions);
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ImageReferenceResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final List<ResourceTypeOption.Builder> getResourceTypeOptions() {
            List<ResourceTypeOption.Builder> result = ResourceTypeOptionListCopier.copyToBuilder(this.resourceTypeOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourceTypeOptions(Collection<ResourceTypeOption.BuilderImpl> resourceTypeOptions) {
            this.resourceTypeOptions = ResourceTypeOptionListCopier.copyFromBuilder(resourceTypeOptions);
        }

        @Override
        public final Builder resourceTypeOptions(Collection<ResourceTypeOption> resourceTypeOptions) {
            this.resourceTypeOptions = ResourceTypeOptionListCopier.copy(resourceTypeOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypeOptions(ResourceTypeOption... resourceTypeOptions) {
            resourceTypeOptions(Arrays.asList(resourceTypeOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypeOptions(Consumer<ResourceTypeOption.Builder>... resourceTypeOptions) {
            resourceTypeOptions(Stream.of(resourceTypeOptions).map(c -> ResourceTypeOption.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ResourceTypeRequest build() {
            return new ResourceTypeRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
