/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the availability of capacity for a Capacity Block.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CapacityBlockStatus implements SdkPojo, Serializable,
        ToCopyableBuilder<CapacityBlockStatus.Builder, CapacityBlockStatus> {
    private static final SdkField<String> CAPACITY_BLOCK_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CapacityBlockId")
            .getter(getter(CapacityBlockStatus::capacityBlockId))
            .setter(setter(Builder::capacityBlockId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityBlockId")
                    .unmarshallLocationName("capacityBlockId").build()).build();

    private static final SdkField<String> INTERCONNECT_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InterconnectStatus")
            .getter(getter(CapacityBlockStatus::interconnectStatusAsString))
            .setter(setter(Builder::interconnectStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InterconnectStatus")
                    .unmarshallLocationName("interconnectStatus").build()).build();

    private static final SdkField<Integer> TOTAL_CAPACITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("TotalCapacity")
            .getter(getter(CapacityBlockStatus::totalCapacity))
            .setter(setter(Builder::totalCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalCapacity")
                    .unmarshallLocationName("totalCapacity").build()).build();

    private static final SdkField<Integer> TOTAL_AVAILABLE_CAPACITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("TotalAvailableCapacity")
            .getter(getter(CapacityBlockStatus::totalAvailableCapacity))
            .setter(setter(Builder::totalAvailableCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalAvailableCapacity")
                    .unmarshallLocationName("totalAvailableCapacity").build()).build();

    private static final SdkField<Integer> TOTAL_UNAVAILABLE_CAPACITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("TotalUnavailableCapacity")
            .getter(getter(CapacityBlockStatus::totalUnavailableCapacity))
            .setter(setter(Builder::totalUnavailableCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalUnavailableCapacity")
                    .unmarshallLocationName("totalUnavailableCapacity").build()).build();

    private static final SdkField<List<CapacityReservationStatus>> CAPACITY_RESERVATION_STATUSES_FIELD = SdkField
            .<List<CapacityReservationStatus>> builder(MarshallingType.LIST)
            .memberName("CapacityReservationStatuses")
            .getter(getter(CapacityBlockStatus::capacityReservationStatuses))
            .setter(setter(Builder::capacityReservationStatuses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityReservationStatusSet")
                    .unmarshallLocationName("capacityReservationStatusSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<CapacityReservationStatus> builder(MarshallingType.SDK_POJO)
                                            .constructor(CapacityReservationStatus::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CAPACITY_BLOCK_ID_FIELD,
            INTERCONNECT_STATUS_FIELD, TOTAL_CAPACITY_FIELD, TOTAL_AVAILABLE_CAPACITY_FIELD, TOTAL_UNAVAILABLE_CAPACITY_FIELD,
            CAPACITY_RESERVATION_STATUSES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String capacityBlockId;

    private final String interconnectStatus;

    private final Integer totalCapacity;

    private final Integer totalAvailableCapacity;

    private final Integer totalUnavailableCapacity;

    private final List<CapacityReservationStatus> capacityReservationStatuses;

    private CapacityBlockStatus(BuilderImpl builder) {
        this.capacityBlockId = builder.capacityBlockId;
        this.interconnectStatus = builder.interconnectStatus;
        this.totalCapacity = builder.totalCapacity;
        this.totalAvailableCapacity = builder.totalAvailableCapacity;
        this.totalUnavailableCapacity = builder.totalUnavailableCapacity;
        this.capacityReservationStatuses = builder.capacityReservationStatuses;
    }

    /**
     * <p>
     * The ID of the Capacity Block.
     * </p>
     * 
     * @return The ID of the Capacity Block.
     */
    public final String capacityBlockId() {
        return capacityBlockId;
    }

    /**
     * <p>
     * The status of the high-bandwidth accelerator interconnect. Possible states include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ok</code> the accelerator interconnect is healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>impaired</code> - accelerator interconnect communication is impaired.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>insufficient-data</code> - insufficient data to determine accelerator interconnect status.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #interconnectStatus} will return {@link CapacityBlockInterconnectStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #interconnectStatusAsString}.
     * </p>
     * 
     * @return The status of the high-bandwidth accelerator interconnect. Possible states include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ok</code> the accelerator interconnect is healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>impaired</code> - accelerator interconnect communication is impaired.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>insufficient-data</code> - insufficient data to determine accelerator interconnect status.
     *         </p>
     *         </li>
     * @see CapacityBlockInterconnectStatus
     */
    public final CapacityBlockInterconnectStatus interconnectStatus() {
        return CapacityBlockInterconnectStatus.fromValue(interconnectStatus);
    }

    /**
     * <p>
     * The status of the high-bandwidth accelerator interconnect. Possible states include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ok</code> the accelerator interconnect is healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>impaired</code> - accelerator interconnect communication is impaired.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>insufficient-data</code> - insufficient data to determine accelerator interconnect status.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #interconnectStatus} will return {@link CapacityBlockInterconnectStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #interconnectStatusAsString}.
     * </p>
     * 
     * @return The status of the high-bandwidth accelerator interconnect. Possible states include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ok</code> the accelerator interconnect is healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>impaired</code> - accelerator interconnect communication is impaired.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>insufficient-data</code> - insufficient data to determine accelerator interconnect status.
     *         </p>
     *         </li>
     * @see CapacityBlockInterconnectStatus
     */
    public final String interconnectStatusAsString() {
        return interconnectStatus;
    }

    /**
     * <p>
     * The combined amount of <code>Available</code> and <code>Unavailable</code> capacity in the Capacity Block.
     * </p>
     * 
     * @return The combined amount of <code>Available</code> and <code>Unavailable</code> capacity in the Capacity
     *         Block.
     */
    public final Integer totalCapacity() {
        return totalCapacity;
    }

    /**
     * <p>
     * The remaining capacity. Indicates the number of resources that can be launched into the Capacity Block.
     * </p>
     * 
     * @return The remaining capacity. Indicates the number of resources that can be launched into the Capacity Block.
     */
    public final Integer totalAvailableCapacity() {
        return totalAvailableCapacity;
    }

    /**
     * <p>
     * The unavailable capacity. Indicates the instance capacity that is unavailable for use due to a system status
     * check failure.
     * </p>
     * 
     * @return The unavailable capacity. Indicates the instance capacity that is unavailable for use due to a system
     *         status check failure.
     */
    public final Integer totalUnavailableCapacity() {
        return totalUnavailableCapacity;
    }

    /**
     * For responses, this returns true if the service returned a value for the CapacityReservationStatuses property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCapacityReservationStatuses() {
        return capacityReservationStatuses != null && !(capacityReservationStatuses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The availability of capacity for the Capacity Block reservations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCapacityReservationStatuses} method.
     * </p>
     * 
     * @return The availability of capacity for the Capacity Block reservations.
     */
    public final List<CapacityReservationStatus> capacityReservationStatuses() {
        return capacityReservationStatuses;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(capacityBlockId());
        hashCode = 31 * hashCode + Objects.hashCode(interconnectStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(totalCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(totalAvailableCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(totalUnavailableCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(hasCapacityReservationStatuses() ? capacityReservationStatuses() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CapacityBlockStatus)) {
            return false;
        }
        CapacityBlockStatus other = (CapacityBlockStatus) obj;
        return Objects.equals(capacityBlockId(), other.capacityBlockId())
                && Objects.equals(interconnectStatusAsString(), other.interconnectStatusAsString())
                && Objects.equals(totalCapacity(), other.totalCapacity())
                && Objects.equals(totalAvailableCapacity(), other.totalAvailableCapacity())
                && Objects.equals(totalUnavailableCapacity(), other.totalUnavailableCapacity())
                && hasCapacityReservationStatuses() == other.hasCapacityReservationStatuses()
                && Objects.equals(capacityReservationStatuses(), other.capacityReservationStatuses());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CapacityBlockStatus").add("CapacityBlockId", capacityBlockId())
                .add("InterconnectStatus", interconnectStatusAsString()).add("TotalCapacity", totalCapacity())
                .add("TotalAvailableCapacity", totalAvailableCapacity())
                .add("TotalUnavailableCapacity", totalUnavailableCapacity())
                .add("CapacityReservationStatuses", hasCapacityReservationStatuses() ? capacityReservationStatuses() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CapacityBlockId":
            return Optional.ofNullable(clazz.cast(capacityBlockId()));
        case "InterconnectStatus":
            return Optional.ofNullable(clazz.cast(interconnectStatusAsString()));
        case "TotalCapacity":
            return Optional.ofNullable(clazz.cast(totalCapacity()));
        case "TotalAvailableCapacity":
            return Optional.ofNullable(clazz.cast(totalAvailableCapacity()));
        case "TotalUnavailableCapacity":
            return Optional.ofNullable(clazz.cast(totalUnavailableCapacity()));
        case "CapacityReservationStatuses":
            return Optional.ofNullable(clazz.cast(capacityReservationStatuses()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CapacityBlockId", CAPACITY_BLOCK_ID_FIELD);
        map.put("InterconnectStatus", INTERCONNECT_STATUS_FIELD);
        map.put("TotalCapacity", TOTAL_CAPACITY_FIELD);
        map.put("TotalAvailableCapacity", TOTAL_AVAILABLE_CAPACITY_FIELD);
        map.put("TotalUnavailableCapacity", TOTAL_UNAVAILABLE_CAPACITY_FIELD);
        map.put("CapacityReservationStatusSet", CAPACITY_RESERVATION_STATUSES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CapacityBlockStatus, T> g) {
        return obj -> g.apply((CapacityBlockStatus) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CapacityBlockStatus> {
        /**
         * <p>
         * The ID of the Capacity Block.
         * </p>
         * 
         * @param capacityBlockId
         *        The ID of the Capacity Block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityBlockId(String capacityBlockId);

        /**
         * <p>
         * The status of the high-bandwidth accelerator interconnect. Possible states include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ok</code> the accelerator interconnect is healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>impaired</code> - accelerator interconnect communication is impaired.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>insufficient-data</code> - insufficient data to determine accelerator interconnect status.
         * </p>
         * </li>
         * </ul>
         * 
         * @param interconnectStatus
         *        The status of the high-bandwidth accelerator interconnect. Possible states include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ok</code> the accelerator interconnect is healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>impaired</code> - accelerator interconnect communication is impaired.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>insufficient-data</code> - insufficient data to determine accelerator interconnect status.
         *        </p>
         *        </li>
         * @see CapacityBlockInterconnectStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapacityBlockInterconnectStatus
         */
        Builder interconnectStatus(String interconnectStatus);

        /**
         * <p>
         * The status of the high-bandwidth accelerator interconnect. Possible states include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ok</code> the accelerator interconnect is healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>impaired</code> - accelerator interconnect communication is impaired.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>insufficient-data</code> - insufficient data to determine accelerator interconnect status.
         * </p>
         * </li>
         * </ul>
         * 
         * @param interconnectStatus
         *        The status of the high-bandwidth accelerator interconnect. Possible states include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ok</code> the accelerator interconnect is healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>impaired</code> - accelerator interconnect communication is impaired.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>insufficient-data</code> - insufficient data to determine accelerator interconnect status.
         *        </p>
         *        </li>
         * @see CapacityBlockInterconnectStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapacityBlockInterconnectStatus
         */
        Builder interconnectStatus(CapacityBlockInterconnectStatus interconnectStatus);

        /**
         * <p>
         * The combined amount of <code>Available</code> and <code>Unavailable</code> capacity in the Capacity Block.
         * </p>
         * 
         * @param totalCapacity
         *        The combined amount of <code>Available</code> and <code>Unavailable</code> capacity in the Capacity
         *        Block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalCapacity(Integer totalCapacity);

        /**
         * <p>
         * The remaining capacity. Indicates the number of resources that can be launched into the Capacity Block.
         * </p>
         * 
         * @param totalAvailableCapacity
         *        The remaining capacity. Indicates the number of resources that can be launched into the Capacity
         *        Block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalAvailableCapacity(Integer totalAvailableCapacity);

        /**
         * <p>
         * The unavailable capacity. Indicates the instance capacity that is unavailable for use due to a system status
         * check failure.
         * </p>
         * 
         * @param totalUnavailableCapacity
         *        The unavailable capacity. Indicates the instance capacity that is unavailable for use due to a system
         *        status check failure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalUnavailableCapacity(Integer totalUnavailableCapacity);

        /**
         * <p>
         * The availability of capacity for the Capacity Block reservations.
         * </p>
         * 
         * @param capacityReservationStatuses
         *        The availability of capacity for the Capacity Block reservations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityReservationStatuses(Collection<CapacityReservationStatus> capacityReservationStatuses);

        /**
         * <p>
         * The availability of capacity for the Capacity Block reservations.
         * </p>
         * 
         * @param capacityReservationStatuses
         *        The availability of capacity for the Capacity Block reservations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityReservationStatuses(CapacityReservationStatus... capacityReservationStatuses);

        /**
         * <p>
         * The availability of capacity for the Capacity Block reservations.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.CapacityReservationStatus.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.ec2.model.CapacityReservationStatus#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.CapacityReservationStatus.Builder#build()} is called
         * immediately and its result is passed to {@link #capacityReservationStatuses(List<CapacityReservationStatus>)}.
         * 
         * @param capacityReservationStatuses
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.CapacityReservationStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capacityReservationStatuses(java.util.Collection<CapacityReservationStatus>)
         */
        Builder capacityReservationStatuses(Consumer<CapacityReservationStatus.Builder>... capacityReservationStatuses);
    }

    static final class BuilderImpl implements Builder {
        private String capacityBlockId;

        private String interconnectStatus;

        private Integer totalCapacity;

        private Integer totalAvailableCapacity;

        private Integer totalUnavailableCapacity;

        private List<CapacityReservationStatus> capacityReservationStatuses = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CapacityBlockStatus model) {
            capacityBlockId(model.capacityBlockId);
            interconnectStatus(model.interconnectStatus);
            totalCapacity(model.totalCapacity);
            totalAvailableCapacity(model.totalAvailableCapacity);
            totalUnavailableCapacity(model.totalUnavailableCapacity);
            capacityReservationStatuses(model.capacityReservationStatuses);
        }

        public final String getCapacityBlockId() {
            return capacityBlockId;
        }

        public final void setCapacityBlockId(String capacityBlockId) {
            this.capacityBlockId = capacityBlockId;
        }

        @Override
        public final Builder capacityBlockId(String capacityBlockId) {
            this.capacityBlockId = capacityBlockId;
            return this;
        }

        public final String getInterconnectStatus() {
            return interconnectStatus;
        }

        public final void setInterconnectStatus(String interconnectStatus) {
            this.interconnectStatus = interconnectStatus;
        }

        @Override
        public final Builder interconnectStatus(String interconnectStatus) {
            this.interconnectStatus = interconnectStatus;
            return this;
        }

        @Override
        public final Builder interconnectStatus(CapacityBlockInterconnectStatus interconnectStatus) {
            this.interconnectStatus(interconnectStatus == null ? null : interconnectStatus.toString());
            return this;
        }

        public final Integer getTotalCapacity() {
            return totalCapacity;
        }

        public final void setTotalCapacity(Integer totalCapacity) {
            this.totalCapacity = totalCapacity;
        }

        @Override
        public final Builder totalCapacity(Integer totalCapacity) {
            this.totalCapacity = totalCapacity;
            return this;
        }

        public final Integer getTotalAvailableCapacity() {
            return totalAvailableCapacity;
        }

        public final void setTotalAvailableCapacity(Integer totalAvailableCapacity) {
            this.totalAvailableCapacity = totalAvailableCapacity;
        }

        @Override
        public final Builder totalAvailableCapacity(Integer totalAvailableCapacity) {
            this.totalAvailableCapacity = totalAvailableCapacity;
            return this;
        }

        public final Integer getTotalUnavailableCapacity() {
            return totalUnavailableCapacity;
        }

        public final void setTotalUnavailableCapacity(Integer totalUnavailableCapacity) {
            this.totalUnavailableCapacity = totalUnavailableCapacity;
        }

        @Override
        public final Builder totalUnavailableCapacity(Integer totalUnavailableCapacity) {
            this.totalUnavailableCapacity = totalUnavailableCapacity;
            return this;
        }

        public final List<CapacityReservationStatus.Builder> getCapacityReservationStatuses() {
            List<CapacityReservationStatus.Builder> result = CapacityReservationStatusSetCopier
                    .copyToBuilder(this.capacityReservationStatuses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCapacityReservationStatuses(
                Collection<CapacityReservationStatus.BuilderImpl> capacityReservationStatuses) {
            this.capacityReservationStatuses = CapacityReservationStatusSetCopier.copyFromBuilder(capacityReservationStatuses);
        }

        @Override
        public final Builder capacityReservationStatuses(Collection<CapacityReservationStatus> capacityReservationStatuses) {
            this.capacityReservationStatuses = CapacityReservationStatusSetCopier.copy(capacityReservationStatuses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capacityReservationStatuses(CapacityReservationStatus... capacityReservationStatuses) {
            capacityReservationStatuses(Arrays.asList(capacityReservationStatuses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capacityReservationStatuses(
                Consumer<CapacityReservationStatus.Builder>... capacityReservationStatuses) {
            capacityReservationStatuses(Stream.of(capacityReservationStatuses)
                    .map(c -> CapacityReservationStatus.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public CapacityBlockStatus build() {
            return new CapacityBlockStatus(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
