/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about a snapshot's storage tier.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SnapshotTierStatus implements SdkPojo, Serializable,
        ToCopyableBuilder<SnapshotTierStatus.Builder, SnapshotTierStatus> {
    private static final SdkField<String> SNAPSHOT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SnapshotId")
            .getter(getter(SnapshotTierStatus::snapshotId))
            .setter(setter(Builder::snapshotId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotId")
                    .unmarshallLocationName("snapshotId").build()).build();

    private static final SdkField<String> VOLUME_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VolumeId")
            .getter(getter(SnapshotTierStatus::volumeId))
            .setter(setter(Builder::volumeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeId")
                    .unmarshallLocationName("volumeId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Status")
            .getter(getter(SnapshotTierStatus::statusAsString))
            .setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status")
                    .unmarshallLocationName("status").build()).build();

    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(SnapshotTierStatus::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(SnapshotTierStatus::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> STORAGE_TIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StorageTier")
            .getter(getter(SnapshotTierStatus::storageTierAsString))
            .setter(setter(Builder::storageTier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageTier")
                    .unmarshallLocationName("storageTier").build()).build();

    private static final SdkField<Instant> LAST_TIERING_START_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastTieringStartTime")
            .getter(getter(SnapshotTierStatus::lastTieringStartTime))
            .setter(setter(Builder::lastTieringStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastTieringStartTime")
                    .unmarshallLocationName("lastTieringStartTime").build()).build();

    private static final SdkField<Integer> LAST_TIERING_PROGRESS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("LastTieringProgress")
            .getter(getter(SnapshotTierStatus::lastTieringProgress))
            .setter(setter(Builder::lastTieringProgress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastTieringProgress")
                    .unmarshallLocationName("lastTieringProgress").build()).build();

    private static final SdkField<String> LAST_TIERING_OPERATION_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LastTieringOperationStatus")
            .getter(getter(SnapshotTierStatus::lastTieringOperationStatusAsString))
            .setter(setter(Builder::lastTieringOperationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastTieringOperationStatus")
                    .unmarshallLocationName("lastTieringOperationStatus").build()).build();

    private static final SdkField<String> LAST_TIERING_OPERATION_STATUS_DETAIL_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LastTieringOperationStatusDetail")
            .getter(getter(SnapshotTierStatus::lastTieringOperationStatusDetail))
            .setter(setter(Builder::lastTieringOperationStatusDetail))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastTieringOperationStatusDetail")
                    .unmarshallLocationName("lastTieringOperationStatusDetail").build()).build();

    private static final SdkField<Instant> ARCHIVAL_COMPLETE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ArchivalCompleteTime")
            .getter(getter(SnapshotTierStatus::archivalCompleteTime))
            .setter(setter(Builder::archivalCompleteTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ArchivalCompleteTime")
                    .unmarshallLocationName("archivalCompleteTime").build()).build();

    private static final SdkField<Instant> RESTORE_EXPIRY_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("RestoreExpiryTime")
            .getter(getter(SnapshotTierStatus::restoreExpiryTime))
            .setter(setter(Builder::restoreExpiryTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RestoreExpiryTime")
                    .unmarshallLocationName("restoreExpiryTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SNAPSHOT_ID_FIELD,
            VOLUME_ID_FIELD, STATUS_FIELD, OWNER_ID_FIELD, TAGS_FIELD, STORAGE_TIER_FIELD, LAST_TIERING_START_TIME_FIELD,
            LAST_TIERING_PROGRESS_FIELD, LAST_TIERING_OPERATION_STATUS_FIELD, LAST_TIERING_OPERATION_STATUS_DETAIL_FIELD,
            ARCHIVAL_COMPLETE_TIME_FIELD, RESTORE_EXPIRY_TIME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String snapshotId;

    private final String volumeId;

    private final String status;

    private final String ownerId;

    private final List<Tag> tags;

    private final String storageTier;

    private final Instant lastTieringStartTime;

    private final Integer lastTieringProgress;

    private final String lastTieringOperationStatus;

    private final String lastTieringOperationStatusDetail;

    private final Instant archivalCompleteTime;

    private final Instant restoreExpiryTime;

    private SnapshotTierStatus(BuilderImpl builder) {
        this.snapshotId = builder.snapshotId;
        this.volumeId = builder.volumeId;
        this.status = builder.status;
        this.ownerId = builder.ownerId;
        this.tags = builder.tags;
        this.storageTier = builder.storageTier;
        this.lastTieringStartTime = builder.lastTieringStartTime;
        this.lastTieringProgress = builder.lastTieringProgress;
        this.lastTieringOperationStatus = builder.lastTieringOperationStatus;
        this.lastTieringOperationStatusDetail = builder.lastTieringOperationStatusDetail;
        this.archivalCompleteTime = builder.archivalCompleteTime;
        this.restoreExpiryTime = builder.restoreExpiryTime;
    }

    /**
     * <p>
     * The ID of the snapshot.
     * </p>
     * 
     * @return The ID of the snapshot.
     */
    public final String snapshotId() {
        return snapshotId;
    }

    /**
     * <p>
     * The ID of the volume from which the snapshot was created.
     * </p>
     * 
     * @return The ID of the volume from which the snapshot was created.
     */
    public final String volumeId() {
        return volumeId;
    }

    /**
     * <p>
     * The state of the snapshot.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link SnapshotState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The state of the snapshot.
     * @see SnapshotState
     */
    public final SnapshotState status() {
        return SnapshotState.fromValue(status);
    }

    /**
     * <p>
     * The state of the snapshot.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link SnapshotState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The state of the snapshot.
     * @see SnapshotState
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services account that owns the snapshot.
     * </p>
     * 
     * @return The ID of the Amazon Web Services account that owns the snapshot.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags that are assigned to the snapshot.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags that are assigned to the snapshot.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The storage tier in which the snapshot is stored. <code>standard</code> indicates that the snapshot is stored in
     * the standard snapshot storage tier and that it is ready for use. <code>archive</code> indicates that the snapshot
     * is currently archived and that it must be restored before it can be used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageTier} will
     * return {@link StorageTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTierAsString}.
     * </p>
     * 
     * @return The storage tier in which the snapshot is stored. <code>standard</code> indicates that the snapshot is
     *         stored in the standard snapshot storage tier and that it is ready for use. <code>archive</code> indicates
     *         that the snapshot is currently archived and that it must be restored before it can be used.
     * @see StorageTier
     */
    public final StorageTier storageTier() {
        return StorageTier.fromValue(storageTier);
    }

    /**
     * <p>
     * The storage tier in which the snapshot is stored. <code>standard</code> indicates that the snapshot is stored in
     * the standard snapshot storage tier and that it is ready for use. <code>archive</code> indicates that the snapshot
     * is currently archived and that it must be restored before it can be used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageTier} will
     * return {@link StorageTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTierAsString}.
     * </p>
     * 
     * @return The storage tier in which the snapshot is stored. <code>standard</code> indicates that the snapshot is
     *         stored in the standard snapshot storage tier and that it is ready for use. <code>archive</code> indicates
     *         that the snapshot is currently archived and that it must be restored before it can be used.
     * @see StorageTier
     */
    public final String storageTierAsString() {
        return storageTier;
    }

    /**
     * <p>
     * The date and time when the last archive or restore process was started.
     * </p>
     * 
     * @return The date and time when the last archive or restore process was started.
     */
    public final Instant lastTieringStartTime() {
        return lastTieringStartTime;
    }

    /**
     * <p>
     * The progress of the last archive or restore process, as a percentage.
     * </p>
     * 
     * @return The progress of the last archive or restore process, as a percentage.
     */
    public final Integer lastTieringProgress() {
        return lastTieringProgress;
    }

    /**
     * <p>
     * The status of the last archive or restore process.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #lastTieringOperationStatus} will return {@link TieringOperationStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #lastTieringOperationStatusAsString}.
     * </p>
     * 
     * @return The status of the last archive or restore process.
     * @see TieringOperationStatus
     */
    public final TieringOperationStatus lastTieringOperationStatus() {
        return TieringOperationStatus.fromValue(lastTieringOperationStatus);
    }

    /**
     * <p>
     * The status of the last archive or restore process.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #lastTieringOperationStatus} will return {@link TieringOperationStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #lastTieringOperationStatusAsString}.
     * </p>
     * 
     * @return The status of the last archive or restore process.
     * @see TieringOperationStatus
     */
    public final String lastTieringOperationStatusAsString() {
        return lastTieringOperationStatus;
    }

    /**
     * <p>
     * A message describing the status of the last archive or restore process.
     * </p>
     * 
     * @return A message describing the status of the last archive or restore process.
     */
    public final String lastTieringOperationStatusDetail() {
        return lastTieringOperationStatusDetail;
    }

    /**
     * <p>
     * The date and time when the last archive process was completed.
     * </p>
     * 
     * @return The date and time when the last archive process was completed.
     */
    public final Instant archivalCompleteTime() {
        return archivalCompleteTime;
    }

    /**
     * <p>
     * Only for archived snapshots that are temporarily restored. Indicates the date and time when a temporarily
     * restored snapshot will be automatically re-archived.
     * </p>
     * 
     * @return Only for archived snapshots that are temporarily restored. Indicates the date and time when a temporarily
     *         restored snapshot will be automatically re-archived.
     */
    public final Instant restoreExpiryTime() {
        return restoreExpiryTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(snapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(volumeId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(storageTierAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastTieringStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastTieringProgress());
        hashCode = 31 * hashCode + Objects.hashCode(lastTieringOperationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastTieringOperationStatusDetail());
        hashCode = 31 * hashCode + Objects.hashCode(archivalCompleteTime());
        hashCode = 31 * hashCode + Objects.hashCode(restoreExpiryTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SnapshotTierStatus)) {
            return false;
        }
        SnapshotTierStatus other = (SnapshotTierStatus) obj;
        return Objects.equals(snapshotId(), other.snapshotId()) && Objects.equals(volumeId(), other.volumeId())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(ownerId(), other.ownerId())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(storageTierAsString(), other.storageTierAsString())
                && Objects.equals(lastTieringStartTime(), other.lastTieringStartTime())
                && Objects.equals(lastTieringProgress(), other.lastTieringProgress())
                && Objects.equals(lastTieringOperationStatusAsString(), other.lastTieringOperationStatusAsString())
                && Objects.equals(lastTieringOperationStatusDetail(), other.lastTieringOperationStatusDetail())
                && Objects.equals(archivalCompleteTime(), other.archivalCompleteTime())
                && Objects.equals(restoreExpiryTime(), other.restoreExpiryTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SnapshotTierStatus").add("SnapshotId", snapshotId()).add("VolumeId", volumeId())
                .add("Status", statusAsString()).add("OwnerId", ownerId()).add("Tags", hasTags() ? tags() : null)
                .add("StorageTier", storageTierAsString()).add("LastTieringStartTime", lastTieringStartTime())
                .add("LastTieringProgress", lastTieringProgress())
                .add("LastTieringOperationStatus", lastTieringOperationStatusAsString())
                .add("LastTieringOperationStatusDetail", lastTieringOperationStatusDetail())
                .add("ArchivalCompleteTime", archivalCompleteTime()).add("RestoreExpiryTime", restoreExpiryTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SnapshotId":
            return Optional.ofNullable(clazz.cast(snapshotId()));
        case "VolumeId":
            return Optional.ofNullable(clazz.cast(volumeId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "StorageTier":
            return Optional.ofNullable(clazz.cast(storageTierAsString()));
        case "LastTieringStartTime":
            return Optional.ofNullable(clazz.cast(lastTieringStartTime()));
        case "LastTieringProgress":
            return Optional.ofNullable(clazz.cast(lastTieringProgress()));
        case "LastTieringOperationStatus":
            return Optional.ofNullable(clazz.cast(lastTieringOperationStatusAsString()));
        case "LastTieringOperationStatusDetail":
            return Optional.ofNullable(clazz.cast(lastTieringOperationStatusDetail()));
        case "ArchivalCompleteTime":
            return Optional.ofNullable(clazz.cast(archivalCompleteTime()));
        case "RestoreExpiryTime":
            return Optional.ofNullable(clazz.cast(restoreExpiryTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SnapshotId", SNAPSHOT_ID_FIELD);
        map.put("VolumeId", VOLUME_ID_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("OwnerId", OWNER_ID_FIELD);
        map.put("TagSet", TAGS_FIELD);
        map.put("StorageTier", STORAGE_TIER_FIELD);
        map.put("LastTieringStartTime", LAST_TIERING_START_TIME_FIELD);
        map.put("LastTieringProgress", LAST_TIERING_PROGRESS_FIELD);
        map.put("LastTieringOperationStatus", LAST_TIERING_OPERATION_STATUS_FIELD);
        map.put("LastTieringOperationStatusDetail", LAST_TIERING_OPERATION_STATUS_DETAIL_FIELD);
        map.put("ArchivalCompleteTime", ARCHIVAL_COMPLETE_TIME_FIELD);
        map.put("RestoreExpiryTime", RESTORE_EXPIRY_TIME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SnapshotTierStatus, T> g) {
        return obj -> g.apply((SnapshotTierStatus) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SnapshotTierStatus> {
        /**
         * <p>
         * The ID of the snapshot.
         * </p>
         * 
         * @param snapshotId
         *        The ID of the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotId(String snapshotId);

        /**
         * <p>
         * The ID of the volume from which the snapshot was created.
         * </p>
         * 
         * @param volumeId
         *        The ID of the volume from which the snapshot was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeId(String volumeId);

        /**
         * <p>
         * The state of the snapshot.
         * </p>
         * 
         * @param status
         *        The state of the snapshot.
         * @see SnapshotState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SnapshotState
         */
        Builder status(String status);

        /**
         * <p>
         * The state of the snapshot.
         * </p>
         * 
         * @param status
         *        The state of the snapshot.
         * @see SnapshotState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SnapshotState
         */
        Builder status(SnapshotState status);

        /**
         * <p>
         * The ID of the Amazon Web Services account that owns the snapshot.
         * </p>
         * 
         * @param ownerId
         *        The ID of the Amazon Web Services account that owns the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The tags that are assigned to the snapshot.
         * </p>
         * 
         * @param tags
         *        The tags that are assigned to the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags that are assigned to the snapshot.
         * </p>
         * 
         * @param tags
         *        The tags that are assigned to the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags that are assigned to the snapshot.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The storage tier in which the snapshot is stored. <code>standard</code> indicates that the snapshot is stored
         * in the standard snapshot storage tier and that it is ready for use. <code>archive</code> indicates that the
         * snapshot is currently archived and that it must be restored before it can be used.
         * </p>
         * 
         * @param storageTier
         *        The storage tier in which the snapshot is stored. <code>standard</code> indicates that the snapshot is
         *        stored in the standard snapshot storage tier and that it is ready for use. <code>archive</code>
         *        indicates that the snapshot is currently archived and that it must be restored before it can be used.
         * @see StorageTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageTier
         */
        Builder storageTier(String storageTier);

        /**
         * <p>
         * The storage tier in which the snapshot is stored. <code>standard</code> indicates that the snapshot is stored
         * in the standard snapshot storage tier and that it is ready for use. <code>archive</code> indicates that the
         * snapshot is currently archived and that it must be restored before it can be used.
         * </p>
         * 
         * @param storageTier
         *        The storage tier in which the snapshot is stored. <code>standard</code> indicates that the snapshot is
         *        stored in the standard snapshot storage tier and that it is ready for use. <code>archive</code>
         *        indicates that the snapshot is currently archived and that it must be restored before it can be used.
         * @see StorageTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageTier
         */
        Builder storageTier(StorageTier storageTier);

        /**
         * <p>
         * The date and time when the last archive or restore process was started.
         * </p>
         * 
         * @param lastTieringStartTime
         *        The date and time when the last archive or restore process was started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastTieringStartTime(Instant lastTieringStartTime);

        /**
         * <p>
         * The progress of the last archive or restore process, as a percentage.
         * </p>
         * 
         * @param lastTieringProgress
         *        The progress of the last archive or restore process, as a percentage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastTieringProgress(Integer lastTieringProgress);

        /**
         * <p>
         * The status of the last archive or restore process.
         * </p>
         * 
         * @param lastTieringOperationStatus
         *        The status of the last archive or restore process.
         * @see TieringOperationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TieringOperationStatus
         */
        Builder lastTieringOperationStatus(String lastTieringOperationStatus);

        /**
         * <p>
         * The status of the last archive or restore process.
         * </p>
         * 
         * @param lastTieringOperationStatus
         *        The status of the last archive or restore process.
         * @see TieringOperationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TieringOperationStatus
         */
        Builder lastTieringOperationStatus(TieringOperationStatus lastTieringOperationStatus);

        /**
         * <p>
         * A message describing the status of the last archive or restore process.
         * </p>
         * 
         * @param lastTieringOperationStatusDetail
         *        A message describing the status of the last archive or restore process.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastTieringOperationStatusDetail(String lastTieringOperationStatusDetail);

        /**
         * <p>
         * The date and time when the last archive process was completed.
         * </p>
         * 
         * @param archivalCompleteTime
         *        The date and time when the last archive process was completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder archivalCompleteTime(Instant archivalCompleteTime);

        /**
         * <p>
         * Only for archived snapshots that are temporarily restored. Indicates the date and time when a temporarily
         * restored snapshot will be automatically re-archived.
         * </p>
         * 
         * @param restoreExpiryTime
         *        Only for archived snapshots that are temporarily restored. Indicates the date and time when a
         *        temporarily restored snapshot will be automatically re-archived.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder restoreExpiryTime(Instant restoreExpiryTime);
    }

    static final class BuilderImpl implements Builder {
        private String snapshotId;

        private String volumeId;

        private String status;

        private String ownerId;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String storageTier;

        private Instant lastTieringStartTime;

        private Integer lastTieringProgress;

        private String lastTieringOperationStatus;

        private String lastTieringOperationStatusDetail;

        private Instant archivalCompleteTime;

        private Instant restoreExpiryTime;

        private BuilderImpl() {
        }

        private BuilderImpl(SnapshotTierStatus model) {
            snapshotId(model.snapshotId);
            volumeId(model.volumeId);
            status(model.status);
            ownerId(model.ownerId);
            tags(model.tags);
            storageTier(model.storageTier);
            lastTieringStartTime(model.lastTieringStartTime);
            lastTieringProgress(model.lastTieringProgress);
            lastTieringOperationStatus(model.lastTieringOperationStatus);
            lastTieringOperationStatusDetail(model.lastTieringOperationStatusDetail);
            archivalCompleteTime(model.archivalCompleteTime);
            restoreExpiryTime(model.restoreExpiryTime);
        }

        public final String getSnapshotId() {
            return snapshotId;
        }

        public final void setSnapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
        }

        @Override
        public final Builder snapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
            return this;
        }

        public final String getVolumeId() {
            return volumeId;
        }

        public final void setVolumeId(String volumeId) {
            this.volumeId = volumeId;
        }

        @Override
        public final Builder volumeId(String volumeId) {
            this.volumeId = volumeId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(SnapshotState status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getStorageTier() {
            return storageTier;
        }

        public final void setStorageTier(String storageTier) {
            this.storageTier = storageTier;
        }

        @Override
        public final Builder storageTier(String storageTier) {
            this.storageTier = storageTier;
            return this;
        }

        @Override
        public final Builder storageTier(StorageTier storageTier) {
            this.storageTier(storageTier == null ? null : storageTier.toString());
            return this;
        }

        public final Instant getLastTieringStartTime() {
            return lastTieringStartTime;
        }

        public final void setLastTieringStartTime(Instant lastTieringStartTime) {
            this.lastTieringStartTime = lastTieringStartTime;
        }

        @Override
        public final Builder lastTieringStartTime(Instant lastTieringStartTime) {
            this.lastTieringStartTime = lastTieringStartTime;
            return this;
        }

        public final Integer getLastTieringProgress() {
            return lastTieringProgress;
        }

        public final void setLastTieringProgress(Integer lastTieringProgress) {
            this.lastTieringProgress = lastTieringProgress;
        }

        @Override
        public final Builder lastTieringProgress(Integer lastTieringProgress) {
            this.lastTieringProgress = lastTieringProgress;
            return this;
        }

        public final String getLastTieringOperationStatus() {
            return lastTieringOperationStatus;
        }

        public final void setLastTieringOperationStatus(String lastTieringOperationStatus) {
            this.lastTieringOperationStatus = lastTieringOperationStatus;
        }

        @Override
        public final Builder lastTieringOperationStatus(String lastTieringOperationStatus) {
            this.lastTieringOperationStatus = lastTieringOperationStatus;
            return this;
        }

        @Override
        public final Builder lastTieringOperationStatus(TieringOperationStatus lastTieringOperationStatus) {
            this.lastTieringOperationStatus(lastTieringOperationStatus == null ? null : lastTieringOperationStatus.toString());
            return this;
        }

        public final String getLastTieringOperationStatusDetail() {
            return lastTieringOperationStatusDetail;
        }

        public final void setLastTieringOperationStatusDetail(String lastTieringOperationStatusDetail) {
            this.lastTieringOperationStatusDetail = lastTieringOperationStatusDetail;
        }

        @Override
        public final Builder lastTieringOperationStatusDetail(String lastTieringOperationStatusDetail) {
            this.lastTieringOperationStatusDetail = lastTieringOperationStatusDetail;
            return this;
        }

        public final Instant getArchivalCompleteTime() {
            return archivalCompleteTime;
        }

        public final void setArchivalCompleteTime(Instant archivalCompleteTime) {
            this.archivalCompleteTime = archivalCompleteTime;
        }

        @Override
        public final Builder archivalCompleteTime(Instant archivalCompleteTime) {
            this.archivalCompleteTime = archivalCompleteTime;
            return this;
        }

        public final Instant getRestoreExpiryTime() {
            return restoreExpiryTime;
        }

        public final void setRestoreExpiryTime(Instant restoreExpiryTime) {
            this.restoreExpiryTime = restoreExpiryTime;
        }

        @Override
        public final Builder restoreExpiryTime(Instant restoreExpiryTime) {
            this.restoreExpiryTime = restoreExpiryTime;
            return this;
        }

        @Override
        public SnapshotTierStatus build() {
            return new SnapshotTierStatus(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
