/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A public IP Address discovered by IPAM.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamDiscoveredPublicAddress implements SdkPojo, Serializable,
        ToCopyableBuilder<IpamDiscoveredPublicAddress.Builder, IpamDiscoveredPublicAddress> {
    private static final SdkField<String> IPAM_RESOURCE_DISCOVERY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamResourceDiscoveryId")
            .getter(getter(IpamDiscoveredPublicAddress::ipamResourceDiscoveryId))
            .setter(setter(Builder::ipamResourceDiscoveryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamResourceDiscoveryId")
                    .unmarshallLocationName("ipamResourceDiscoveryId").build()).build();

    private static final SdkField<String> ADDRESS_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AddressRegion")
            .getter(getter(IpamDiscoveredPublicAddress::addressRegion))
            .setter(setter(Builder::addressRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddressRegion")
                    .unmarshallLocationName("addressRegion").build()).build();

    private static final SdkField<String> ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Address")
            .getter(getter(IpamDiscoveredPublicAddress::address))
            .setter(setter(Builder::address))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Address")
                    .unmarshallLocationName("address").build()).build();

    private static final SdkField<String> ADDRESS_OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AddressOwnerId")
            .getter(getter(IpamDiscoveredPublicAddress::addressOwnerId))
            .setter(setter(Builder::addressOwnerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddressOwnerId")
                    .unmarshallLocationName("addressOwnerId").build()).build();

    private static final SdkField<String> ADDRESS_ALLOCATION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AddressAllocationId")
            .getter(getter(IpamDiscoveredPublicAddress::addressAllocationId))
            .setter(setter(Builder::addressAllocationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddressAllocationId")
                    .unmarshallLocationName("addressAllocationId").build()).build();

    private static final SdkField<String> ASSOCIATION_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AssociationStatus")
            .getter(getter(IpamDiscoveredPublicAddress::associationStatusAsString))
            .setter(setter(Builder::associationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationStatus")
                    .unmarshallLocationName("associationStatus").build()).build();

    private static final SdkField<String> ADDRESS_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AddressType")
            .getter(getter(IpamDiscoveredPublicAddress::addressTypeAsString))
            .setter(setter(Builder::addressType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddressType")
                    .unmarshallLocationName("addressType").build()).build();

    private static final SdkField<String> SERVICE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Service")
            .getter(getter(IpamDiscoveredPublicAddress::serviceAsString))
            .setter(setter(Builder::service))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Service")
                    .unmarshallLocationName("service").build()).build();

    private static final SdkField<String> SERVICE_RESOURCE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ServiceResource")
            .getter(getter(IpamDiscoveredPublicAddress::serviceResource))
            .setter(setter(Builder::serviceResource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceResource")
                    .unmarshallLocationName("serviceResource").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcId")
            .getter(getter(IpamDiscoveredPublicAddress::vpcId))
            .setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId")
                    .unmarshallLocationName("vpcId").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SubnetId")
            .getter(getter(IpamDiscoveredPublicAddress::subnetId))
            .setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId")
                    .unmarshallLocationName("subnetId").build()).build();

    private static final SdkField<String> PUBLIC_IPV4_POOL_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PublicIpv4PoolId")
            .getter(getter(IpamDiscoveredPublicAddress::publicIpv4PoolId))
            .setter(setter(Builder::publicIpv4PoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicIpv4PoolId")
                    .unmarshallLocationName("publicIpv4PoolId").build()).build();

    private static final SdkField<String> NETWORK_INTERFACE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NetworkInterfaceId")
            .getter(getter(IpamDiscoveredPublicAddress::networkInterfaceId))
            .setter(setter(Builder::networkInterfaceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceId")
                    .unmarshallLocationName("networkInterfaceId").build()).build();

    private static final SdkField<String> NETWORK_INTERFACE_DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NetworkInterfaceDescription")
            .getter(getter(IpamDiscoveredPublicAddress::networkInterfaceDescription))
            .setter(setter(Builder::networkInterfaceDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceDescription")
                    .unmarshallLocationName("networkInterfaceDescription").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceId")
            .getter(getter(IpamDiscoveredPublicAddress::instanceId))
            .setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId")
                    .unmarshallLocationName("instanceId").build()).build();

    private static final SdkField<IpamPublicAddressTags> TAGS_FIELD = SdkField
            .<IpamPublicAddressTags> builder(MarshallingType.SDK_POJO)
            .memberName("Tags")
            .getter(getter(IpamDiscoveredPublicAddress::tags))
            .setter(setter(Builder::tags))
            .constructor(IpamPublicAddressTags::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags")
                    .unmarshallLocationName("tags").build()).build();

    private static final SdkField<String> NETWORK_BORDER_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NetworkBorderGroup")
            .getter(getter(IpamDiscoveredPublicAddress::networkBorderGroup))
            .setter(setter(Builder::networkBorderGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkBorderGroup")
                    .unmarshallLocationName("networkBorderGroup").build()).build();

    private static final SdkField<List<IpamPublicAddressSecurityGroup>> SECURITY_GROUPS_FIELD = SdkField
            .<List<IpamPublicAddressSecurityGroup>> builder(MarshallingType.LIST)
            .memberName("SecurityGroups")
            .getter(getter(IpamDiscoveredPublicAddress::securityGroups))
            .setter(setter(Builder::securityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupSet")
                    .unmarshallLocationName("securityGroupSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<IpamPublicAddressSecurityGroup> builder(MarshallingType.SDK_POJO)
                                            .constructor(IpamPublicAddressSecurityGroup::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Instant> SAMPLE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("SampleTime")
            .getter(getter(IpamDiscoveredPublicAddress::sampleTime))
            .setter(setter(Builder::sampleTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SampleTime")
                    .unmarshallLocationName("sampleTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            IPAM_RESOURCE_DISCOVERY_ID_FIELD, ADDRESS_REGION_FIELD, ADDRESS_FIELD, ADDRESS_OWNER_ID_FIELD,
            ADDRESS_ALLOCATION_ID_FIELD, ASSOCIATION_STATUS_FIELD, ADDRESS_TYPE_FIELD, SERVICE_FIELD, SERVICE_RESOURCE_FIELD,
            VPC_ID_FIELD, SUBNET_ID_FIELD, PUBLIC_IPV4_POOL_ID_FIELD, NETWORK_INTERFACE_ID_FIELD,
            NETWORK_INTERFACE_DESCRIPTION_FIELD, INSTANCE_ID_FIELD, TAGS_FIELD, NETWORK_BORDER_GROUP_FIELD,
            SECURITY_GROUPS_FIELD, SAMPLE_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ipamResourceDiscoveryId;

    private final String addressRegion;

    private final String address;

    private final String addressOwnerId;

    private final String addressAllocationId;

    private final String associationStatus;

    private final String addressType;

    private final String service;

    private final String serviceResource;

    private final String vpcId;

    private final String subnetId;

    private final String publicIpv4PoolId;

    private final String networkInterfaceId;

    private final String networkInterfaceDescription;

    private final String instanceId;

    private final IpamPublicAddressTags tags;

    private final String networkBorderGroup;

    private final List<IpamPublicAddressSecurityGroup> securityGroups;

    private final Instant sampleTime;

    private IpamDiscoveredPublicAddress(BuilderImpl builder) {
        this.ipamResourceDiscoveryId = builder.ipamResourceDiscoveryId;
        this.addressRegion = builder.addressRegion;
        this.address = builder.address;
        this.addressOwnerId = builder.addressOwnerId;
        this.addressAllocationId = builder.addressAllocationId;
        this.associationStatus = builder.associationStatus;
        this.addressType = builder.addressType;
        this.service = builder.service;
        this.serviceResource = builder.serviceResource;
        this.vpcId = builder.vpcId;
        this.subnetId = builder.subnetId;
        this.publicIpv4PoolId = builder.publicIpv4PoolId;
        this.networkInterfaceId = builder.networkInterfaceId;
        this.networkInterfaceDescription = builder.networkInterfaceDescription;
        this.instanceId = builder.instanceId;
        this.tags = builder.tags;
        this.networkBorderGroup = builder.networkBorderGroup;
        this.securityGroups = builder.securityGroups;
        this.sampleTime = builder.sampleTime;
    }

    /**
     * <p>
     * The resource discovery ID.
     * </p>
     * 
     * @return The resource discovery ID.
     */
    public final String ipamResourceDiscoveryId() {
        return ipamResourceDiscoveryId;
    }

    /**
     * <p>
     * The Region of the resource the IP address is assigned to.
     * </p>
     * 
     * @return The Region of the resource the IP address is assigned to.
     */
    public final String addressRegion() {
        return addressRegion;
    }

    /**
     * <p>
     * The IP address.
     * </p>
     * 
     * @return The IP address.
     */
    public final String address() {
        return address;
    }

    /**
     * <p>
     * The ID of the owner of the resource the IP address is assigned to.
     * </p>
     * 
     * @return The ID of the owner of the resource the IP address is assigned to.
     */
    public final String addressOwnerId() {
        return addressOwnerId;
    }

    /**
     * <p>
     * The allocation ID of the resource the IP address is assigned to.
     * </p>
     * 
     * @return The allocation ID of the resource the IP address is assigned to.
     */
    public final String addressAllocationId() {
        return addressAllocationId;
    }

    /**
     * <p>
     * The association status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #associationStatus}
     * will return {@link IpamPublicAddressAssociationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #associationStatusAsString}.
     * </p>
     * 
     * @return The association status.
     * @see IpamPublicAddressAssociationStatus
     */
    public final IpamPublicAddressAssociationStatus associationStatus() {
        return IpamPublicAddressAssociationStatus.fromValue(associationStatus);
    }

    /**
     * <p>
     * The association status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #associationStatus}
     * will return {@link IpamPublicAddressAssociationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #associationStatusAsString}.
     * </p>
     * 
     * @return The association status.
     * @see IpamPublicAddressAssociationStatus
     */
    public final String associationStatusAsString() {
        return associationStatus;
    }

    /**
     * <p>
     * The IP address type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #addressType} will
     * return {@link IpamPublicAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #addressTypeAsString}.
     * </p>
     * 
     * @return The IP address type.
     * @see IpamPublicAddressType
     */
    public final IpamPublicAddressType addressType() {
        return IpamPublicAddressType.fromValue(addressType);
    }

    /**
     * <p>
     * The IP address type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #addressType} will
     * return {@link IpamPublicAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #addressTypeAsString}.
     * </p>
     * 
     * @return The IP address type.
     * @see IpamPublicAddressType
     */
    public final String addressTypeAsString() {
        return addressType;
    }

    /**
     * <p>
     * The Amazon Web Services service associated with the IP address.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #service} will
     * return {@link IpamPublicAddressAwsService#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #serviceAsString}.
     * </p>
     * 
     * @return The Amazon Web Services service associated with the IP address.
     * @see IpamPublicAddressAwsService
     */
    public final IpamPublicAddressAwsService service() {
        return IpamPublicAddressAwsService.fromValue(service);
    }

    /**
     * <p>
     * The Amazon Web Services service associated with the IP address.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #service} will
     * return {@link IpamPublicAddressAwsService#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #serviceAsString}.
     * </p>
     * 
     * @return The Amazon Web Services service associated with the IP address.
     * @see IpamPublicAddressAwsService
     */
    public final String serviceAsString() {
        return service;
    }

    /**
     * <p>
     * The resource ARN or ID.
     * </p>
     * 
     * @return The resource ARN or ID.
     */
    public final String serviceResource() {
        return serviceResource;
    }

    /**
     * <p>
     * The ID of the VPC that the resource with the assigned IP address is in.
     * </p>
     * 
     * @return The ID of the VPC that the resource with the assigned IP address is in.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The ID of the subnet that the resource with the assigned IP address is in.
     * </p>
     * 
     * @return The ID of the subnet that the resource with the assigned IP address is in.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The ID of the public IPv4 pool that the resource with the assigned IP address is from.
     * </p>
     * 
     * @return The ID of the public IPv4 pool that the resource with the assigned IP address is from.
     */
    public final String publicIpv4PoolId() {
        return publicIpv4PoolId;
    }

    /**
     * <p>
     * The network interface ID of the resource with the assigned IP address.
     * </p>
     * 
     * @return The network interface ID of the resource with the assigned IP address.
     */
    public final String networkInterfaceId() {
        return networkInterfaceId;
    }

    /**
     * <p>
     * The description of the network interface that IP address is assigned to.
     * </p>
     * 
     * @return The description of the network interface that IP address is assigned to.
     */
    public final String networkInterfaceDescription() {
        return networkInterfaceDescription;
    }

    /**
     * <p>
     * The instance ID of the instance the assigned IP address is assigned to.
     * </p>
     * 
     * @return The instance ID of the instance the assigned IP address is assigned to.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * Tags associated with the IP address.
     * </p>
     * 
     * @return Tags associated with the IP address.
     */
    public final IpamPublicAddressTags tags() {
        return tags;
    }

    /**
     * <p>
     * The Availability Zone (AZ) or Local Zone (LZ) network border group that the resource that the IP address is
     * assigned to is in. Defaults to an AZ network border group. For more information on available Local Zones, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">Local Zone
     * availability</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * 
     * @return The Availability Zone (AZ) or Local Zone (LZ) network border group that the resource that the IP address
     *         is assigned to is in. Defaults to an AZ network border group. For more information on available Local
     *         Zones, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">Local Zone
     *         availability</a> in the <i>Amazon EC2 User Guide</i>.
     */
    public final String networkBorderGroup() {
        return networkBorderGroup;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroups() {
        return securityGroups != null && !(securityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Security groups associated with the resource that the IP address is assigned to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroups} method.
     * </p>
     * 
     * @return Security groups associated with the resource that the IP address is assigned to.
     */
    public final List<IpamPublicAddressSecurityGroup> securityGroups() {
        return securityGroups;
    }

    /**
     * <p>
     * The last successful resource discovery time.
     * </p>
     * 
     * @return The last successful resource discovery time.
     */
    public final Instant sampleTime() {
        return sampleTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ipamResourceDiscoveryId());
        hashCode = 31 * hashCode + Objects.hashCode(addressRegion());
        hashCode = 31 * hashCode + Objects.hashCode(address());
        hashCode = 31 * hashCode + Objects.hashCode(addressOwnerId());
        hashCode = 31 * hashCode + Objects.hashCode(addressAllocationId());
        hashCode = 31 * hashCode + Objects.hashCode(associationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(addressTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(serviceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(serviceResource());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(publicIpv4PoolId());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaceId());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaceDescription());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(networkBorderGroup());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroups() ? securityGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sampleTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamDiscoveredPublicAddress)) {
            return false;
        }
        IpamDiscoveredPublicAddress other = (IpamDiscoveredPublicAddress) obj;
        return Objects.equals(ipamResourceDiscoveryId(), other.ipamResourceDiscoveryId())
                && Objects.equals(addressRegion(), other.addressRegion()) && Objects.equals(address(), other.address())
                && Objects.equals(addressOwnerId(), other.addressOwnerId())
                && Objects.equals(addressAllocationId(), other.addressAllocationId())
                && Objects.equals(associationStatusAsString(), other.associationStatusAsString())
                && Objects.equals(addressTypeAsString(), other.addressTypeAsString())
                && Objects.equals(serviceAsString(), other.serviceAsString())
                && Objects.equals(serviceResource(), other.serviceResource()) && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(subnetId(), other.subnetId()) && Objects.equals(publicIpv4PoolId(), other.publicIpv4PoolId())
                && Objects.equals(networkInterfaceId(), other.networkInterfaceId())
                && Objects.equals(networkInterfaceDescription(), other.networkInterfaceDescription())
                && Objects.equals(instanceId(), other.instanceId()) && Objects.equals(tags(), other.tags())
                && Objects.equals(networkBorderGroup(), other.networkBorderGroup())
                && hasSecurityGroups() == other.hasSecurityGroups() && Objects.equals(securityGroups(), other.securityGroups())
                && Objects.equals(sampleTime(), other.sampleTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamDiscoveredPublicAddress").add("IpamResourceDiscoveryId", ipamResourceDiscoveryId())
                .add("AddressRegion", addressRegion()).add("Address", address()).add("AddressOwnerId", addressOwnerId())
                .add("AddressAllocationId", addressAllocationId()).add("AssociationStatus", associationStatusAsString())
                .add("AddressType", addressTypeAsString()).add("Service", serviceAsString())
                .add("ServiceResource", serviceResource()).add("VpcId", vpcId()).add("SubnetId", subnetId())
                .add("PublicIpv4PoolId", publicIpv4PoolId()).add("NetworkInterfaceId", networkInterfaceId())
                .add("NetworkInterfaceDescription", networkInterfaceDescription()).add("InstanceId", instanceId())
                .add("Tags", tags()).add("NetworkBorderGroup", networkBorderGroup())
                .add("SecurityGroups", hasSecurityGroups() ? securityGroups() : null).add("SampleTime", sampleTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IpamResourceDiscoveryId":
            return Optional.ofNullable(clazz.cast(ipamResourceDiscoveryId()));
        case "AddressRegion":
            return Optional.ofNullable(clazz.cast(addressRegion()));
        case "Address":
            return Optional.ofNullable(clazz.cast(address()));
        case "AddressOwnerId":
            return Optional.ofNullable(clazz.cast(addressOwnerId()));
        case "AddressAllocationId":
            return Optional.ofNullable(clazz.cast(addressAllocationId()));
        case "AssociationStatus":
            return Optional.ofNullable(clazz.cast(associationStatusAsString()));
        case "AddressType":
            return Optional.ofNullable(clazz.cast(addressTypeAsString()));
        case "Service":
            return Optional.ofNullable(clazz.cast(serviceAsString()));
        case "ServiceResource":
            return Optional.ofNullable(clazz.cast(serviceResource()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "PublicIpv4PoolId":
            return Optional.ofNullable(clazz.cast(publicIpv4PoolId()));
        case "NetworkInterfaceId":
            return Optional.ofNullable(clazz.cast(networkInterfaceId()));
        case "NetworkInterfaceDescription":
            return Optional.ofNullable(clazz.cast(networkInterfaceDescription()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "NetworkBorderGroup":
            return Optional.ofNullable(clazz.cast(networkBorderGroup()));
        case "SecurityGroups":
            return Optional.ofNullable(clazz.cast(securityGroups()));
        case "SampleTime":
            return Optional.ofNullable(clazz.cast(sampleTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IpamDiscoveredPublicAddress, T> g) {
        return obj -> g.apply((IpamDiscoveredPublicAddress) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamDiscoveredPublicAddress> {
        /**
         * <p>
         * The resource discovery ID.
         * </p>
         * 
         * @param ipamResourceDiscoveryId
         *        The resource discovery ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamResourceDiscoveryId(String ipamResourceDiscoveryId);

        /**
         * <p>
         * The Region of the resource the IP address is assigned to.
         * </p>
         * 
         * @param addressRegion
         *        The Region of the resource the IP address is assigned to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addressRegion(String addressRegion);

        /**
         * <p>
         * The IP address.
         * </p>
         * 
         * @param address
         *        The IP address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder address(String address);

        /**
         * <p>
         * The ID of the owner of the resource the IP address is assigned to.
         * </p>
         * 
         * @param addressOwnerId
         *        The ID of the owner of the resource the IP address is assigned to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addressOwnerId(String addressOwnerId);

        /**
         * <p>
         * The allocation ID of the resource the IP address is assigned to.
         * </p>
         * 
         * @param addressAllocationId
         *        The allocation ID of the resource the IP address is assigned to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addressAllocationId(String addressAllocationId);

        /**
         * <p>
         * The association status.
         * </p>
         * 
         * @param associationStatus
         *        The association status.
         * @see IpamPublicAddressAssociationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPublicAddressAssociationStatus
         */
        Builder associationStatus(String associationStatus);

        /**
         * <p>
         * The association status.
         * </p>
         * 
         * @param associationStatus
         *        The association status.
         * @see IpamPublicAddressAssociationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPublicAddressAssociationStatus
         */
        Builder associationStatus(IpamPublicAddressAssociationStatus associationStatus);

        /**
         * <p>
         * The IP address type.
         * </p>
         * 
         * @param addressType
         *        The IP address type.
         * @see IpamPublicAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPublicAddressType
         */
        Builder addressType(String addressType);

        /**
         * <p>
         * The IP address type.
         * </p>
         * 
         * @param addressType
         *        The IP address type.
         * @see IpamPublicAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPublicAddressType
         */
        Builder addressType(IpamPublicAddressType addressType);

        /**
         * <p>
         * The Amazon Web Services service associated with the IP address.
         * </p>
         * 
         * @param service
         *        The Amazon Web Services service associated with the IP address.
         * @see IpamPublicAddressAwsService
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPublicAddressAwsService
         */
        Builder service(String service);

        /**
         * <p>
         * The Amazon Web Services service associated with the IP address.
         * </p>
         * 
         * @param service
         *        The Amazon Web Services service associated with the IP address.
         * @see IpamPublicAddressAwsService
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPublicAddressAwsService
         */
        Builder service(IpamPublicAddressAwsService service);

        /**
         * <p>
         * The resource ARN or ID.
         * </p>
         * 
         * @param serviceResource
         *        The resource ARN or ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceResource(String serviceResource);

        /**
         * <p>
         * The ID of the VPC that the resource with the assigned IP address is in.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC that the resource with the assigned IP address is in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The ID of the subnet that the resource with the assigned IP address is in.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet that the resource with the assigned IP address is in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The ID of the public IPv4 pool that the resource with the assigned IP address is from.
         * </p>
         * 
         * @param publicIpv4PoolId
         *        The ID of the public IPv4 pool that the resource with the assigned IP address is from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicIpv4PoolId(String publicIpv4PoolId);

        /**
         * <p>
         * The network interface ID of the resource with the assigned IP address.
         * </p>
         * 
         * @param networkInterfaceId
         *        The network interface ID of the resource with the assigned IP address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceId(String networkInterfaceId);

        /**
         * <p>
         * The description of the network interface that IP address is assigned to.
         * </p>
         * 
         * @param networkInterfaceDescription
         *        The description of the network interface that IP address is assigned to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceDescription(String networkInterfaceDescription);

        /**
         * <p>
         * The instance ID of the instance the assigned IP address is assigned to.
         * </p>
         * 
         * @param instanceId
         *        The instance ID of the instance the assigned IP address is assigned to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * Tags associated with the IP address.
         * </p>
         * 
         * @param tags
         *        Tags associated with the IP address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(IpamPublicAddressTags tags);

        /**
         * <p>
         * Tags associated with the IP address.
         * </p>
         * This is a convenience method that creates an instance of the {@link IpamPublicAddressTags.Builder} avoiding
         * the need to create one manually via {@link IpamPublicAddressTags#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IpamPublicAddressTags.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(IpamPublicAddressTags)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link IpamPublicAddressTags.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(IpamPublicAddressTags)
         */
        default Builder tags(Consumer<IpamPublicAddressTags.Builder> tags) {
            return tags(IpamPublicAddressTags.builder().applyMutation(tags).build());
        }

        /**
         * <p>
         * The Availability Zone (AZ) or Local Zone (LZ) network border group that the resource that the IP address is
         * assigned to is in. Defaults to an AZ network border group. For more information on available Local Zones, see
         * <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">Local Zone
         * availability</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param networkBorderGroup
         *        The Availability Zone (AZ) or Local Zone (LZ) network border group that the resource that the IP
         *        address is assigned to is in. Defaults to an AZ network border group. For more information on
         *        available Local Zones, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">Local Zone
         *        availability</a> in the <i>Amazon EC2 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkBorderGroup(String networkBorderGroup);

        /**
         * <p>
         * Security groups associated with the resource that the IP address is assigned to.
         * </p>
         * 
         * @param securityGroups
         *        Security groups associated with the resource that the IP address is assigned to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(Collection<IpamPublicAddressSecurityGroup> securityGroups);

        /**
         * <p>
         * Security groups associated with the resource that the IP address is assigned to.
         * </p>
         * 
         * @param securityGroups
         *        Security groups associated with the resource that the IP address is assigned to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(IpamPublicAddressSecurityGroup... securityGroups);

        /**
         * <p>
         * Security groups associated with the resource that the IP address is assigned to.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.IpamPublicAddressSecurityGroup.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.IpamPublicAddressSecurityGroup#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.IpamPublicAddressSecurityGroup.Builder#build()} is called
         * immediately and its result is passed to {@link #securityGroups(List<IpamPublicAddressSecurityGroup>)}.
         * 
         * @param securityGroups
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.IpamPublicAddressSecurityGroup.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #securityGroups(java.util.Collection<IpamPublicAddressSecurityGroup>)
         */
        Builder securityGroups(Consumer<IpamPublicAddressSecurityGroup.Builder>... securityGroups);

        /**
         * <p>
         * The last successful resource discovery time.
         * </p>
         * 
         * @param sampleTime
         *        The last successful resource discovery time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleTime(Instant sampleTime);
    }

    static final class BuilderImpl implements Builder {
        private String ipamResourceDiscoveryId;

        private String addressRegion;

        private String address;

        private String addressOwnerId;

        private String addressAllocationId;

        private String associationStatus;

        private String addressType;

        private String service;

        private String serviceResource;

        private String vpcId;

        private String subnetId;

        private String publicIpv4PoolId;

        private String networkInterfaceId;

        private String networkInterfaceDescription;

        private String instanceId;

        private IpamPublicAddressTags tags;

        private String networkBorderGroup;

        private List<IpamPublicAddressSecurityGroup> securityGroups = DefaultSdkAutoConstructList.getInstance();

        private Instant sampleTime;

        private BuilderImpl() {
        }

        private BuilderImpl(IpamDiscoveredPublicAddress model) {
            ipamResourceDiscoveryId(model.ipamResourceDiscoveryId);
            addressRegion(model.addressRegion);
            address(model.address);
            addressOwnerId(model.addressOwnerId);
            addressAllocationId(model.addressAllocationId);
            associationStatus(model.associationStatus);
            addressType(model.addressType);
            service(model.service);
            serviceResource(model.serviceResource);
            vpcId(model.vpcId);
            subnetId(model.subnetId);
            publicIpv4PoolId(model.publicIpv4PoolId);
            networkInterfaceId(model.networkInterfaceId);
            networkInterfaceDescription(model.networkInterfaceDescription);
            instanceId(model.instanceId);
            tags(model.tags);
            networkBorderGroup(model.networkBorderGroup);
            securityGroups(model.securityGroups);
            sampleTime(model.sampleTime);
        }

        public final String getIpamResourceDiscoveryId() {
            return ipamResourceDiscoveryId;
        }

        public final void setIpamResourceDiscoveryId(String ipamResourceDiscoveryId) {
            this.ipamResourceDiscoveryId = ipamResourceDiscoveryId;
        }

        @Override
        public final Builder ipamResourceDiscoveryId(String ipamResourceDiscoveryId) {
            this.ipamResourceDiscoveryId = ipamResourceDiscoveryId;
            return this;
        }

        public final String getAddressRegion() {
            return addressRegion;
        }

        public final void setAddressRegion(String addressRegion) {
            this.addressRegion = addressRegion;
        }

        @Override
        public final Builder addressRegion(String addressRegion) {
            this.addressRegion = addressRegion;
            return this;
        }

        public final String getAddress() {
            return address;
        }

        public final void setAddress(String address) {
            this.address = address;
        }

        @Override
        public final Builder address(String address) {
            this.address = address;
            return this;
        }

        public final String getAddressOwnerId() {
            return addressOwnerId;
        }

        public final void setAddressOwnerId(String addressOwnerId) {
            this.addressOwnerId = addressOwnerId;
        }

        @Override
        public final Builder addressOwnerId(String addressOwnerId) {
            this.addressOwnerId = addressOwnerId;
            return this;
        }

        public final String getAddressAllocationId() {
            return addressAllocationId;
        }

        public final void setAddressAllocationId(String addressAllocationId) {
            this.addressAllocationId = addressAllocationId;
        }

        @Override
        public final Builder addressAllocationId(String addressAllocationId) {
            this.addressAllocationId = addressAllocationId;
            return this;
        }

        public final String getAssociationStatus() {
            return associationStatus;
        }

        public final void setAssociationStatus(String associationStatus) {
            this.associationStatus = associationStatus;
        }

        @Override
        public final Builder associationStatus(String associationStatus) {
            this.associationStatus = associationStatus;
            return this;
        }

        @Override
        public final Builder associationStatus(IpamPublicAddressAssociationStatus associationStatus) {
            this.associationStatus(associationStatus == null ? null : associationStatus.toString());
            return this;
        }

        public final String getAddressType() {
            return addressType;
        }

        public final void setAddressType(String addressType) {
            this.addressType = addressType;
        }

        @Override
        public final Builder addressType(String addressType) {
            this.addressType = addressType;
            return this;
        }

        @Override
        public final Builder addressType(IpamPublicAddressType addressType) {
            this.addressType(addressType == null ? null : addressType.toString());
            return this;
        }

        public final String getService() {
            return service;
        }

        public final void setService(String service) {
            this.service = service;
        }

        @Override
        public final Builder service(String service) {
            this.service = service;
            return this;
        }

        @Override
        public final Builder service(IpamPublicAddressAwsService service) {
            this.service(service == null ? null : service.toString());
            return this;
        }

        public final String getServiceResource() {
            return serviceResource;
        }

        public final void setServiceResource(String serviceResource) {
            this.serviceResource = serviceResource;
        }

        @Override
        public final Builder serviceResource(String serviceResource) {
            this.serviceResource = serviceResource;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final String getPublicIpv4PoolId() {
            return publicIpv4PoolId;
        }

        public final void setPublicIpv4PoolId(String publicIpv4PoolId) {
            this.publicIpv4PoolId = publicIpv4PoolId;
        }

        @Override
        public final Builder publicIpv4PoolId(String publicIpv4PoolId) {
            this.publicIpv4PoolId = publicIpv4PoolId;
            return this;
        }

        public final String getNetworkInterfaceId() {
            return networkInterfaceId;
        }

        public final void setNetworkInterfaceId(String networkInterfaceId) {
            this.networkInterfaceId = networkInterfaceId;
        }

        @Override
        public final Builder networkInterfaceId(String networkInterfaceId) {
            this.networkInterfaceId = networkInterfaceId;
            return this;
        }

        public final String getNetworkInterfaceDescription() {
            return networkInterfaceDescription;
        }

        public final void setNetworkInterfaceDescription(String networkInterfaceDescription) {
            this.networkInterfaceDescription = networkInterfaceDescription;
        }

        @Override
        public final Builder networkInterfaceDescription(String networkInterfaceDescription) {
            this.networkInterfaceDescription = networkInterfaceDescription;
            return this;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final IpamPublicAddressTags.Builder getTags() {
            return tags != null ? tags.toBuilder() : null;
        }

        public final void setTags(IpamPublicAddressTags.BuilderImpl tags) {
            this.tags = tags != null ? tags.build() : null;
        }

        @Override
        public final Builder tags(IpamPublicAddressTags tags) {
            this.tags = tags;
            return this;
        }

        public final String getNetworkBorderGroup() {
            return networkBorderGroup;
        }

        public final void setNetworkBorderGroup(String networkBorderGroup) {
            this.networkBorderGroup = networkBorderGroup;
        }

        @Override
        public final Builder networkBorderGroup(String networkBorderGroup) {
            this.networkBorderGroup = networkBorderGroup;
            return this;
        }

        public final List<IpamPublicAddressSecurityGroup.Builder> getSecurityGroups() {
            List<IpamPublicAddressSecurityGroup.Builder> result = IpamPublicAddressSecurityGroupListCopier
                    .copyToBuilder(this.securityGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSecurityGroups(Collection<IpamPublicAddressSecurityGroup.BuilderImpl> securityGroups) {
            this.securityGroups = IpamPublicAddressSecurityGroupListCopier.copyFromBuilder(securityGroups);
        }

        @Override
        public final Builder securityGroups(Collection<IpamPublicAddressSecurityGroup> securityGroups) {
            this.securityGroups = IpamPublicAddressSecurityGroupListCopier.copy(securityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(IpamPublicAddressSecurityGroup... securityGroups) {
            securityGroups(Arrays.asList(securityGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(Consumer<IpamPublicAddressSecurityGroup.Builder>... securityGroups) {
            securityGroups(Stream.of(securityGroups).map(c -> IpamPublicAddressSecurityGroup.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Instant getSampleTime() {
            return sampleTime;
        }

        public final void setSampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
        }

        @Override
        public final Builder sampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
            return this;
        }

        @Override
        public IpamDiscoveredPublicAddress build() {
            return new IpamDiscoveredPublicAddress(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
