/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Spot Instance replacement strategy to use when Amazon EC2 emits a signal that your Spot Instance is at an
 * elevated risk of being interrupted. For more information, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-fleet-capacity-rebalance.html">Capacity
 * rebalancing</a> in the <i>Amazon EC2 User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SpotCapacityRebalance implements SdkPojo, Serializable,
        ToCopyableBuilder<SpotCapacityRebalance.Builder, SpotCapacityRebalance> {
    private static final SdkField<String> REPLACEMENT_STRATEGY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReplacementStrategy")
            .getter(getter(SpotCapacityRebalance::replacementStrategyAsString))
            .setter(setter(Builder::replacementStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplacementStrategy")
                    .unmarshallLocationName("replacementStrategy").build()).build();

    private static final SdkField<Integer> TERMINATION_DELAY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("TerminationDelay")
            .getter(getter(SpotCapacityRebalance::terminationDelay))
            .setter(setter(Builder::terminationDelay))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TerminationDelay")
                    .unmarshallLocationName("terminationDelay").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REPLACEMENT_STRATEGY_FIELD,
            TERMINATION_DELAY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String replacementStrategy;

    private final Integer terminationDelay;

    private SpotCapacityRebalance(BuilderImpl builder) {
        this.replacementStrategy = builder.replacementStrategy;
        this.terminationDelay = builder.terminationDelay;
    }

    /**
     * <p>
     * The replacement strategy to use. Only available for fleets of type <code>maintain</code>.
     * </p>
     * <p>
     * <code>launch</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance notification is
     * emitted for an existing Spot Instance in the fleet. Spot Fleet does not terminate the instances that receive a
     * rebalance notification. You can terminate the old instances, or you can leave them running. You are charged for
     * all instances while they are running.
     * </p>
     * <p>
     * <code>launch-before-terminate</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance
     * notification is emitted for an existing Spot Instance in the fleet, and then, after a delay that you specify (in
     * <code>TerminationDelay</code>), terminates the instances that received a rebalance notification.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #replacementStrategy} will return {@link ReplacementStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #replacementStrategyAsString}.
     * </p>
     * 
     * @return The replacement strategy to use. Only available for fleets of type <code>maintain</code>.</p>
     *         <p>
     *         <code>launch</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance notification
     *         is emitted for an existing Spot Instance in the fleet. Spot Fleet does not terminate the instances that
     *         receive a rebalance notification. You can terminate the old instances, or you can leave them running. You
     *         are charged for all instances while they are running.
     *         </p>
     *         <p>
     *         <code>launch-before-terminate</code> - Spot Fleet launches a new replacement Spot Instance when a
     *         rebalance notification is emitted for an existing Spot Instance in the fleet, and then, after a delay
     *         that you specify (in <code>TerminationDelay</code>), terminates the instances that received a rebalance
     *         notification.
     * @see ReplacementStrategy
     */
    public final ReplacementStrategy replacementStrategy() {
        return ReplacementStrategy.fromValue(replacementStrategy);
    }

    /**
     * <p>
     * The replacement strategy to use. Only available for fleets of type <code>maintain</code>.
     * </p>
     * <p>
     * <code>launch</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance notification is
     * emitted for an existing Spot Instance in the fleet. Spot Fleet does not terminate the instances that receive a
     * rebalance notification. You can terminate the old instances, or you can leave them running. You are charged for
     * all instances while they are running.
     * </p>
     * <p>
     * <code>launch-before-terminate</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance
     * notification is emitted for an existing Spot Instance in the fleet, and then, after a delay that you specify (in
     * <code>TerminationDelay</code>), terminates the instances that received a rebalance notification.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #replacementStrategy} will return {@link ReplacementStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #replacementStrategyAsString}.
     * </p>
     * 
     * @return The replacement strategy to use. Only available for fleets of type <code>maintain</code>.</p>
     *         <p>
     *         <code>launch</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance notification
     *         is emitted for an existing Spot Instance in the fleet. Spot Fleet does not terminate the instances that
     *         receive a rebalance notification. You can terminate the old instances, or you can leave them running. You
     *         are charged for all instances while they are running.
     *         </p>
     *         <p>
     *         <code>launch-before-terminate</code> - Spot Fleet launches a new replacement Spot Instance when a
     *         rebalance notification is emitted for an existing Spot Instance in the fleet, and then, after a delay
     *         that you specify (in <code>TerminationDelay</code>), terminates the instances that received a rebalance
     *         notification.
     * @see ReplacementStrategy
     */
    public final String replacementStrategyAsString() {
        return replacementStrategy;
    }

    /**
     * <p>
     * The amount of time (in seconds) that Amazon EC2 waits before terminating the old Spot Instance after launching a
     * new replacement Spot Instance.
     * </p>
     * <p>
     * Required when <code>ReplacementStrategy</code> is set to <code>launch-before-terminate</code>.
     * </p>
     * <p>
     * Not valid when <code>ReplacementStrategy</code> is set to <code>launch</code>.
     * </p>
     * <p>
     * Valid values: Minimum value of <code>120</code> seconds. Maximum value of <code>7200</code> seconds.
     * </p>
     * 
     * @return The amount of time (in seconds) that Amazon EC2 waits before terminating the old Spot Instance after
     *         launching a new replacement Spot Instance.</p>
     *         <p>
     *         Required when <code>ReplacementStrategy</code> is set to <code>launch-before-terminate</code>.
     *         </p>
     *         <p>
     *         Not valid when <code>ReplacementStrategy</code> is set to <code>launch</code>.
     *         </p>
     *         <p>
     *         Valid values: Minimum value of <code>120</code> seconds. Maximum value of <code>7200</code> seconds.
     */
    public final Integer terminationDelay() {
        return terminationDelay;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(replacementStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(terminationDelay());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SpotCapacityRebalance)) {
            return false;
        }
        SpotCapacityRebalance other = (SpotCapacityRebalance) obj;
        return Objects.equals(replacementStrategyAsString(), other.replacementStrategyAsString())
                && Objects.equals(terminationDelay(), other.terminationDelay());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SpotCapacityRebalance").add("ReplacementStrategy", replacementStrategyAsString())
                .add("TerminationDelay", terminationDelay()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReplacementStrategy":
            return Optional.ofNullable(clazz.cast(replacementStrategyAsString()));
        case "TerminationDelay":
            return Optional.ofNullable(clazz.cast(terminationDelay()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SpotCapacityRebalance, T> g) {
        return obj -> g.apply((SpotCapacityRebalance) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SpotCapacityRebalance> {
        /**
         * <p>
         * The replacement strategy to use. Only available for fleets of type <code>maintain</code>.
         * </p>
         * <p>
         * <code>launch</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance notification is
         * emitted for an existing Spot Instance in the fleet. Spot Fleet does not terminate the instances that receive
         * a rebalance notification. You can terminate the old instances, or you can leave them running. You are charged
         * for all instances while they are running.
         * </p>
         * <p>
         * <code>launch-before-terminate</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance
         * notification is emitted for an existing Spot Instance in the fleet, and then, after a delay that you specify
         * (in <code>TerminationDelay</code>), terminates the instances that received a rebalance notification.
         * </p>
         * 
         * @param replacementStrategy
         *        The replacement strategy to use. Only available for fleets of type <code>maintain</code>.</p>
         *        <p>
         *        <code>launch</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance
         *        notification is emitted for an existing Spot Instance in the fleet. Spot Fleet does not terminate the
         *        instances that receive a rebalance notification. You can terminate the old instances, or you can leave
         *        them running. You are charged for all instances while they are running.
         *        </p>
         *        <p>
         *        <code>launch-before-terminate</code> - Spot Fleet launches a new replacement Spot Instance when a
         *        rebalance notification is emitted for an existing Spot Instance in the fleet, and then, after a delay
         *        that you specify (in <code>TerminationDelay</code>), terminates the instances that received a
         *        rebalance notification.
         * @see ReplacementStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReplacementStrategy
         */
        Builder replacementStrategy(String replacementStrategy);

        /**
         * <p>
         * The replacement strategy to use. Only available for fleets of type <code>maintain</code>.
         * </p>
         * <p>
         * <code>launch</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance notification is
         * emitted for an existing Spot Instance in the fleet. Spot Fleet does not terminate the instances that receive
         * a rebalance notification. You can terminate the old instances, or you can leave them running. You are charged
         * for all instances while they are running.
         * </p>
         * <p>
         * <code>launch-before-terminate</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance
         * notification is emitted for an existing Spot Instance in the fleet, and then, after a delay that you specify
         * (in <code>TerminationDelay</code>), terminates the instances that received a rebalance notification.
         * </p>
         * 
         * @param replacementStrategy
         *        The replacement strategy to use. Only available for fleets of type <code>maintain</code>.</p>
         *        <p>
         *        <code>launch</code> - Spot Fleet launches a new replacement Spot Instance when a rebalance
         *        notification is emitted for an existing Spot Instance in the fleet. Spot Fleet does not terminate the
         *        instances that receive a rebalance notification. You can terminate the old instances, or you can leave
         *        them running. You are charged for all instances while they are running.
         *        </p>
         *        <p>
         *        <code>launch-before-terminate</code> - Spot Fleet launches a new replacement Spot Instance when a
         *        rebalance notification is emitted for an existing Spot Instance in the fleet, and then, after a delay
         *        that you specify (in <code>TerminationDelay</code>), terminates the instances that received a
         *        rebalance notification.
         * @see ReplacementStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReplacementStrategy
         */
        Builder replacementStrategy(ReplacementStrategy replacementStrategy);

        /**
         * <p>
         * The amount of time (in seconds) that Amazon EC2 waits before terminating the old Spot Instance after
         * launching a new replacement Spot Instance.
         * </p>
         * <p>
         * Required when <code>ReplacementStrategy</code> is set to <code>launch-before-terminate</code>.
         * </p>
         * <p>
         * Not valid when <code>ReplacementStrategy</code> is set to <code>launch</code>.
         * </p>
         * <p>
         * Valid values: Minimum value of <code>120</code> seconds. Maximum value of <code>7200</code> seconds.
         * </p>
         * 
         * @param terminationDelay
         *        The amount of time (in seconds) that Amazon EC2 waits before terminating the old Spot Instance after
         *        launching a new replacement Spot Instance.</p>
         *        <p>
         *        Required when <code>ReplacementStrategy</code> is set to <code>launch-before-terminate</code>.
         *        </p>
         *        <p>
         *        Not valid when <code>ReplacementStrategy</code> is set to <code>launch</code>.
         *        </p>
         *        <p>
         *        Valid values: Minimum value of <code>120</code> seconds. Maximum value of <code>7200</code> seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminationDelay(Integer terminationDelay);
    }

    static final class BuilderImpl implements Builder {
        private String replacementStrategy;

        private Integer terminationDelay;

        private BuilderImpl() {
        }

        private BuilderImpl(SpotCapacityRebalance model) {
            replacementStrategy(model.replacementStrategy);
            terminationDelay(model.terminationDelay);
        }

        public final String getReplacementStrategy() {
            return replacementStrategy;
        }

        public final void setReplacementStrategy(String replacementStrategy) {
            this.replacementStrategy = replacementStrategy;
        }

        @Override
        public final Builder replacementStrategy(String replacementStrategy) {
            this.replacementStrategy = replacementStrategy;
            return this;
        }

        @Override
        public final Builder replacementStrategy(ReplacementStrategy replacementStrategy) {
            this.replacementStrategy(replacementStrategy == null ? null : replacementStrategy.toString());
            return this;
        }

        public final Integer getTerminationDelay() {
            return terminationDelay;
        }

        public final void setTerminationDelay(Integer terminationDelay) {
            this.terminationDelay = terminationDelay;
        }

        @Override
        public final Builder terminationDelay(Integer terminationDelay) {
            this.terminationDelay = terminationDelay;
            return this;
        }

        @Override
        public SpotCapacityRebalance build() {
            return new SpotCapacityRebalance(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
