/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateVerifiedAccessEndpointRequest extends Ec2Request implements
        ToCopyableBuilder<CreateVerifiedAccessEndpointRequest.Builder, CreateVerifiedAccessEndpointRequest> {
    private static final SdkField<String> VERIFIED_ACCESS_GROUP_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VerifiedAccessGroupId")
            .getter(getter(CreateVerifiedAccessEndpointRequest::verifiedAccessGroupId))
            .setter(setter(Builder::verifiedAccessGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VerifiedAccessGroupId")
                    .unmarshallLocationName("VerifiedAccessGroupId").build()).build();

    private static final SdkField<String> ENDPOINT_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EndpointType")
            .getter(getter(CreateVerifiedAccessEndpointRequest::endpointTypeAsString))
            .setter(setter(Builder::endpointType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointType")
                    .unmarshallLocationName("EndpointType").build()).build();

    private static final SdkField<String> ATTACHMENT_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AttachmentType")
            .getter(getter(CreateVerifiedAccessEndpointRequest::attachmentTypeAsString))
            .setter(setter(Builder::attachmentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttachmentType")
                    .unmarshallLocationName("AttachmentType").build()).build();

    private static final SdkField<String> DOMAIN_CERTIFICATE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DomainCertificateArn")
            .getter(getter(CreateVerifiedAccessEndpointRequest::domainCertificateArn))
            .setter(setter(Builder::domainCertificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainCertificateArn")
                    .unmarshallLocationName("DomainCertificateArn").build()).build();

    private static final SdkField<String> APPLICATION_DOMAIN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ApplicationDomain")
            .getter(getter(CreateVerifiedAccessEndpointRequest::applicationDomain))
            .setter(setter(Builder::applicationDomain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationDomain")
                    .unmarshallLocationName("ApplicationDomain").build()).build();

    private static final SdkField<String> ENDPOINT_DOMAIN_PREFIX_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EndpointDomainPrefix")
            .getter(getter(CreateVerifiedAccessEndpointRequest::endpointDomainPrefix))
            .setter(setter(Builder::endpointDomainPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointDomainPrefix")
                    .unmarshallLocationName("EndpointDomainPrefix").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(CreateVerifiedAccessEndpointRequest::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupId")
                    .unmarshallLocationName("SecurityGroupId").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<CreateVerifiedAccessEndpointLoadBalancerOptions> LOAD_BALANCER_OPTIONS_FIELD = SdkField
            .<CreateVerifiedAccessEndpointLoadBalancerOptions> builder(MarshallingType.SDK_POJO)
            .memberName("LoadBalancerOptions")
            .getter(getter(CreateVerifiedAccessEndpointRequest::loadBalancerOptions))
            .setter(setter(Builder::loadBalancerOptions))
            .constructor(CreateVerifiedAccessEndpointLoadBalancerOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoadBalancerOptions")
                    .unmarshallLocationName("LoadBalancerOptions").build()).build();

    private static final SdkField<CreateVerifiedAccessEndpointEniOptions> NETWORK_INTERFACE_OPTIONS_FIELD = SdkField
            .<CreateVerifiedAccessEndpointEniOptions> builder(MarshallingType.SDK_POJO)
            .memberName("NetworkInterfaceOptions")
            .getter(getter(CreateVerifiedAccessEndpointRequest::networkInterfaceOptions))
            .setter(setter(Builder::networkInterfaceOptions))
            .constructor(CreateVerifiedAccessEndpointEniOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceOptions")
                    .unmarshallLocationName("NetworkInterfaceOptions").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(CreateVerifiedAccessEndpointRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("Description").build()).build();

    private static final SdkField<String> POLICY_DOCUMENT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PolicyDocument")
            .getter(getter(CreateVerifiedAccessEndpointRequest::policyDocument))
            .setter(setter(Builder::policyDocument))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyDocument")
                    .unmarshallLocationName("PolicyDocument").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .memberName("TagSpecifications")
            .getter(getter(CreateVerifiedAccessEndpointRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(CreateVerifiedAccessEndpointRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken")
                    .unmarshallLocationName("ClientToken").build(), DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<Boolean> DRY_RUN_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DryRun")
            .getter(getter(CreateVerifiedAccessEndpointRequest::dryRun))
            .setter(setter(Builder::dryRun))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DryRun")
                    .unmarshallLocationName("DryRun").build()).build();

    private static final SdkField<VerifiedAccessSseSpecificationRequest> SSE_SPECIFICATION_FIELD = SdkField
            .<VerifiedAccessSseSpecificationRequest> builder(MarshallingType.SDK_POJO)
            .memberName("SseSpecification")
            .getter(getter(CreateVerifiedAccessEndpointRequest::sseSpecification))
            .setter(setter(Builder::sseSpecification))
            .constructor(VerifiedAccessSseSpecificationRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SseSpecification")
                    .unmarshallLocationName("SseSpecification").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            VERIFIED_ACCESS_GROUP_ID_FIELD, ENDPOINT_TYPE_FIELD, ATTACHMENT_TYPE_FIELD, DOMAIN_CERTIFICATE_ARN_FIELD,
            APPLICATION_DOMAIN_FIELD, ENDPOINT_DOMAIN_PREFIX_FIELD, SECURITY_GROUP_IDS_FIELD, LOAD_BALANCER_OPTIONS_FIELD,
            NETWORK_INTERFACE_OPTIONS_FIELD, DESCRIPTION_FIELD, POLICY_DOCUMENT_FIELD, TAG_SPECIFICATIONS_FIELD,
            CLIENT_TOKEN_FIELD, DRY_RUN_FIELD, SSE_SPECIFICATION_FIELD));

    private final String verifiedAccessGroupId;

    private final String endpointType;

    private final String attachmentType;

    private final String domainCertificateArn;

    private final String applicationDomain;

    private final String endpointDomainPrefix;

    private final List<String> securityGroupIds;

    private final CreateVerifiedAccessEndpointLoadBalancerOptions loadBalancerOptions;

    private final CreateVerifiedAccessEndpointEniOptions networkInterfaceOptions;

    private final String description;

    private final String policyDocument;

    private final List<TagSpecification> tagSpecifications;

    private final String clientToken;

    private final Boolean dryRun;

    private final VerifiedAccessSseSpecificationRequest sseSpecification;

    private CreateVerifiedAccessEndpointRequest(BuilderImpl builder) {
        super(builder);
        this.verifiedAccessGroupId = builder.verifiedAccessGroupId;
        this.endpointType = builder.endpointType;
        this.attachmentType = builder.attachmentType;
        this.domainCertificateArn = builder.domainCertificateArn;
        this.applicationDomain = builder.applicationDomain;
        this.endpointDomainPrefix = builder.endpointDomainPrefix;
        this.securityGroupIds = builder.securityGroupIds;
        this.loadBalancerOptions = builder.loadBalancerOptions;
        this.networkInterfaceOptions = builder.networkInterfaceOptions;
        this.description = builder.description;
        this.policyDocument = builder.policyDocument;
        this.tagSpecifications = builder.tagSpecifications;
        this.clientToken = builder.clientToken;
        this.dryRun = builder.dryRun;
        this.sseSpecification = builder.sseSpecification;
    }

    /**
     * <p>
     * The ID of the Verified Access group to associate the endpoint with.
     * </p>
     * 
     * @return The ID of the Verified Access group to associate the endpoint with.
     */
    public final String verifiedAccessGroupId() {
        return verifiedAccessGroupId;
    }

    /**
     * <p>
     * The type of Verified Access endpoint to create.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointType} will
     * return {@link VerifiedAccessEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #endpointTypeAsString}.
     * </p>
     * 
     * @return The type of Verified Access endpoint to create.
     * @see VerifiedAccessEndpointType
     */
    public final VerifiedAccessEndpointType endpointType() {
        return VerifiedAccessEndpointType.fromValue(endpointType);
    }

    /**
     * <p>
     * The type of Verified Access endpoint to create.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointType} will
     * return {@link VerifiedAccessEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #endpointTypeAsString}.
     * </p>
     * 
     * @return The type of Verified Access endpoint to create.
     * @see VerifiedAccessEndpointType
     */
    public final String endpointTypeAsString() {
        return endpointType;
    }

    /**
     * <p>
     * The type of attachment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #attachmentType}
     * will return {@link VerifiedAccessEndpointAttachmentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #attachmentTypeAsString}.
     * </p>
     * 
     * @return The type of attachment.
     * @see VerifiedAccessEndpointAttachmentType
     */
    public final VerifiedAccessEndpointAttachmentType attachmentType() {
        return VerifiedAccessEndpointAttachmentType.fromValue(attachmentType);
    }

    /**
     * <p>
     * The type of attachment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #attachmentType}
     * will return {@link VerifiedAccessEndpointAttachmentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #attachmentTypeAsString}.
     * </p>
     * 
     * @return The type of attachment.
     * @see VerifiedAccessEndpointAttachmentType
     */
    public final String attachmentTypeAsString() {
        return attachmentType;
    }

    /**
     * <p>
     * The ARN of the public TLS/SSL certificate in Amazon Web Services Certificate Manager to associate with the
     * endpoint. The CN in the certificate must match the DNS name your end users will use to reach your application.
     * </p>
     * 
     * @return The ARN of the public TLS/SSL certificate in Amazon Web Services Certificate Manager to associate with
     *         the endpoint. The CN in the certificate must match the DNS name your end users will use to reach your
     *         application.
     */
    public final String domainCertificateArn() {
        return domainCertificateArn;
    }

    /**
     * <p>
     * The DNS name for users to reach your application.
     * </p>
     * 
     * @return The DNS name for users to reach your application.
     */
    public final String applicationDomain() {
        return applicationDomain;
    }

    /**
     * <p>
     * A custom identifier that is prepended to the DNS name that is generated for the endpoint.
     * </p>
     * 
     * @return A custom identifier that is prepended to the DNS name that is generated for the endpoint.
     */
    public final String endpointDomainPrefix() {
        return endpointDomainPrefix;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the security groups to associate with the Verified Access endpoint. Required if
     * <code>AttachmentType</code> is set to <code>vpc</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return The IDs of the security groups to associate with the Verified Access endpoint. Required if
     *         <code>AttachmentType</code> is set to <code>vpc</code>.
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * <p>
     * The load balancer details. This parameter is required if the endpoint type is <code>load-balancer</code>.
     * </p>
     * 
     * @return The load balancer details. This parameter is required if the endpoint type is <code>load-balancer</code>.
     */
    public final CreateVerifiedAccessEndpointLoadBalancerOptions loadBalancerOptions() {
        return loadBalancerOptions;
    }

    /**
     * <p>
     * The network interface details. This parameter is required if the endpoint type is <code>network-interface</code>.
     * </p>
     * 
     * @return The network interface details. This parameter is required if the endpoint type is
     *         <code>network-interface</code>.
     */
    public final CreateVerifiedAccessEndpointEniOptions networkInterfaceOptions() {
        return networkInterfaceOptions;
    }

    /**
     * <p>
     * A description for the Verified Access endpoint.
     * </p>
     * 
     * @return A description for the Verified Access endpoint.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The Verified Access policy document.
     * </p>
     * 
     * @return The Verified Access policy document.
     */
    public final String policyDocument() {
        return policyDocument;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to assign to the Verified Access endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagSpecifications} method.
     * </p>
     * 
     * @return The tags to assign to the Verified Access endpoint.
     */
    public final List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    /**
     * <p>
     * A unique, case-sensitive token that you provide to ensure idempotency of your modification request. For more
     * information, see <a href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">Ensuring
     * idempotency</a>.
     * </p>
     * 
     * @return A unique, case-sensitive token that you provide to ensure idempotency of your modification request. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">Ensuring idempotency</a>.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * Checks whether you have the required permissions for the action, without actually making the request, and
     * provides an error response. If you have the required permissions, the error response is
     * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     * </p>
     * 
     * @return Checks whether you have the required permissions for the action, without actually making the request, and
     *         provides an error response. If you have the required permissions, the error response is
     *         <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     */
    public final Boolean dryRun() {
        return dryRun;
    }

    /**
     * <p>
     * The options for server side encryption.
     * </p>
     * 
     * @return The options for server side encryption.
     */
    public final VerifiedAccessSseSpecificationRequest sseSpecification() {
        return sseSpecification;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(verifiedAccessGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(endpointTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(attachmentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(domainCertificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(applicationDomain());
        hashCode = 31 * hashCode + Objects.hashCode(endpointDomainPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(loadBalancerOptions());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaceOptions());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(policyDocument());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagSpecifications() ? tagSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(dryRun());
        hashCode = 31 * hashCode + Objects.hashCode(sseSpecification());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateVerifiedAccessEndpointRequest)) {
            return false;
        }
        CreateVerifiedAccessEndpointRequest other = (CreateVerifiedAccessEndpointRequest) obj;
        return Objects.equals(verifiedAccessGroupId(), other.verifiedAccessGroupId())
                && Objects.equals(endpointTypeAsString(), other.endpointTypeAsString())
                && Objects.equals(attachmentTypeAsString(), other.attachmentTypeAsString())
                && Objects.equals(domainCertificateArn(), other.domainCertificateArn())
                && Objects.equals(applicationDomain(), other.applicationDomain())
                && Objects.equals(endpointDomainPrefix(), other.endpointDomainPrefix())
                && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds())
                && Objects.equals(loadBalancerOptions(), other.loadBalancerOptions())
                && Objects.equals(networkInterfaceOptions(), other.networkInterfaceOptions())
                && Objects.equals(description(), other.description()) && Objects.equals(policyDocument(), other.policyDocument())
                && hasTagSpecifications() == other.hasTagSpecifications()
                && Objects.equals(tagSpecifications(), other.tagSpecifications())
                && Objects.equals(clientToken(), other.clientToken()) && Objects.equals(dryRun(), other.dryRun())
                && Objects.equals(sseSpecification(), other.sseSpecification());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateVerifiedAccessEndpointRequest").add("VerifiedAccessGroupId", verifiedAccessGroupId())
                .add("EndpointType", endpointTypeAsString()).add("AttachmentType", attachmentTypeAsString())
                .add("DomainCertificateArn", domainCertificateArn()).add("ApplicationDomain", applicationDomain())
                .add("EndpointDomainPrefix", endpointDomainPrefix())
                .add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null)
                .add("LoadBalancerOptions", loadBalancerOptions()).add("NetworkInterfaceOptions", networkInterfaceOptions())
                .add("Description", description()).add("PolicyDocument", policyDocument())
                .add("TagSpecifications", hasTagSpecifications() ? tagSpecifications() : null).add("ClientToken", clientToken())
                .add("DryRun", dryRun()).add("SseSpecification", sseSpecification()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VerifiedAccessGroupId":
            return Optional.ofNullable(clazz.cast(verifiedAccessGroupId()));
        case "EndpointType":
            return Optional.ofNullable(clazz.cast(endpointTypeAsString()));
        case "AttachmentType":
            return Optional.ofNullable(clazz.cast(attachmentTypeAsString()));
        case "DomainCertificateArn":
            return Optional.ofNullable(clazz.cast(domainCertificateArn()));
        case "ApplicationDomain":
            return Optional.ofNullable(clazz.cast(applicationDomain()));
        case "EndpointDomainPrefix":
            return Optional.ofNullable(clazz.cast(endpointDomainPrefix()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "LoadBalancerOptions":
            return Optional.ofNullable(clazz.cast(loadBalancerOptions()));
        case "NetworkInterfaceOptions":
            return Optional.ofNullable(clazz.cast(networkInterfaceOptions()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "PolicyDocument":
            return Optional.ofNullable(clazz.cast(policyDocument()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "DryRun":
            return Optional.ofNullable(clazz.cast(dryRun()));
        case "SseSpecification":
            return Optional.ofNullable(clazz.cast(sseSpecification()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateVerifiedAccessEndpointRequest, T> g) {
        return obj -> g.apply((CreateVerifiedAccessEndpointRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateVerifiedAccessEndpointRequest> {
        /**
         * <p>
         * The ID of the Verified Access group to associate the endpoint with.
         * </p>
         * 
         * @param verifiedAccessGroupId
         *        The ID of the Verified Access group to associate the endpoint with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder verifiedAccessGroupId(String verifiedAccessGroupId);

        /**
         * <p>
         * The type of Verified Access endpoint to create.
         * </p>
         * 
         * @param endpointType
         *        The type of Verified Access endpoint to create.
         * @see VerifiedAccessEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifiedAccessEndpointType
         */
        Builder endpointType(String endpointType);

        /**
         * <p>
         * The type of Verified Access endpoint to create.
         * </p>
         * 
         * @param endpointType
         *        The type of Verified Access endpoint to create.
         * @see VerifiedAccessEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifiedAccessEndpointType
         */
        Builder endpointType(VerifiedAccessEndpointType endpointType);

        /**
         * <p>
         * The type of attachment.
         * </p>
         * 
         * @param attachmentType
         *        The type of attachment.
         * @see VerifiedAccessEndpointAttachmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifiedAccessEndpointAttachmentType
         */
        Builder attachmentType(String attachmentType);

        /**
         * <p>
         * The type of attachment.
         * </p>
         * 
         * @param attachmentType
         *        The type of attachment.
         * @see VerifiedAccessEndpointAttachmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifiedAccessEndpointAttachmentType
         */
        Builder attachmentType(VerifiedAccessEndpointAttachmentType attachmentType);

        /**
         * <p>
         * The ARN of the public TLS/SSL certificate in Amazon Web Services Certificate Manager to associate with the
         * endpoint. The CN in the certificate must match the DNS name your end users will use to reach your
         * application.
         * </p>
         * 
         * @param domainCertificateArn
         *        The ARN of the public TLS/SSL certificate in Amazon Web Services Certificate Manager to associate with
         *        the endpoint. The CN in the certificate must match the DNS name your end users will use to reach your
         *        application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainCertificateArn(String domainCertificateArn);

        /**
         * <p>
         * The DNS name for users to reach your application.
         * </p>
         * 
         * @param applicationDomain
         *        The DNS name for users to reach your application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationDomain(String applicationDomain);

        /**
         * <p>
         * A custom identifier that is prepended to the DNS name that is generated for the endpoint.
         * </p>
         * 
         * @param endpointDomainPrefix
         *        A custom identifier that is prepended to the DNS name that is generated for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointDomainPrefix(String endpointDomainPrefix);

        /**
         * <p>
         * The IDs of the security groups to associate with the Verified Access endpoint. Required if
         * <code>AttachmentType</code> is set to <code>vpc</code>.
         * </p>
         * 
         * @param securityGroupIds
         *        The IDs of the security groups to associate with the Verified Access endpoint. Required if
         *        <code>AttachmentType</code> is set to <code>vpc</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The IDs of the security groups to associate with the Verified Access endpoint. Required if
         * <code>AttachmentType</code> is set to <code>vpc</code>.
         * </p>
         * 
         * @param securityGroupIds
         *        The IDs of the security groups to associate with the Verified Access endpoint. Required if
         *        <code>AttachmentType</code> is set to <code>vpc</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * The load balancer details. This parameter is required if the endpoint type is <code>load-balancer</code>.
         * </p>
         * 
         * @param loadBalancerOptions
         *        The load balancer details. This parameter is required if the endpoint type is
         *        <code>load-balancer</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancerOptions(CreateVerifiedAccessEndpointLoadBalancerOptions loadBalancerOptions);

        /**
         * <p>
         * The load balancer details. This parameter is required if the endpoint type is <code>load-balancer</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link CreateVerifiedAccessEndpointLoadBalancerOptions.Builder} avoiding the need to create one manually via
         * {@link CreateVerifiedAccessEndpointLoadBalancerOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CreateVerifiedAccessEndpointLoadBalancerOptions.Builder#build()}
         * is called immediately and its result is passed to
         * {@link #loadBalancerOptions(CreateVerifiedAccessEndpointLoadBalancerOptions)}.
         * 
         * @param loadBalancerOptions
         *        a consumer that will call methods on {@link CreateVerifiedAccessEndpointLoadBalancerOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #loadBalancerOptions(CreateVerifiedAccessEndpointLoadBalancerOptions)
         */
        default Builder loadBalancerOptions(Consumer<CreateVerifiedAccessEndpointLoadBalancerOptions.Builder> loadBalancerOptions) {
            return loadBalancerOptions(CreateVerifiedAccessEndpointLoadBalancerOptions.builder()
                    .applyMutation(loadBalancerOptions).build());
        }

        /**
         * <p>
         * The network interface details. This parameter is required if the endpoint type is
         * <code>network-interface</code>.
         * </p>
         * 
         * @param networkInterfaceOptions
         *        The network interface details. This parameter is required if the endpoint type is
         *        <code>network-interface</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceOptions(CreateVerifiedAccessEndpointEniOptions networkInterfaceOptions);

        /**
         * <p>
         * The network interface details. This parameter is required if the endpoint type is
         * <code>network-interface</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link CreateVerifiedAccessEndpointEniOptions.Builder} avoiding the need to create one manually via
         * {@link CreateVerifiedAccessEndpointEniOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CreateVerifiedAccessEndpointEniOptions.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #networkInterfaceOptions(CreateVerifiedAccessEndpointEniOptions)}.
         * 
         * @param networkInterfaceOptions
         *        a consumer that will call methods on {@link CreateVerifiedAccessEndpointEniOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaceOptions(CreateVerifiedAccessEndpointEniOptions)
         */
        default Builder networkInterfaceOptions(Consumer<CreateVerifiedAccessEndpointEniOptions.Builder> networkInterfaceOptions) {
            return networkInterfaceOptions(CreateVerifiedAccessEndpointEniOptions.builder()
                    .applyMutation(networkInterfaceOptions).build());
        }

        /**
         * <p>
         * A description for the Verified Access endpoint.
         * </p>
         * 
         * @param description
         *        A description for the Verified Access endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The Verified Access policy document.
         * </p>
         * 
         * @param policyDocument
         *        The Verified Access policy document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyDocument(String policyDocument);

        /**
         * <p>
         * The tags to assign to the Verified Access endpoint.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to assign to the Verified Access endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The tags to assign to the Verified Access endpoint.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to assign to the Verified Access endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The tags to assign to the Verified Access endpoint.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.TagSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(java.util.Collection<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        /**
         * <p>
         * A unique, case-sensitive token that you provide to ensure idempotency of your modification request. For more
         * information, see <a href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">Ensuring
         * idempotency</a>.
         * </p>
         * 
         * @param clientToken
         *        A unique, case-sensitive token that you provide to ensure idempotency of your modification request.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">Ensuring
         *        idempotency</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * Checks whether you have the required permissions for the action, without actually making the request, and
         * provides an error response. If you have the required permissions, the error response is
         * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * </p>
         * 
         * @param dryRun
         *        Checks whether you have the required permissions for the action, without actually making the request,
         *        and provides an error response. If you have the required permissions, the error response is
         *        <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dryRun(Boolean dryRun);

        /**
         * <p>
         * The options for server side encryption.
         * </p>
         * 
         * @param sseSpecification
         *        The options for server side encryption.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sseSpecification(VerifiedAccessSseSpecificationRequest sseSpecification);

        /**
         * <p>
         * The options for server side encryption.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link VerifiedAccessSseSpecificationRequest.Builder} avoiding the need to create one manually via
         * {@link VerifiedAccessSseSpecificationRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VerifiedAccessSseSpecificationRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #sseSpecification(VerifiedAccessSseSpecificationRequest)}.
         * 
         * @param sseSpecification
         *        a consumer that will call methods on {@link VerifiedAccessSseSpecificationRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sseSpecification(VerifiedAccessSseSpecificationRequest)
         */
        default Builder sseSpecification(Consumer<VerifiedAccessSseSpecificationRequest.Builder> sseSpecification) {
            return sseSpecification(VerifiedAccessSseSpecificationRequest.builder().applyMutation(sseSpecification).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String verifiedAccessGroupId;

        private String endpointType;

        private String attachmentType;

        private String domainCertificateArn;

        private String applicationDomain;

        private String endpointDomainPrefix;

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private CreateVerifiedAccessEndpointLoadBalancerOptions loadBalancerOptions;

        private CreateVerifiedAccessEndpointEniOptions networkInterfaceOptions;

        private String description;

        private String policyDocument;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private String clientToken;

        private Boolean dryRun;

        private VerifiedAccessSseSpecificationRequest sseSpecification;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateVerifiedAccessEndpointRequest model) {
            super(model);
            verifiedAccessGroupId(model.verifiedAccessGroupId);
            endpointType(model.endpointType);
            attachmentType(model.attachmentType);
            domainCertificateArn(model.domainCertificateArn);
            applicationDomain(model.applicationDomain);
            endpointDomainPrefix(model.endpointDomainPrefix);
            securityGroupIds(model.securityGroupIds);
            loadBalancerOptions(model.loadBalancerOptions);
            networkInterfaceOptions(model.networkInterfaceOptions);
            description(model.description);
            policyDocument(model.policyDocument);
            tagSpecifications(model.tagSpecifications);
            clientToken(model.clientToken);
            dryRun(model.dryRun);
            sseSpecification(model.sseSpecification);
        }

        public final String getVerifiedAccessGroupId() {
            return verifiedAccessGroupId;
        }

        public final void setVerifiedAccessGroupId(String verifiedAccessGroupId) {
            this.verifiedAccessGroupId = verifiedAccessGroupId;
        }

        @Override
        public final Builder verifiedAccessGroupId(String verifiedAccessGroupId) {
            this.verifiedAccessGroupId = verifiedAccessGroupId;
            return this;
        }

        public final String getEndpointType() {
            return endpointType;
        }

        public final void setEndpointType(String endpointType) {
            this.endpointType = endpointType;
        }

        @Override
        public final Builder endpointType(String endpointType) {
            this.endpointType = endpointType;
            return this;
        }

        @Override
        public final Builder endpointType(VerifiedAccessEndpointType endpointType) {
            this.endpointType(endpointType == null ? null : endpointType.toString());
            return this;
        }

        public final String getAttachmentType() {
            return attachmentType;
        }

        public final void setAttachmentType(String attachmentType) {
            this.attachmentType = attachmentType;
        }

        @Override
        public final Builder attachmentType(String attachmentType) {
            this.attachmentType = attachmentType;
            return this;
        }

        @Override
        public final Builder attachmentType(VerifiedAccessEndpointAttachmentType attachmentType) {
            this.attachmentType(attachmentType == null ? null : attachmentType.toString());
            return this;
        }

        public final String getDomainCertificateArn() {
            return domainCertificateArn;
        }

        public final void setDomainCertificateArn(String domainCertificateArn) {
            this.domainCertificateArn = domainCertificateArn;
        }

        @Override
        public final Builder domainCertificateArn(String domainCertificateArn) {
            this.domainCertificateArn = domainCertificateArn;
            return this;
        }

        public final String getApplicationDomain() {
            return applicationDomain;
        }

        public final void setApplicationDomain(String applicationDomain) {
            this.applicationDomain = applicationDomain;
        }

        @Override
        public final Builder applicationDomain(String applicationDomain) {
            this.applicationDomain = applicationDomain;
            return this;
        }

        public final String getEndpointDomainPrefix() {
            return endpointDomainPrefix;
        }

        public final void setEndpointDomainPrefix(String endpointDomainPrefix) {
            this.endpointDomainPrefix = endpointDomainPrefix;
        }

        @Override
        public final Builder endpointDomainPrefix(String endpointDomainPrefix) {
            this.endpointDomainPrefix = endpointDomainPrefix;
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdListCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdListCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final CreateVerifiedAccessEndpointLoadBalancerOptions.Builder getLoadBalancerOptions() {
            return loadBalancerOptions != null ? loadBalancerOptions.toBuilder() : null;
        }

        public final void setLoadBalancerOptions(CreateVerifiedAccessEndpointLoadBalancerOptions.BuilderImpl loadBalancerOptions) {
            this.loadBalancerOptions = loadBalancerOptions != null ? loadBalancerOptions.build() : null;
        }

        @Override
        public final Builder loadBalancerOptions(CreateVerifiedAccessEndpointLoadBalancerOptions loadBalancerOptions) {
            this.loadBalancerOptions = loadBalancerOptions;
            return this;
        }

        public final CreateVerifiedAccessEndpointEniOptions.Builder getNetworkInterfaceOptions() {
            return networkInterfaceOptions != null ? networkInterfaceOptions.toBuilder() : null;
        }

        public final void setNetworkInterfaceOptions(CreateVerifiedAccessEndpointEniOptions.BuilderImpl networkInterfaceOptions) {
            this.networkInterfaceOptions = networkInterfaceOptions != null ? networkInterfaceOptions.build() : null;
        }

        @Override
        public final Builder networkInterfaceOptions(CreateVerifiedAccessEndpointEniOptions networkInterfaceOptions) {
            this.networkInterfaceOptions = networkInterfaceOptions;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getPolicyDocument() {
            return policyDocument;
        }

        public final void setPolicyDocument(String policyDocument) {
            this.policyDocument = policyDocument;
        }

        @Override
        public final Builder policyDocument(String policyDocument) {
            this.policyDocument = policyDocument;
            return this;
        }

        public final List<TagSpecification.Builder> getTagSpecifications() {
            List<TagSpecification.Builder> result = TagSpecificationListCopier.copyToBuilder(this.tagSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final Boolean getDryRun() {
            return dryRun;
        }

        public final void setDryRun(Boolean dryRun) {
            this.dryRun = dryRun;
        }

        @Override
        public final Builder dryRun(Boolean dryRun) {
            this.dryRun = dryRun;
            return this;
        }

        public final VerifiedAccessSseSpecificationRequest.Builder getSseSpecification() {
            return sseSpecification != null ? sseSpecification.toBuilder() : null;
        }

        public final void setSseSpecification(VerifiedAccessSseSpecificationRequest.BuilderImpl sseSpecification) {
            this.sseSpecification = sseSpecification != null ? sseSpecification.build() : null;
        }

        @Override
        public final Builder sseSpecification(VerifiedAccessSseSpecificationRequest sseSpecification) {
            this.sseSpecification = sseSpecification;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateVerifiedAccessEndpointRequest build() {
            return new CreateVerifiedAccessEndpointRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
