/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the configuration of On-Demand Instances in an EC2 Fleet.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OnDemandOptionsRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<OnDemandOptionsRequest.Builder, OnDemandOptionsRequest> {
    private static final SdkField<String> ALLOCATION_STRATEGY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AllocationStrategy")
            .getter(getter(OnDemandOptionsRequest::allocationStrategyAsString))
            .setter(setter(Builder::allocationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationStrategy")
                    .unmarshallLocationName("AllocationStrategy").build()).build();

    private static final SdkField<CapacityReservationOptionsRequest> CAPACITY_RESERVATION_OPTIONS_FIELD = SdkField
            .<CapacityReservationOptionsRequest> builder(MarshallingType.SDK_POJO)
            .memberName("CapacityReservationOptions")
            .getter(getter(OnDemandOptionsRequest::capacityReservationOptions))
            .setter(setter(Builder::capacityReservationOptions))
            .constructor(CapacityReservationOptionsRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityReservationOptions")
                    .unmarshallLocationName("CapacityReservationOptions").build()).build();

    private static final SdkField<Boolean> SINGLE_INSTANCE_TYPE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SingleInstanceType")
            .getter(getter(OnDemandOptionsRequest::singleInstanceType))
            .setter(setter(Builder::singleInstanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SingleInstanceType")
                    .unmarshallLocationName("SingleInstanceType").build()).build();

    private static final SdkField<Boolean> SINGLE_AVAILABILITY_ZONE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SingleAvailabilityZone")
            .getter(getter(OnDemandOptionsRequest::singleAvailabilityZone))
            .setter(setter(Builder::singleAvailabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SingleAvailabilityZone")
                    .unmarshallLocationName("SingleAvailabilityZone").build()).build();

    private static final SdkField<Integer> MIN_TARGET_CAPACITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinTargetCapacity")
            .getter(getter(OnDemandOptionsRequest::minTargetCapacity))
            .setter(setter(Builder::minTargetCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinTargetCapacity")
                    .unmarshallLocationName("MinTargetCapacity").build()).build();

    private static final SdkField<String> MAX_TOTAL_PRICE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("MaxTotalPrice")
            .getter(getter(OnDemandOptionsRequest::maxTotalPrice))
            .setter(setter(Builder::maxTotalPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxTotalPrice")
                    .unmarshallLocationName("MaxTotalPrice").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALLOCATION_STRATEGY_FIELD,
            CAPACITY_RESERVATION_OPTIONS_FIELD, SINGLE_INSTANCE_TYPE_FIELD, SINGLE_AVAILABILITY_ZONE_FIELD,
            MIN_TARGET_CAPACITY_FIELD, MAX_TOTAL_PRICE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String allocationStrategy;

    private final CapacityReservationOptionsRequest capacityReservationOptions;

    private final Boolean singleInstanceType;

    private final Boolean singleAvailabilityZone;

    private final Integer minTargetCapacity;

    private final String maxTotalPrice;

    private OnDemandOptionsRequest(BuilderImpl builder) {
        this.allocationStrategy = builder.allocationStrategy;
        this.capacityReservationOptions = builder.capacityReservationOptions;
        this.singleInstanceType = builder.singleInstanceType;
        this.singleAvailabilityZone = builder.singleAvailabilityZone;
        this.minTargetCapacity = builder.minTargetCapacity;
        this.maxTotalPrice = builder.maxTotalPrice;
    }

    /**
     * <p>
     * The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand capacity.
     * </p>
     * <p>
     * <code>lowest-price</code> - EC2 Fleet uses price to determine the order, launching the lowest price first.
     * </p>
     * <p>
     * <code>prioritized</code> - EC2 Fleet uses the priority that you assigned to each launch template override,
     * launching the highest priority first.
     * </p>
     * <p>
     * Default: <code>lowest-price</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link FleetOnDemandAllocationStrategy#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand
     *         capacity.</p>
     *         <p>
     *         <code>lowest-price</code> - EC2 Fleet uses price to determine the order, launching the lowest price
     *         first.
     *         </p>
     *         <p>
     *         <code>prioritized</code> - EC2 Fleet uses the priority that you assigned to each launch template
     *         override, launching the highest priority first.
     *         </p>
     *         <p>
     *         Default: <code>lowest-price</code>
     * @see FleetOnDemandAllocationStrategy
     */
    public final FleetOnDemandAllocationStrategy allocationStrategy() {
        return FleetOnDemandAllocationStrategy.fromValue(allocationStrategy);
    }

    /**
     * <p>
     * The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand capacity.
     * </p>
     * <p>
     * <code>lowest-price</code> - EC2 Fleet uses price to determine the order, launching the lowest price first.
     * </p>
     * <p>
     * <code>prioritized</code> - EC2 Fleet uses the priority that you assigned to each launch template override,
     * launching the highest priority first.
     * </p>
     * <p>
     * Default: <code>lowest-price</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link FleetOnDemandAllocationStrategy#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand
     *         capacity.</p>
     *         <p>
     *         <code>lowest-price</code> - EC2 Fleet uses price to determine the order, launching the lowest price
     *         first.
     *         </p>
     *         <p>
     *         <code>prioritized</code> - EC2 Fleet uses the priority that you assigned to each launch template
     *         override, launching the highest priority first.
     *         </p>
     *         <p>
     *         Default: <code>lowest-price</code>
     * @see FleetOnDemandAllocationStrategy
     */
    public final String allocationStrategyAsString() {
        return allocationStrategy;
    }

    /**
     * <p>
     * The strategy for using unused Capacity Reservations for fulfilling On-Demand capacity.
     * </p>
     * <p>
     * Supported only for fleets of type <code>instant</code>.
     * </p>
     * 
     * @return The strategy for using unused Capacity Reservations for fulfilling On-Demand capacity.</p>
     *         <p>
     *         Supported only for fleets of type <code>instant</code>.
     */
    public final CapacityReservationOptionsRequest capacityReservationOptions() {
        return capacityReservationOptions;
    }

    /**
     * <p>
     * Indicates that the fleet uses a single instance type to launch all On-Demand Instances in the fleet.
     * </p>
     * <p>
     * Supported only for fleets of type <code>instant</code>.
     * </p>
     * 
     * @return Indicates that the fleet uses a single instance type to launch all On-Demand Instances in the fleet.</p>
     *         <p>
     *         Supported only for fleets of type <code>instant</code>.
     */
    public final Boolean singleInstanceType() {
        return singleInstanceType;
    }

    /**
     * <p>
     * Indicates that the fleet launches all On-Demand Instances into a single Availability Zone.
     * </p>
     * <p>
     * Supported only for fleets of type <code>instant</code>.
     * </p>
     * 
     * @return Indicates that the fleet launches all On-Demand Instances into a single Availability Zone.</p>
     *         <p>
     *         Supported only for fleets of type <code>instant</code>.
     */
    public final Boolean singleAvailabilityZone() {
        return singleAvailabilityZone;
    }

    /**
     * <p>
     * The minimum target capacity for On-Demand Instances in the fleet. If the minimum target capacity is not reached,
     * the fleet launches no instances.
     * </p>
     * <p>
     * Supported only for fleets of type <code>instant</code>.
     * </p>
     * <p>
     * At least one of the following must be specified: <code>SingleAvailabilityZone</code> |
     * <code>SingleInstanceType</code>
     * </p>
     * 
     * @return The minimum target capacity for On-Demand Instances in the fleet. If the minimum target capacity is not
     *         reached, the fleet launches no instances.</p>
     *         <p>
     *         Supported only for fleets of type <code>instant</code>.
     *         </p>
     *         <p>
     *         At least one of the following must be specified: <code>SingleAvailabilityZone</code> |
     *         <code>SingleInstanceType</code>
     */
    public final Integer minTargetCapacity() {
        return minTargetCapacity;
    }

    /**
     * <p>
     * The maximum amount per hour for On-Demand Instances that you're willing to pay.
     * </p>
     * <note>
     * <p>
     * If your fleet includes T instances that are configured as <code>unlimited</code>, and if their average CPU usage
     * exceeds the baseline utilization, you will incur a charge for surplus credits. The <code>MaxTotalPrice</code>
     * does not account for surplus credits, and, if you use surplus credits, your final cost might be higher than what
     * you specified for <code>MaxTotalPrice</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances-unlimited-mode-concepts.html#unlimited-mode-surplus-credits"
     * >Surplus credits can incur charges</a> in the <i>EC2 User Guide</i>.
     * </p>
     * </note>
     * 
     * @return The maximum amount per hour for On-Demand Instances that you're willing to pay.</p> <note>
     *         <p>
     *         If your fleet includes T instances that are configured as <code>unlimited</code>, and if their average
     *         CPU usage exceeds the baseline utilization, you will incur a charge for surplus credits. The
     *         <code>MaxTotalPrice</code> does not account for surplus credits, and, if you use surplus credits, your
     *         final cost might be higher than what you specified for <code>MaxTotalPrice</code>. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances-unlimited-mode-concepts.html#unlimited-mode-surplus-credits"
     *         >Surplus credits can incur charges</a> in the <i>EC2 User Guide</i>.
     *         </p>
     */
    public final String maxTotalPrice() {
        return maxTotalPrice;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(allocationStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(capacityReservationOptions());
        hashCode = 31 * hashCode + Objects.hashCode(singleInstanceType());
        hashCode = 31 * hashCode + Objects.hashCode(singleAvailabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(minTargetCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(maxTotalPrice());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OnDemandOptionsRequest)) {
            return false;
        }
        OnDemandOptionsRequest other = (OnDemandOptionsRequest) obj;
        return Objects.equals(allocationStrategyAsString(), other.allocationStrategyAsString())
                && Objects.equals(capacityReservationOptions(), other.capacityReservationOptions())
                && Objects.equals(singleInstanceType(), other.singleInstanceType())
                && Objects.equals(singleAvailabilityZone(), other.singleAvailabilityZone())
                && Objects.equals(minTargetCapacity(), other.minTargetCapacity())
                && Objects.equals(maxTotalPrice(), other.maxTotalPrice());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OnDemandOptionsRequest").add("AllocationStrategy", allocationStrategyAsString())
                .add("CapacityReservationOptions", capacityReservationOptions()).add("SingleInstanceType", singleInstanceType())
                .add("SingleAvailabilityZone", singleAvailabilityZone()).add("MinTargetCapacity", minTargetCapacity())
                .add("MaxTotalPrice", maxTotalPrice()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AllocationStrategy":
            return Optional.ofNullable(clazz.cast(allocationStrategyAsString()));
        case "CapacityReservationOptions":
            return Optional.ofNullable(clazz.cast(capacityReservationOptions()));
        case "SingleInstanceType":
            return Optional.ofNullable(clazz.cast(singleInstanceType()));
        case "SingleAvailabilityZone":
            return Optional.ofNullable(clazz.cast(singleAvailabilityZone()));
        case "MinTargetCapacity":
            return Optional.ofNullable(clazz.cast(minTargetCapacity()));
        case "MaxTotalPrice":
            return Optional.ofNullable(clazz.cast(maxTotalPrice()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OnDemandOptionsRequest, T> g) {
        return obj -> g.apply((OnDemandOptionsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OnDemandOptionsRequest> {
        /**
         * <p>
         * The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand
         * capacity.
         * </p>
         * <p>
         * <code>lowest-price</code> - EC2 Fleet uses price to determine the order, launching the lowest price first.
         * </p>
         * <p>
         * <code>prioritized</code> - EC2 Fleet uses the priority that you assigned to each launch template override,
         * launching the highest priority first.
         * </p>
         * <p>
         * Default: <code>lowest-price</code>
         * </p>
         * 
         * @param allocationStrategy
         *        The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand
         *        capacity.</p>
         *        <p>
         *        <code>lowest-price</code> - EC2 Fleet uses price to determine the order, launching the lowest price
         *        first.
         *        </p>
         *        <p>
         *        <code>prioritized</code> - EC2 Fleet uses the priority that you assigned to each launch template
         *        override, launching the highest priority first.
         *        </p>
         *        <p>
         *        Default: <code>lowest-price</code>
         * @see FleetOnDemandAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetOnDemandAllocationStrategy
         */
        Builder allocationStrategy(String allocationStrategy);

        /**
         * <p>
         * The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand
         * capacity.
         * </p>
         * <p>
         * <code>lowest-price</code> - EC2 Fleet uses price to determine the order, launching the lowest price first.
         * </p>
         * <p>
         * <code>prioritized</code> - EC2 Fleet uses the priority that you assigned to each launch template override,
         * launching the highest priority first.
         * </p>
         * <p>
         * Default: <code>lowest-price</code>
         * </p>
         * 
         * @param allocationStrategy
         *        The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand
         *        capacity.</p>
         *        <p>
         *        <code>lowest-price</code> - EC2 Fleet uses price to determine the order, launching the lowest price
         *        first.
         *        </p>
         *        <p>
         *        <code>prioritized</code> - EC2 Fleet uses the priority that you assigned to each launch template
         *        override, launching the highest priority first.
         *        </p>
         *        <p>
         *        Default: <code>lowest-price</code>
         * @see FleetOnDemandAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetOnDemandAllocationStrategy
         */
        Builder allocationStrategy(FleetOnDemandAllocationStrategy allocationStrategy);

        /**
         * <p>
         * The strategy for using unused Capacity Reservations for fulfilling On-Demand capacity.
         * </p>
         * <p>
         * Supported only for fleets of type <code>instant</code>.
         * </p>
         * 
         * @param capacityReservationOptions
         *        The strategy for using unused Capacity Reservations for fulfilling On-Demand capacity.</p>
         *        <p>
         *        Supported only for fleets of type <code>instant</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityReservationOptions(CapacityReservationOptionsRequest capacityReservationOptions);

        /**
         * <p>
         * The strategy for using unused Capacity Reservations for fulfilling On-Demand capacity.
         * </p>
         * <p>
         * Supported only for fleets of type <code>instant</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link CapacityReservationOptionsRequest.Builder} avoiding the need to create one manually via
         * {@link CapacityReservationOptionsRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CapacityReservationOptionsRequest.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #capacityReservationOptions(CapacityReservationOptionsRequest)}.
         * 
         * @param capacityReservationOptions
         *        a consumer that will call methods on {@link CapacityReservationOptionsRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capacityReservationOptions(CapacityReservationOptionsRequest)
         */
        default Builder capacityReservationOptions(Consumer<CapacityReservationOptionsRequest.Builder> capacityReservationOptions) {
            return capacityReservationOptions(CapacityReservationOptionsRequest.builder()
                    .applyMutation(capacityReservationOptions).build());
        }

        /**
         * <p>
         * Indicates that the fleet uses a single instance type to launch all On-Demand Instances in the fleet.
         * </p>
         * <p>
         * Supported only for fleets of type <code>instant</code>.
         * </p>
         * 
         * @param singleInstanceType
         *        Indicates that the fleet uses a single instance type to launch all On-Demand Instances in the
         *        fleet.</p>
         *        <p>
         *        Supported only for fleets of type <code>instant</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder singleInstanceType(Boolean singleInstanceType);

        /**
         * <p>
         * Indicates that the fleet launches all On-Demand Instances into a single Availability Zone.
         * </p>
         * <p>
         * Supported only for fleets of type <code>instant</code>.
         * </p>
         * 
         * @param singleAvailabilityZone
         *        Indicates that the fleet launches all On-Demand Instances into a single Availability Zone.</p>
         *        <p>
         *        Supported only for fleets of type <code>instant</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder singleAvailabilityZone(Boolean singleAvailabilityZone);

        /**
         * <p>
         * The minimum target capacity for On-Demand Instances in the fleet. If the minimum target capacity is not
         * reached, the fleet launches no instances.
         * </p>
         * <p>
         * Supported only for fleets of type <code>instant</code>.
         * </p>
         * <p>
         * At least one of the following must be specified: <code>SingleAvailabilityZone</code> |
         * <code>SingleInstanceType</code>
         * </p>
         * 
         * @param minTargetCapacity
         *        The minimum target capacity for On-Demand Instances in the fleet. If the minimum target capacity is
         *        not reached, the fleet launches no instances.</p>
         *        <p>
         *        Supported only for fleets of type <code>instant</code>.
         *        </p>
         *        <p>
         *        At least one of the following must be specified: <code>SingleAvailabilityZone</code> |
         *        <code>SingleInstanceType</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minTargetCapacity(Integer minTargetCapacity);

        /**
         * <p>
         * The maximum amount per hour for On-Demand Instances that you're willing to pay.
         * </p>
         * <note>
         * <p>
         * If your fleet includes T instances that are configured as <code>unlimited</code>, and if their average CPU
         * usage exceeds the baseline utilization, you will incur a charge for surplus credits. The
         * <code>MaxTotalPrice</code> does not account for surplus credits, and, if you use surplus credits, your final
         * cost might be higher than what you specified for <code>MaxTotalPrice</code>. For more information, see <a
         * href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances-unlimited-mode-concepts.html#unlimited-mode-surplus-credits"
         * >Surplus credits can incur charges</a> in the <i>EC2 User Guide</i>.
         * </p>
         * </note>
         * 
         * @param maxTotalPrice
         *        The maximum amount per hour for On-Demand Instances that you're willing to pay.</p> <note>
         *        <p>
         *        If your fleet includes T instances that are configured as <code>unlimited</code>, and if their average
         *        CPU usage exceeds the baseline utilization, you will incur a charge for surplus credits. The
         *        <code>MaxTotalPrice</code> does not account for surplus credits, and, if you use surplus credits, your
         *        final cost might be higher than what you specified for <code>MaxTotalPrice</code>. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances-unlimited-mode-concepts.html#unlimited-mode-surplus-credits"
         *        >Surplus credits can incur charges</a> in the <i>EC2 User Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxTotalPrice(String maxTotalPrice);
    }

    static final class BuilderImpl implements Builder {
        private String allocationStrategy;

        private CapacityReservationOptionsRequest capacityReservationOptions;

        private Boolean singleInstanceType;

        private Boolean singleAvailabilityZone;

        private Integer minTargetCapacity;

        private String maxTotalPrice;

        private BuilderImpl() {
        }

        private BuilderImpl(OnDemandOptionsRequest model) {
            allocationStrategy(model.allocationStrategy);
            capacityReservationOptions(model.capacityReservationOptions);
            singleInstanceType(model.singleInstanceType);
            singleAvailabilityZone(model.singleAvailabilityZone);
            minTargetCapacity(model.minTargetCapacity);
            maxTotalPrice(model.maxTotalPrice);
        }

        public final String getAllocationStrategy() {
            return allocationStrategy;
        }

        public final void setAllocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
        }

        @Override
        public final Builder allocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
            return this;
        }

        @Override
        public final Builder allocationStrategy(FleetOnDemandAllocationStrategy allocationStrategy) {
            this.allocationStrategy(allocationStrategy == null ? null : allocationStrategy.toString());
            return this;
        }

        public final CapacityReservationOptionsRequest.Builder getCapacityReservationOptions() {
            return capacityReservationOptions != null ? capacityReservationOptions.toBuilder() : null;
        }

        public final void setCapacityReservationOptions(CapacityReservationOptionsRequest.BuilderImpl capacityReservationOptions) {
            this.capacityReservationOptions = capacityReservationOptions != null ? capacityReservationOptions.build() : null;
        }

        @Override
        public final Builder capacityReservationOptions(CapacityReservationOptionsRequest capacityReservationOptions) {
            this.capacityReservationOptions = capacityReservationOptions;
            return this;
        }

        public final Boolean getSingleInstanceType() {
            return singleInstanceType;
        }

        public final void setSingleInstanceType(Boolean singleInstanceType) {
            this.singleInstanceType = singleInstanceType;
        }

        @Override
        public final Builder singleInstanceType(Boolean singleInstanceType) {
            this.singleInstanceType = singleInstanceType;
            return this;
        }

        public final Boolean getSingleAvailabilityZone() {
            return singleAvailabilityZone;
        }

        public final void setSingleAvailabilityZone(Boolean singleAvailabilityZone) {
            this.singleAvailabilityZone = singleAvailabilityZone;
        }

        @Override
        public final Builder singleAvailabilityZone(Boolean singleAvailabilityZone) {
            this.singleAvailabilityZone = singleAvailabilityZone;
            return this;
        }

        public final Integer getMinTargetCapacity() {
            return minTargetCapacity;
        }

        public final void setMinTargetCapacity(Integer minTargetCapacity) {
            this.minTargetCapacity = minTargetCapacity;
        }

        @Override
        public final Builder minTargetCapacity(Integer minTargetCapacity) {
            this.minTargetCapacity = minTargetCapacity;
            return this;
        }

        public final String getMaxTotalPrice() {
            return maxTotalPrice;
        }

        public final void setMaxTotalPrice(String maxTotalPrice) {
            this.maxTotalPrice = maxTotalPrice;
        }

        @Override
        public final Builder maxTotalPrice(String maxTotalPrice) {
            this.maxTotalPrice = maxTotalPrice;
            return this;
        }

        @Override
        public OnDemandOptionsRequest build() {
            return new OnDemandOptionsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
