/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateRestoreImageTaskRequest extends Ec2Request implements
        ToCopyableBuilder<CreateRestoreImageTaskRequest.Builder, CreateRestoreImageTaskRequest> {
    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Bucket")
            .getter(getter(CreateRestoreImageTaskRequest::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build()).build();

    private static final SdkField<String> OBJECT_KEY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ObjectKey")
            .getter(getter(CreateRestoreImageTaskRequest::objectKey))
            .setter(setter(Builder::objectKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectKey")
                    .unmarshallLocationName("ObjectKey").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(CreateRestoreImageTaskRequest::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name")
                    .unmarshallLocationName("Name").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .memberName("TagSpecifications")
            .getter(getter(CreateRestoreImageTaskRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_FIELD,
            OBJECT_KEY_FIELD, NAME_FIELD, TAG_SPECIFICATIONS_FIELD));

    private final String bucket;

    private final String objectKey;

    private final String name;

    private final List<TagSpecification> tagSpecifications;

    private CreateRestoreImageTaskRequest(BuilderImpl builder) {
        super(builder);
        this.bucket = builder.bucket;
        this.objectKey = builder.objectKey;
        this.name = builder.name;
        this.tagSpecifications = builder.tagSpecifications;
    }

    /**
     * <p>
     * The name of the Amazon S3 bucket that contains the stored AMI object.
     * </p>
     * 
     * @return The name of the Amazon S3 bucket that contains the stored AMI object.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * The name of the stored AMI object in the bucket.
     * </p>
     * 
     * @return The name of the stored AMI object in the bucket.
     */
    public final String objectKey() {
        return objectKey;
    }

    /**
     * <p>
     * The name for the restored AMI. The name must be unique for AMIs in the Region for this account. If you do not
     * provide a name, the new AMI gets the same name as the original AMI.
     * </p>
     * 
     * @return The name for the restored AMI. The name must be unique for AMIs in the Region for this account. If you do
     *         not provide a name, the new AMI gets the same name as the original AMI.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to apply to the AMI and snapshots on restoration. You can tag the AMI, the snapshots, or both.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To tag the AMI, the value for <code>ResourceType</code> must be <code>image</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To tag the snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same tag is
     * applied to all of the snapshots that are created.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagSpecifications} method.
     * </p>
     * 
     * @return The tags to apply to the AMI and snapshots on restoration. You can tag the AMI, the snapshots, or
     *         both.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To tag the AMI, the value for <code>ResourceType</code> must be <code>image</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To tag the snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same tag
     *         is applied to all of the snapshots that are created.
     *         </p>
     *         </li>
     */
    public final List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(objectKey());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagSpecifications() ? tagSpecifications() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateRestoreImageTaskRequest)) {
            return false;
        }
        CreateRestoreImageTaskRequest other = (CreateRestoreImageTaskRequest) obj;
        return Objects.equals(bucket(), other.bucket()) && Objects.equals(objectKey(), other.objectKey())
                && Objects.equals(name(), other.name()) && hasTagSpecifications() == other.hasTagSpecifications()
                && Objects.equals(tagSpecifications(), other.tagSpecifications());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateRestoreImageTaskRequest").add("Bucket", bucket()).add("ObjectKey", objectKey())
                .add("Name", name()).add("TagSpecifications", hasTagSpecifications() ? tagSpecifications() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "ObjectKey":
            return Optional.ofNullable(clazz.cast(objectKey()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateRestoreImageTaskRequest, T> g) {
        return obj -> g.apply((CreateRestoreImageTaskRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateRestoreImageTaskRequest> {
        /**
         * <p>
         * The name of the Amazon S3 bucket that contains the stored AMI object.
         * </p>
         * 
         * @param bucket
         *        The name of the Amazon S3 bucket that contains the stored AMI object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * The name of the stored AMI object in the bucket.
         * </p>
         * 
         * @param objectKey
         *        The name of the stored AMI object in the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectKey(String objectKey);

        /**
         * <p>
         * The name for the restored AMI. The name must be unique for AMIs in the Region for this account. If you do not
         * provide a name, the new AMI gets the same name as the original AMI.
         * </p>
         * 
         * @param name
         *        The name for the restored AMI. The name must be unique for AMIs in the Region for this account. If you
         *        do not provide a name, the new AMI gets the same name as the original AMI.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The tags to apply to the AMI and snapshots on restoration. You can tag the AMI, the snapshots, or both.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To tag the AMI, the value for <code>ResourceType</code> must be <code>image</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To tag the snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same tag is
         * applied to all of the snapshots that are created.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tagSpecifications
         *        The tags to apply to the AMI and snapshots on restoration. You can tag the AMI, the snapshots, or
         *        both.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To tag the AMI, the value for <code>ResourceType</code> must be <code>image</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To tag the snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same
         *        tag is applied to all of the snapshots that are created.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The tags to apply to the AMI and snapshots on restoration. You can tag the AMI, the snapshots, or both.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To tag the AMI, the value for <code>ResourceType</code> must be <code>image</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To tag the snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same tag is
         * applied to all of the snapshots that are created.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tagSpecifications
         *        The tags to apply to the AMI and snapshots on restoration. You can tag the AMI, the snapshots, or
         *        both.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To tag the AMI, the value for <code>ResourceType</code> must be <code>image</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To tag the snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same
         *        tag is applied to all of the snapshots that are created.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The tags to apply to the AMI and snapshots on restoration. You can tag the AMI, the snapshots, or both.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To tag the AMI, the value for <code>ResourceType</code> must be <code>image</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To tag the snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same tag is
         * applied to all of the snapshots that are created.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link List<TagSpecification>.Builder} avoiding the
         * need to create one manually via {@link List<TagSpecification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TagSpecification>.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on {@link List<TagSpecification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(List<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String bucket;

        private String objectKey;

        private String name;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateRestoreImageTaskRequest model) {
            super(model);
            bucket(model.bucket);
            objectKey(model.objectKey);
            name(model.name);
            tagSpecifications(model.tagSpecifications);
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        @Transient
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final String getObjectKey() {
            return objectKey;
        }

        public final void setObjectKey(String objectKey) {
            this.objectKey = objectKey;
        }

        @Override
        @Transient
        public final Builder objectKey(String objectKey) {
            this.objectKey = objectKey;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final List<TagSpecification.Builder> getTagSpecifications() {
            List<TagSpecification.Builder> result = TagSpecificationListCopier.copyToBuilder(this.tagSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        @Transient
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateRestoreImageTaskRequest build() {
            return new CreateRestoreImageTaskRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
