/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The CIDR for an IPAM resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamResourceCidr implements SdkPojo, Serializable,
        ToCopyableBuilder<IpamResourceCidr.Builder, IpamResourceCidr> {
    private static final SdkField<String> IPAM_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamId")
            .getter(getter(IpamResourceCidr::ipamId))
            .setter(setter(Builder::ipamId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamId")
                    .unmarshallLocationName("ipamId").build()).build();

    private static final SdkField<String> IPAM_SCOPE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamScopeId")
            .getter(getter(IpamResourceCidr::ipamScopeId))
            .setter(setter(Builder::ipamScopeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamScopeId")
                    .unmarshallLocationName("ipamScopeId").build()).build();

    private static final SdkField<String> IPAM_POOL_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamPoolId")
            .getter(getter(IpamResourceCidr::ipamPoolId))
            .setter(setter(Builder::ipamPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamPoolId")
                    .unmarshallLocationName("ipamPoolId").build()).build();

    private static final SdkField<String> RESOURCE_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceRegion")
            .getter(getter(IpamResourceCidr::resourceRegion))
            .setter(setter(Builder::resourceRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceRegion")
                    .unmarshallLocationName("resourceRegion").build()).build();

    private static final SdkField<String> RESOURCE_OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceOwnerId")
            .getter(getter(IpamResourceCidr::resourceOwnerId))
            .setter(setter(Builder::resourceOwnerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceOwnerId")
                    .unmarshallLocationName("resourceOwnerId").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceId")
            .getter(getter(IpamResourceCidr::resourceId))
            .setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId")
                    .unmarshallLocationName("resourceId").build()).build();

    private static final SdkField<String> RESOURCE_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceName")
            .getter(getter(IpamResourceCidr::resourceName))
            .setter(setter(Builder::resourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceName")
                    .unmarshallLocationName("resourceName").build()).build();

    private static final SdkField<String> RESOURCE_CIDR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceCidr")
            .getter(getter(IpamResourceCidr::resourceCidr))
            .setter(setter(Builder::resourceCidr))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceCidr")
                    .unmarshallLocationName("resourceCidr").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceType")
            .getter(getter(IpamResourceCidr::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType")
                    .unmarshallLocationName("resourceType").build()).build();

    private static final SdkField<List<IpamResourceTag>> RESOURCE_TAGS_FIELD = SdkField
            .<List<IpamResourceTag>> builder(MarshallingType.LIST)
            .memberName("ResourceTags")
            .getter(getter(IpamResourceCidr::resourceTags))
            .setter(setter(Builder::resourceTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTagSet")
                    .unmarshallLocationName("resourceTagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<IpamResourceTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(IpamResourceTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Double> IP_USAGE_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .memberName("IpUsage")
            .getter(getter(IpamResourceCidr::ipUsage))
            .setter(setter(Builder::ipUsage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpUsage")
                    .unmarshallLocationName("ipUsage").build()).build();

    private static final SdkField<String> COMPLIANCE_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ComplianceStatus")
            .getter(getter(IpamResourceCidr::complianceStatusAsString))
            .setter(setter(Builder::complianceStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComplianceStatus")
                    .unmarshallLocationName("complianceStatus").build()).build();

    private static final SdkField<String> MANAGEMENT_STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ManagementState")
            .getter(getter(IpamResourceCidr::managementStateAsString))
            .setter(setter(Builder::managementState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManagementState")
                    .unmarshallLocationName("managementState").build()).build();

    private static final SdkField<String> OVERLAP_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OverlapStatus")
            .getter(getter(IpamResourceCidr::overlapStatusAsString))
            .setter(setter(Builder::overlapStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OverlapStatus")
                    .unmarshallLocationName("overlapStatus").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcId")
            .getter(getter(IpamResourceCidr::vpcId))
            .setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId")
                    .unmarshallLocationName("vpcId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IPAM_ID_FIELD,
            IPAM_SCOPE_ID_FIELD, IPAM_POOL_ID_FIELD, RESOURCE_REGION_FIELD, RESOURCE_OWNER_ID_FIELD, RESOURCE_ID_FIELD,
            RESOURCE_NAME_FIELD, RESOURCE_CIDR_FIELD, RESOURCE_TYPE_FIELD, RESOURCE_TAGS_FIELD, IP_USAGE_FIELD,
            COMPLIANCE_STATUS_FIELD, MANAGEMENT_STATE_FIELD, OVERLAP_STATUS_FIELD, VPC_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ipamId;

    private final String ipamScopeId;

    private final String ipamPoolId;

    private final String resourceRegion;

    private final String resourceOwnerId;

    private final String resourceId;

    private final String resourceName;

    private final String resourceCidr;

    private final String resourceType;

    private final List<IpamResourceTag> resourceTags;

    private final Double ipUsage;

    private final String complianceStatus;

    private final String managementState;

    private final String overlapStatus;

    private final String vpcId;

    private IpamResourceCidr(BuilderImpl builder) {
        this.ipamId = builder.ipamId;
        this.ipamScopeId = builder.ipamScopeId;
        this.ipamPoolId = builder.ipamPoolId;
        this.resourceRegion = builder.resourceRegion;
        this.resourceOwnerId = builder.resourceOwnerId;
        this.resourceId = builder.resourceId;
        this.resourceName = builder.resourceName;
        this.resourceCidr = builder.resourceCidr;
        this.resourceType = builder.resourceType;
        this.resourceTags = builder.resourceTags;
        this.ipUsage = builder.ipUsage;
        this.complianceStatus = builder.complianceStatus;
        this.managementState = builder.managementState;
        this.overlapStatus = builder.overlapStatus;
        this.vpcId = builder.vpcId;
    }

    /**
     * <p>
     * The IPAM ID for an IPAM resource.
     * </p>
     * 
     * @return The IPAM ID for an IPAM resource.
     */
    public final String ipamId() {
        return ipamId;
    }

    /**
     * <p>
     * The scope ID for an IPAM resource.
     * </p>
     * 
     * @return The scope ID for an IPAM resource.
     */
    public final String ipamScopeId() {
        return ipamScopeId;
    }

    /**
     * <p>
     * The pool ID for an IPAM resource.
     * </p>
     * 
     * @return The pool ID for an IPAM resource.
     */
    public final String ipamPoolId() {
        return ipamPoolId;
    }

    /**
     * <p>
     * The Amazon Web Services Region for an IPAM resource.
     * </p>
     * 
     * @return The Amazon Web Services Region for an IPAM resource.
     */
    public final String resourceRegion() {
        return resourceRegion;
    }

    /**
     * <p>
     * The Amazon Web Services account number of the owner of an IPAM resource.
     * </p>
     * 
     * @return The Amazon Web Services account number of the owner of an IPAM resource.
     */
    public final String resourceOwnerId() {
        return resourceOwnerId;
    }

    /**
     * <p>
     * The ID of an IPAM resource.
     * </p>
     * 
     * @return The ID of an IPAM resource.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The name of an IPAM resource.
     * </p>
     * 
     * @return The name of an IPAM resource.
     */
    public final String resourceName() {
        return resourceName;
    }

    /**
     * <p>
     * The CIDR for an IPAM resource.
     * </p>
     * 
     * @return The CIDR for an IPAM resource.
     */
    public final String resourceCidr() {
        return resourceCidr;
    }

    /**
     * <p>
     * The type of IPAM resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link IpamResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of IPAM resource.
     * @see IpamResourceType
     */
    public final IpamResourceType resourceType() {
        return IpamResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of IPAM resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link IpamResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of IPAM resource.
     * @see IpamResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTags() {
        return resourceTags != null && !(resourceTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags for an IPAM resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTags} method.
     * </p>
     * 
     * @return The tags for an IPAM resource.
     */
    public final List<IpamResourceTag> resourceTags() {
        return resourceTags;
    }

    /**
     * <p>
     * The percentage of IP address space in use. To convert the decimal to a percentage, multiply the decimal by 100.
     * Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For a resources that are VPCs, this is the percentage of IP address space in the VPC that's taken up by subnet
     * CIDRs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For resources that are subnets, if the subnet has an IPv4 CIDR provisioned to it, this is the percentage of IPv4
     * address space in the subnet that's in use. If the subnet has an IPv6 CIDR provisioned to it, the percentage of
     * IPv6 address space in use is not represented. The percentage of IPv6 address space in use cannot currently be
     * calculated.
     * </p>
     * </li>
     * <li>
     * <p>
     * For resources that are public IPv4 pools, this is the percentage of IP address space in the pool that's been
     * allocated to Elastic IP addresses (EIPs).
     * </p>
     * </li>
     * </ul>
     * 
     * @return The percentage of IP address space in use. To convert the decimal to a percentage, multiply the decimal
     *         by 100. Note the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For a resources that are VPCs, this is the percentage of IP address space in the VPC that's taken up by
     *         subnet CIDRs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For resources that are subnets, if the subnet has an IPv4 CIDR provisioned to it, this is the percentage
     *         of IPv4 address space in the subnet that's in use. If the subnet has an IPv6 CIDR provisioned to it, the
     *         percentage of IPv6 address space in use is not represented. The percentage of IPv6 address space in use
     *         cannot currently be calculated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For resources that are public IPv4 pools, this is the percentage of IP address space in the pool that's
     *         been allocated to Elastic IP addresses (EIPs).
     *         </p>
     *         </li>
     */
    public final Double ipUsage() {
        return ipUsage;
    }

    /**
     * <p>
     * The compliance status of the IPAM resource. For more information on compliance statuses, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
     * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #complianceStatus}
     * will return {@link IpamComplianceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #complianceStatusAsString}.
     * </p>
     * 
     * @return The compliance status of the IPAM resource. For more information on compliance statuses, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage
     *         by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * @see IpamComplianceStatus
     */
    public final IpamComplianceStatus complianceStatus() {
        return IpamComplianceStatus.fromValue(complianceStatus);
    }

    /**
     * <p>
     * The compliance status of the IPAM resource. For more information on compliance statuses, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
     * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #complianceStatus}
     * will return {@link IpamComplianceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #complianceStatusAsString}.
     * </p>
     * 
     * @return The compliance status of the IPAM resource. For more information on compliance statuses, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage
     *         by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * @see IpamComplianceStatus
     */
    public final String complianceStatusAsString() {
        return complianceStatus;
    }

    /**
     * <p>
     * The management state of the resource. For more information about management states, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
     * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #managementState}
     * will return {@link IpamManagementState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #managementStateAsString}.
     * </p>
     * 
     * @return The management state of the resource. For more information about management states, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage
     *         by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * @see IpamManagementState
     */
    public final IpamManagementState managementState() {
        return IpamManagementState.fromValue(managementState);
    }

    /**
     * <p>
     * The management state of the resource. For more information about management states, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
     * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #managementState}
     * will return {@link IpamManagementState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #managementStateAsString}.
     * </p>
     * 
     * @return The management state of the resource. For more information about management states, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage
     *         by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * @see IpamManagementState
     */
    public final String managementStateAsString() {
        return managementState;
    }

    /**
     * <p>
     * The overlap status of an IPAM resource. The overlap status tells you if the CIDR for a resource overlaps with
     * another CIDR in the scope. For more information on overlap statuses, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
     * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #overlapStatus}
     * will return {@link IpamOverlapStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #overlapStatusAsString}.
     * </p>
     * 
     * @return The overlap status of an IPAM resource. The overlap status tells you if the CIDR for a resource overlaps
     *         with another CIDR in the scope. For more information on overlap statuses, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage
     *         by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * @see IpamOverlapStatus
     */
    public final IpamOverlapStatus overlapStatus() {
        return IpamOverlapStatus.fromValue(overlapStatus);
    }

    /**
     * <p>
     * The overlap status of an IPAM resource. The overlap status tells you if the CIDR for a resource overlaps with
     * another CIDR in the scope. For more information on overlap statuses, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
     * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #overlapStatus}
     * will return {@link IpamOverlapStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #overlapStatusAsString}.
     * </p>
     * 
     * @return The overlap status of an IPAM resource. The overlap status tells you if the CIDR for a resource overlaps
     *         with another CIDR in the scope. For more information on overlap statuses, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage
     *         by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * @see IpamOverlapStatus
     */
    public final String overlapStatusAsString() {
        return overlapStatus;
    }

    /**
     * <p>
     * The ID of a VPC.
     * </p>
     * 
     * @return The ID of a VPC.
     */
    public final String vpcId() {
        return vpcId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ipamId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamScopeId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceRegion());
        hashCode = 31 * hashCode + Objects.hashCode(resourceOwnerId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceName());
        hashCode = 31 * hashCode + Objects.hashCode(resourceCidr());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTags() ? resourceTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ipUsage());
        hashCode = 31 * hashCode + Objects.hashCode(complianceStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(managementStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(overlapStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamResourceCidr)) {
            return false;
        }
        IpamResourceCidr other = (IpamResourceCidr) obj;
        return Objects.equals(ipamId(), other.ipamId()) && Objects.equals(ipamScopeId(), other.ipamScopeId())
                && Objects.equals(ipamPoolId(), other.ipamPoolId()) && Objects.equals(resourceRegion(), other.resourceRegion())
                && Objects.equals(resourceOwnerId(), other.resourceOwnerId()) && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(resourceName(), other.resourceName()) && Objects.equals(resourceCidr(), other.resourceCidr())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && hasResourceTags() == other.hasResourceTags() && Objects.equals(resourceTags(), other.resourceTags())
                && Objects.equals(ipUsage(), other.ipUsage())
                && Objects.equals(complianceStatusAsString(), other.complianceStatusAsString())
                && Objects.equals(managementStateAsString(), other.managementStateAsString())
                && Objects.equals(overlapStatusAsString(), other.overlapStatusAsString())
                && Objects.equals(vpcId(), other.vpcId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamResourceCidr").add("IpamId", ipamId()).add("IpamScopeId", ipamScopeId())
                .add("IpamPoolId", ipamPoolId()).add("ResourceRegion", resourceRegion())
                .add("ResourceOwnerId", resourceOwnerId()).add("ResourceId", resourceId()).add("ResourceName", resourceName())
                .add("ResourceCidr", resourceCidr()).add("ResourceType", resourceTypeAsString())
                .add("ResourceTags", hasResourceTags() ? resourceTags() : null).add("IpUsage", ipUsage())
                .add("ComplianceStatus", complianceStatusAsString()).add("ManagementState", managementStateAsString())
                .add("OverlapStatus", overlapStatusAsString()).add("VpcId", vpcId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IpamId":
            return Optional.ofNullable(clazz.cast(ipamId()));
        case "IpamScopeId":
            return Optional.ofNullable(clazz.cast(ipamScopeId()));
        case "IpamPoolId":
            return Optional.ofNullable(clazz.cast(ipamPoolId()));
        case "ResourceRegion":
            return Optional.ofNullable(clazz.cast(resourceRegion()));
        case "ResourceOwnerId":
            return Optional.ofNullable(clazz.cast(resourceOwnerId()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "ResourceName":
            return Optional.ofNullable(clazz.cast(resourceName()));
        case "ResourceCidr":
            return Optional.ofNullable(clazz.cast(resourceCidr()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "ResourceTags":
            return Optional.ofNullable(clazz.cast(resourceTags()));
        case "IpUsage":
            return Optional.ofNullable(clazz.cast(ipUsage()));
        case "ComplianceStatus":
            return Optional.ofNullable(clazz.cast(complianceStatusAsString()));
        case "ManagementState":
            return Optional.ofNullable(clazz.cast(managementStateAsString()));
        case "OverlapStatus":
            return Optional.ofNullable(clazz.cast(overlapStatusAsString()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IpamResourceCidr, T> g) {
        return obj -> g.apply((IpamResourceCidr) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamResourceCidr> {
        /**
         * <p>
         * The IPAM ID for an IPAM resource.
         * </p>
         * 
         * @param ipamId
         *        The IPAM ID for an IPAM resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamId(String ipamId);

        /**
         * <p>
         * The scope ID for an IPAM resource.
         * </p>
         * 
         * @param ipamScopeId
         *        The scope ID for an IPAM resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamScopeId(String ipamScopeId);

        /**
         * <p>
         * The pool ID for an IPAM resource.
         * </p>
         * 
         * @param ipamPoolId
         *        The pool ID for an IPAM resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamPoolId(String ipamPoolId);

        /**
         * <p>
         * The Amazon Web Services Region for an IPAM resource.
         * </p>
         * 
         * @param resourceRegion
         *        The Amazon Web Services Region for an IPAM resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRegion(String resourceRegion);

        /**
         * <p>
         * The Amazon Web Services account number of the owner of an IPAM resource.
         * </p>
         * 
         * @param resourceOwnerId
         *        The Amazon Web Services account number of the owner of an IPAM resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceOwnerId(String resourceOwnerId);

        /**
         * <p>
         * The ID of an IPAM resource.
         * </p>
         * 
         * @param resourceId
         *        The ID of an IPAM resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The name of an IPAM resource.
         * </p>
         * 
         * @param resourceName
         *        The name of an IPAM resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceName(String resourceName);

        /**
         * <p>
         * The CIDR for an IPAM resource.
         * </p>
         * 
         * @param resourceCidr
         *        The CIDR for an IPAM resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceCidr(String resourceCidr);

        /**
         * <p>
         * The type of IPAM resource.
         * </p>
         * 
         * @param resourceType
         *        The type of IPAM resource.
         * @see IpamResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of IPAM resource.
         * </p>
         * 
         * @param resourceType
         *        The type of IPAM resource.
         * @see IpamResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamResourceType
         */
        Builder resourceType(IpamResourceType resourceType);

        /**
         * <p>
         * The tags for an IPAM resource.
         * </p>
         * 
         * @param resourceTags
         *        The tags for an IPAM resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(Collection<IpamResourceTag> resourceTags);

        /**
         * <p>
         * The tags for an IPAM resource.
         * </p>
         * 
         * @param resourceTags
         *        The tags for an IPAM resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(IpamResourceTag... resourceTags);

        /**
         * <p>
         * The tags for an IPAM resource.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<IpamResourceTag>.Builder} avoiding
         * the need to create one manually via {@link List<IpamResourceTag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<IpamResourceTag>.Builder#build()} is called immediately and
         * its result is passed to {@link #resourceTags(List<IpamResourceTag>)}.
         * 
         * @param resourceTags
         *        a consumer that will call methods on {@link List<IpamResourceTag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceTags(List<IpamResourceTag>)
         */
        Builder resourceTags(Consumer<IpamResourceTag.Builder>... resourceTags);

        /**
         * <p>
         * The percentage of IP address space in use. To convert the decimal to a percentage, multiply the decimal by
         * 100. Note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For a resources that are VPCs, this is the percentage of IP address space in the VPC that's taken up by
         * subnet CIDRs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For resources that are subnets, if the subnet has an IPv4 CIDR provisioned to it, this is the percentage of
         * IPv4 address space in the subnet that's in use. If the subnet has an IPv6 CIDR provisioned to it, the
         * percentage of IPv6 address space in use is not represented. The percentage of IPv6 address space in use
         * cannot currently be calculated.
         * </p>
         * </li>
         * <li>
         * <p>
         * For resources that are public IPv4 pools, this is the percentage of IP address space in the pool that's been
         * allocated to Elastic IP addresses (EIPs).
         * </p>
         * </li>
         * </ul>
         * 
         * @param ipUsage
         *        The percentage of IP address space in use. To convert the decimal to a percentage, multiply the
         *        decimal by 100. Note the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For a resources that are VPCs, this is the percentage of IP address space in the VPC that's taken up
         *        by subnet CIDRs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For resources that are subnets, if the subnet has an IPv4 CIDR provisioned to it, this is the
         *        percentage of IPv4 address space in the subnet that's in use. If the subnet has an IPv6 CIDR
         *        provisioned to it, the percentage of IPv6 address space in use is not represented. The percentage of
         *        IPv6 address space in use cannot currently be calculated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For resources that are public IPv4 pools, this is the percentage of IP address space in the pool
         *        that's been allocated to Elastic IP addresses (EIPs).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipUsage(Double ipUsage);

        /**
         * <p>
         * The compliance status of the IPAM resource. For more information on compliance statuses, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
         * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param complianceStatus
         *        The compliance status of the IPAM resource. For more information on compliance statuses, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR
         *        usage by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * @see IpamComplianceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamComplianceStatus
         */
        Builder complianceStatus(String complianceStatus);

        /**
         * <p>
         * The compliance status of the IPAM resource. For more information on compliance statuses, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
         * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param complianceStatus
         *        The compliance status of the IPAM resource. For more information on compliance statuses, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR
         *        usage by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * @see IpamComplianceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamComplianceStatus
         */
        Builder complianceStatus(IpamComplianceStatus complianceStatus);

        /**
         * <p>
         * The management state of the resource. For more information about management states, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
         * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param managementState
         *        The management state of the resource. For more information about management states, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR
         *        usage by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * @see IpamManagementState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamManagementState
         */
        Builder managementState(String managementState);

        /**
         * <p>
         * The management state of the resource. For more information about management states, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
         * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param managementState
         *        The management state of the resource. For more information about management states, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR
         *        usage by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * @see IpamManagementState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamManagementState
         */
        Builder managementState(IpamManagementState managementState);

        /**
         * <p>
         * The overlap status of an IPAM resource. The overlap status tells you if the CIDR for a resource overlaps with
         * another CIDR in the scope. For more information on overlap statuses, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
         * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param overlapStatus
         *        The overlap status of an IPAM resource. The overlap status tells you if the CIDR for a resource
         *        overlaps with another CIDR in the scope. For more information on overlap statuses, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR
         *        usage by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * @see IpamOverlapStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamOverlapStatus
         */
        Builder overlapStatus(String overlapStatus);

        /**
         * <p>
         * The overlap status of an IPAM resource. The overlap status tells you if the CIDR for a resource overlaps with
         * another CIDR in the scope. For more information on overlap statuses, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR usage by
         * resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param overlapStatus
         *        The overlap status of an IPAM resource. The overlap status tells you if the CIDR for a resource
         *        overlaps with another CIDR in the scope. For more information on overlap statuses, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/monitor-cidr-compliance-ipam.html">Monitor CIDR
         *        usage by resource</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * @see IpamOverlapStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamOverlapStatus
         */
        Builder overlapStatus(IpamOverlapStatus overlapStatus);

        /**
         * <p>
         * The ID of a VPC.
         * </p>
         * 
         * @param vpcId
         *        The ID of a VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);
    }

    static final class BuilderImpl implements Builder {
        private String ipamId;

        private String ipamScopeId;

        private String ipamPoolId;

        private String resourceRegion;

        private String resourceOwnerId;

        private String resourceId;

        private String resourceName;

        private String resourceCidr;

        private String resourceType;

        private List<IpamResourceTag> resourceTags = DefaultSdkAutoConstructList.getInstance();

        private Double ipUsage;

        private String complianceStatus;

        private String managementState;

        private String overlapStatus;

        private String vpcId;

        private BuilderImpl() {
        }

        private BuilderImpl(IpamResourceCidr model) {
            ipamId(model.ipamId);
            ipamScopeId(model.ipamScopeId);
            ipamPoolId(model.ipamPoolId);
            resourceRegion(model.resourceRegion);
            resourceOwnerId(model.resourceOwnerId);
            resourceId(model.resourceId);
            resourceName(model.resourceName);
            resourceCidr(model.resourceCidr);
            resourceType(model.resourceType);
            resourceTags(model.resourceTags);
            ipUsage(model.ipUsage);
            complianceStatus(model.complianceStatus);
            managementState(model.managementState);
            overlapStatus(model.overlapStatus);
            vpcId(model.vpcId);
        }

        public final String getIpamId() {
            return ipamId;
        }

        public final void setIpamId(String ipamId) {
            this.ipamId = ipamId;
        }

        @Override
        public final Builder ipamId(String ipamId) {
            this.ipamId = ipamId;
            return this;
        }

        public final String getIpamScopeId() {
            return ipamScopeId;
        }

        public final void setIpamScopeId(String ipamScopeId) {
            this.ipamScopeId = ipamScopeId;
        }

        @Override
        public final Builder ipamScopeId(String ipamScopeId) {
            this.ipamScopeId = ipamScopeId;
            return this;
        }

        public final String getIpamPoolId() {
            return ipamPoolId;
        }

        public final void setIpamPoolId(String ipamPoolId) {
            this.ipamPoolId = ipamPoolId;
        }

        @Override
        public final Builder ipamPoolId(String ipamPoolId) {
            this.ipamPoolId = ipamPoolId;
            return this;
        }

        public final String getResourceRegion() {
            return resourceRegion;
        }

        public final void setResourceRegion(String resourceRegion) {
            this.resourceRegion = resourceRegion;
        }

        @Override
        public final Builder resourceRegion(String resourceRegion) {
            this.resourceRegion = resourceRegion;
            return this;
        }

        public final String getResourceOwnerId() {
            return resourceOwnerId;
        }

        public final void setResourceOwnerId(String resourceOwnerId) {
            this.resourceOwnerId = resourceOwnerId;
        }

        @Override
        public final Builder resourceOwnerId(String resourceOwnerId) {
            this.resourceOwnerId = resourceOwnerId;
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getResourceName() {
            return resourceName;
        }

        public final void setResourceName(String resourceName) {
            this.resourceName = resourceName;
        }

        @Override
        public final Builder resourceName(String resourceName) {
            this.resourceName = resourceName;
            return this;
        }

        public final String getResourceCidr() {
            return resourceCidr;
        }

        public final void setResourceCidr(String resourceCidr) {
            this.resourceCidr = resourceCidr;
        }

        @Override
        public final Builder resourceCidr(String resourceCidr) {
            this.resourceCidr = resourceCidr;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(IpamResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final List<IpamResourceTag.Builder> getResourceTags() {
            List<IpamResourceTag.Builder> result = IpamResourceTagListCopier.copyToBuilder(this.resourceTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourceTags(Collection<IpamResourceTag.BuilderImpl> resourceTags) {
            this.resourceTags = IpamResourceTagListCopier.copyFromBuilder(resourceTags);
        }

        @Override
        public final Builder resourceTags(Collection<IpamResourceTag> resourceTags) {
            this.resourceTags = IpamResourceTagListCopier.copy(resourceTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(IpamResourceTag... resourceTags) {
            resourceTags(Arrays.asList(resourceTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(Consumer<IpamResourceTag.Builder>... resourceTags) {
            resourceTags(Stream.of(resourceTags).map(c -> IpamResourceTag.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Double getIpUsage() {
            return ipUsage;
        }

        public final void setIpUsage(Double ipUsage) {
            this.ipUsage = ipUsage;
        }

        @Override
        public final Builder ipUsage(Double ipUsage) {
            this.ipUsage = ipUsage;
            return this;
        }

        public final String getComplianceStatus() {
            return complianceStatus;
        }

        public final void setComplianceStatus(String complianceStatus) {
            this.complianceStatus = complianceStatus;
        }

        @Override
        public final Builder complianceStatus(String complianceStatus) {
            this.complianceStatus = complianceStatus;
            return this;
        }

        @Override
        public final Builder complianceStatus(IpamComplianceStatus complianceStatus) {
            this.complianceStatus(complianceStatus == null ? null : complianceStatus.toString());
            return this;
        }

        public final String getManagementState() {
            return managementState;
        }

        public final void setManagementState(String managementState) {
            this.managementState = managementState;
        }

        @Override
        public final Builder managementState(String managementState) {
            this.managementState = managementState;
            return this;
        }

        @Override
        public final Builder managementState(IpamManagementState managementState) {
            this.managementState(managementState == null ? null : managementState.toString());
            return this;
        }

        public final String getOverlapStatus() {
            return overlapStatus;
        }

        public final void setOverlapStatus(String overlapStatus) {
            this.overlapStatus = overlapStatus;
        }

        @Override
        public final Builder overlapStatus(String overlapStatus) {
            this.overlapStatus = overlapStatus;
            return this;
        }

        @Override
        public final Builder overlapStatus(IpamOverlapStatus overlapStatus) {
            this.overlapStatus(overlapStatus == null ? null : overlapStatus.toString());
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        @Override
        public IpamResourceCidr build() {
            return new IpamResourceCidr(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
