/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a target network associated with a Client VPN endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TargetNetwork implements SdkPojo, Serializable, ToCopyableBuilder<TargetNetwork.Builder, TargetNetwork> {
    private static final SdkField<String> ASSOCIATION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AssociationId")
            .getter(getter(TargetNetwork::associationId))
            .setter(setter(Builder::associationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationId")
                    .unmarshallLocationName("associationId").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcId")
            .getter(getter(TargetNetwork::vpcId))
            .setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId")
                    .unmarshallLocationName("vpcId").build()).build();

    private static final SdkField<String> TARGET_NETWORK_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TargetNetworkId")
            .getter(getter(TargetNetwork::targetNetworkId))
            .setter(setter(Builder::targetNetworkId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetNetworkId")
                    .unmarshallLocationName("targetNetworkId").build()).build();

    private static final SdkField<String> CLIENT_VPN_ENDPOINT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientVpnEndpointId")
            .getter(getter(TargetNetwork::clientVpnEndpointId))
            .setter(setter(Builder::clientVpnEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientVpnEndpointId")
                    .unmarshallLocationName("clientVpnEndpointId").build()).build();

    private static final SdkField<AssociationStatus> STATUS_FIELD = SdkField
            .<AssociationStatus> builder(MarshallingType.SDK_POJO)
            .memberName("Status")
            .getter(getter(TargetNetwork::status))
            .setter(setter(Builder::status))
            .constructor(AssociationStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status")
                    .unmarshallLocationName("status").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroups")
            .getter(getter(TargetNetwork::securityGroups))
            .setter(setter(Builder::securityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroups")
                    .unmarshallLocationName("securityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASSOCIATION_ID_FIELD,
            VPC_ID_FIELD, TARGET_NETWORK_ID_FIELD, CLIENT_VPN_ENDPOINT_ID_FIELD, STATUS_FIELD, SECURITY_GROUPS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String associationId;

    private final String vpcId;

    private final String targetNetworkId;

    private final String clientVpnEndpointId;

    private final AssociationStatus status;

    private final List<String> securityGroups;

    private TargetNetwork(BuilderImpl builder) {
        this.associationId = builder.associationId;
        this.vpcId = builder.vpcId;
        this.targetNetworkId = builder.targetNetworkId;
        this.clientVpnEndpointId = builder.clientVpnEndpointId;
        this.status = builder.status;
        this.securityGroups = builder.securityGroups;
    }

    /**
     * <p>
     * The ID of the association.
     * </p>
     * 
     * @return The ID of the association.
     */
    public final String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The ID of the VPC in which the target network (subnet) is located.
     * </p>
     * 
     * @return The ID of the VPC in which the target network (subnet) is located.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The ID of the subnet specified as the target network.
     * </p>
     * 
     * @return The ID of the subnet specified as the target network.
     */
    public final String targetNetworkId() {
        return targetNetworkId;
    }

    /**
     * <p>
     * The ID of the Client VPN endpoint with which the target network is associated.
     * </p>
     * 
     * @return The ID of the Client VPN endpoint with which the target network is associated.
     */
    public final String clientVpnEndpointId() {
        return clientVpnEndpointId;
    }

    /**
     * <p>
     * The current state of the target network association.
     * </p>
     * 
     * @return The current state of the target network association.
     */
    public final AssociationStatus status() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroups() {
        return securityGroups != null && !(securityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the security groups applied to the target network association.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroups} method.
     * </p>
     * 
     * @return The IDs of the security groups applied to the target network association.
     */
    public final List<String> securityGroups() {
        return securityGroups;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(targetNetworkId());
        hashCode = 31 * hashCode + Objects.hashCode(clientVpnEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroups() ? securityGroups() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TargetNetwork)) {
            return false;
        }
        TargetNetwork other = (TargetNetwork) obj;
        return Objects.equals(associationId(), other.associationId()) && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(targetNetworkId(), other.targetNetworkId())
                && Objects.equals(clientVpnEndpointId(), other.clientVpnEndpointId()) && Objects.equals(status(), other.status())
                && hasSecurityGroups() == other.hasSecurityGroups() && Objects.equals(securityGroups(), other.securityGroups());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TargetNetwork").add("AssociationId", associationId()).add("VpcId", vpcId())
                .add("TargetNetworkId", targetNetworkId()).add("ClientVpnEndpointId", clientVpnEndpointId())
                .add("Status", status()).add("SecurityGroups", hasSecurityGroups() ? securityGroups() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssociationId":
            return Optional.ofNullable(clazz.cast(associationId()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "TargetNetworkId":
            return Optional.ofNullable(clazz.cast(targetNetworkId()));
        case "ClientVpnEndpointId":
            return Optional.ofNullable(clazz.cast(clientVpnEndpointId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "SecurityGroups":
            return Optional.ofNullable(clazz.cast(securityGroups()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TargetNetwork, T> g) {
        return obj -> g.apply((TargetNetwork) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TargetNetwork> {
        /**
         * <p>
         * The ID of the association.
         * </p>
         * 
         * @param associationId
         *        The ID of the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The ID of the VPC in which the target network (subnet) is located.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC in which the target network (subnet) is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The ID of the subnet specified as the target network.
         * </p>
         * 
         * @param targetNetworkId
         *        The ID of the subnet specified as the target network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetNetworkId(String targetNetworkId);

        /**
         * <p>
         * The ID of the Client VPN endpoint with which the target network is associated.
         * </p>
         * 
         * @param clientVpnEndpointId
         *        The ID of the Client VPN endpoint with which the target network is associated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientVpnEndpointId(String clientVpnEndpointId);

        /**
         * <p>
         * The current state of the target network association.
         * </p>
         * 
         * @param status
         *        The current state of the target network association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(AssociationStatus status);

        /**
         * <p>
         * The current state of the target network association.
         * </p>
         * This is a convenience method that creates an instance of the {@link AssociationStatus.Builder} avoiding the
         * need to create one manually via {@link AssociationStatus#builder()}.
         *
         * When the {@link Consumer} completes, {@link AssociationStatus.Builder#build()} is called immediately and its
         * result is passed to {@link #status(AssociationStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link AssociationStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(AssociationStatus)
         */
        default Builder status(Consumer<AssociationStatus.Builder> status) {
            return status(AssociationStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * The IDs of the security groups applied to the target network association.
         * </p>
         * 
         * @param securityGroups
         *        The IDs of the security groups applied to the target network association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(Collection<String> securityGroups);

        /**
         * <p>
         * The IDs of the security groups applied to the target network association.
         * </p>
         * 
         * @param securityGroups
         *        The IDs of the security groups applied to the target network association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(String... securityGroups);
    }

    static final class BuilderImpl implements Builder {
        private String associationId;

        private String vpcId;

        private String targetNetworkId;

        private String clientVpnEndpointId;

        private AssociationStatus status;

        private List<String> securityGroups = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TargetNetwork model) {
            associationId(model.associationId);
            vpcId(model.vpcId);
            targetNetworkId(model.targetNetworkId);
            clientVpnEndpointId(model.clientVpnEndpointId);
            status(model.status);
            securityGroups(model.securityGroups);
        }

        public final String getAssociationId() {
            return associationId;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getTargetNetworkId() {
            return targetNetworkId;
        }

        public final void setTargetNetworkId(String targetNetworkId) {
            this.targetNetworkId = targetNetworkId;
        }

        @Override
        public final Builder targetNetworkId(String targetNetworkId) {
            this.targetNetworkId = targetNetworkId;
            return this;
        }

        public final String getClientVpnEndpointId() {
            return clientVpnEndpointId;
        }

        public final void setClientVpnEndpointId(String clientVpnEndpointId) {
            this.clientVpnEndpointId = clientVpnEndpointId;
        }

        @Override
        public final Builder clientVpnEndpointId(String clientVpnEndpointId) {
            this.clientVpnEndpointId = clientVpnEndpointId;
            return this;
        }

        public final AssociationStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(AssociationStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(AssociationStatus status) {
            this.status = status;
            return this;
        }

        public final Collection<String> getSecurityGroups() {
            if (securityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroups;
        }

        public final void setSecurityGroups(Collection<String> securityGroups) {
            this.securityGroups = ValueStringListCopier.copy(securityGroups);
        }

        @Override
        public final Builder securityGroups(Collection<String> securityGroups) {
            this.securityGroups = ValueStringListCopier.copy(securityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(String... securityGroups) {
            securityGroups(Arrays.asList(securityGroups));
            return this;
        }

        @Override
        public TargetNetwork build() {
            return new TargetNetwork(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
