/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a key pair.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KeyPairInfo implements SdkPojo, Serializable, ToCopyableBuilder<KeyPairInfo.Builder, KeyPairInfo> {
    private static final SdkField<String> KEY_PAIR_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KeyPairId")
            .getter(getter(KeyPairInfo::keyPairId))
            .setter(setter(Builder::keyPairId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyPairId")
                    .unmarshallLocationName("keyPairId").build()).build();

    private static final SdkField<String> KEY_FINGERPRINT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KeyFingerprint")
            .getter(getter(KeyPairInfo::keyFingerprint))
            .setter(setter(Builder::keyFingerprint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyFingerprint")
                    .unmarshallLocationName("keyFingerprint").build()).build();

    private static final SdkField<String> KEY_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KeyName")
            .getter(getter(KeyPairInfo::keyName))
            .setter(setter(Builder::keyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyName")
                    .unmarshallLocationName("keyName").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(KeyPairInfo::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_PAIR_ID_FIELD,
            KEY_FINGERPRINT_FIELD, KEY_NAME_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String keyPairId;

    private final String keyFingerprint;

    private final String keyName;

    private final List<Tag> tags;

    private KeyPairInfo(BuilderImpl builder) {
        this.keyPairId = builder.keyPairId;
        this.keyFingerprint = builder.keyFingerprint;
        this.keyName = builder.keyName;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the key pair.
     * </p>
     * 
     * @return The ID of the key pair.
     */
    public final String keyPairId() {
        return keyPairId;
    }

    /**
     * <p>
     * If you used <a>CreateKeyPair</a> to create the key pair, this is the SHA-1 digest of the DER encoded private key.
     * If you used <a>ImportKeyPair</a> to provide Amazon Web Services the public key, this is the MD5 public key
     * fingerprint as specified in section 4 of RFC4716.
     * </p>
     * 
     * @return If you used <a>CreateKeyPair</a> to create the key pair, this is the SHA-1 digest of the DER encoded
     *         private key. If you used <a>ImportKeyPair</a> to provide Amazon Web Services the public key, this is the
     *         MD5 public key fingerprint as specified in section 4 of RFC4716.
     */
    public final String keyFingerprint() {
        return keyFingerprint;
    }

    /**
     * <p>
     * The name of the key pair.
     * </p>
     * 
     * @return The name of the key pair.
     */
    public final String keyName() {
        return keyName;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any tags applied to the key pair.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Any tags applied to the key pair.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(keyPairId());
        hashCode = 31 * hashCode + Objects.hashCode(keyFingerprint());
        hashCode = 31 * hashCode + Objects.hashCode(keyName());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KeyPairInfo)) {
            return false;
        }
        KeyPairInfo other = (KeyPairInfo) obj;
        return Objects.equals(keyPairId(), other.keyPairId()) && Objects.equals(keyFingerprint(), other.keyFingerprint())
                && Objects.equals(keyName(), other.keyName()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KeyPairInfo").add("KeyPairId", keyPairId()).add("KeyFingerprint", keyFingerprint())
                .add("KeyName", keyName()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KeyPairId":
            return Optional.ofNullable(clazz.cast(keyPairId()));
        case "KeyFingerprint":
            return Optional.ofNullable(clazz.cast(keyFingerprint()));
        case "KeyName":
            return Optional.ofNullable(clazz.cast(keyName()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<KeyPairInfo, T> g) {
        return obj -> g.apply((KeyPairInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KeyPairInfo> {
        /**
         * <p>
         * The ID of the key pair.
         * </p>
         * 
         * @param keyPairId
         *        The ID of the key pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyPairId(String keyPairId);

        /**
         * <p>
         * If you used <a>CreateKeyPair</a> to create the key pair, this is the SHA-1 digest of the DER encoded private
         * key. If you used <a>ImportKeyPair</a> to provide Amazon Web Services the public key, this is the MD5 public
         * key fingerprint as specified in section 4 of RFC4716.
         * </p>
         * 
         * @param keyFingerprint
         *        If you used <a>CreateKeyPair</a> to create the key pair, this is the SHA-1 digest of the DER encoded
         *        private key. If you used <a>ImportKeyPair</a> to provide Amazon Web Services the public key, this is
         *        the MD5 public key fingerprint as specified in section 4 of RFC4716.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyFingerprint(String keyFingerprint);

        /**
         * <p>
         * The name of the key pair.
         * </p>
         * 
         * @param keyName
         *        The name of the key pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyName(String keyName);

        /**
         * <p>
         * Any tags applied to the key pair.
         * </p>
         * 
         * @param tags
         *        Any tags applied to the key pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags applied to the key pair.
         * </p>
         * 
         * @param tags
         *        Any tags applied to the key pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags applied to the key pair.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String keyPairId;

        private String keyFingerprint;

        private String keyName;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(KeyPairInfo model) {
            keyPairId(model.keyPairId);
            keyFingerprint(model.keyFingerprint);
            keyName(model.keyName);
            tags(model.tags);
        }

        public final String getKeyPairId() {
            return keyPairId;
        }

        public final void setKeyPairId(String keyPairId) {
            this.keyPairId = keyPairId;
        }

        @Override
        @Transient
        public final Builder keyPairId(String keyPairId) {
            this.keyPairId = keyPairId;
            return this;
        }

        public final String getKeyFingerprint() {
            return keyFingerprint;
        }

        public final void setKeyFingerprint(String keyFingerprint) {
            this.keyFingerprint = keyFingerprint;
        }

        @Override
        @Transient
        public final Builder keyFingerprint(String keyFingerprint) {
            this.keyFingerprint = keyFingerprint;
            return this;
        }

        public final String getKeyName() {
            return keyName;
        }

        public final void setKeyName(String keyName) {
            this.keyName = keyName;
        }

        @Override
        @Transient
        public final Builder keyName(String keyName) {
            this.keyName = keyName;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public KeyPairInfo build() {
            return new KeyPairInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
