/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a VPC endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VpcEndpoint implements SdkPojo, Serializable, ToCopyableBuilder<VpcEndpoint.Builder, VpcEndpoint> {
    private static final SdkField<String> VPC_ENDPOINT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcEndpointId")
            .getter(getter(VpcEndpoint::vpcEndpointId))
            .setter(setter(Builder::vpcEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcEndpointId")
                    .unmarshallLocationName("vpcEndpointId").build()).build();

    private static final SdkField<String> VPC_ENDPOINT_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcEndpointType")
            .getter(getter(VpcEndpoint::vpcEndpointTypeAsString))
            .setter(setter(Builder::vpcEndpointType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcEndpointType")
                    .unmarshallLocationName("vpcEndpointType").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcId")
            .getter(getter(VpcEndpoint::vpcId))
            .setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId")
                    .unmarshallLocationName("vpcId").build()).build();

    private static final SdkField<String> SERVICE_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ServiceName")
            .getter(getter(VpcEndpoint::serviceName))
            .setter(setter(Builder::serviceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceName")
                    .unmarshallLocationName("serviceName").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(VpcEndpoint::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<String> POLICY_DOCUMENT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PolicyDocument")
            .getter(getter(VpcEndpoint::policyDocument))
            .setter(setter(Builder::policyDocument))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyDocument")
                    .unmarshallLocationName("policyDocument").build()).build();

    private static final SdkField<List<String>> ROUTE_TABLE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RouteTableIds")
            .getter(getter(VpcEndpoint::routeTableIds))
            .setter(setter(Builder::routeTableIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteTableIdSet")
                    .unmarshallLocationName("routeTableIdSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SubnetIds")
            .getter(getter(VpcEndpoint::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetIdSet")
                    .unmarshallLocationName("subnetIdSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<SecurityGroupIdentifier>> GROUPS_FIELD = SdkField
            .<List<SecurityGroupIdentifier>> builder(MarshallingType.LIST)
            .memberName("Groups")
            .getter(getter(VpcEndpoint::groups))
            .setter(setter(Builder::groups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupSet")
                    .unmarshallLocationName("groupSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<SecurityGroupIdentifier> builder(MarshallingType.SDK_POJO)
                                            .constructor(SecurityGroupIdentifier::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> PRIVATE_DNS_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PrivateDnsEnabled")
            .getter(getter(VpcEndpoint::privateDnsEnabled))
            .setter(setter(Builder::privateDnsEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateDnsEnabled")
                    .unmarshallLocationName("privateDnsEnabled").build()).build();

    private static final SdkField<Boolean> REQUESTER_MANAGED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RequesterManaged")
            .getter(getter(VpcEndpoint::requesterManaged))
            .setter(setter(Builder::requesterManaged))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequesterManaged")
                    .unmarshallLocationName("requesterManaged").build()).build();

    private static final SdkField<List<String>> NETWORK_INTERFACE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NetworkInterfaceIds")
            .getter(getter(VpcEndpoint::networkInterfaceIds))
            .setter(setter(Builder::networkInterfaceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceIdSet")
                    .unmarshallLocationName("networkInterfaceIdSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<DnsEntry>> DNS_ENTRIES_FIELD = SdkField
            .<List<DnsEntry>> builder(MarshallingType.LIST)
            .memberName("DnsEntries")
            .getter(getter(VpcEndpoint::dnsEntries))
            .setter(setter(Builder::dnsEntries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsEntrySet")
                    .unmarshallLocationName("dnsEntrySet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<DnsEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(DnsEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Instant> CREATION_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTimestamp")
            .getter(getter(VpcEndpoint::creationTimestamp))
            .setter(setter(Builder::creationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTimestamp")
                    .unmarshallLocationName("creationTimestamp").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(VpcEndpoint::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(VpcEndpoint::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<LastError> LAST_ERROR_FIELD = SdkField
            .<LastError> builder(MarshallingType.SDK_POJO)
            .memberName("LastError")
            .getter(getter(VpcEndpoint::lastError))
            .setter(setter(Builder::lastError))
            .constructor(LastError::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastError")
                    .unmarshallLocationName("lastError").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VPC_ENDPOINT_ID_FIELD,
            VPC_ENDPOINT_TYPE_FIELD, VPC_ID_FIELD, SERVICE_NAME_FIELD, STATE_FIELD, POLICY_DOCUMENT_FIELD, ROUTE_TABLE_IDS_FIELD,
            SUBNET_IDS_FIELD, GROUPS_FIELD, PRIVATE_DNS_ENABLED_FIELD, REQUESTER_MANAGED_FIELD, NETWORK_INTERFACE_IDS_FIELD,
            DNS_ENTRIES_FIELD, CREATION_TIMESTAMP_FIELD, TAGS_FIELD, OWNER_ID_FIELD, LAST_ERROR_FIELD));

    private static final long serialVersionUID = 1L;

    private final String vpcEndpointId;

    private final String vpcEndpointType;

    private final String vpcId;

    private final String serviceName;

    private final String state;

    private final String policyDocument;

    private final List<String> routeTableIds;

    private final List<String> subnetIds;

    private final List<SecurityGroupIdentifier> groups;

    private final Boolean privateDnsEnabled;

    private final Boolean requesterManaged;

    private final List<String> networkInterfaceIds;

    private final List<DnsEntry> dnsEntries;

    private final Instant creationTimestamp;

    private final List<Tag> tags;

    private final String ownerId;

    private final LastError lastError;

    private VpcEndpoint(BuilderImpl builder) {
        this.vpcEndpointId = builder.vpcEndpointId;
        this.vpcEndpointType = builder.vpcEndpointType;
        this.vpcId = builder.vpcId;
        this.serviceName = builder.serviceName;
        this.state = builder.state;
        this.policyDocument = builder.policyDocument;
        this.routeTableIds = builder.routeTableIds;
        this.subnetIds = builder.subnetIds;
        this.groups = builder.groups;
        this.privateDnsEnabled = builder.privateDnsEnabled;
        this.requesterManaged = builder.requesterManaged;
        this.networkInterfaceIds = builder.networkInterfaceIds;
        this.dnsEntries = builder.dnsEntries;
        this.creationTimestamp = builder.creationTimestamp;
        this.tags = builder.tags;
        this.ownerId = builder.ownerId;
        this.lastError = builder.lastError;
    }

    /**
     * <p>
     * The ID of the VPC endpoint.
     * </p>
     * 
     * @return The ID of the VPC endpoint.
     */
    public String vpcEndpointId() {
        return vpcEndpointId;
    }

    /**
     * <p>
     * The type of endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vpcEndpointType}
     * will return {@link VpcEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #vpcEndpointTypeAsString}.
     * </p>
     * 
     * @return The type of endpoint.
     * @see VpcEndpointType
     */
    public VpcEndpointType vpcEndpointType() {
        return VpcEndpointType.fromValue(vpcEndpointType);
    }

    /**
     * <p>
     * The type of endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vpcEndpointType}
     * will return {@link VpcEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #vpcEndpointTypeAsString}.
     * </p>
     * 
     * @return The type of endpoint.
     * @see VpcEndpointType
     */
    public String vpcEndpointTypeAsString() {
        return vpcEndpointType;
    }

    /**
     * <p>
     * The ID of the VPC to which the endpoint is associated.
     * </p>
     * 
     * @return The ID of the VPC to which the endpoint is associated.
     */
    public String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The name of the service to which the endpoint is associated.
     * </p>
     * 
     * @return The name of the service to which the endpoint is associated.
     */
    public String serviceName() {
        return serviceName;
    }

    /**
     * <p>
     * The state of the VPC endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link State#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the VPC endpoint.
     * @see State
     */
    public State state() {
        return State.fromValue(state);
    }

    /**
     * <p>
     * The state of the VPC endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link State#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the VPC endpoint.
     * @see State
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The policy document associated with the endpoint, if applicable.
     * </p>
     * 
     * @return The policy document associated with the endpoint, if applicable.
     */
    public String policyDocument() {
        return policyDocument;
    }

    /**
     * Returns true if the RouteTableIds property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRouteTableIds() {
        return routeTableIds != null && !(routeTableIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Gateway endpoint) One or more route tables associated with the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRouteTableIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return (Gateway endpoint) One or more route tables associated with the endpoint.
     */
    public List<String> routeTableIds() {
        return routeTableIds;
    }

    /**
     * Returns true if the SubnetIds property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Interface endpoint) One or more subnets in which the endpoint is located.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubnetIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return (Interface endpoint) One or more subnets in which the endpoint is located.
     */
    public List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * Returns true if the Groups property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasGroups() {
        return groups != null && !(groups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Interface endpoint) Information about the security groups that are associated with the network interface.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return (Interface endpoint) Information about the security groups that are associated with the network
     *         interface.
     */
    public List<SecurityGroupIdentifier> groups() {
        return groups;
    }

    /**
     * <p>
     * (Interface endpoint) Indicates whether the VPC is associated with a private hosted zone.
     * </p>
     * 
     * @return (Interface endpoint) Indicates whether the VPC is associated with a private hosted zone.
     */
    public Boolean privateDnsEnabled() {
        return privateDnsEnabled;
    }

    /**
     * <p>
     * Indicates whether the VPC endpoint is being managed by its service.
     * </p>
     * 
     * @return Indicates whether the VPC endpoint is being managed by its service.
     */
    public Boolean requesterManaged() {
        return requesterManaged;
    }

    /**
     * Returns true if the NetworkInterfaceIds property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasNetworkInterfaceIds() {
        return networkInterfaceIds != null && !(networkInterfaceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Interface endpoint) One or more network interfaces for the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNetworkInterfaceIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return (Interface endpoint) One or more network interfaces for the endpoint.
     */
    public List<String> networkInterfaceIds() {
        return networkInterfaceIds;
    }

    /**
     * Returns true if the DnsEntries property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDnsEntries() {
        return dnsEntries != null && !(dnsEntries instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Interface endpoint) The DNS entries for the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDnsEntries()} to see if a value was sent in this field.
     * </p>
     * 
     * @return (Interface endpoint) The DNS entries for the endpoint.
     */
    public List<DnsEntry> dnsEntries() {
        return dnsEntries;
    }

    /**
     * <p>
     * The date and time that the VPC endpoint was created.
     * </p>
     * 
     * @return The date and time that the VPC endpoint was created.
     */
    public Instant creationTimestamp() {
        return creationTimestamp;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any tags assigned to the VPC endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Any tags assigned to the VPC endpoint.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The ID of the AWS account that owns the VPC endpoint.
     * </p>
     * 
     * @return The ID of the AWS account that owns the VPC endpoint.
     */
    public String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The last error that occurred for VPC endpoint.
     * </p>
     * 
     * @return The last error that occurred for VPC endpoint.
     */
    public LastError lastError() {
        return lastError;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vpcEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(vpcEndpointTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(serviceName());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(policyDocument());
        hashCode = 31 * hashCode + Objects.hashCode(hasRouteTableIds() ? routeTableIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetIds() ? subnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasGroups() ? groups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(privateDnsEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(requesterManaged());
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaceIds() ? networkInterfaceIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsEntries() ? dnsEntries() : null);
        hashCode = 31 * hashCode + Objects.hashCode(creationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(lastError());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VpcEndpoint)) {
            return false;
        }
        VpcEndpoint other = (VpcEndpoint) obj;
        return Objects.equals(vpcEndpointId(), other.vpcEndpointId())
                && Objects.equals(vpcEndpointTypeAsString(), other.vpcEndpointTypeAsString())
                && Objects.equals(vpcId(), other.vpcId()) && Objects.equals(serviceName(), other.serviceName())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(policyDocument(), other.policyDocument()) && hasRouteTableIds() == other.hasRouteTableIds()
                && Objects.equals(routeTableIds(), other.routeTableIds()) && hasSubnetIds() == other.hasSubnetIds()
                && Objects.equals(subnetIds(), other.subnetIds()) && hasGroups() == other.hasGroups()
                && Objects.equals(groups(), other.groups()) && Objects.equals(privateDnsEnabled(), other.privateDnsEnabled())
                && Objects.equals(requesterManaged(), other.requesterManaged())
                && hasNetworkInterfaceIds() == other.hasNetworkInterfaceIds()
                && Objects.equals(networkInterfaceIds(), other.networkInterfaceIds()) && hasDnsEntries() == other.hasDnsEntries()
                && Objects.equals(dnsEntries(), other.dnsEntries())
                && Objects.equals(creationTimestamp(), other.creationTimestamp()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(ownerId(), other.ownerId())
                && Objects.equals(lastError(), other.lastError());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("VpcEndpoint").add("VpcEndpointId", vpcEndpointId())
                .add("VpcEndpointType", vpcEndpointTypeAsString()).add("VpcId", vpcId()).add("ServiceName", serviceName())
                .add("State", stateAsString()).add("PolicyDocument", policyDocument())
                .add("RouteTableIds", hasRouteTableIds() ? routeTableIds() : null)
                .add("SubnetIds", hasSubnetIds() ? subnetIds() : null).add("Groups", hasGroups() ? groups() : null)
                .add("PrivateDnsEnabled", privateDnsEnabled()).add("RequesterManaged", requesterManaged())
                .add("NetworkInterfaceIds", hasNetworkInterfaceIds() ? networkInterfaceIds() : null)
                .add("DnsEntries", hasDnsEntries() ? dnsEntries() : null).add("CreationTimestamp", creationTimestamp())
                .add("Tags", hasTags() ? tags() : null).add("OwnerId", ownerId()).add("LastError", lastError()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VpcEndpointId":
            return Optional.ofNullable(clazz.cast(vpcEndpointId()));
        case "VpcEndpointType":
            return Optional.ofNullable(clazz.cast(vpcEndpointTypeAsString()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "ServiceName":
            return Optional.ofNullable(clazz.cast(serviceName()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "PolicyDocument":
            return Optional.ofNullable(clazz.cast(policyDocument()));
        case "RouteTableIds":
            return Optional.ofNullable(clazz.cast(routeTableIds()));
        case "SubnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "Groups":
            return Optional.ofNullable(clazz.cast(groups()));
        case "PrivateDnsEnabled":
            return Optional.ofNullable(clazz.cast(privateDnsEnabled()));
        case "RequesterManaged":
            return Optional.ofNullable(clazz.cast(requesterManaged()));
        case "NetworkInterfaceIds":
            return Optional.ofNullable(clazz.cast(networkInterfaceIds()));
        case "DnsEntries":
            return Optional.ofNullable(clazz.cast(dnsEntries()));
        case "CreationTimestamp":
            return Optional.ofNullable(clazz.cast(creationTimestamp()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "LastError":
            return Optional.ofNullable(clazz.cast(lastError()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<VpcEndpoint, T> g) {
        return obj -> g.apply((VpcEndpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VpcEndpoint> {
        /**
         * <p>
         * The ID of the VPC endpoint.
         * </p>
         * 
         * @param vpcEndpointId
         *        The ID of the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcEndpointId(String vpcEndpointId);

        /**
         * <p>
         * The type of endpoint.
         * </p>
         * 
         * @param vpcEndpointType
         *        The type of endpoint.
         * @see VpcEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEndpointType
         */
        Builder vpcEndpointType(String vpcEndpointType);

        /**
         * <p>
         * The type of endpoint.
         * </p>
         * 
         * @param vpcEndpointType
         *        The type of endpoint.
         * @see VpcEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcEndpointType
         */
        Builder vpcEndpointType(VpcEndpointType vpcEndpointType);

        /**
         * <p>
         * The ID of the VPC to which the endpoint is associated.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC to which the endpoint is associated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The name of the service to which the endpoint is associated.
         * </p>
         * 
         * @param serviceName
         *        The name of the service to which the endpoint is associated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceName(String serviceName);

        /**
         * <p>
         * The state of the VPC endpoint.
         * </p>
         * 
         * @param state
         *        The state of the VPC endpoint.
         * @see State
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see State
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the VPC endpoint.
         * </p>
         * 
         * @param state
         *        The state of the VPC endpoint.
         * @see State
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see State
         */
        Builder state(State state);

        /**
         * <p>
         * The policy document associated with the endpoint, if applicable.
         * </p>
         * 
         * @param policyDocument
         *        The policy document associated with the endpoint, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyDocument(String policyDocument);

        /**
         * <p>
         * (Gateway endpoint) One or more route tables associated with the endpoint.
         * </p>
         * 
         * @param routeTableIds
         *        (Gateway endpoint) One or more route tables associated with the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeTableIds(Collection<String> routeTableIds);

        /**
         * <p>
         * (Gateway endpoint) One or more route tables associated with the endpoint.
         * </p>
         * 
         * @param routeTableIds
         *        (Gateway endpoint) One or more route tables associated with the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeTableIds(String... routeTableIds);

        /**
         * <p>
         * (Interface endpoint) One or more subnets in which the endpoint is located.
         * </p>
         * 
         * @param subnetIds
         *        (Interface endpoint) One or more subnets in which the endpoint is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * (Interface endpoint) One or more subnets in which the endpoint is located.
         * </p>
         * 
         * @param subnetIds
         *        (Interface endpoint) One or more subnets in which the endpoint is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * (Interface endpoint) Information about the security groups that are associated with the network interface.
         * </p>
         * 
         * @param groups
         *        (Interface endpoint) Information about the security groups that are associated with the network
         *        interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(Collection<SecurityGroupIdentifier> groups);

        /**
         * <p>
         * (Interface endpoint) Information about the security groups that are associated with the network interface.
         * </p>
         * 
         * @param groups
         *        (Interface endpoint) Information about the security groups that are associated with the network
         *        interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(SecurityGroupIdentifier... groups);

        /**
         * <p>
         * (Interface endpoint) Information about the security groups that are associated with the network interface.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SecurityGroupIdentifier>.Builder} avoiding
         * the need to create one manually via {@link List<SecurityGroupIdentifier>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SecurityGroupIdentifier>.Builder#build()} is called
         * immediately and its result is passed to {@link #groups(List<SecurityGroupIdentifier>)}.
         * 
         * @param groups
         *        a consumer that will call methods on {@link List<SecurityGroupIdentifier>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #groups(List<SecurityGroupIdentifier>)
         */
        Builder groups(Consumer<SecurityGroupIdentifier.Builder>... groups);

        /**
         * <p>
         * (Interface endpoint) Indicates whether the VPC is associated with a private hosted zone.
         * </p>
         * 
         * @param privateDnsEnabled
         *        (Interface endpoint) Indicates whether the VPC is associated with a private hosted zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateDnsEnabled(Boolean privateDnsEnabled);

        /**
         * <p>
         * Indicates whether the VPC endpoint is being managed by its service.
         * </p>
         * 
         * @param requesterManaged
         *        Indicates whether the VPC endpoint is being managed by its service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requesterManaged(Boolean requesterManaged);

        /**
         * <p>
         * (Interface endpoint) One or more network interfaces for the endpoint.
         * </p>
         * 
         * @param networkInterfaceIds
         *        (Interface endpoint) One or more network interfaces for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceIds(Collection<String> networkInterfaceIds);

        /**
         * <p>
         * (Interface endpoint) One or more network interfaces for the endpoint.
         * </p>
         * 
         * @param networkInterfaceIds
         *        (Interface endpoint) One or more network interfaces for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceIds(String... networkInterfaceIds);

        /**
         * <p>
         * (Interface endpoint) The DNS entries for the endpoint.
         * </p>
         * 
         * @param dnsEntries
         *        (Interface endpoint) The DNS entries for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsEntries(Collection<DnsEntry> dnsEntries);

        /**
         * <p>
         * (Interface endpoint) The DNS entries for the endpoint.
         * </p>
         * 
         * @param dnsEntries
         *        (Interface endpoint) The DNS entries for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsEntries(DnsEntry... dnsEntries);

        /**
         * <p>
         * (Interface endpoint) The DNS entries for the endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DnsEntry>.Builder} avoiding the need to
         * create one manually via {@link List<DnsEntry>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DnsEntry>.Builder#build()} is called immediately and its
         * result is passed to {@link #dnsEntries(List<DnsEntry>)}.
         * 
         * @param dnsEntries
         *        a consumer that will call methods on {@link List<DnsEntry>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dnsEntries(List<DnsEntry>)
         */
        Builder dnsEntries(Consumer<DnsEntry.Builder>... dnsEntries);

        /**
         * <p>
         * The date and time that the VPC endpoint was created.
         * </p>
         * 
         * @param creationTimestamp
         *        The date and time that the VPC endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTimestamp(Instant creationTimestamp);

        /**
         * <p>
         * Any tags assigned to the VPC endpoint.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the VPC endpoint.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the VPC endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The ID of the AWS account that owns the VPC endpoint.
         * </p>
         * 
         * @param ownerId
         *        The ID of the AWS account that owns the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The last error that occurred for VPC endpoint.
         * </p>
         * 
         * @param lastError
         *        The last error that occurred for VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastError(LastError lastError);

        /**
         * <p>
         * The last error that occurred for VPC endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link LastError.Builder} avoiding the need to create
         * one manually via {@link LastError#builder()}.
         *
         * When the {@link Consumer} completes, {@link LastError.Builder#build()} is called immediately and its result
         * is passed to {@link #lastError(LastError)}.
         * 
         * @param lastError
         *        a consumer that will call methods on {@link LastError.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lastError(LastError)
         */
        default Builder lastError(Consumer<LastError.Builder> lastError) {
            return lastError(LastError.builder().applyMutation(lastError).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String vpcEndpointId;

        private String vpcEndpointType;

        private String vpcId;

        private String serviceName;

        private String state;

        private String policyDocument;

        private List<String> routeTableIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private List<SecurityGroupIdentifier> groups = DefaultSdkAutoConstructList.getInstance();

        private Boolean privateDnsEnabled;

        private Boolean requesterManaged;

        private List<String> networkInterfaceIds = DefaultSdkAutoConstructList.getInstance();

        private List<DnsEntry> dnsEntries = DefaultSdkAutoConstructList.getInstance();

        private Instant creationTimestamp;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String ownerId;

        private LastError lastError;

        private BuilderImpl() {
        }

        private BuilderImpl(VpcEndpoint model) {
            vpcEndpointId(model.vpcEndpointId);
            vpcEndpointType(model.vpcEndpointType);
            vpcId(model.vpcId);
            serviceName(model.serviceName);
            state(model.state);
            policyDocument(model.policyDocument);
            routeTableIds(model.routeTableIds);
            subnetIds(model.subnetIds);
            groups(model.groups);
            privateDnsEnabled(model.privateDnsEnabled);
            requesterManaged(model.requesterManaged);
            networkInterfaceIds(model.networkInterfaceIds);
            dnsEntries(model.dnsEntries);
            creationTimestamp(model.creationTimestamp);
            tags(model.tags);
            ownerId(model.ownerId);
            lastError(model.lastError);
        }

        public final String getVpcEndpointId() {
            return vpcEndpointId;
        }

        @Override
        public final Builder vpcEndpointId(String vpcEndpointId) {
            this.vpcEndpointId = vpcEndpointId;
            return this;
        }

        public final void setVpcEndpointId(String vpcEndpointId) {
            this.vpcEndpointId = vpcEndpointId;
        }

        public final String getVpcEndpointType() {
            return vpcEndpointType;
        }

        @Override
        public final Builder vpcEndpointType(String vpcEndpointType) {
            this.vpcEndpointType = vpcEndpointType;
            return this;
        }

        @Override
        public final Builder vpcEndpointType(VpcEndpointType vpcEndpointType) {
            this.vpcEndpointType(vpcEndpointType == null ? null : vpcEndpointType.toString());
            return this;
        }

        public final void setVpcEndpointType(String vpcEndpointType) {
            this.vpcEndpointType = vpcEndpointType;
        }

        public final String getVpcId() {
            return vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        public final String getServiceName() {
            return serviceName;
        }

        @Override
        public final Builder serviceName(String serviceName) {
            this.serviceName = serviceName;
            return this;
        }

        public final void setServiceName(String serviceName) {
            this.serviceName = serviceName;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(State state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getPolicyDocument() {
            return policyDocument;
        }

        @Override
        public final Builder policyDocument(String policyDocument) {
            this.policyDocument = policyDocument;
            return this;
        }

        public final void setPolicyDocument(String policyDocument) {
            this.policyDocument = policyDocument;
        }

        public final Collection<String> getRouteTableIds() {
            if (routeTableIds instanceof SdkAutoConstructList) {
                return null;
            }
            return routeTableIds;
        }

        @Override
        public final Builder routeTableIds(Collection<String> routeTableIds) {
            this.routeTableIds = ValueStringListCopier.copy(routeTableIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder routeTableIds(String... routeTableIds) {
            routeTableIds(Arrays.asList(routeTableIds));
            return this;
        }

        public final void setRouteTableIds(Collection<String> routeTableIds) {
            this.routeTableIds = ValueStringListCopier.copy(routeTableIds);
        }

        public final Collection<String> getSubnetIds() {
            if (subnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetIds;
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = ValueStringListCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = ValueStringListCopier.copy(subnetIds);
        }

        public final Collection<SecurityGroupIdentifier.Builder> getGroups() {
            if (groups instanceof SdkAutoConstructList) {
                return null;
            }
            return groups != null ? groups.stream().map(SecurityGroupIdentifier::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder groups(Collection<SecurityGroupIdentifier> groups) {
            this.groups = GroupIdentifierSetCopier.copy(groups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(SecurityGroupIdentifier... groups) {
            groups(Arrays.asList(groups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(Consumer<SecurityGroupIdentifier.Builder>... groups) {
            groups(Stream.of(groups).map(c -> SecurityGroupIdentifier.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setGroups(Collection<SecurityGroupIdentifier.BuilderImpl> groups) {
            this.groups = GroupIdentifierSetCopier.copyFromBuilder(groups);
        }

        public final Boolean getPrivateDnsEnabled() {
            return privateDnsEnabled;
        }

        @Override
        public final Builder privateDnsEnabled(Boolean privateDnsEnabled) {
            this.privateDnsEnabled = privateDnsEnabled;
            return this;
        }

        public final void setPrivateDnsEnabled(Boolean privateDnsEnabled) {
            this.privateDnsEnabled = privateDnsEnabled;
        }

        public final Boolean getRequesterManaged() {
            return requesterManaged;
        }

        @Override
        public final Builder requesterManaged(Boolean requesterManaged) {
            this.requesterManaged = requesterManaged;
            return this;
        }

        public final void setRequesterManaged(Boolean requesterManaged) {
            this.requesterManaged = requesterManaged;
        }

        public final Collection<String> getNetworkInterfaceIds() {
            if (networkInterfaceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return networkInterfaceIds;
        }

        @Override
        public final Builder networkInterfaceIds(Collection<String> networkInterfaceIds) {
            this.networkInterfaceIds = ValueStringListCopier.copy(networkInterfaceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaceIds(String... networkInterfaceIds) {
            networkInterfaceIds(Arrays.asList(networkInterfaceIds));
            return this;
        }

        public final void setNetworkInterfaceIds(Collection<String> networkInterfaceIds) {
            this.networkInterfaceIds = ValueStringListCopier.copy(networkInterfaceIds);
        }

        public final Collection<DnsEntry.Builder> getDnsEntries() {
            if (dnsEntries instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsEntries != null ? dnsEntries.stream().map(DnsEntry::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dnsEntries(Collection<DnsEntry> dnsEntries) {
            this.dnsEntries = DnsEntrySetCopier.copy(dnsEntries);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsEntries(DnsEntry... dnsEntries) {
            dnsEntries(Arrays.asList(dnsEntries));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsEntries(Consumer<DnsEntry.Builder>... dnsEntries) {
            dnsEntries(Stream.of(dnsEntries).map(c -> DnsEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDnsEntries(Collection<DnsEntry.BuilderImpl> dnsEntries) {
            this.dnsEntries = DnsEntrySetCopier.copyFromBuilder(dnsEntries);
        }

        public final Instant getCreationTimestamp() {
            return creationTimestamp;
        }

        @Override
        public final Builder creationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
            return this;
        }

        public final void setCreationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
        }

        public final Collection<Tag.Builder> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        public final LastError.Builder getLastError() {
            return lastError != null ? lastError.toBuilder() : null;
        }

        @Override
        public final Builder lastError(LastError lastError) {
            this.lastError = lastError;
            return this;
        }

        public final void setLastError(LastError.BuilderImpl lastError) {
            this.lastError = lastError != null ? lastError.build() : null;
        }

        @Override
        public VpcEndpoint build() {
            return new VpcEndpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
