/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the strategy for using unused Capacity Reservations for fulfilling On-Demand capacity.
 * </p>
 * <note>
 * <p>
 * This strategy can only be used if the EC2 Fleet is of type <code>instant</code>.
 * </p>
 * </note>
 * <p>
 * For more information about Capacity Reservations, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html">On-Demand Capacity
 * Reservations</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>. For examples of using Capacity Reservations
 * in an EC2 Fleet, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-examples.html">EC2 Fleet
 * Example Configurations</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CapacityReservationOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<CapacityReservationOptions.Builder, CapacityReservationOptions> {
    private static final SdkField<String> USAGE_STRATEGY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CapacityReservationOptions::usageStrategyAsString))
            .setter(setter(Builder::usageStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsageStrategy")
                    .unmarshallLocationName("usageStrategy").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USAGE_STRATEGY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String usageStrategy;

    private CapacityReservationOptions(BuilderImpl builder) {
        this.usageStrategy = builder.usageStrategy;
    }

    /**
     * <p>
     * Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity.
     * </p>
     * <p>
     * If you specify <code>use-capacity-reservations-first</code>, the fleet uses unused Capacity Reservations to
     * fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance pools have unused Capacity
     * Reservations, the On-Demand allocation strategy (<code>lowest-price</code> or <code>prioritized</code>) is
     * applied. If the number of unused Capacity Reservations is less than the On-Demand target capacity, the remaining
     * On-Demand target capacity is launched according to the On-Demand allocation strategy (<code>lowest-price</code>
     * or <code>prioritized</code>).
     * </p>
     * <p>
     * If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen On-Demand
     * allocation strategy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #usageStrategy}
     * will return {@link FleetCapacityReservationUsageStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #usageStrategyAsString}.
     * </p>
     * 
     * @return Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity.</p>
     *         <p>
     *         If you specify <code>use-capacity-reservations-first</code>, the fleet uses unused Capacity Reservations
     *         to fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance pools have unused
     *         Capacity Reservations, the On-Demand allocation strategy (<code>lowest-price</code> or
     *         <code>prioritized</code>) is applied. If the number of unused Capacity Reservations is less than the
     *         On-Demand target capacity, the remaining On-Demand target capacity is launched according to the On-Demand
     *         allocation strategy (<code>lowest-price</code> or <code>prioritized</code>).
     *         </p>
     *         <p>
     *         If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen On-Demand
     *         allocation strategy.
     * @see FleetCapacityReservationUsageStrategy
     */
    public FleetCapacityReservationUsageStrategy usageStrategy() {
        return FleetCapacityReservationUsageStrategy.fromValue(usageStrategy);
    }

    /**
     * <p>
     * Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity.
     * </p>
     * <p>
     * If you specify <code>use-capacity-reservations-first</code>, the fleet uses unused Capacity Reservations to
     * fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance pools have unused Capacity
     * Reservations, the On-Demand allocation strategy (<code>lowest-price</code> or <code>prioritized</code>) is
     * applied. If the number of unused Capacity Reservations is less than the On-Demand target capacity, the remaining
     * On-Demand target capacity is launched according to the On-Demand allocation strategy (<code>lowest-price</code>
     * or <code>prioritized</code>).
     * </p>
     * <p>
     * If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen On-Demand
     * allocation strategy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #usageStrategy}
     * will return {@link FleetCapacityReservationUsageStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #usageStrategyAsString}.
     * </p>
     * 
     * @return Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity.</p>
     *         <p>
     *         If you specify <code>use-capacity-reservations-first</code>, the fleet uses unused Capacity Reservations
     *         to fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance pools have unused
     *         Capacity Reservations, the On-Demand allocation strategy (<code>lowest-price</code> or
     *         <code>prioritized</code>) is applied. If the number of unused Capacity Reservations is less than the
     *         On-Demand target capacity, the remaining On-Demand target capacity is launched according to the On-Demand
     *         allocation strategy (<code>lowest-price</code> or <code>prioritized</code>).
     *         </p>
     *         <p>
     *         If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen On-Demand
     *         allocation strategy.
     * @see FleetCapacityReservationUsageStrategy
     */
    public String usageStrategyAsString() {
        return usageStrategy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(usageStrategyAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CapacityReservationOptions)) {
            return false;
        }
        CapacityReservationOptions other = (CapacityReservationOptions) obj;
        return Objects.equals(usageStrategyAsString(), other.usageStrategyAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CapacityReservationOptions").add("UsageStrategy", usageStrategyAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UsageStrategy":
            return Optional.ofNullable(clazz.cast(usageStrategyAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CapacityReservationOptions, T> g) {
        return obj -> g.apply((CapacityReservationOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CapacityReservationOptions> {
        /**
         * <p>
         * Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity.
         * </p>
         * <p>
         * If you specify <code>use-capacity-reservations-first</code>, the fleet uses unused Capacity Reservations to
         * fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance pools have unused
         * Capacity Reservations, the On-Demand allocation strategy (<code>lowest-price</code> or
         * <code>prioritized</code>) is applied. If the number of unused Capacity Reservations is less than the
         * On-Demand target capacity, the remaining On-Demand target capacity is launched according to the On-Demand
         * allocation strategy (<code>lowest-price</code> or <code>prioritized</code>).
         * </p>
         * <p>
         * If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen On-Demand
         * allocation strategy.
         * </p>
         * 
         * @param usageStrategy
         *        Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity.</p>
         *        <p>
         *        If you specify <code>use-capacity-reservations-first</code>, the fleet uses unused Capacity
         *        Reservations to fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance
         *        pools have unused Capacity Reservations, the On-Demand allocation strategy (<code>lowest-price</code>
         *        or <code>prioritized</code>) is applied. If the number of unused Capacity Reservations is less than
         *        the On-Demand target capacity, the remaining On-Demand target capacity is launched according to the
         *        On-Demand allocation strategy (<code>lowest-price</code> or <code>prioritized</code>).
         *        </p>
         *        <p>
         *        If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen
         *        On-Demand allocation strategy.
         * @see FleetCapacityReservationUsageStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetCapacityReservationUsageStrategy
         */
        Builder usageStrategy(String usageStrategy);

        /**
         * <p>
         * Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity.
         * </p>
         * <p>
         * If you specify <code>use-capacity-reservations-first</code>, the fleet uses unused Capacity Reservations to
         * fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance pools have unused
         * Capacity Reservations, the On-Demand allocation strategy (<code>lowest-price</code> or
         * <code>prioritized</code>) is applied. If the number of unused Capacity Reservations is less than the
         * On-Demand target capacity, the remaining On-Demand target capacity is launched according to the On-Demand
         * allocation strategy (<code>lowest-price</code> or <code>prioritized</code>).
         * </p>
         * <p>
         * If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen On-Demand
         * allocation strategy.
         * </p>
         * 
         * @param usageStrategy
         *        Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity.</p>
         *        <p>
         *        If you specify <code>use-capacity-reservations-first</code>, the fleet uses unused Capacity
         *        Reservations to fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance
         *        pools have unused Capacity Reservations, the On-Demand allocation strategy (<code>lowest-price</code>
         *        or <code>prioritized</code>) is applied. If the number of unused Capacity Reservations is less than
         *        the On-Demand target capacity, the remaining On-Demand target capacity is launched according to the
         *        On-Demand allocation strategy (<code>lowest-price</code> or <code>prioritized</code>).
         *        </p>
         *        <p>
         *        If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen
         *        On-Demand allocation strategy.
         * @see FleetCapacityReservationUsageStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetCapacityReservationUsageStrategy
         */
        Builder usageStrategy(FleetCapacityReservationUsageStrategy usageStrategy);
    }

    static final class BuilderImpl implements Builder {
        private String usageStrategy;

        private BuilderImpl() {
        }

        private BuilderImpl(CapacityReservationOptions model) {
            usageStrategy(model.usageStrategy);
        }

        public final String getUsageStrategy() {
            return usageStrategy;
        }

        @Override
        public final Builder usageStrategy(String usageStrategy) {
            this.usageStrategy = usageStrategy;
            return this;
        }

        @Override
        public final Builder usageStrategy(FleetCapacityReservationUsageStrategy usageStrategy) {
            this.usageStrategy(usageStrategy == null ? null : usageStrategy.toString());
            return this;
        }

        public final void setUsageStrategy(String usageStrategy) {
            this.usageStrategy = usageStrategy;
        }

        @Override
        public CapacityReservationOptions build() {
            return new CapacityReservationOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
