/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for CreateNetworkInterface.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateNetworkInterfaceRequest extends Ec2Request implements
        ToCopyableBuilder<CreateNetworkInterfaceRequest.Builder, CreateNetworkInterfaceRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateNetworkInterfaceRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<List<String>> GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CreateNetworkInterfaceRequest::groups))
            .setter(setter(Builder::groups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupId")
                    .unmarshallLocationName("SecurityGroupId").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("SecurityGroupId")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("SecurityGroupId").unmarshallLocationName("SecurityGroupId")
                                                    .build()).build()).build()).build();

    private static final SdkField<Integer> IPV6_ADDRESS_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CreateNetworkInterfaceRequest::ipv6AddressCount))
            .setter(setter(Builder::ipv6AddressCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6AddressCount")
                    .unmarshallLocationName("ipv6AddressCount").build()).build();

    private static final SdkField<List<InstanceIpv6Address>> IPV6_ADDRESSES_FIELD = SdkField
            .<List<InstanceIpv6Address>> builder(MarshallingType.LIST)
            .getter(getter(CreateNetworkInterfaceRequest::ipv6Addresses))
            .setter(setter(Builder::ipv6Addresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6Addresses")
                    .unmarshallLocationName("ipv6Addresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<InstanceIpv6Address> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceIpv6Address::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> PRIVATE_IP_ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateNetworkInterfaceRequest::privateIpAddress))
            .setter(setter(Builder::privateIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateIpAddress")
                    .unmarshallLocationName("privateIpAddress").build()).build();

    private static final SdkField<List<PrivateIpAddressSpecification>> PRIVATE_IP_ADDRESSES_FIELD = SdkField
            .<List<PrivateIpAddressSpecification>> builder(MarshallingType.LIST)
            .getter(getter(CreateNetworkInterfaceRequest::privateIpAddresses))
            .setter(setter(Builder::privateIpAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateIpAddresses")
                    .unmarshallLocationName("privateIpAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<PrivateIpAddressSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(PrivateIpAddressSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Integer> SECONDARY_PRIVATE_IP_ADDRESS_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CreateNetworkInterfaceRequest::secondaryPrivateIpAddressCount))
            .setter(setter(Builder::secondaryPrivateIpAddressCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecondaryPrivateIpAddressCount")
                    .unmarshallLocationName("secondaryPrivateIpAddressCount").build()).build();

    private static final SdkField<String> INTERFACE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateNetworkInterfaceRequest::interfaceTypeAsString))
            .setter(setter(Builder::interfaceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InterfaceType")
                    .unmarshallLocationName("InterfaceType").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateNetworkInterfaceRequest::subnetId))
            .setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId")
                    .unmarshallLocationName("subnetId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            GROUPS_FIELD, IPV6_ADDRESS_COUNT_FIELD, IPV6_ADDRESSES_FIELD, PRIVATE_IP_ADDRESS_FIELD, PRIVATE_IP_ADDRESSES_FIELD,
            SECONDARY_PRIVATE_IP_ADDRESS_COUNT_FIELD, INTERFACE_TYPE_FIELD, SUBNET_ID_FIELD));

    private final String description;

    private final List<String> groups;

    private final Integer ipv6AddressCount;

    private final List<InstanceIpv6Address> ipv6Addresses;

    private final String privateIpAddress;

    private final List<PrivateIpAddressSpecification> privateIpAddresses;

    private final Integer secondaryPrivateIpAddressCount;

    private final String interfaceType;

    private final String subnetId;

    private CreateNetworkInterfaceRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.groups = builder.groups;
        this.ipv6AddressCount = builder.ipv6AddressCount;
        this.ipv6Addresses = builder.ipv6Addresses;
        this.privateIpAddress = builder.privateIpAddress;
        this.privateIpAddresses = builder.privateIpAddresses;
        this.secondaryPrivateIpAddressCount = builder.secondaryPrivateIpAddressCount;
        this.interfaceType = builder.interfaceType;
        this.subnetId = builder.subnetId;
    }

    /**
     * <p>
     * A description for the network interface.
     * </p>
     * 
     * @return A description for the network interface.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the Groups property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasGroups() {
        return groups != null && !(groups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of one or more security groups.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The IDs of one or more security groups.
     */
    public List<String> groups() {
        return groups;
    }

    /**
     * <p>
     * The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the IPv6
     * addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses. If your subnet
     * has the <code>AssignIpv6AddressOnCreation</code> attribute set to <code>true</code>, you can specify
     * <code>0</code> to override this setting.
     * </p>
     * 
     * @return The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the IPv6
     *         addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses. If your
     *         subnet has the <code>AssignIpv6AddressOnCreation</code> attribute set to <code>true</code>, you can
     *         specify <code>0</code> to override this setting.
     */
    public Integer ipv6AddressCount() {
        return ipv6AddressCount;
    }

    /**
     * Returns true if the Ipv6Addresses property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasIpv6Addresses() {
        return ipv6Addresses != null && !(ipv6Addresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this option if
     * you're specifying a number of IPv6 addresses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasIpv6Addresses()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this
     *         option if you're specifying a number of IPv6 addresses.
     */
    public List<InstanceIpv6Address> ipv6Addresses() {
        return ipv6Addresses;
    }

    /**
     * <p>
     * The primary private IPv4 address of the network interface. If you don't specify an IPv4 address, Amazon EC2
     * selects one for you from the subnet's IPv4 CIDR range. If you specify an IP address, you cannot indicate any IP
     * addresses specified in <code>privateIpAddresses</code> as primary (only one IP address can be designated as
     * primary).
     * </p>
     * 
     * @return The primary private IPv4 address of the network interface. If you don't specify an IPv4 address, Amazon
     *         EC2 selects one for you from the subnet's IPv4 CIDR range. If you specify an IP address, you cannot
     *         indicate any IP addresses specified in <code>privateIpAddresses</code> as primary (only one IP address
     *         can be designated as primary).
     */
    public String privateIpAddress() {
        return privateIpAddress;
    }

    /**
     * Returns true if the PrivateIpAddresses property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasPrivateIpAddresses() {
        return privateIpAddresses != null && !(privateIpAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more private IPv4 addresses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPrivateIpAddresses()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One or more private IPv4 addresses.
     */
    public List<PrivateIpAddressSpecification> privateIpAddresses() {
        return privateIpAddresses;
    }

    /**
     * <p>
     * The number of secondary private IPv4 addresses to assign to a network interface. When you specify a number of
     * secondary IPv4 addresses, Amazon EC2 selects these IP addresses within the subnet's IPv4 CIDR range. You can't
     * specify this option and specify more than one private IP address using <code>privateIpAddresses</code>.
     * </p>
     * <p>
     * The number of IP addresses you can assign to a network interface varies by instance type. For more information,
     * see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html#AvailableIpPerENI">IP Addresses
     * Per ENI Per Instance Type</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
     * </p>
     * 
     * @return The number of secondary private IPv4 addresses to assign to a network interface. When you specify a
     *         number of secondary IPv4 addresses, Amazon EC2 selects these IP addresses within the subnet's IPv4 CIDR
     *         range. You can't specify this option and specify more than one private IP address using
     *         <code>privateIpAddresses</code>.</p>
     *         <p>
     *         The number of IP addresses you can assign to a network interface varies by instance type. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html#AvailableIpPerENI">IP Addresses
     *         Per ENI Per Instance Type</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
     */
    public Integer secondaryPrivateIpAddressCount() {
        return secondaryPrivateIpAddressCount;
    }

    /**
     * <p>
     * Indicates the type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>. For
     * more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html"> Elastic Fabric
     * Adapter</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #interfaceType}
     * will return {@link NetworkInterfaceCreationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #interfaceTypeAsString}.
     * </p>
     * 
     * @return Indicates the type of network interface. To create an Elastic Fabric Adapter (EFA), specify
     *         <code>efa</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html"> Elastic Fabric Adapter</a> in the
     *         <i>Amazon Elastic Compute Cloud User Guide</i>.
     * @see NetworkInterfaceCreationType
     */
    public NetworkInterfaceCreationType interfaceType() {
        return NetworkInterfaceCreationType.fromValue(interfaceType);
    }

    /**
     * <p>
     * Indicates the type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>. For
     * more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html"> Elastic Fabric
     * Adapter</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #interfaceType}
     * will return {@link NetworkInterfaceCreationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #interfaceTypeAsString}.
     * </p>
     * 
     * @return Indicates the type of network interface. To create an Elastic Fabric Adapter (EFA), specify
     *         <code>efa</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html"> Elastic Fabric Adapter</a> in the
     *         <i>Amazon Elastic Compute Cloud User Guide</i>.
     * @see NetworkInterfaceCreationType
     */
    public String interfaceTypeAsString() {
        return interfaceType;
    }

    /**
     * <p>
     * The ID of the subnet to associate with the network interface.
     * </p>
     * 
     * @return The ID of the subnet to associate with the network interface.
     */
    public String subnetId() {
        return subnetId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(groups());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6AddressCount());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6Addresses());
        hashCode = 31 * hashCode + Objects.hashCode(privateIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(privateIpAddresses());
        hashCode = 31 * hashCode + Objects.hashCode(secondaryPrivateIpAddressCount());
        hashCode = 31 * hashCode + Objects.hashCode(interfaceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateNetworkInterfaceRequest)) {
            return false;
        }
        CreateNetworkInterfaceRequest other = (CreateNetworkInterfaceRequest) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(groups(), other.groups())
                && Objects.equals(ipv6AddressCount(), other.ipv6AddressCount())
                && Objects.equals(ipv6Addresses(), other.ipv6Addresses())
                && Objects.equals(privateIpAddress(), other.privateIpAddress())
                && Objects.equals(privateIpAddresses(), other.privateIpAddresses())
                && Objects.equals(secondaryPrivateIpAddressCount(), other.secondaryPrivateIpAddressCount())
                && Objects.equals(interfaceTypeAsString(), other.interfaceTypeAsString())
                && Objects.equals(subnetId(), other.subnetId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateNetworkInterfaceRequest").add("Description", description()).add("Groups", groups())
                .add("Ipv6AddressCount", ipv6AddressCount()).add("Ipv6Addresses", ipv6Addresses())
                .add("PrivateIpAddress", privateIpAddress()).add("PrivateIpAddresses", privateIpAddresses())
                .add("SecondaryPrivateIpAddressCount", secondaryPrivateIpAddressCount())
                .add("InterfaceType", interfaceTypeAsString()).add("SubnetId", subnetId()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Groups":
            return Optional.ofNullable(clazz.cast(groups()));
        case "Ipv6AddressCount":
            return Optional.ofNullable(clazz.cast(ipv6AddressCount()));
        case "Ipv6Addresses":
            return Optional.ofNullable(clazz.cast(ipv6Addresses()));
        case "PrivateIpAddress":
            return Optional.ofNullable(clazz.cast(privateIpAddress()));
        case "PrivateIpAddresses":
            return Optional.ofNullable(clazz.cast(privateIpAddresses()));
        case "SecondaryPrivateIpAddressCount":
            return Optional.ofNullable(clazz.cast(secondaryPrivateIpAddressCount()));
        case "InterfaceType":
            return Optional.ofNullable(clazz.cast(interfaceTypeAsString()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateNetworkInterfaceRequest, T> g) {
        return obj -> g.apply((CreateNetworkInterfaceRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateNetworkInterfaceRequest> {
        /**
         * <p>
         * A description for the network interface.
         * </p>
         * 
         * @param description
         *        A description for the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The IDs of one or more security groups.
         * </p>
         * 
         * @param groups
         *        The IDs of one or more security groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(Collection<String> groups);

        /**
         * <p>
         * The IDs of one or more security groups.
         * </p>
         * 
         * @param groups
         *        The IDs of one or more security groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(String... groups);

        /**
         * <p>
         * The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the IPv6
         * addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses. If your
         * subnet has the <code>AssignIpv6AddressOnCreation</code> attribute set to <code>true</code>, you can specify
         * <code>0</code> to override this setting.
         * </p>
         * 
         * @param ipv6AddressCount
         *        The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the
         *        IPv6 addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses.
         *        If your subnet has the <code>AssignIpv6AddressOnCreation</code> attribute set to <code>true</code>,
         *        you can specify <code>0</code> to override this setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6AddressCount(Integer ipv6AddressCount);

        /**
         * <p>
         * One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this option
         * if you're specifying a number of IPv6 addresses.
         * </p>
         * 
         * @param ipv6Addresses
         *        One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this
         *        option if you're specifying a number of IPv6 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Addresses(Collection<InstanceIpv6Address> ipv6Addresses);

        /**
         * <p>
         * One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this option
         * if you're specifying a number of IPv6 addresses.
         * </p>
         * 
         * @param ipv6Addresses
         *        One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this
         *        option if you're specifying a number of IPv6 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Addresses(InstanceIpv6Address... ipv6Addresses);

        /**
         * <p>
         * One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this option
         * if you're specifying a number of IPv6 addresses.
         * </p>
         * This is a convenience that creates an instance of the {@link List<InstanceIpv6Address>.Builder} avoiding the
         * need to create one manually via {@link List<InstanceIpv6Address>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<InstanceIpv6Address>.Builder#build()} is called immediately
         * and its result is passed to {@link #ipv6Addresses(List<InstanceIpv6Address>)}.
         * 
         * @param ipv6Addresses
         *        a consumer that will call methods on {@link List<InstanceIpv6Address>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ipv6Addresses(List<InstanceIpv6Address>)
         */
        Builder ipv6Addresses(Consumer<InstanceIpv6Address.Builder>... ipv6Addresses);

        /**
         * <p>
         * The primary private IPv4 address of the network interface. If you don't specify an IPv4 address, Amazon EC2
         * selects one for you from the subnet's IPv4 CIDR range. If you specify an IP address, you cannot indicate any
         * IP addresses specified in <code>privateIpAddresses</code> as primary (only one IP address can be designated
         * as primary).
         * </p>
         * 
         * @param privateIpAddress
         *        The primary private IPv4 address of the network interface. If you don't specify an IPv4 address,
         *        Amazon EC2 selects one for you from the subnet's IPv4 CIDR range. If you specify an IP address, you
         *        cannot indicate any IP addresses specified in <code>privateIpAddresses</code> as primary (only one IP
         *        address can be designated as primary).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddress(String privateIpAddress);

        /**
         * <p>
         * One or more private IPv4 addresses.
         * </p>
         * 
         * @param privateIpAddresses
         *        One or more private IPv4 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddresses(Collection<PrivateIpAddressSpecification> privateIpAddresses);

        /**
         * <p>
         * One or more private IPv4 addresses.
         * </p>
         * 
         * @param privateIpAddresses
         *        One or more private IPv4 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddresses(PrivateIpAddressSpecification... privateIpAddresses);

        /**
         * <p>
         * One or more private IPv4 addresses.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PrivateIpAddressSpecification>.Builder}
         * avoiding the need to create one manually via {@link List<PrivateIpAddressSpecification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PrivateIpAddressSpecification>.Builder#build()} is called
         * immediately and its result is passed to {@link #privateIpAddresses(List<PrivateIpAddressSpecification>)}.
         * 
         * @param privateIpAddresses
         *        a consumer that will call methods on {@link List<PrivateIpAddressSpecification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #privateIpAddresses(List<PrivateIpAddressSpecification>)
         */
        Builder privateIpAddresses(Consumer<PrivateIpAddressSpecification.Builder>... privateIpAddresses);

        /**
         * <p>
         * The number of secondary private IPv4 addresses to assign to a network interface. When you specify a number of
         * secondary IPv4 addresses, Amazon EC2 selects these IP addresses within the subnet's IPv4 CIDR range. You
         * can't specify this option and specify more than one private IP address using <code>privateIpAddresses</code>.
         * </p>
         * <p>
         * The number of IP addresses you can assign to a network interface varies by instance type. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html#AvailableIpPerENI">IP Addresses Per
         * ENI Per Instance Type</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * </p>
         * 
         * @param secondaryPrivateIpAddressCount
         *        The number of secondary private IPv4 addresses to assign to a network interface. When you specify a
         *        number of secondary IPv4 addresses, Amazon EC2 selects these IP addresses within the subnet's IPv4
         *        CIDR range. You can't specify this option and specify more than one private IP address using
         *        <code>privateIpAddresses</code>.</p>
         *        <p>
         *        The number of IP addresses you can assign to a network interface varies by instance type. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html#AvailableIpPerENI">IP
         *        Addresses Per ENI Per Instance Type</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount);

        /**
         * <p>
         * Indicates the type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>.
         * For more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html"> Elastic
         * Fabric Adapter</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * 
         * @param interfaceType
         *        Indicates the type of network interface. To create an Elastic Fabric Adapter (EFA), specify
         *        <code>efa</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html"> Elastic Fabric Adapter</a> in the
         *        <i>Amazon Elastic Compute Cloud User Guide</i>.
         * @see NetworkInterfaceCreationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkInterfaceCreationType
         */
        Builder interfaceType(String interfaceType);

        /**
         * <p>
         * Indicates the type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>.
         * For more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html"> Elastic
         * Fabric Adapter</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * 
         * @param interfaceType
         *        Indicates the type of network interface. To create an Elastic Fabric Adapter (EFA), specify
         *        <code>efa</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html"> Elastic Fabric Adapter</a> in the
         *        <i>Amazon Elastic Compute Cloud User Guide</i>.
         * @see NetworkInterfaceCreationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkInterfaceCreationType
         */
        Builder interfaceType(NetworkInterfaceCreationType interfaceType);

        /**
         * <p>
         * The ID of the subnet to associate with the network interface.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet to associate with the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String description;

        private List<String> groups = DefaultSdkAutoConstructList.getInstance();

        private Integer ipv6AddressCount;

        private List<InstanceIpv6Address> ipv6Addresses = DefaultSdkAutoConstructList.getInstance();

        private String privateIpAddress;

        private List<PrivateIpAddressSpecification> privateIpAddresses = DefaultSdkAutoConstructList.getInstance();

        private Integer secondaryPrivateIpAddressCount;

        private String interfaceType;

        private String subnetId;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateNetworkInterfaceRequest model) {
            super(model);
            description(model.description);
            groups(model.groups);
            ipv6AddressCount(model.ipv6AddressCount);
            ipv6Addresses(model.ipv6Addresses);
            privateIpAddress(model.privateIpAddress);
            privateIpAddresses(model.privateIpAddresses);
            secondaryPrivateIpAddressCount(model.secondaryPrivateIpAddressCount);
            interfaceType(model.interfaceType);
            subnetId(model.subnetId);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<String> getGroups() {
            return groups;
        }

        @Override
        public final Builder groups(Collection<String> groups) {
            this.groups = SecurityGroupIdStringListCopier.copy(groups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(String... groups) {
            groups(Arrays.asList(groups));
            return this;
        }

        public final void setGroups(Collection<String> groups) {
            this.groups = SecurityGroupIdStringListCopier.copy(groups);
        }

        public final Integer getIpv6AddressCount() {
            return ipv6AddressCount;
        }

        @Override
        public final Builder ipv6AddressCount(Integer ipv6AddressCount) {
            this.ipv6AddressCount = ipv6AddressCount;
            return this;
        }

        public final void setIpv6AddressCount(Integer ipv6AddressCount) {
            this.ipv6AddressCount = ipv6AddressCount;
        }

        public final Collection<InstanceIpv6Address.Builder> getIpv6Addresses() {
            return ipv6Addresses != null ? ipv6Addresses.stream().map(InstanceIpv6Address::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder ipv6Addresses(Collection<InstanceIpv6Address> ipv6Addresses) {
            this.ipv6Addresses = InstanceIpv6AddressListCopier.copy(ipv6Addresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Addresses(InstanceIpv6Address... ipv6Addresses) {
            ipv6Addresses(Arrays.asList(ipv6Addresses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Addresses(Consumer<InstanceIpv6Address.Builder>... ipv6Addresses) {
            ipv6Addresses(Stream.of(ipv6Addresses).map(c -> InstanceIpv6Address.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setIpv6Addresses(Collection<InstanceIpv6Address.BuilderImpl> ipv6Addresses) {
            this.ipv6Addresses = InstanceIpv6AddressListCopier.copyFromBuilder(ipv6Addresses);
        }

        public final String getPrivateIpAddress() {
            return privateIpAddress;
        }

        @Override
        public final Builder privateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
            return this;
        }

        public final void setPrivateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
        }

        public final Collection<PrivateIpAddressSpecification.Builder> getPrivateIpAddresses() {
            return privateIpAddresses != null ? privateIpAddresses.stream().map(PrivateIpAddressSpecification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder privateIpAddresses(Collection<PrivateIpAddressSpecification> privateIpAddresses) {
            this.privateIpAddresses = PrivateIpAddressSpecificationListCopier.copy(privateIpAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder privateIpAddresses(PrivateIpAddressSpecification... privateIpAddresses) {
            privateIpAddresses(Arrays.asList(privateIpAddresses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder privateIpAddresses(Consumer<PrivateIpAddressSpecification.Builder>... privateIpAddresses) {
            privateIpAddresses(Stream.of(privateIpAddresses)
                    .map(c -> PrivateIpAddressSpecification.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setPrivateIpAddresses(Collection<PrivateIpAddressSpecification.BuilderImpl> privateIpAddresses) {
            this.privateIpAddresses = PrivateIpAddressSpecificationListCopier.copyFromBuilder(privateIpAddresses);
        }

        public final Integer getSecondaryPrivateIpAddressCount() {
            return secondaryPrivateIpAddressCount;
        }

        @Override
        public final Builder secondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount) {
            this.secondaryPrivateIpAddressCount = secondaryPrivateIpAddressCount;
            return this;
        }

        public final void setSecondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount) {
            this.secondaryPrivateIpAddressCount = secondaryPrivateIpAddressCount;
        }

        public final String getInterfaceTypeAsString() {
            return interfaceType;
        }

        @Override
        public final Builder interfaceType(String interfaceType) {
            this.interfaceType = interfaceType;
            return this;
        }

        @Override
        public final Builder interfaceType(NetworkInterfaceCreationType interfaceType) {
            this.interfaceType(interfaceType == null ? null : interfaceType.toString());
            return this;
        }

        public final void setInterfaceType(String interfaceType) {
            this.interfaceType = interfaceType;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateNetworkInterfaceRequest build() {
            return new CreateNetworkInterfaceRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
