/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateTrafficMirrorFilterRequest extends Ec2Request implements
        ToCopyableBuilder<CreateTrafficMirrorFilterRequest.Builder, CreateTrafficMirrorFilterRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrafficMirrorFilterRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("Description").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .getter(getter(CreateTrafficMirrorFilterRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrafficMirrorFilterRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken")
                    .unmarshallLocationName("ClientToken").build(), DefaultValueTrait.idempotencyToken()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            TAG_SPECIFICATIONS_FIELD, CLIENT_TOKEN_FIELD));

    private final String description;

    private final List<TagSpecification> tagSpecifications;

    private final String clientToken;

    private CreateTrafficMirrorFilterRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.tagSpecifications = builder.tagSpecifications;
        this.clientToken = builder.clientToken;
    }

    /**
     * <p>
     * The description of the Traffic Mirror filter.
     * </p>
     * 
     * @return The description of the Traffic Mirror filter.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the TagSpecifications property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to assign to a Traffic Mirror filter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTagSpecifications()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tags to assign to a Traffic Mirror filter.
     */
    public List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    /**
     * <p>
     * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">How to Ensure
     * Idempotency</a>.
     * </p>
     * 
     * @return Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">How to Ensure
     *         Idempotency</a>.
     */
    public String clientToken() {
        return clientToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(tagSpecifications());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateTrafficMirrorFilterRequest)) {
            return false;
        }
        CreateTrafficMirrorFilterRequest other = (CreateTrafficMirrorFilterRequest) obj;
        return Objects.equals(description(), other.description())
                && Objects.equals(tagSpecifications(), other.tagSpecifications())
                && Objects.equals(clientToken(), other.clientToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateTrafficMirrorFilterRequest").add("Description", description())
                .add("TagSpecifications", tagSpecifications()).add("ClientToken", clientToken()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateTrafficMirrorFilterRequest, T> g) {
        return obj -> g.apply((CreateTrafficMirrorFilterRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateTrafficMirrorFilterRequest> {
        /**
         * <p>
         * The description of the Traffic Mirror filter.
         * </p>
         * 
         * @param description
         *        The description of the Traffic Mirror filter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The tags to assign to a Traffic Mirror filter.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to assign to a Traffic Mirror filter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The tags to assign to a Traffic Mirror filter.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to assign to a Traffic Mirror filter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The tags to assign to a Traffic Mirror filter.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TagSpecification>.Builder} avoiding the
         * need to create one manually via {@link List<TagSpecification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TagSpecification>.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on {@link List<TagSpecification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(List<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        /**
         * <p>
         * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">How to Ensure
         * Idempotency</a>.
         * </p>
         * 
         * @param clientToken
         *        Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">How to
         *        Ensure Idempotency</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String description;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private String clientToken;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateTrafficMirrorFilterRequest model) {
            super(model);
            description(model.description);
            tagSpecifications(model.tagSpecifications);
            clientToken(model.clientToken);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<TagSpecification.Builder> getTagSpecifications() {
            return tagSpecifications != null ? tagSpecifications.stream().map(TagSpecification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        public final String getClientToken() {
            return clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateTrafficMirrorFilterRequest build() {
            return new CreateTrafficMirrorFilterRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
