/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.core.AwsRequestOverrideConfig;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for CreateNetworkInterface.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class CreateNetworkInterfaceRequest extends EC2Request implements
        ToCopyableBuilder<CreateNetworkInterfaceRequest.Builder, CreateNetworkInterfaceRequest> {
    private final String description;

    private final List<String> groups;

    private final Integer ipv6AddressCount;

    private final List<InstanceIpv6Address> ipv6Addresses;

    private final String privateIpAddress;

    private final List<PrivateIpAddressSpecification> privateIpAddresses;

    private final Integer secondaryPrivateIpAddressCount;

    private final String subnetId;

    private CreateNetworkInterfaceRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.groups = builder.groups;
        this.ipv6AddressCount = builder.ipv6AddressCount;
        this.ipv6Addresses = builder.ipv6Addresses;
        this.privateIpAddress = builder.privateIpAddress;
        this.privateIpAddresses = builder.privateIpAddresses;
        this.secondaryPrivateIpAddressCount = builder.secondaryPrivateIpAddressCount;
        this.subnetId = builder.subnetId;
    }

    /**
     * <p>
     * A description for the network interface.
     * </p>
     * 
     * @return A description for the network interface.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The IDs of one or more security groups.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The IDs of one or more security groups.
     */
    public List<String> groups() {
        return groups;
    }

    /**
     * <p>
     * The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the IPv6
     * addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses. If your subnet
     * has the <code>AssignIpv6AddressOnCreation</code> attribute set to <code>true</code>, you can specify
     * <code>0</code> to override this setting.
     * </p>
     * 
     * @return The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the IPv6
     *         addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses. If your
     *         subnet has the <code>AssignIpv6AddressOnCreation</code> attribute set to <code>true</code>, you can
     *         specify <code>0</code> to override this setting.
     */
    public Integer ipv6AddressCount() {
        return ipv6AddressCount;
    }

    /**
     * <p>
     * One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this option if
     * you're specifying a number of IPv6 addresses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this
     *         option if you're specifying a number of IPv6 addresses.
     */
    public List<InstanceIpv6Address> ipv6Addresses() {
        return ipv6Addresses;
    }

    /**
     * <p>
     * The primary private IPv4 address of the network interface. If you don't specify an IPv4 address, Amazon EC2
     * selects one for you from the subnet's IPv4 CIDR range. If you specify an IP address, you cannot indicate any IP
     * addresses specified in <code>privateIpAddresses</code> as primary (only one IP address can be designated as
     * primary).
     * </p>
     * 
     * @return The primary private IPv4 address of the network interface. If you don't specify an IPv4 address, Amazon
     *         EC2 selects one for you from the subnet's IPv4 CIDR range. If you specify an IP address, you cannot
     *         indicate any IP addresses specified in <code>privateIpAddresses</code> as primary (only one IP address
     *         can be designated as primary).
     */
    public String privateIpAddress() {
        return privateIpAddress;
    }

    /**
     * <p>
     * One or more private IPv4 addresses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more private IPv4 addresses.
     */
    public List<PrivateIpAddressSpecification> privateIpAddresses() {
        return privateIpAddresses;
    }

    /**
     * <p>
     * The number of secondary private IPv4 addresses to assign to a network interface. When you specify a number of
     * secondary IPv4 addresses, Amazon EC2 selects these IP addresses within the subnet's IPv4 CIDR range. You can't
     * specify this option and specify more than one private IP address using <code>privateIpAddresses</code>.
     * </p>
     * <p>
     * The number of IP addresses you can assign to a network interface varies by instance type. For more information,
     * see <a href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html#AvailableIpPerENI">IP Addresses
     * Per ENI Per Instance Type</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
     * </p>
     * 
     * @return The number of secondary private IPv4 addresses to assign to a network interface. When you specify a
     *         number of secondary IPv4 addresses, Amazon EC2 selects these IP addresses within the subnet's IPv4 CIDR
     *         range. You can't specify this option and specify more than one private IP address using
     *         <code>privateIpAddresses</code>.</p>
     *         <p>
     *         The number of IP addresses you can assign to a network interface varies by instance type. For more
     *         information, see <a
     *         href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html#AvailableIpPerENI">IP Addresses
     *         Per ENI Per Instance Type</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
     */
    public Integer secondaryPrivateIpAddressCount() {
        return secondaryPrivateIpAddressCount;
    }

    /**
     * <p>
     * The ID of the subnet to associate with the network interface.
     * </p>
     * 
     * @return The ID of the subnet to associate with the network interface.
     */
    public String subnetId() {
        return subnetId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(groups());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6AddressCount());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6Addresses());
        hashCode = 31 * hashCode + Objects.hashCode(privateIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(privateIpAddresses());
        hashCode = 31 * hashCode + Objects.hashCode(secondaryPrivateIpAddressCount());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateNetworkInterfaceRequest)) {
            return false;
        }
        CreateNetworkInterfaceRequest other = (CreateNetworkInterfaceRequest) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(groups(), other.groups())
                && Objects.equals(ipv6AddressCount(), other.ipv6AddressCount())
                && Objects.equals(ipv6Addresses(), other.ipv6Addresses())
                && Objects.equals(privateIpAddress(), other.privateIpAddress())
                && Objects.equals(privateIpAddresses(), other.privateIpAddresses())
                && Objects.equals(secondaryPrivateIpAddressCount(), other.secondaryPrivateIpAddressCount())
                && Objects.equals(subnetId(), other.subnetId());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (description() != null) {
            sb.append("Description: ").append(description()).append(",");
        }
        if (groups() != null) {
            sb.append("Groups: ").append(groups()).append(",");
        }
        if (ipv6AddressCount() != null) {
            sb.append("Ipv6AddressCount: ").append(ipv6AddressCount()).append(",");
        }
        if (ipv6Addresses() != null) {
            sb.append("Ipv6Addresses: ").append(ipv6Addresses()).append(",");
        }
        if (privateIpAddress() != null) {
            sb.append("PrivateIpAddress: ").append(privateIpAddress()).append(",");
        }
        if (privateIpAddresses() != null) {
            sb.append("PrivateIpAddresses: ").append(privateIpAddresses()).append(",");
        }
        if (secondaryPrivateIpAddressCount() != null) {
            sb.append("SecondaryPrivateIpAddressCount: ").append(secondaryPrivateIpAddressCount()).append(",");
        }
        if (subnetId() != null) {
            sb.append("SubnetId: ").append(subnetId()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.of(clazz.cast(description()));
        case "Groups":
            return Optional.of(clazz.cast(groups()));
        case "Ipv6AddressCount":
            return Optional.of(clazz.cast(ipv6AddressCount()));
        case "Ipv6Addresses":
            return Optional.of(clazz.cast(ipv6Addresses()));
        case "PrivateIpAddress":
            return Optional.of(clazz.cast(privateIpAddress()));
        case "PrivateIpAddresses":
            return Optional.of(clazz.cast(privateIpAddresses()));
        case "SecondaryPrivateIpAddressCount":
            return Optional.of(clazz.cast(secondaryPrivateIpAddressCount()));
        case "SubnetId":
            return Optional.of(clazz.cast(subnetId()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends EC2Request.Builder, CopyableBuilder<Builder, CreateNetworkInterfaceRequest> {
        /**
         * <p>
         * A description for the network interface.
         * </p>
         * 
         * @param description
         *        A description for the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The IDs of one or more security groups.
         * </p>
         * 
         * @param groups
         *        The IDs of one or more security groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(Collection<String> groups);

        /**
         * <p>
         * The IDs of one or more security groups.
         * </p>
         * 
         * @param groups
         *        The IDs of one or more security groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(String... groups);

        /**
         * <p>
         * The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the IPv6
         * addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses. If your
         * subnet has the <code>AssignIpv6AddressOnCreation</code> attribute set to <code>true</code>, you can specify
         * <code>0</code> to override this setting.
         * </p>
         * 
         * @param ipv6AddressCount
         *        The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the
         *        IPv6 addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses.
         *        If your subnet has the <code>AssignIpv6AddressOnCreation</code> attribute set to <code>true</code>,
         *        you can specify <code>0</code> to override this setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6AddressCount(Integer ipv6AddressCount);

        /**
         * <p>
         * One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this option
         * if you're specifying a number of IPv6 addresses.
         * </p>
         * 
         * @param ipv6Addresses
         *        One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this
         *        option if you're specifying a number of IPv6 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Addresses(Collection<InstanceIpv6Address> ipv6Addresses);

        /**
         * <p>
         * One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this option
         * if you're specifying a number of IPv6 addresses.
         * </p>
         * 
         * @param ipv6Addresses
         *        One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this
         *        option if you're specifying a number of IPv6 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Addresses(InstanceIpv6Address... ipv6Addresses);

        /**
         * <p>
         * The primary private IPv4 address of the network interface. If you don't specify an IPv4 address, Amazon EC2
         * selects one for you from the subnet's IPv4 CIDR range. If you specify an IP address, you cannot indicate any
         * IP addresses specified in <code>privateIpAddresses</code> as primary (only one IP address can be designated
         * as primary).
         * </p>
         * 
         * @param privateIpAddress
         *        The primary private IPv4 address of the network interface. If you don't specify an IPv4 address,
         *        Amazon EC2 selects one for you from the subnet's IPv4 CIDR range. If you specify an IP address, you
         *        cannot indicate any IP addresses specified in <code>privateIpAddresses</code> as primary (only one IP
         *        address can be designated as primary).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddress(String privateIpAddress);

        /**
         * <p>
         * One or more private IPv4 addresses.
         * </p>
         * 
         * @param privateIpAddresses
         *        One or more private IPv4 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddresses(Collection<PrivateIpAddressSpecification> privateIpAddresses);

        /**
         * <p>
         * One or more private IPv4 addresses.
         * </p>
         * 
         * @param privateIpAddresses
         *        One or more private IPv4 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddresses(PrivateIpAddressSpecification... privateIpAddresses);

        /**
         * <p>
         * The number of secondary private IPv4 addresses to assign to a network interface. When you specify a number of
         * secondary IPv4 addresses, Amazon EC2 selects these IP addresses within the subnet's IPv4 CIDR range. You
         * can't specify this option and specify more than one private IP address using <code>privateIpAddresses</code>.
         * </p>
         * <p>
         * The number of IP addresses you can assign to a network interface varies by instance type. For more
         * information, see <a
         * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html#AvailableIpPerENI">IP Addresses Per
         * ENI Per Instance Type</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * </p>
         * 
         * @param secondaryPrivateIpAddressCount
         *        The number of secondary private IPv4 addresses to assign to a network interface. When you specify a
         *        number of secondary IPv4 addresses, Amazon EC2 selects these IP addresses within the subnet's IPv4
         *        CIDR range. You can't specify this option and specify more than one private IP address using
         *        <code>privateIpAddresses</code>.</p>
         *        <p>
         *        The number of IP addresses you can assign to a network interface varies by instance type. For more
         *        information, see <a
         *        href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html#AvailableIpPerENI">IP
         *        Addresses Per ENI Per Instance Type</a> in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount);

        /**
         * <p>
         * The ID of the subnet to associate with the network interface.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet to associate with the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);
    }

    static final class BuilderImpl extends EC2Request.BuilderImpl implements Builder {
        private String description;

        private List<String> groups;

        private Integer ipv6AddressCount;

        private List<InstanceIpv6Address> ipv6Addresses;

        private String privateIpAddress;

        private List<PrivateIpAddressSpecification> privateIpAddresses;

        private Integer secondaryPrivateIpAddressCount;

        private String subnetId;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateNetworkInterfaceRequest model) {
            description(model.description);
            groups(model.groups);
            ipv6AddressCount(model.ipv6AddressCount);
            ipv6Addresses(model.ipv6Addresses);
            privateIpAddress(model.privateIpAddress);
            privateIpAddresses(model.privateIpAddresses);
            secondaryPrivateIpAddressCount(model.secondaryPrivateIpAddressCount);
            subnetId(model.subnetId);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<String> getGroups() {
            return groups;
        }

        @Override
        public final Builder groups(Collection<String> groups) {
            this.groups = SecurityGroupIdStringListCopier.copy(groups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(String... groups) {
            groups(Arrays.asList(groups));
            return this;
        }

        public final void setGroups(Collection<String> groups) {
            this.groups = SecurityGroupIdStringListCopier.copy(groups);
        }

        public final Integer getIpv6AddressCount() {
            return ipv6AddressCount;
        }

        @Override
        public final Builder ipv6AddressCount(Integer ipv6AddressCount) {
            this.ipv6AddressCount = ipv6AddressCount;
            return this;
        }

        public final void setIpv6AddressCount(Integer ipv6AddressCount) {
            this.ipv6AddressCount = ipv6AddressCount;
        }

        public final Collection<InstanceIpv6Address.Builder> getIpv6Addresses() {
            return ipv6Addresses != null ? ipv6Addresses.stream().map(InstanceIpv6Address::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder ipv6Addresses(Collection<InstanceIpv6Address> ipv6Addresses) {
            this.ipv6Addresses = InstanceIpv6AddressListCopier.copy(ipv6Addresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Addresses(InstanceIpv6Address... ipv6Addresses) {
            ipv6Addresses(Arrays.asList(ipv6Addresses));
            return this;
        }

        public final void setIpv6Addresses(Collection<InstanceIpv6Address.BuilderImpl> ipv6Addresses) {
            this.ipv6Addresses = InstanceIpv6AddressListCopier.copyFromBuilder(ipv6Addresses);
        }

        public final String getPrivateIpAddress() {
            return privateIpAddress;
        }

        @Override
        public final Builder privateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
            return this;
        }

        public final void setPrivateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
        }

        public final Collection<PrivateIpAddressSpecification.Builder> getPrivateIpAddresses() {
            return privateIpAddresses != null ? privateIpAddresses.stream().map(PrivateIpAddressSpecification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder privateIpAddresses(Collection<PrivateIpAddressSpecification> privateIpAddresses) {
            this.privateIpAddresses = PrivateIpAddressSpecificationListCopier.copy(privateIpAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder privateIpAddresses(PrivateIpAddressSpecification... privateIpAddresses) {
            privateIpAddresses(Arrays.asList(privateIpAddresses));
            return this;
        }

        public final void setPrivateIpAddresses(Collection<PrivateIpAddressSpecification.BuilderImpl> privateIpAddresses) {
            this.privateIpAddresses = PrivateIpAddressSpecificationListCopier.copyFromBuilder(privateIpAddresses);
        }

        public final Integer getSecondaryPrivateIpAddressCount() {
            return secondaryPrivateIpAddressCount;
        }

        @Override
        public final Builder secondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount) {
            this.secondaryPrivateIpAddressCount = secondaryPrivateIpAddressCount;
            return this;
        }

        public final void setSecondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount) {
            this.secondaryPrivateIpAddressCount = secondaryPrivateIpAddressCount;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public CreateNetworkInterfaceRequest build() {
            return new CreateNetworkInterfaceRequest(this);
        }
    }
}
