/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a NAT gateway.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class NatGateway implements ToCopyableBuilder<NatGateway.Builder, NatGateway> {
    private final Instant createTime;

    private final Instant deleteTime;

    private final String failureCode;

    private final String failureMessage;

    private final List<NatGatewayAddress> natGatewayAddresses;

    private final String natGatewayId;

    private final ProvisionedBandwidth provisionedBandwidth;

    private final String state;

    private final String subnetId;

    private final String vpcId;

    private NatGateway(BuilderImpl builder) {
        this.createTime = builder.createTime;
        this.deleteTime = builder.deleteTime;
        this.failureCode = builder.failureCode;
        this.failureMessage = builder.failureMessage;
        this.natGatewayAddresses = builder.natGatewayAddresses;
        this.natGatewayId = builder.natGatewayId;
        this.provisionedBandwidth = builder.provisionedBandwidth;
        this.state = builder.state;
        this.subnetId = builder.subnetId;
        this.vpcId = builder.vpcId;
    }

    /**
     * <p>
     * The date and time the NAT gateway was created.
     * </p>
     * 
     * @return The date and time the NAT gateway was created.
     */
    public Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * The date and time the NAT gateway was deleted, if applicable.
     * </p>
     * 
     * @return The date and time the NAT gateway was deleted, if applicable.
     */
    public Instant deleteTime() {
        return deleteTime;
    }

    /**
     * <p>
     * If the NAT gateway could not be created, specifies the error code for the failure. (
     * <code>InsufficientFreeAddressesInSubnet</code> | <code>Gateway.NotAttached</code> |
     * <code>InvalidAllocationID.NotFound</code> | <code>Resource.AlreadyAssociated</code> | <code>InternalError</code>
     * | <code>InvalidSubnetID.NotFound</code>)
     * </p>
     * 
     * @return If the NAT gateway could not be created, specifies the error code for the failure. (
     *         <code>InsufficientFreeAddressesInSubnet</code> | <code>Gateway.NotAttached</code> |
     *         <code>InvalidAllocationID.NotFound</code> | <code>Resource.AlreadyAssociated</code> |
     *         <code>InternalError</code> | <code>InvalidSubnetID.NotFound</code>)
     */
    public String failureCode() {
        return failureCode;
    }

    /**
     * <p>
     * If the NAT gateway could not be created, specifies the error message for the failure, that corresponds to the
     * error code.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For InsufficientFreeAddressesInSubnet: "Subnet has insufficient free addresses to create this NAT gateway"
     * </p>
     * </li>
     * <li>
     * <p>
     * For Gateway.NotAttached: "Network vpc-xxxxxxxx has no Internet gateway attached"
     * </p>
     * </li>
     * <li>
     * <p>
     * For InvalidAllocationID.NotFound:
     * "Elastic IP address eipalloc-xxxxxxxx could not be associated with this NAT gateway"
     * </p>
     * </li>
     * <li>
     * <p>
     * For Resource.AlreadyAssociated: "Elastic IP address eipalloc-xxxxxxxx is already associated"
     * </p>
     * </li>
     * <li>
     * <p>
     * For InternalError:
     * "Network interface eni-xxxxxxxx, created and used internally by this NAT gateway is in an invalid state. Please try again."
     * </p>
     * </li>
     * <li>
     * <p>
     * For InvalidSubnetID.NotFound: "The specified subnet subnet-xxxxxxxx does not exist or could not be found."
     * </p>
     * </li>
     * </ul>
     * 
     * @return If the NAT gateway could not be created, specifies the error message for the failure, that corresponds to
     *         the error code.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For InsufficientFreeAddressesInSubnet:
     *         "Subnet has insufficient free addresses to create this NAT gateway"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Gateway.NotAttached: "Network vpc-xxxxxxxx has no Internet gateway attached"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For InvalidAllocationID.NotFound:
     *         "Elastic IP address eipalloc-xxxxxxxx could not be associated with this NAT gateway"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Resource.AlreadyAssociated: "Elastic IP address eipalloc-xxxxxxxx is already associated"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For InternalError:
     *         "Network interface eni-xxxxxxxx, created and used internally by this NAT gateway is in an invalid state. Please try again."
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For InvalidSubnetID.NotFound:
     *         "The specified subnet subnet-xxxxxxxx does not exist or could not be found."
     *         </p>
     *         </li>
     */
    public String failureMessage() {
        return failureMessage;
    }

    /**
     * <p>
     * Information about the IP addresses and network interface associated with the NAT gateway.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Information about the IP addresses and network interface associated with the NAT gateway.
     */
    public List<NatGatewayAddress> natGatewayAddresses() {
        return natGatewayAddresses;
    }

    /**
     * <p>
     * The ID of the NAT gateway.
     * </p>
     * 
     * @return The ID of the NAT gateway.
     */
    public String natGatewayId() {
        return natGatewayId;
    }

    /**
     * <p>
     * Reserved. If you need to sustain traffic greater than the <a
     * href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-nat-gateway.html">documented limits</a>, contact
     * us through the <a href="https://console.aws.amazon.com/support/home?">Support Center</a>.
     * </p>
     * 
     * @return Reserved. If you need to sustain traffic greater than the <a
     *         href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-nat-gateway.html">documented limits</a>,
     *         contact us through the <a href="https://console.aws.amazon.com/support/home?">Support Center</a>.
     */
    public ProvisionedBandwidth provisionedBandwidth() {
        return provisionedBandwidth;
    }

    /**
     * <p>
     * The state of the NAT gateway.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
     * <code>failureMessage</code> fields for the reason.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>available</code>: The NAT gateway is able to process traffic. This status remains until you delete the NAT
     * gateway, and does not indicate the health of the NAT gateway.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be processing traffic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The state of the NAT gateway.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
     *         <code>failureMessage</code> fields for the reason.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>available</code>: The NAT gateway is able to process traffic. This status remains until you delete
     *         the NAT gateway, and does not indicate the health of the NAT gateway.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be processing
     *         traffic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
     *         </p>
     *         </li>
     * @see NatGatewayState
     */
    public String state() {
        return state;
    }

    /**
     * <p>
     * The ID of the subnet in which the NAT gateway is located.
     * </p>
     * 
     * @return The ID of the subnet in which the NAT gateway is located.
     */
    public String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The ID of the VPC in which the NAT gateway is located.
     * </p>
     * 
     * @return The ID of the VPC in which the NAT gateway is located.
     */
    public String vpcId() {
        return vpcId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((createTime() == null) ? 0 : createTime().hashCode());
        hashCode = 31 * hashCode + ((deleteTime() == null) ? 0 : deleteTime().hashCode());
        hashCode = 31 * hashCode + ((failureCode() == null) ? 0 : failureCode().hashCode());
        hashCode = 31 * hashCode + ((failureMessage() == null) ? 0 : failureMessage().hashCode());
        hashCode = 31 * hashCode + ((natGatewayAddresses() == null) ? 0 : natGatewayAddresses().hashCode());
        hashCode = 31 * hashCode + ((natGatewayId() == null) ? 0 : natGatewayId().hashCode());
        hashCode = 31 * hashCode + ((provisionedBandwidth() == null) ? 0 : provisionedBandwidth().hashCode());
        hashCode = 31 * hashCode + ((state() == null) ? 0 : state().hashCode());
        hashCode = 31 * hashCode + ((subnetId() == null) ? 0 : subnetId().hashCode());
        hashCode = 31 * hashCode + ((vpcId() == null) ? 0 : vpcId().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NatGateway)) {
            return false;
        }
        NatGateway other = (NatGateway) obj;
        if (other.createTime() == null ^ this.createTime() == null) {
            return false;
        }
        if (other.createTime() != null && !other.createTime().equals(this.createTime())) {
            return false;
        }
        if (other.deleteTime() == null ^ this.deleteTime() == null) {
            return false;
        }
        if (other.deleteTime() != null && !other.deleteTime().equals(this.deleteTime())) {
            return false;
        }
        if (other.failureCode() == null ^ this.failureCode() == null) {
            return false;
        }
        if (other.failureCode() != null && !other.failureCode().equals(this.failureCode())) {
            return false;
        }
        if (other.failureMessage() == null ^ this.failureMessage() == null) {
            return false;
        }
        if (other.failureMessage() != null && !other.failureMessage().equals(this.failureMessage())) {
            return false;
        }
        if (other.natGatewayAddresses() == null ^ this.natGatewayAddresses() == null) {
            return false;
        }
        if (other.natGatewayAddresses() != null && !other.natGatewayAddresses().equals(this.natGatewayAddresses())) {
            return false;
        }
        if (other.natGatewayId() == null ^ this.natGatewayId() == null) {
            return false;
        }
        if (other.natGatewayId() != null && !other.natGatewayId().equals(this.natGatewayId())) {
            return false;
        }
        if (other.provisionedBandwidth() == null ^ this.provisionedBandwidth() == null) {
            return false;
        }
        if (other.provisionedBandwidth() != null && !other.provisionedBandwidth().equals(this.provisionedBandwidth())) {
            return false;
        }
        if (other.state() == null ^ this.state() == null) {
            return false;
        }
        if (other.state() != null && !other.state().equals(this.state())) {
            return false;
        }
        if (other.subnetId() == null ^ this.subnetId() == null) {
            return false;
        }
        if (other.subnetId() != null && !other.subnetId().equals(this.subnetId())) {
            return false;
        }
        if (other.vpcId() == null ^ this.vpcId() == null) {
            return false;
        }
        if (other.vpcId() != null && !other.vpcId().equals(this.vpcId())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (createTime() != null) {
            sb.append("CreateTime: ").append(createTime()).append(",");
        }
        if (deleteTime() != null) {
            sb.append("DeleteTime: ").append(deleteTime()).append(",");
        }
        if (failureCode() != null) {
            sb.append("FailureCode: ").append(failureCode()).append(",");
        }
        if (failureMessage() != null) {
            sb.append("FailureMessage: ").append(failureMessage()).append(",");
        }
        if (natGatewayAddresses() != null) {
            sb.append("NatGatewayAddresses: ").append(natGatewayAddresses()).append(",");
        }
        if (natGatewayId() != null) {
            sb.append("NatGatewayId: ").append(natGatewayId()).append(",");
        }
        if (provisionedBandwidth() != null) {
            sb.append("ProvisionedBandwidth: ").append(provisionedBandwidth()).append(",");
        }
        if (state() != null) {
            sb.append("State: ").append(state()).append(",");
        }
        if (subnetId() != null) {
            sb.append("SubnetId: ").append(subnetId()).append(",");
        }
        if (vpcId() != null) {
            sb.append("VpcId: ").append(vpcId()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    public interface Builder extends CopyableBuilder<Builder, NatGateway> {
        /**
         * <p>
         * The date and time the NAT gateway was created.
         * </p>
         * 
         * @param createTime
         *        The date and time the NAT gateway was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * The date and time the NAT gateway was deleted, if applicable.
         * </p>
         * 
         * @param deleteTime
         *        The date and time the NAT gateway was deleted, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteTime(Instant deleteTime);

        /**
         * <p>
         * If the NAT gateway could not be created, specifies the error code for the failure. (
         * <code>InsufficientFreeAddressesInSubnet</code> | <code>Gateway.NotAttached</code> |
         * <code>InvalidAllocationID.NotFound</code> | <code>Resource.AlreadyAssociated</code> |
         * <code>InternalError</code> | <code>InvalidSubnetID.NotFound</code>)
         * </p>
         * 
         * @param failureCode
         *        If the NAT gateway could not be created, specifies the error code for the failure. (
         *        <code>InsufficientFreeAddressesInSubnet</code> | <code>Gateway.NotAttached</code> |
         *        <code>InvalidAllocationID.NotFound</code> | <code>Resource.AlreadyAssociated</code> |
         *        <code>InternalError</code> | <code>InvalidSubnetID.NotFound</code>)
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureCode(String failureCode);

        /**
         * <p>
         * If the NAT gateway could not be created, specifies the error message for the failure, that corresponds to the
         * error code.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For InsufficientFreeAddressesInSubnet: "Subnet has insufficient free addresses to create this NAT gateway"
         * </p>
         * </li>
         * <li>
         * <p>
         * For Gateway.NotAttached: "Network vpc-xxxxxxxx has no Internet gateway attached"
         * </p>
         * </li>
         * <li>
         * <p>
         * For InvalidAllocationID.NotFound:
         * "Elastic IP address eipalloc-xxxxxxxx could not be associated with this NAT gateway"
         * </p>
         * </li>
         * <li>
         * <p>
         * For Resource.AlreadyAssociated: "Elastic IP address eipalloc-xxxxxxxx is already associated"
         * </p>
         * </li>
         * <li>
         * <p>
         * For InternalError:
         * "Network interface eni-xxxxxxxx, created and used internally by this NAT gateway is in an invalid state. Please try again."
         * </p>
         * </li>
         * <li>
         * <p>
         * For InvalidSubnetID.NotFound: "The specified subnet subnet-xxxxxxxx does not exist or could not be found."
         * </p>
         * </li>
         * </ul>
         * 
         * @param failureMessage
         *        If the NAT gateway could not be created, specifies the error message for the failure, that corresponds
         *        to the error code.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For InsufficientFreeAddressesInSubnet:
         *        "Subnet has insufficient free addresses to create this NAT gateway"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Gateway.NotAttached: "Network vpc-xxxxxxxx has no Internet gateway attached"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For InvalidAllocationID.NotFound:
         *        "Elastic IP address eipalloc-xxxxxxxx could not be associated with this NAT gateway"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Resource.AlreadyAssociated: "Elastic IP address eipalloc-xxxxxxxx is already associated"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For InternalError:
         *        "Network interface eni-xxxxxxxx, created and used internally by this NAT gateway is in an invalid state. Please try again."
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For InvalidSubnetID.NotFound:
         *        "The specified subnet subnet-xxxxxxxx does not exist or could not be found."
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureMessage(String failureMessage);

        /**
         * <p>
         * Information about the IP addresses and network interface associated with the NAT gateway.
         * </p>
         * 
         * @param natGatewayAddresses
         *        Information about the IP addresses and network interface associated with the NAT gateway.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder natGatewayAddresses(Collection<NatGatewayAddress> natGatewayAddresses);

        /**
         * <p>
         * Information about the IP addresses and network interface associated with the NAT gateway.
         * </p>
         * 
         * @param natGatewayAddresses
         *        Information about the IP addresses and network interface associated with the NAT gateway.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder natGatewayAddresses(NatGatewayAddress... natGatewayAddresses);

        /**
         * <p>
         * The ID of the NAT gateway.
         * </p>
         * 
         * @param natGatewayId
         *        The ID of the NAT gateway.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder natGatewayId(String natGatewayId);

        /**
         * <p>
         * Reserved. If you need to sustain traffic greater than the <a
         * href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-nat-gateway.html">documented limits</a>,
         * contact us through the <a href="https://console.aws.amazon.com/support/home?">Support Center</a>.
         * </p>
         * 
         * @param provisionedBandwidth
         *        Reserved. If you need to sustain traffic greater than the <a
         *        href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-nat-gateway.html">documented
         *        limits</a>, contact us through the <a href="https://console.aws.amazon.com/support/home?">Support
         *        Center</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedBandwidth(ProvisionedBandwidth provisionedBandwidth);

        /**
         * <p>
         * The state of the NAT gateway.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
         * <code>failureMessage</code> fields for the reason.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>available</code>: The NAT gateway is able to process traffic. This status remains until you delete the
         * NAT gateway, and does not indicate the health of the NAT gateway.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be processing
         * traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the NAT gateway.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
         *        <code>failureMessage</code> fields for the reason.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>available</code>: The NAT gateway is able to process traffic. This status remains until you
         *        delete the NAT gateway, and does not indicate the health of the NAT gateway.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be
         *        processing traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
         *        </p>
         *        </li>
         * @see NatGatewayState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NatGatewayState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the NAT gateway.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
         * <code>failureMessage</code> fields for the reason.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>available</code>: The NAT gateway is able to process traffic. This status remains until you delete the
         * NAT gateway, and does not indicate the health of the NAT gateway.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be processing
         * traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the NAT gateway.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
         *        <code>failureMessage</code> fields for the reason.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>available</code>: The NAT gateway is able to process traffic. This status remains until you
         *        delete the NAT gateway, and does not indicate the health of the NAT gateway.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be
         *        processing traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
         *        </p>
         *        </li>
         * @see NatGatewayState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NatGatewayState
         */
        Builder state(NatGatewayState state);

        /**
         * <p>
         * The ID of the subnet in which the NAT gateway is located.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet in which the NAT gateway is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The ID of the VPC in which the NAT gateway is located.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC in which the NAT gateway is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);
    }

    private static final class BuilderImpl implements Builder {
        private Instant createTime;

        private Instant deleteTime;

        private String failureCode;

        private String failureMessage;

        private List<NatGatewayAddress> natGatewayAddresses;

        private String natGatewayId;

        private ProvisionedBandwidth provisionedBandwidth;

        private String state;

        private String subnetId;

        private String vpcId;

        private BuilderImpl() {
        }

        private BuilderImpl(NatGateway model) {
            setCreateTime(model.createTime);
            setDeleteTime(model.deleteTime);
            setFailureCode(model.failureCode);
            setFailureMessage(model.failureMessage);
            setNatGatewayAddresses(model.natGatewayAddresses);
            setNatGatewayId(model.natGatewayId);
            setProvisionedBandwidth(model.provisionedBandwidth);
            setState(model.state);
            setSubnetId(model.subnetId);
            setVpcId(model.vpcId);
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        public final Instant getDeleteTime() {
            return deleteTime;
        }

        @Override
        public final Builder deleteTime(Instant deleteTime) {
            this.deleteTime = deleteTime;
            return this;
        }

        public final void setDeleteTime(Instant deleteTime) {
            this.deleteTime = deleteTime;
        }

        public final String getFailureCode() {
            return failureCode;
        }

        @Override
        public final Builder failureCode(String failureCode) {
            this.failureCode = failureCode;
            return this;
        }

        public final void setFailureCode(String failureCode) {
            this.failureCode = failureCode;
        }

        public final String getFailureMessage() {
            return failureMessage;
        }

        @Override
        public final Builder failureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
            return this;
        }

        public final void setFailureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
        }

        public final Collection<NatGatewayAddress> getNatGatewayAddresses() {
            return natGatewayAddresses;
        }

        @Override
        public final Builder natGatewayAddresses(Collection<NatGatewayAddress> natGatewayAddresses) {
            this.natGatewayAddresses = NatGatewayAddressListCopier.copy(natGatewayAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder natGatewayAddresses(NatGatewayAddress... natGatewayAddresses) {
            natGatewayAddresses(Arrays.asList(natGatewayAddresses));
            return this;
        }

        public final void setNatGatewayAddresses(Collection<NatGatewayAddress> natGatewayAddresses) {
            this.natGatewayAddresses = NatGatewayAddressListCopier.copy(natGatewayAddresses);
        }

        public final String getNatGatewayId() {
            return natGatewayId;
        }

        @Override
        public final Builder natGatewayId(String natGatewayId) {
            this.natGatewayId = natGatewayId;
            return this;
        }

        public final void setNatGatewayId(String natGatewayId) {
            this.natGatewayId = natGatewayId;
        }

        public final ProvisionedBandwidth getProvisionedBandwidth() {
            return provisionedBandwidth;
        }

        @Override
        public final Builder provisionedBandwidth(ProvisionedBandwidth provisionedBandwidth) {
            this.provisionedBandwidth = provisionedBandwidth;
            return this;
        }

        public final void setProvisionedBandwidth(ProvisionedBandwidth provisionedBandwidth) {
            this.provisionedBandwidth = provisionedBandwidth;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(NatGatewayState state) {
            this.state(state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        public final String getVpcId() {
            return vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public NatGateway build() {
            return new NatGateway(this);
        }
    }
}
