/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfig;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for ModifyNetworkInterfaceAttribute.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ModifyNetworkInterfaceAttributeRequest extends EC2Request implements
        ToCopyableBuilder<ModifyNetworkInterfaceAttributeRequest.Builder, ModifyNetworkInterfaceAttributeRequest> {
    private final NetworkInterfaceAttachmentChanges attachment;

    private final String description;

    private final List<String> groups;

    private final String networkInterfaceId;

    private final Boolean sourceDestCheck;

    private ModifyNetworkInterfaceAttributeRequest(BuilderImpl builder) {
        super(builder);
        this.attachment = builder.attachment;
        this.description = builder.description;
        this.groups = builder.groups;
        this.networkInterfaceId = builder.networkInterfaceId;
        this.sourceDestCheck = builder.sourceDestCheck;
    }

    /**
     * <p>
     * Information about the interface attachment. If modifying the 'delete on termination' attribute, you must specify
     * the ID of the interface attachment.
     * </p>
     * 
     * @return Information about the interface attachment. If modifying the 'delete on termination' attribute, you must
     *         specify the ID of the interface attachment.
     */
    public NetworkInterfaceAttachmentChanges attachment() {
        return attachment;
    }

    /**
     * <p>
     * A description for the network interface.
     * </p>
     * 
     * @return A description for the network interface.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * Changes the security groups for the network interface. The new set of groups you specify replaces the current
     * set. You must specify at least one group, even if it's just the default security group in the VPC. You must
     * specify the ID of the security group, not the name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Changes the security groups for the network interface. The new set of groups you specify replaces the
     *         current set. You must specify at least one group, even if it's just the default security group in the
     *         VPC. You must specify the ID of the security group, not the name.
     */
    public List<String> groups() {
        return groups;
    }

    /**
     * <p>
     * The ID of the network interface.
     * </p>
     * 
     * @return The ID of the network interface.
     */
    public String networkInterfaceId() {
        return networkInterfaceId;
    }

    /**
     * <p>
     * Indicates whether source/destination checking is enabled. A value of <code>true</code> means checking is enabled,
     * and <code>false</code> means checking is disabled. This value must be <code>false</code> for a NAT instance to
     * perform NAT. For more information, see <a
     * href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_NAT_Instance.html">NAT Instances</a> in the
     * <i>Amazon Virtual Private Cloud User Guide</i>.
     * </p>
     * 
     * @return Indicates whether source/destination checking is enabled. A value of <code>true</code> means checking is
     *         enabled, and <code>false</code> means checking is disabled. This value must be <code>false</code> for a
     *         NAT instance to perform NAT. For more information, see <a
     *         href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_NAT_Instance.html">NAT Instances</a> in
     *         the <i>Amazon Virtual Private Cloud User Guide</i>.
     */
    public Boolean sourceDestCheck() {
        return sourceDestCheck;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(attachment());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(groups());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaceId());
        hashCode = 31 * hashCode + Objects.hashCode(sourceDestCheck());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ModifyNetworkInterfaceAttributeRequest)) {
            return false;
        }
        ModifyNetworkInterfaceAttributeRequest other = (ModifyNetworkInterfaceAttributeRequest) obj;
        return Objects.equals(attachment(), other.attachment()) && Objects.equals(description(), other.description())
                && Objects.equals(groups(), other.groups()) && Objects.equals(networkInterfaceId(), other.networkInterfaceId())
                && Objects.equals(sourceDestCheck(), other.sourceDestCheck());
    }

    @Override
    public String toString() {
        return ToString.builder("ModifyNetworkInterfaceAttributeRequest").add("Attachment", attachment())
                .add("Description", description()).add("Groups", groups()).add("NetworkInterfaceId", networkInterfaceId())
                .add("SourceDestCheck", sourceDestCheck()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Attachment":
            return Optional.of(clazz.cast(attachment()));
        case "Description":
            return Optional.of(clazz.cast(description()));
        case "Groups":
            return Optional.of(clazz.cast(groups()));
        case "NetworkInterfaceId":
            return Optional.of(clazz.cast(networkInterfaceId()));
        case "SourceDestCheck":
            return Optional.of(clazz.cast(sourceDestCheck()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends EC2Request.Builder, CopyableBuilder<Builder, ModifyNetworkInterfaceAttributeRequest> {
        /**
         * <p>
         * Information about the interface attachment. If modifying the 'delete on termination' attribute, you must
         * specify the ID of the interface attachment.
         * </p>
         * 
         * @param attachment
         *        Information about the interface attachment. If modifying the 'delete on termination' attribute, you
         *        must specify the ID of the interface attachment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachment(NetworkInterfaceAttachmentChanges attachment);

        /**
         * <p>
         * Information about the interface attachment. If modifying the 'delete on termination' attribute, you must
         * specify the ID of the interface attachment.
         * </p>
         * This is a convenience that creates an instance of the {@link NetworkInterfaceAttachmentChanges.Builder}
         * avoiding the need to create one manually via {@link NetworkInterfaceAttachmentChanges#builder()}.
         *
         * When the {@link Consumer} completes, {@link NetworkInterfaceAttachmentChanges.Builder#build()} is called
         * immediately and its result is passed to {@link #attachment(NetworkInterfaceAttachmentChanges)}.
         * 
         * @param attachment
         *        a consumer that will call methods on {@link NetworkInterfaceAttachmentChanges.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachment(NetworkInterfaceAttachmentChanges)
         */
        default Builder attachment(Consumer<NetworkInterfaceAttachmentChanges.Builder> attachment) {
            return attachment(NetworkInterfaceAttachmentChanges.builder().apply(attachment).build());
        }

        /**
         * <p>
         * A description for the network interface.
         * </p>
         * 
         * @param description
         *        A description for the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Changes the security groups for the network interface. The new set of groups you specify replaces the current
         * set. You must specify at least one group, even if it's just the default security group in the VPC. You must
         * specify the ID of the security group, not the name.
         * </p>
         * 
         * @param groups
         *        Changes the security groups for the network interface. The new set of groups you specify replaces the
         *        current set. You must specify at least one group, even if it's just the default security group in the
         *        VPC. You must specify the ID of the security group, not the name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(Collection<String> groups);

        /**
         * <p>
         * Changes the security groups for the network interface. The new set of groups you specify replaces the current
         * set. You must specify at least one group, even if it's just the default security group in the VPC. You must
         * specify the ID of the security group, not the name.
         * </p>
         * 
         * @param groups
         *        Changes the security groups for the network interface. The new set of groups you specify replaces the
         *        current set. You must specify at least one group, even if it's just the default security group in the
         *        VPC. You must specify the ID of the security group, not the name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(String... groups);

        /**
         * <p>
         * The ID of the network interface.
         * </p>
         * 
         * @param networkInterfaceId
         *        The ID of the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceId(String networkInterfaceId);

        /**
         * <p>
         * Indicates whether source/destination checking is enabled. A value of <code>true</code> means checking is
         * enabled, and <code>false</code> means checking is disabled. This value must be <code>false</code> for a NAT
         * instance to perform NAT. For more information, see <a
         * href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_NAT_Instance.html">NAT Instances</a> in the
         * <i>Amazon Virtual Private Cloud User Guide</i>.
         * </p>
         * 
         * @param sourceDestCheck
         *        Indicates whether source/destination checking is enabled. A value of <code>true</code> means checking
         *        is enabled, and <code>false</code> means checking is disabled. This value must be <code>false</code>
         *        for a NAT instance to perform NAT. For more information, see <a
         *        href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_NAT_Instance.html">NAT Instances</a>
         *        in the <i>Amazon Virtual Private Cloud User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDestCheck(Boolean sourceDestCheck);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);

        @Override
        Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer);
    }

    static final class BuilderImpl extends EC2Request.BuilderImpl implements Builder {
        private NetworkInterfaceAttachmentChanges attachment;

        private String description;

        private List<String> groups;

        private String networkInterfaceId;

        private Boolean sourceDestCheck;

        private BuilderImpl() {
        }

        private BuilderImpl(ModifyNetworkInterfaceAttributeRequest model) {
            super(model);
            attachment(model.attachment);
            description(model.description);
            groups(model.groups);
            networkInterfaceId(model.networkInterfaceId);
            sourceDestCheck(model.sourceDestCheck);
        }

        public final NetworkInterfaceAttachmentChanges.Builder getAttachment() {
            return attachment != null ? attachment.toBuilder() : null;
        }

        @Override
        public final Builder attachment(NetworkInterfaceAttachmentChanges attachment) {
            this.attachment = attachment;
            return this;
        }

        public final void setAttachment(NetworkInterfaceAttachmentChanges.BuilderImpl attachment) {
            this.attachment = attachment != null ? attachment.build() : null;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<String> getGroups() {
            return groups;
        }

        @Override
        public final Builder groups(Collection<String> groups) {
            this.groups = SecurityGroupIdStringListCopier.copy(groups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(String... groups) {
            groups(Arrays.asList(groups));
            return this;
        }

        public final void setGroups(Collection<String> groups) {
            this.groups = SecurityGroupIdStringListCopier.copy(groups);
        }

        public final String getNetworkInterfaceId() {
            return networkInterfaceId;
        }

        @Override
        public final Builder networkInterfaceId(String networkInterfaceId) {
            this.networkInterfaceId = networkInterfaceId;
            return this;
        }

        public final void setNetworkInterfaceId(String networkInterfaceId) {
            this.networkInterfaceId = networkInterfaceId;
        }

        public final Boolean getSourceDestCheck() {
            return sourceDestCheck;
        }

        @Override
        public final Builder sourceDestCheck(Boolean sourceDestCheck) {
            this.sourceDestCheck = sourceDestCheck;
            return this;
        }

        public final void setSourceDestCheck(Boolean sourceDestCheck) {
            this.sourceDestCheck = sourceDestCheck;
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public ModifyNetworkInterfaceAttributeRequest build() {
            return new ModifyNetworkInterfaceAttributeRequest(this);
        }
    }
}
