/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a conversion task.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ConversionTask implements ToCopyableBuilder<ConversionTask.Builder, ConversionTask> {
    private final String conversionTaskId;

    private final String expirationTime;

    private final ImportInstanceTaskDetails importInstance;

    private final ImportVolumeTaskDetails importVolume;

    private final String state;

    private final String statusMessage;

    private final List<Tag> tags;

    private ConversionTask(BuilderImpl builder) {
        this.conversionTaskId = builder.conversionTaskId;
        this.expirationTime = builder.expirationTime;
        this.importInstance = builder.importInstance;
        this.importVolume = builder.importVolume;
        this.state = builder.state;
        this.statusMessage = builder.statusMessage;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the conversion task.
     * </p>
     * 
     * @return The ID of the conversion task.
     */
    public String conversionTaskId() {
        return conversionTaskId;
    }

    /**
     * <p>
     * The time when the task expires. If the upload isn't complete before the expiration time, we automatically cancel
     * the task.
     * </p>
     * 
     * @return The time when the task expires. If the upload isn't complete before the expiration time, we automatically
     *         cancel the task.
     */
    public String expirationTime() {
        return expirationTime;
    }

    /**
     * <p>
     * If the task is for importing an instance, this contains information about the import instance task.
     * </p>
     * 
     * @return If the task is for importing an instance, this contains information about the import instance task.
     */
    public ImportInstanceTaskDetails importInstance() {
        return importInstance;
    }

    /**
     * <p>
     * If the task is for importing a volume, this contains information about the import volume task.
     * </p>
     * 
     * @return If the task is for importing a volume, this contains information about the import volume task.
     */
    public ImportVolumeTaskDetails importVolume() {
        return importVolume;
    }

    /**
     * <p>
     * The state of the conversion task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ConversionTaskState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the conversion task.
     * @see ConversionTaskState
     */
    public ConversionTaskState state() {
        return ConversionTaskState.fromValue(state);
    }

    /**
     * <p>
     * The state of the conversion task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ConversionTaskState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the conversion task.
     * @see ConversionTaskState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The status message related to the conversion task.
     * </p>
     * 
     * @return The status message related to the conversion task.
     */
    public String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * Any tags assigned to the task.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Any tags assigned to the task.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(conversionTaskId());
        hashCode = 31 * hashCode + Objects.hashCode(expirationTime());
        hashCode = 31 * hashCode + Objects.hashCode(importInstance());
        hashCode = 31 * hashCode + Objects.hashCode(importVolume());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConversionTask)) {
            return false;
        }
        ConversionTask other = (ConversionTask) obj;
        return Objects.equals(conversionTaskId(), other.conversionTaskId())
                && Objects.equals(expirationTime(), other.expirationTime())
                && Objects.equals(importInstance(), other.importInstance())
                && Objects.equals(importVolume(), other.importVolume()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(statusMessage(), other.statusMessage()) && Objects.equals(tags(), other.tags());
    }

    @Override
    public String toString() {
        return ToString.builder("ConversionTask").add("ConversionTaskId", conversionTaskId())
                .add("ExpirationTime", expirationTime()).add("ImportInstance", importInstance())
                .add("ImportVolume", importVolume()).add("State", stateAsString()).add("StatusMessage", statusMessage())
                .add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConversionTaskId":
            return Optional.of(clazz.cast(conversionTaskId()));
        case "ExpirationTime":
            return Optional.of(clazz.cast(expirationTime()));
        case "ImportInstance":
            return Optional.of(clazz.cast(importInstance()));
        case "ImportVolume":
            return Optional.of(clazz.cast(importVolume()));
        case "State":
            return Optional.of(clazz.cast(stateAsString()));
        case "StatusMessage":
            return Optional.of(clazz.cast(statusMessage()));
        case "Tags":
            return Optional.of(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, ConversionTask> {
        /**
         * <p>
         * The ID of the conversion task.
         * </p>
         * 
         * @param conversionTaskId
         *        The ID of the conversion task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conversionTaskId(String conversionTaskId);

        /**
         * <p>
         * The time when the task expires. If the upload isn't complete before the expiration time, we automatically
         * cancel the task.
         * </p>
         * 
         * @param expirationTime
         *        The time when the task expires. If the upload isn't complete before the expiration time, we
         *        automatically cancel the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expirationTime(String expirationTime);

        /**
         * <p>
         * If the task is for importing an instance, this contains information about the import instance task.
         * </p>
         * 
         * @param importInstance
         *        If the task is for importing an instance, this contains information about the import instance task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importInstance(ImportInstanceTaskDetails importInstance);

        /**
         * <p>
         * If the task is for importing an instance, this contains information about the import instance task.
         * </p>
         * This is a convenience that creates an instance of the {@link ImportInstanceTaskDetails.Builder} avoiding the
         * need to create one manually via {@link ImportInstanceTaskDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link ImportInstanceTaskDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #importInstance(ImportInstanceTaskDetails)}.
         * 
         * @param importInstance
         *        a consumer that will call methods on {@link ImportInstanceTaskDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #importInstance(ImportInstanceTaskDetails)
         */
        default Builder importInstance(Consumer<ImportInstanceTaskDetails.Builder> importInstance) {
            return importInstance(ImportInstanceTaskDetails.builder().apply(importInstance).build());
        }

        /**
         * <p>
         * If the task is for importing a volume, this contains information about the import volume task.
         * </p>
         * 
         * @param importVolume
         *        If the task is for importing a volume, this contains information about the import volume task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importVolume(ImportVolumeTaskDetails importVolume);

        /**
         * <p>
         * If the task is for importing a volume, this contains information about the import volume task.
         * </p>
         * This is a convenience that creates an instance of the {@link ImportVolumeTaskDetails.Builder} avoiding the
         * need to create one manually via {@link ImportVolumeTaskDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link ImportVolumeTaskDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #importVolume(ImportVolumeTaskDetails)}.
         * 
         * @param importVolume
         *        a consumer that will call methods on {@link ImportVolumeTaskDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #importVolume(ImportVolumeTaskDetails)
         */
        default Builder importVolume(Consumer<ImportVolumeTaskDetails.Builder> importVolume) {
            return importVolume(ImportVolumeTaskDetails.builder().apply(importVolume).build());
        }

        /**
         * <p>
         * The state of the conversion task.
         * </p>
         * 
         * @param state
         *        The state of the conversion task.
         * @see ConversionTaskState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConversionTaskState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the conversion task.
         * </p>
         * 
         * @param state
         *        The state of the conversion task.
         * @see ConversionTaskState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConversionTaskState
         */
        Builder state(ConversionTaskState state);

        /**
         * <p>
         * The status message related to the conversion task.
         * </p>
         * 
         * @param statusMessage
         *        The status message related to the conversion task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * Any tags assigned to the task.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the task.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the task.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String conversionTaskId;

        private String expirationTime;

        private ImportInstanceTaskDetails importInstance;

        private ImportVolumeTaskDetails importVolume;

        private String state;

        private String statusMessage;

        private List<Tag> tags;

        private BuilderImpl() {
        }

        private BuilderImpl(ConversionTask model) {
            conversionTaskId(model.conversionTaskId);
            expirationTime(model.expirationTime);
            importInstance(model.importInstance);
            importVolume(model.importVolume);
            state(model.state);
            statusMessage(model.statusMessage);
            tags(model.tags);
        }

        public final String getConversionTaskId() {
            return conversionTaskId;
        }

        @Override
        public final Builder conversionTaskId(String conversionTaskId) {
            this.conversionTaskId = conversionTaskId;
            return this;
        }

        public final void setConversionTaskId(String conversionTaskId) {
            this.conversionTaskId = conversionTaskId;
        }

        public final String getExpirationTime() {
            return expirationTime;
        }

        @Override
        public final Builder expirationTime(String expirationTime) {
            this.expirationTime = expirationTime;
            return this;
        }

        public final void setExpirationTime(String expirationTime) {
            this.expirationTime = expirationTime;
        }

        public final ImportInstanceTaskDetails.Builder getImportInstance() {
            return importInstance != null ? importInstance.toBuilder() : null;
        }

        @Override
        public final Builder importInstance(ImportInstanceTaskDetails importInstance) {
            this.importInstance = importInstance;
            return this;
        }

        public final void setImportInstance(ImportInstanceTaskDetails.BuilderImpl importInstance) {
            this.importInstance = importInstance != null ? importInstance.build() : null;
        }

        public final ImportVolumeTaskDetails.Builder getImportVolume() {
            return importVolume != null ? importVolume.toBuilder() : null;
        }

        @Override
        public final Builder importVolume(ImportVolumeTaskDetails importVolume) {
            this.importVolume = importVolume;
            return this;
        }

        public final void setImportVolume(ImportVolumeTaskDetails.BuilderImpl importVolume) {
            this.importVolume = importVolume != null ? importVolume.build() : null;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(ConversionTaskState state) {
            this.state(state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public ConversionTask build() {
            return new ConversionTask(this);
        }
    }
}
