/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.endpoints.internal;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletionException;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsExecutionAttribute;
import software.amazon.awssdk.awscore.endpoints.AccountIdEndpointMode;
import software.amazon.awssdk.awscore.endpoints.AwsEndpointAttribute;
import software.amazon.awssdk.awscore.endpoints.authscheme.EndpointAuthScheme;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4AuthScheme;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4aAuthScheme;
import software.amazon.awssdk.awscore.internal.useragent.BusinessMetricsUtils;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.SelectedAuthScheme;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.interceptor.Context;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.interceptor.ExecutionInterceptor;
import software.amazon.awssdk.core.interceptor.SdkExecutionAttribute;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.useragent.BusinessMetricFeatureId;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.http.SdkHttpRequest;
import software.amazon.awssdk.http.auth.aws.signer.AwsV4HttpSigner;
import software.amazon.awssdk.http.auth.aws.signer.AwsV4aHttpSigner;
import software.amazon.awssdk.http.auth.aws.signer.RegionSet;
import software.amazon.awssdk.http.auth.spi.scheme.AuthSchemeOption;
import software.amazon.awssdk.identity.spi.AwsCredentialsIdentity;
import software.amazon.awssdk.identity.spi.Identity;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.services.dynamodb.endpoints.DynamoDbEndpointParams;
import software.amazon.awssdk.services.dynamodb.endpoints.DynamoDbEndpointProvider;
import software.amazon.awssdk.services.dynamodb.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.dynamodb.model.BatchGetItemRequest;
import software.amazon.awssdk.services.dynamodb.model.BatchWriteItemRequest;
import software.amazon.awssdk.services.dynamodb.model.CreateBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.CreateGlobalTableRequest;
import software.amazon.awssdk.services.dynamodb.model.CreateTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteItemRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeContinuousBackupsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeContributorInsightsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeExportRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeGlobalTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeGlobalTableSettingsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeImportRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeKinesisStreamingDestinationRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableReplicaAutoScalingRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTimeToLiveRequest;
import software.amazon.awssdk.services.dynamodb.model.DisableKinesisStreamingDestinationRequest;
import software.amazon.awssdk.services.dynamodb.model.EnableKinesisStreamingDestinationRequest;
import software.amazon.awssdk.services.dynamodb.model.ExportTableToPointInTimeRequest;
import software.amazon.awssdk.services.dynamodb.model.GetItemRequest;
import software.amazon.awssdk.services.dynamodb.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.dynamodb.model.ImportTableRequest;
import software.amazon.awssdk.services.dynamodb.model.ListBackupsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListContributorInsightsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListExportsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListImportsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListTagsOfResourceRequest;
import software.amazon.awssdk.services.dynamodb.model.PutItemRequest;
import software.amazon.awssdk.services.dynamodb.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.dynamodb.model.QueryRequest;
import software.amazon.awssdk.services.dynamodb.model.RestoreTableFromBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.RestoreTableToPointInTimeRequest;
import software.amazon.awssdk.services.dynamodb.model.ScanRequest;
import software.amazon.awssdk.services.dynamodb.model.TagResourceRequest;
import software.amazon.awssdk.services.dynamodb.model.TransactGetItemsRequest;
import software.amazon.awssdk.services.dynamodb.model.UntagResourceRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateContinuousBackupsRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateContributorInsightsRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateGlobalTableRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateGlobalTableSettingsRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateItemRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateKinesisStreamingDestinationRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateTableReplicaAutoScalingRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateTableRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateTimeToLiveRequest;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.CompletableFutureUtils;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DynamoDbResolveEndpointInterceptor implements ExecutionInterceptor {
    @Override
    public SdkRequest modifyRequest(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        SdkRequest result = context.request();
        if (AwsEndpointProviderUtils.endpointIsDiscovered(executionAttributes)) {
            return result;
        }
        DynamoDbEndpointProvider provider = (DynamoDbEndpointProvider) executionAttributes
                .getAttribute(SdkInternalExecutionAttribute.ENDPOINT_PROVIDER);
        try {
            long resolveEndpointStart = System.nanoTime();
            DynamoDbEndpointParams endpointParams = ruleParams(result, executionAttributes);
            Endpoint endpoint = provider.resolveEndpoint(endpointParams).join();
            Duration resolveEndpointDuration = Duration.ofNanos(System.nanoTime() - resolveEndpointStart);
            Optional<MetricCollector> metricCollector = executionAttributes
                    .getOptionalAttribute(SdkExecutionAttribute.API_CALL_METRIC_COLLECTOR);
            metricCollector.ifPresent(mc -> mc.reportMetric(CoreMetric.ENDPOINT_RESOLVE_DURATION, resolveEndpointDuration));
            if (!AwsEndpointProviderUtils.disableHostPrefixInjection(executionAttributes)) {
                Optional<String> hostPrefix = hostPrefix(executionAttributes.getAttribute(SdkExecutionAttribute.OPERATION_NAME),
                        result);
                if (hostPrefix.isPresent()) {
                    endpoint = AwsEndpointProviderUtils.addHostPrefix(endpoint, hostPrefix.get());
                }
            }
            List<EndpointAuthScheme> endpointAuthSchemes = endpoint.attribute(AwsEndpointAttribute.AUTH_SCHEMES);
            SelectedAuthScheme<?> selectedAuthScheme = executionAttributes
                    .getAttribute(SdkInternalExecutionAttribute.SELECTED_AUTH_SCHEME);
            if (endpointAuthSchemes != null && selectedAuthScheme != null) {
                selectedAuthScheme = authSchemeWithEndpointSignerProperties(endpointAuthSchemes, selectedAuthScheme);
                executionAttributes.putAttribute(SdkInternalExecutionAttribute.SELECTED_AUTH_SCHEME, selectedAuthScheme);
            }
            executionAttributes.putAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT, endpoint);
            return result;
        } catch (CompletionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof SdkClientException) {
                throw (SdkClientException) cause;
            } else {
                throw SdkClientException.create("Endpoint resolution failed", cause);
            }
        }
    }

    @Override
    public SdkHttpRequest modifyHttpRequest(Context.ModifyHttpRequest context, ExecutionAttributes executionAttributes) {
        Endpoint resolvedEndpoint = executionAttributes.getAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT);
        if (resolvedEndpoint.headers().isEmpty()) {
            return context.httpRequest();
        }
        SdkHttpRequest.Builder httpRequestBuilder = context.httpRequest().toBuilder();
        resolvedEndpoint.headers().forEach((name, values) -> {
            values.forEach(v -> httpRequestBuilder.appendHeader(name, v));
        });
        return httpRequestBuilder.build();
    }

    public static DynamoDbEndpointParams ruleParams(SdkRequest request, ExecutionAttributes executionAttributes) {
        DynamoDbEndpointParams.Builder builder = DynamoDbEndpointParams.builder();
        builder.region(AwsEndpointProviderUtils.regionBuiltIn(executionAttributes));
        builder.useDualStack(AwsEndpointProviderUtils.dualStackEnabledBuiltIn(executionAttributes));
        builder.useFips(AwsEndpointProviderUtils.fipsEnabledBuiltIn(executionAttributes));
        builder.endpoint(AwsEndpointProviderUtils.endpointBuiltIn(executionAttributes));
        builder.accountId(resolveAndRecordAccountIdFromIdentity(executionAttributes));
        builder.accountIdEndpointMode(recordAccountIdEndpointMode(executionAttributes));
        setContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), request);
        setStaticContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME));
        setOperationContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), request);
        return builder.build();
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, String operationName, SdkRequest request) {
        switch (operationName) {
        case "CreateBackup":
            setContextParams(params, (CreateBackupRequest) request);
            break;
        case "CreateGlobalTable":
            setContextParams(params, (CreateGlobalTableRequest) request);
            break;
        case "CreateTable":
            setContextParams(params, (CreateTableRequest) request);
            break;
        case "DeleteBackup":
            setContextParams(params, (DeleteBackupRequest) request);
            break;
        case "DeleteItem":
            setContextParams(params, (DeleteItemRequest) request);
            break;
        case "DeleteResourcePolicy":
            setContextParams(params, (DeleteResourcePolicyRequest) request);
            break;
        case "DeleteTable":
            setContextParams(params, (DeleteTableRequest) request);
            break;
        case "DescribeBackup":
            setContextParams(params, (DescribeBackupRequest) request);
            break;
        case "DescribeContinuousBackups":
            setContextParams(params, (DescribeContinuousBackupsRequest) request);
            break;
        case "DescribeContributorInsights":
            setContextParams(params, (DescribeContributorInsightsRequest) request);
            break;
        case "DescribeExport":
            setContextParams(params, (DescribeExportRequest) request);
            break;
        case "DescribeGlobalTable":
            setContextParams(params, (DescribeGlobalTableRequest) request);
            break;
        case "DescribeGlobalTableSettings":
            setContextParams(params, (DescribeGlobalTableSettingsRequest) request);
            break;
        case "DescribeImport":
            setContextParams(params, (DescribeImportRequest) request);
            break;
        case "DescribeKinesisStreamingDestination":
            setContextParams(params, (DescribeKinesisStreamingDestinationRequest) request);
            break;
        case "DescribeTable":
            setContextParams(params, (DescribeTableRequest) request);
            break;
        case "DescribeTableReplicaAutoScaling":
            setContextParams(params, (DescribeTableReplicaAutoScalingRequest) request);
            break;
        case "DescribeTimeToLive":
            setContextParams(params, (DescribeTimeToLiveRequest) request);
            break;
        case "DisableKinesisStreamingDestination":
            setContextParams(params, (DisableKinesisStreamingDestinationRequest) request);
            break;
        case "EnableKinesisStreamingDestination":
            setContextParams(params, (EnableKinesisStreamingDestinationRequest) request);
            break;
        case "ExportTableToPointInTime":
            setContextParams(params, (ExportTableToPointInTimeRequest) request);
            break;
        case "GetItem":
            setContextParams(params, (GetItemRequest) request);
            break;
        case "GetResourcePolicy":
            setContextParams(params, (GetResourcePolicyRequest) request);
            break;
        case "ListBackups":
            setContextParams(params, (ListBackupsRequest) request);
            break;
        case "ListContributorInsights":
            setContextParams(params, (ListContributorInsightsRequest) request);
            break;
        case "ListExports":
            setContextParams(params, (ListExportsRequest) request);
            break;
        case "ListImports":
            setContextParams(params, (ListImportsRequest) request);
            break;
        case "ListTagsOfResource":
            setContextParams(params, (ListTagsOfResourceRequest) request);
            break;
        case "PutItem":
            setContextParams(params, (PutItemRequest) request);
            break;
        case "PutResourcePolicy":
            setContextParams(params, (PutResourcePolicyRequest) request);
            break;
        case "Query":
            setContextParams(params, (QueryRequest) request);
            break;
        case "RestoreTableFromBackup":
            setContextParams(params, (RestoreTableFromBackupRequest) request);
            break;
        case "RestoreTableToPointInTime":
            setContextParams(params, (RestoreTableToPointInTimeRequest) request);
            break;
        case "Scan":
            setContextParams(params, (ScanRequest) request);
            break;
        case "TagResource":
            setContextParams(params, (TagResourceRequest) request);
            break;
        case "UntagResource":
            setContextParams(params, (UntagResourceRequest) request);
            break;
        case "UpdateContinuousBackups":
            setContextParams(params, (UpdateContinuousBackupsRequest) request);
            break;
        case "UpdateContributorInsights":
            setContextParams(params, (UpdateContributorInsightsRequest) request);
            break;
        case "UpdateGlobalTable":
            setContextParams(params, (UpdateGlobalTableRequest) request);
            break;
        case "UpdateGlobalTableSettings":
            setContextParams(params, (UpdateGlobalTableSettingsRequest) request);
            break;
        case "UpdateItem":
            setContextParams(params, (UpdateItemRequest) request);
            break;
        case "UpdateKinesisStreamingDestination":
            setContextParams(params, (UpdateKinesisStreamingDestinationRequest) request);
            break;
        case "UpdateTable":
            setContextParams(params, (UpdateTableRequest) request);
            break;
        case "UpdateTableReplicaAutoScaling":
            setContextParams(params, (UpdateTableReplicaAutoScalingRequest) request);
            break;
        case "UpdateTimeToLive":
            setContextParams(params, (UpdateTimeToLiveRequest) request);
            break;
        default:
            break;
        }
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, CreateBackupRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, CreateGlobalTableRequest request) {
        params.resourceArn(request.globalTableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, CreateTableRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DeleteBackupRequest request) {
        params.resourceArn(request.backupArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DeleteItemRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DeleteResourcePolicyRequest request) {
        params.resourceArn(request.resourceArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DeleteTableRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DescribeBackupRequest request) {
        params.resourceArn(request.backupArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DescribeContinuousBackupsRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DescribeContributorInsightsRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DescribeExportRequest request) {
        params.resourceArn(request.exportArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DescribeGlobalTableRequest request) {
        params.resourceArn(request.globalTableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DescribeGlobalTableSettingsRequest request) {
        params.resourceArn(request.globalTableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DescribeImportRequest request) {
        params.resourceArn(request.importArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DescribeKinesisStreamingDestinationRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DescribeTableRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DescribeTableReplicaAutoScalingRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DescribeTimeToLiveRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, DisableKinesisStreamingDestinationRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, EnableKinesisStreamingDestinationRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, ExportTableToPointInTimeRequest request) {
        params.resourceArn(request.tableArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, GetItemRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, GetResourcePolicyRequest request) {
        params.resourceArn(request.resourceArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, ListBackupsRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, ListContributorInsightsRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, ListExportsRequest request) {
        params.resourceArn(request.tableArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, ListImportsRequest request) {
        params.resourceArn(request.tableArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, ListTagsOfResourceRequest request) {
        params.resourceArn(request.resourceArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, PutItemRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, PutResourcePolicyRequest request) {
        params.resourceArn(request.resourceArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, QueryRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, RestoreTableFromBackupRequest request) {
        params.resourceArn(request.targetTableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, RestoreTableToPointInTimeRequest request) {
        params.resourceArn(request.targetTableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, ScanRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, TagResourceRequest request) {
        params.resourceArn(request.resourceArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, UntagResourceRequest request) {
        params.resourceArn(request.resourceArn());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, UpdateContinuousBackupsRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, UpdateContributorInsightsRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, UpdateGlobalTableRequest request) {
        params.resourceArn(request.globalTableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, UpdateGlobalTableSettingsRequest request) {
        params.resourceArn(request.globalTableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, UpdateItemRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, UpdateKinesisStreamingDestinationRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, UpdateTableRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, UpdateTableReplicaAutoScalingRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setContextParams(DynamoDbEndpointParams.Builder params, UpdateTimeToLiveRequest request) {
        params.resourceArn(request.tableName());
    }

    private static void setStaticContextParams(DynamoDbEndpointParams.Builder params, String operationName) {
    }

    private <T extends Identity> SelectedAuthScheme<T> authSchemeWithEndpointSignerProperties(
            List<EndpointAuthScheme> endpointAuthSchemes, SelectedAuthScheme<T> selectedAuthScheme) {
        for (EndpointAuthScheme endpointAuthScheme : endpointAuthSchemes) {
            if (!endpointAuthScheme.schemeId().equals(selectedAuthScheme.authSchemeOption().schemeId())) {
                continue;
            }
            AuthSchemeOption.Builder option = selectedAuthScheme.authSchemeOption().toBuilder();
            if (endpointAuthScheme instanceof SigV4AuthScheme) {
                SigV4AuthScheme v4AuthScheme = (SigV4AuthScheme) endpointAuthScheme;
                if (v4AuthScheme.isDisableDoubleEncodingSet()) {
                    option.putSignerProperty(AwsV4HttpSigner.DOUBLE_URL_ENCODE, !v4AuthScheme.disableDoubleEncoding());
                }
                if (v4AuthScheme.signingRegion() != null) {
                    option.putSignerProperty(AwsV4HttpSigner.REGION_NAME, v4AuthScheme.signingRegion());
                }
                if (v4AuthScheme.signingName() != null) {
                    option.putSignerProperty(AwsV4HttpSigner.SERVICE_SIGNING_NAME, v4AuthScheme.signingName());
                }
                return new SelectedAuthScheme<>(selectedAuthScheme.identity(), selectedAuthScheme.signer(), option.build());
            }
            if (endpointAuthScheme instanceof SigV4aAuthScheme) {
                SigV4aAuthScheme v4aAuthScheme = (SigV4aAuthScheme) endpointAuthScheme;
                if (v4aAuthScheme.isDisableDoubleEncodingSet()) {
                    option.putSignerProperty(AwsV4aHttpSigner.DOUBLE_URL_ENCODE, !v4aAuthScheme.disableDoubleEncoding());
                }
                if (!CollectionUtils.isNullOrEmpty(v4aAuthScheme.signingRegionSet())) {
                    RegionSet regionSet = RegionSet.create(v4aAuthScheme.signingRegionSet());
                    option.putSignerProperty(AwsV4aHttpSigner.REGION_SET, regionSet);
                }
                if (v4aAuthScheme.signingName() != null) {
                    option.putSignerProperty(AwsV4aHttpSigner.SERVICE_SIGNING_NAME, v4aAuthScheme.signingName());
                }
                return new SelectedAuthScheme<>(selectedAuthScheme.identity(), selectedAuthScheme.signer(), option.build());
            }
            throw new IllegalArgumentException("Endpoint auth scheme '" + endpointAuthScheme.name()
                    + "' cannot be mapped to the SDK auth scheme. Was it declared in the service's model?");
        }
        return selectedAuthScheme;
    }

    private static void setOperationContextParams(DynamoDbEndpointParams.Builder params, String operationName, SdkRequest request) {
        switch (operationName) {
        case "BatchGetItem":
            setOperationContextParams(params, (BatchGetItemRequest) request);
            break;
        case "BatchWriteItem":
            setOperationContextParams(params, (BatchWriteItemRequest) request);
            break;
        case "ImportTable":
            setOperationContextParams(params, (ImportTableRequest) request);
            break;
        case "TransactGetItems":
            setOperationContextParams(params, (TransactGetItemsRequest) request);
            break;
        default:
            break;
        }
    }

    private static void setOperationContextParams(DynamoDbEndpointParams.Builder params, BatchGetItemRequest request) {
        JmesPathRuntime.Value input = new JmesPathRuntime.Value(request);
        params.resourceArnList(input.field("RequestItems").keys().stringValues());
    }

    private static void setOperationContextParams(DynamoDbEndpointParams.Builder params, BatchWriteItemRequest request) {
        JmesPathRuntime.Value input = new JmesPathRuntime.Value(request);
        params.resourceArnList(input.field("RequestItems").keys().stringValues());
    }

    private static void setOperationContextParams(DynamoDbEndpointParams.Builder params, ImportTableRequest request) {
        JmesPathRuntime.Value input = new JmesPathRuntime.Value(request);
        params.resourceArn(input.field("TableCreationParameters").field("TableName").stringValue());
    }

    private static void setOperationContextParams(DynamoDbEndpointParams.Builder params, TransactGetItemsRequest request) {
        JmesPathRuntime.Value input = new JmesPathRuntime.Value(request);
        params.resourceArnList(input.field("TransactItems").wildcard().field("Get").field("TableName").stringValues());
    }

    private static Optional<String> hostPrefix(String operationName, SdkRequest request) {
        return Optional.empty();
    }

    private static String resolveAndRecordAccountIdFromIdentity(ExecutionAttributes executionAttributes) {
        String accountId = accountIdFromIdentity(executionAttributes
                .getAttribute(SdkInternalExecutionAttribute.SELECTED_AUTH_SCHEME));
        executionAttributes.getAttribute(SdkInternalExecutionAttribute.BUSINESS_METRICS).addMetric(
                BusinessMetricFeatureId.RESOLVED_ACCOUNT_ID.value());
        return accountId;
    }

    private static <T extends Identity> String accountIdFromIdentity(SelectedAuthScheme<T> selectedAuthScheme) {
        T identity = CompletableFutureUtils.joinLikeSync(selectedAuthScheme.identity());
        String accountId = null;
        if (identity instanceof AwsCredentialsIdentity) {
            accountId = ((AwsCredentialsIdentity) identity).accountId().orElse(null);
        }
        return accountId;
    }

    private static String recordAccountIdEndpointMode(ExecutionAttributes executionAttributes) {
        AccountIdEndpointMode mode = executionAttributes.getAttribute(AwsExecutionAttribute.AWS_AUTH_ACCOUNT_ID_ENDPOINT_MODE);
        BusinessMetricsUtils.resolveAccountIdEndpointModeMetric(mode).ifPresent(
                m -> executionAttributes.getAttribute(SdkInternalExecutionAttribute.BUSINESS_METRICS).addMetric(m));
        return mode.name().toLowerCase();
    }
}
