/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the properties of a global secondary index.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GlobalSecondaryIndexDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<GlobalSecondaryIndexDescription.Builder, GlobalSecondaryIndexDescription> {
    private static final SdkField<String> INDEX_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GlobalSecondaryIndexDescription::indexName)).setter(setter(Builder::indexName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IndexName").build()).build();

    private static final SdkField<List<KeySchemaElement>> KEY_SCHEMA_FIELD = SdkField
            .<List<KeySchemaElement>> builder(MarshallingType.LIST)
            .getter(getter(GlobalSecondaryIndexDescription::keySchema))
            .setter(setter(Builder::keySchema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeySchema").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeySchemaElement> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeySchemaElement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Projection> PROJECTION_FIELD = SdkField.<Projection> builder(MarshallingType.SDK_POJO)
            .getter(getter(GlobalSecondaryIndexDescription::projection)).setter(setter(Builder::projection))
            .constructor(Projection::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Projection").build()).build();

    private static final SdkField<String> INDEX_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GlobalSecondaryIndexDescription::indexStatusAsString)).setter(setter(Builder::indexStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IndexStatus").build()).build();

    private static final SdkField<Boolean> BACKFILLING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(GlobalSecondaryIndexDescription::backfilling)).setter(setter(Builder::backfilling))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Backfilling").build()).build();

    private static final SdkField<ProvisionedThroughputDescription> PROVISIONED_THROUGHPUT_FIELD = SdkField
            .<ProvisionedThroughputDescription> builder(MarshallingType.SDK_POJO)
            .getter(getter(GlobalSecondaryIndexDescription::provisionedThroughput))
            .setter(setter(Builder::provisionedThroughput)).constructor(ProvisionedThroughputDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProvisionedThroughput").build())
            .build();

    private static final SdkField<Long> INDEX_SIZE_BYTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(GlobalSecondaryIndexDescription::indexSizeBytes)).setter(setter(Builder::indexSizeBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IndexSizeBytes").build()).build();

    private static final SdkField<Long> ITEM_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(GlobalSecondaryIndexDescription::itemCount)).setter(setter(Builder::itemCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ItemCount").build()).build();

    private static final SdkField<String> INDEX_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GlobalSecondaryIndexDescription::indexArn)).setter(setter(Builder::indexArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IndexArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INDEX_NAME_FIELD,
            KEY_SCHEMA_FIELD, PROJECTION_FIELD, INDEX_STATUS_FIELD, BACKFILLING_FIELD, PROVISIONED_THROUGHPUT_FIELD,
            INDEX_SIZE_BYTES_FIELD, ITEM_COUNT_FIELD, INDEX_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String indexName;

    private final List<KeySchemaElement> keySchema;

    private final Projection projection;

    private final String indexStatus;

    private final Boolean backfilling;

    private final ProvisionedThroughputDescription provisionedThroughput;

    private final Long indexSizeBytes;

    private final Long itemCount;

    private final String indexArn;

    private GlobalSecondaryIndexDescription(BuilderImpl builder) {
        this.indexName = builder.indexName;
        this.keySchema = builder.keySchema;
        this.projection = builder.projection;
        this.indexStatus = builder.indexStatus;
        this.backfilling = builder.backfilling;
        this.provisionedThroughput = builder.provisionedThroughput;
        this.indexSizeBytes = builder.indexSizeBytes;
        this.itemCount = builder.itemCount;
        this.indexArn = builder.indexArn;
    }

    /**
     * <p>
     * The name of the global secondary index.
     * </p>
     * 
     * @return The name of the global secondary index.
     */
    public String indexName() {
        return indexName;
    }

    /**
     * Returns true if the KeySchema property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasKeySchema() {
        return keySchema != null && !(keySchema instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The complete key schema for a global secondary index, which consists of one or more pairs of attribute names and
     * key types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HASH</code> - partition key
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RANGE</code> - sort key
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives from
     * DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based on their
     * partition key values.
     * </p>
     * <p>
     * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from the
     * way DynamoDB stores items with the same partition key physically close together, in sorted order by the sort key
     * value.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasKeySchema()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The complete key schema for a global secondary index, which consists of one or more pairs of attribute
     *         names and key types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HASH</code> - partition key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RANGE</code> - sort key
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
     *         derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
     *         partitions, based on their partition key values.
     *         </p>
     *         <p>
     *         The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives
     *         from the way DynamoDB stores items with the same partition key physically close together, in sorted order
     *         by the sort key value.
     *         </p>
     */
    public List<KeySchemaElement> keySchema() {
        return keySchema;
    }

    /**
     * <p>
     * Represents attributes that are copied (projected) from the table into the global secondary index. These are in
     * addition to the primary key attributes and index key attributes, which are automatically projected.
     * </p>
     * 
     * @return Represents attributes that are copied (projected) from the table into the global secondary index. These
     *         are in addition to the primary key attributes and index key attributes, which are automatically
     *         projected.
     */
    public Projection projection() {
        return projection;
    }

    /**
     * <p>
     * The current state of the global secondary index:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The index is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The index is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The index is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The index is ready for use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #indexStatus} will
     * return {@link IndexStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #indexStatusAsString}.
     * </p>
     * 
     * @return The current state of the global secondary index:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The index is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The index is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The index is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The index is ready for use.
     *         </p>
     *         </li>
     * @see IndexStatus
     */
    public IndexStatus indexStatus() {
        return IndexStatus.fromValue(indexStatus);
    }

    /**
     * <p>
     * The current state of the global secondary index:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The index is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The index is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The index is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The index is ready for use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #indexStatus} will
     * return {@link IndexStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #indexStatusAsString}.
     * </p>
     * 
     * @return The current state of the global secondary index:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The index is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The index is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The index is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The index is ready for use.
     *         </p>
     *         </li>
     * @see IndexStatus
     */
    public String indexStatusAsString() {
        return indexStatus;
    }

    /**
     * <p>
     * Indicates whether the index is currently backfilling. <i>Backfilling</i> is the process of reading items from the
     * table and determining whether they can be added to the index. (Not all items will qualify: For example, a
     * partition key cannot have any duplicate values.) If an item can be added to the index, DynamoDB will do so. After
     * all items have been processed, the backfilling operation is complete and <code>Backfilling</code> is false.
     * </p>
     * <p>
     * You can delete an index that is being created during the <code>Backfilling</code> phase when
     * <code>IndexStatus</code> is set to CREATING and <code>Backfilling</code> is true. You can't delete the index that
     * is being created when <code>IndexStatus</code> is set to CREATING and <code>Backfilling</code> is false.
     * </p>
     * <note>
     * <p>
     * For indexes that were created during a <code>CreateTable</code> operation, the <code>Backfilling</code> attribute
     * does not appear in the <code>DescribeTable</code> output.
     * </p>
     * </note>
     * 
     * @return Indicates whether the index is currently backfilling. <i>Backfilling</i> is the process of reading items
     *         from the table and determining whether they can be added to the index. (Not all items will qualify: For
     *         example, a partition key cannot have any duplicate values.) If an item can be added to the index,
     *         DynamoDB will do so. After all items have been processed, the backfilling operation is complete and
     *         <code>Backfilling</code> is false.</p>
     *         <p>
     *         You can delete an index that is being created during the <code>Backfilling</code> phase when
     *         <code>IndexStatus</code> is set to CREATING and <code>Backfilling</code> is true. You can't delete the
     *         index that is being created when <code>IndexStatus</code> is set to CREATING and <code>Backfilling</code>
     *         is false.
     *         </p>
     *         <note>
     *         <p>
     *         For indexes that were created during a <code>CreateTable</code> operation, the <code>Backfilling</code>
     *         attribute does not appear in the <code>DescribeTable</code> output.
     *         </p>
     */
    public Boolean backfilling() {
        return backfilling;
    }

    /**
     * <p>
     * Represents the provisioned throughput settings for the specified global secondary index.
     * </p>
     * <p>
     * For current minimum and maximum provisioned throughput values, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Limits</a> in the <i>Amazon
     * DynamoDB Developer Guide</i>.
     * </p>
     * 
     * @return Represents the provisioned throughput settings for the specified global secondary index.</p>
     *         <p>
     *         For current minimum and maximum provisioned throughput values, see <a
     *         href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Limits</a> in the
     *         <i>Amazon DynamoDB Developer Guide</i>.
     */
    public ProvisionedThroughputDescription provisionedThroughput() {
        return provisionedThroughput;
    }

    /**
     * <p>
     * The total size of the specified index, in bytes. DynamoDB updates this value approximately every six hours.
     * Recent changes might not be reflected in this value.
     * </p>
     * 
     * @return The total size of the specified index, in bytes. DynamoDB updates this value approximately every six
     *         hours. Recent changes might not be reflected in this value.
     */
    public Long indexSizeBytes() {
        return indexSizeBytes;
    }

    /**
     * <p>
     * The number of items in the specified index. DynamoDB updates this value approximately every six hours. Recent
     * changes might not be reflected in this value.
     * </p>
     * 
     * @return The number of items in the specified index. DynamoDB updates this value approximately every six hours.
     *         Recent changes might not be reflected in this value.
     */
    public Long itemCount() {
        return itemCount;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that uniquely identifies the index.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that uniquely identifies the index.
     */
    public String indexArn() {
        return indexArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(indexName());
        hashCode = 31 * hashCode + Objects.hashCode(keySchema());
        hashCode = 31 * hashCode + Objects.hashCode(projection());
        hashCode = 31 * hashCode + Objects.hashCode(indexStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(backfilling());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedThroughput());
        hashCode = 31 * hashCode + Objects.hashCode(indexSizeBytes());
        hashCode = 31 * hashCode + Objects.hashCode(itemCount());
        hashCode = 31 * hashCode + Objects.hashCode(indexArn());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GlobalSecondaryIndexDescription)) {
            return false;
        }
        GlobalSecondaryIndexDescription other = (GlobalSecondaryIndexDescription) obj;
        return Objects.equals(indexName(), other.indexName()) && Objects.equals(keySchema(), other.keySchema())
                && Objects.equals(projection(), other.projection())
                && Objects.equals(indexStatusAsString(), other.indexStatusAsString())
                && Objects.equals(backfilling(), other.backfilling())
                && Objects.equals(provisionedThroughput(), other.provisionedThroughput())
                && Objects.equals(indexSizeBytes(), other.indexSizeBytes()) && Objects.equals(itemCount(), other.itemCount())
                && Objects.equals(indexArn(), other.indexArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GlobalSecondaryIndexDescription").add("IndexName", indexName()).add("KeySchema", keySchema())
                .add("Projection", projection()).add("IndexStatus", indexStatusAsString()).add("Backfilling", backfilling())
                .add("ProvisionedThroughput", provisionedThroughput()).add("IndexSizeBytes", indexSizeBytes())
                .add("ItemCount", itemCount()).add("IndexArn", indexArn()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IndexName":
            return Optional.ofNullable(clazz.cast(indexName()));
        case "KeySchema":
            return Optional.ofNullable(clazz.cast(keySchema()));
        case "Projection":
            return Optional.ofNullable(clazz.cast(projection()));
        case "IndexStatus":
            return Optional.ofNullable(clazz.cast(indexStatusAsString()));
        case "Backfilling":
            return Optional.ofNullable(clazz.cast(backfilling()));
        case "ProvisionedThroughput":
            return Optional.ofNullable(clazz.cast(provisionedThroughput()));
        case "IndexSizeBytes":
            return Optional.ofNullable(clazz.cast(indexSizeBytes()));
        case "ItemCount":
            return Optional.ofNullable(clazz.cast(itemCount()));
        case "IndexArn":
            return Optional.ofNullable(clazz.cast(indexArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GlobalSecondaryIndexDescription, T> g) {
        return obj -> g.apply((GlobalSecondaryIndexDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GlobalSecondaryIndexDescription> {
        /**
         * <p>
         * The name of the global secondary index.
         * </p>
         * 
         * @param indexName
         *        The name of the global secondary index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder indexName(String indexName);

        /**
         * <p>
         * The complete key schema for a global secondary index, which consists of one or more pairs of attribute names
         * and key types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based
         * on their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note>
         * 
         * @param keySchema
         *        The complete key schema for a global secondary index, which consists of one or more pairs of attribute
         *        names and key types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(Collection<KeySchemaElement> keySchema);

        /**
         * <p>
         * The complete key schema for a global secondary index, which consists of one or more pairs of attribute names
         * and key types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based
         * on their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note>
         * 
         * @param keySchema
         *        The complete key schema for a global secondary index, which consists of one or more pairs of attribute
         *        names and key types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(KeySchemaElement... keySchema);

        /**
         * <p>
         * The complete key schema for a global secondary index, which consists of one or more pairs of attribute names
         * and key types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based
         * on their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<KeySchemaElement>.Builder} avoiding
         * the need to create one manually via {@link List<KeySchemaElement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeySchemaElement>.Builder#build()} is called immediately and
         * its result is passed to {@link #keySchema(List<KeySchemaElement>)}.
         * 
         * @param keySchema
         *        a consumer that will call methods on {@link List<KeySchemaElement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keySchema(List<KeySchemaElement>)
         */
        Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema);

        /**
         * <p>
         * Represents attributes that are copied (projected) from the table into the global secondary index. These are
         * in addition to the primary key attributes and index key attributes, which are automatically projected.
         * </p>
         * 
         * @param projection
         *        Represents attributes that are copied (projected) from the table into the global secondary index.
         *        These are in addition to the primary key attributes and index key attributes, which are automatically
         *        projected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projection(Projection projection);

        /**
         * <p>
         * Represents attributes that are copied (projected) from the table into the global secondary index. These are
         * in addition to the primary key attributes and index key attributes, which are automatically projected.
         * </p>
         * This is a convenience that creates an instance of the {@link Projection.Builder} avoiding the need to create
         * one manually via {@link Projection#builder()}.
         *
         * When the {@link Consumer} completes, {@link Projection.Builder#build()} is called immediately and its result
         * is passed to {@link #projection(Projection)}.
         * 
         * @param projection
         *        a consumer that will call methods on {@link Projection.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #projection(Projection)
         */
        default Builder projection(Consumer<Projection.Builder> projection) {
            return projection(Projection.builder().applyMutation(projection).build());
        }

        /**
         * <p>
         * The current state of the global secondary index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The index is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The index is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The index is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The index is ready for use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param indexStatus
         *        The current state of the global secondary index:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The index is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The index is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The index is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The index is ready for use.
         *        </p>
         *        </li>
         * @see IndexStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IndexStatus
         */
        Builder indexStatus(String indexStatus);

        /**
         * <p>
         * The current state of the global secondary index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The index is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The index is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The index is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The index is ready for use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param indexStatus
         *        The current state of the global secondary index:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The index is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The index is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The index is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The index is ready for use.
         *        </p>
         *        </li>
         * @see IndexStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IndexStatus
         */
        Builder indexStatus(IndexStatus indexStatus);

        /**
         * <p>
         * Indicates whether the index is currently backfilling. <i>Backfilling</i> is the process of reading items from
         * the table and determining whether they can be added to the index. (Not all items will qualify: For example, a
         * partition key cannot have any duplicate values.) If an item can be added to the index, DynamoDB will do so.
         * After all items have been processed, the backfilling operation is complete and <code>Backfilling</code> is
         * false.
         * </p>
         * <p>
         * You can delete an index that is being created during the <code>Backfilling</code> phase when
         * <code>IndexStatus</code> is set to CREATING and <code>Backfilling</code> is true. You can't delete the index
         * that is being created when <code>IndexStatus</code> is set to CREATING and <code>Backfilling</code> is false.
         * </p>
         * <note>
         * <p>
         * For indexes that were created during a <code>CreateTable</code> operation, the <code>Backfilling</code>
         * attribute does not appear in the <code>DescribeTable</code> output.
         * </p>
         * </note>
         * 
         * @param backfilling
         *        Indicates whether the index is currently backfilling. <i>Backfilling</i> is the process of reading
         *        items from the table and determining whether they can be added to the index. (Not all items will
         *        qualify: For example, a partition key cannot have any duplicate values.) If an item can be added to
         *        the index, DynamoDB will do so. After all items have been processed, the backfilling operation is
         *        complete and <code>Backfilling</code> is false.</p>
         *        <p>
         *        You can delete an index that is being created during the <code>Backfilling</code> phase when
         *        <code>IndexStatus</code> is set to CREATING and <code>Backfilling</code> is true. You can't delete the
         *        index that is being created when <code>IndexStatus</code> is set to CREATING and
         *        <code>Backfilling</code> is false.
         *        </p>
         *        <note>
         *        <p>
         *        For indexes that were created during a <code>CreateTable</code> operation, the
         *        <code>Backfilling</code> attribute does not appear in the <code>DescribeTable</code> output.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backfilling(Boolean backfilling);

        /**
         * <p>
         * Represents the provisioned throughput settings for the specified global secondary index.
         * </p>
         * <p>
         * For current minimum and maximum provisioned throughput values, see <a
         * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Limits</a> in the
         * <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param provisionedThroughput
         *        Represents the provisioned throughput settings for the specified global secondary index.</p>
         *        <p>
         *        For current minimum and maximum provisioned throughput values, see <a
         *        href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Limits</a> in the
         *        <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedThroughput(ProvisionedThroughputDescription provisionedThroughput);

        /**
         * <p>
         * Represents the provisioned throughput settings for the specified global secondary index.
         * </p>
         * <p>
         * For current minimum and maximum provisioned throughput values, see <a
         * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Limits</a> in the
         * <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link ProvisionedThroughputDescription.Builder}
         * avoiding the need to create one manually via {@link ProvisionedThroughputDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProvisionedThroughputDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #provisionedThroughput(ProvisionedThroughputDescription)}.
         * 
         * @param provisionedThroughput
         *        a consumer that will call methods on {@link ProvisionedThroughputDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #provisionedThroughput(ProvisionedThroughputDescription)
         */
        default Builder provisionedThroughput(Consumer<ProvisionedThroughputDescription.Builder> provisionedThroughput) {
            return provisionedThroughput(ProvisionedThroughputDescription.builder().applyMutation(provisionedThroughput).build());
        }

        /**
         * <p>
         * The total size of the specified index, in bytes. DynamoDB updates this value approximately every six hours.
         * Recent changes might not be reflected in this value.
         * </p>
         * 
         * @param indexSizeBytes
         *        The total size of the specified index, in bytes. DynamoDB updates this value approximately every six
         *        hours. Recent changes might not be reflected in this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder indexSizeBytes(Long indexSizeBytes);

        /**
         * <p>
         * The number of items in the specified index. DynamoDB updates this value approximately every six hours. Recent
         * changes might not be reflected in this value.
         * </p>
         * 
         * @param itemCount
         *        The number of items in the specified index. DynamoDB updates this value approximately every six hours.
         *        Recent changes might not be reflected in this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemCount(Long itemCount);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that uniquely identifies the index.
         * </p>
         * 
         * @param indexArn
         *        The Amazon Resource Name (ARN) that uniquely identifies the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder indexArn(String indexArn);
    }

    static final class BuilderImpl implements Builder {
        private String indexName;

        private List<KeySchemaElement> keySchema = DefaultSdkAutoConstructList.getInstance();

        private Projection projection;

        private String indexStatus;

        private Boolean backfilling;

        private ProvisionedThroughputDescription provisionedThroughput;

        private Long indexSizeBytes;

        private Long itemCount;

        private String indexArn;

        private BuilderImpl() {
        }

        private BuilderImpl(GlobalSecondaryIndexDescription model) {
            indexName(model.indexName);
            keySchema(model.keySchema);
            projection(model.projection);
            indexStatus(model.indexStatus);
            backfilling(model.backfilling);
            provisionedThroughput(model.provisionedThroughput);
            indexSizeBytes(model.indexSizeBytes);
            itemCount(model.itemCount);
            indexArn(model.indexArn);
        }

        public final String getIndexName() {
            return indexName;
        }

        @Override
        public final Builder indexName(String indexName) {
            this.indexName = indexName;
            return this;
        }

        public final void setIndexName(String indexName) {
            this.indexName = indexName;
        }

        public final Collection<KeySchemaElement.Builder> getKeySchema() {
            return keySchema != null ? keySchema.stream().map(KeySchemaElement::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder keySchema(Collection<KeySchemaElement> keySchema) {
            this.keySchema = KeySchemaCopier.copy(keySchema);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keySchema(KeySchemaElement... keySchema) {
            keySchema(Arrays.asList(keySchema));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema) {
            keySchema(Stream.of(keySchema).map(c -> KeySchemaElement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setKeySchema(Collection<KeySchemaElement.BuilderImpl> keySchema) {
            this.keySchema = KeySchemaCopier.copyFromBuilder(keySchema);
        }

        public final Projection.Builder getProjection() {
            return projection != null ? projection.toBuilder() : null;
        }

        @Override
        public final Builder projection(Projection projection) {
            this.projection = projection;
            return this;
        }

        public final void setProjection(Projection.BuilderImpl projection) {
            this.projection = projection != null ? projection.build() : null;
        }

        public final String getIndexStatus() {
            return indexStatus;
        }

        @Override
        public final Builder indexStatus(String indexStatus) {
            this.indexStatus = indexStatus;
            return this;
        }

        @Override
        public final Builder indexStatus(IndexStatus indexStatus) {
            this.indexStatus(indexStatus == null ? null : indexStatus.toString());
            return this;
        }

        public final void setIndexStatus(String indexStatus) {
            this.indexStatus = indexStatus;
        }

        public final Boolean getBackfilling() {
            return backfilling;
        }

        @Override
        public final Builder backfilling(Boolean backfilling) {
            this.backfilling = backfilling;
            return this;
        }

        public final void setBackfilling(Boolean backfilling) {
            this.backfilling = backfilling;
        }

        public final ProvisionedThroughputDescription.Builder getProvisionedThroughput() {
            return provisionedThroughput != null ? provisionedThroughput.toBuilder() : null;
        }

        @Override
        public final Builder provisionedThroughput(ProvisionedThroughputDescription provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput;
            return this;
        }

        public final void setProvisionedThroughput(ProvisionedThroughputDescription.BuilderImpl provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput != null ? provisionedThroughput.build() : null;
        }

        public final Long getIndexSizeBytes() {
            return indexSizeBytes;
        }

        @Override
        public final Builder indexSizeBytes(Long indexSizeBytes) {
            this.indexSizeBytes = indexSizeBytes;
            return this;
        }

        public final void setIndexSizeBytes(Long indexSizeBytes) {
            this.indexSizeBytes = indexSizeBytes;
        }

        public final Long getItemCount() {
            return itemCount;
        }

        @Override
        public final Builder itemCount(Long itemCount) {
            this.itemCount = itemCount;
            return this;
        }

        public final void setItemCount(Long itemCount) {
            this.itemCount = itemCount;
        }

        public final String getIndexArn() {
            return indexArn;
        }

        @Override
        public final Builder indexArn(String indexArn) {
            this.indexArn = indexArn;
            return this;
        }

        public final void setIndexArn(String indexArn) {
            this.indexArn = indexArn;
        }

        @Override
        public GlobalSecondaryIndexDescription build() {
            return new GlobalSecondaryIndexDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
