/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.runtime.StandardMemberCopier;
import software.amazon.awssdk.services.dynamodb.transform.AttributeValueMarshaller;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the data for an attribute.
 * </p>
 * <p>
 * Each attribute value is described as a name-value pair. The name is the data type, and the value is the data itself.
 * </p>
 * <p>
 * For more information, see <a href=
 * "http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
 * >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class AttributeValue implements StructuredPojo, ToCopyableBuilder<AttributeValue.Builder, AttributeValue> {
    private final String s;

    private final String n;

    private final ByteBuffer b;

    private final List<String> ss;

    private final List<String> ns;

    private final List<ByteBuffer> bs;

    private final Map<String, AttributeValue> m;

    private final List<AttributeValue> l;

    private final Boolean bool;

    private final Boolean nul;

    private AttributeValue(BuilderImpl builder) {
        this.s = builder.s;
        this.n = builder.n;
        this.b = builder.b;
        this.ss = builder.ss;
        this.ns = builder.ns;
        this.bs = builder.bs;
        this.m = builder.m;
        this.l = builder.l;
        this.bool = builder.bool;
        this.nul = builder.nul;
    }

    /**
     * <p>
     * An attribute of type String. For example:
     * </p>
     * <p>
     * <code>"S": "Hello"</code>
     * </p>
     * 
     * @return An attribute of type String. For example:</p>
     *         <p>
     *         <code>"S": "Hello"</code>
     */
    public String s() {
        return s;
    }

    /**
     * <p>
     * An attribute of type Number. For example:
     * </p>
     * <p>
     * <code>"N": "123.45"</code>
     * </p>
     * <p>
     * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
     * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     * </p>
     * 
     * @return An attribute of type Number. For example:</p>
     *         <p>
     *         <code>"N": "123.45"</code>
     *         </p>
     *         <p>
     *         Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages
     *         and libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    public String n() {
        return n;
    }

    /**
     * <p>
     * An attribute of type Binary. For example:
     * </p>
     * <p>
     * <code>"B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"</code>
     * </p>
     * <p>
     * This method will return a new read-only {@code ByteBuffer} each time it is invoked.
     * </p>
     * 
     * @return An attribute of type Binary. For example:</p>
     *         <p>
     *         <code>"B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"</code>
     */
    public ByteBuffer b() {
        return b == null ? null : b.asReadOnlyBuffer();
    }

    /**
     * <p>
     * An attribute of type String Set. For example:
     * </p>
     * <p>
     * <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An attribute of type String Set. For example:</p>
     *         <p>
     *         <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
     */
    public List<String> ss() {
        return ss;
    }

    /**
     * <p>
     * An attribute of type Number Set. For example:
     * </p>
     * <p>
     * <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
     * </p>
     * <p>
     * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
     * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An attribute of type Number Set. For example:</p>
     *         <p>
     *         <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
     *         </p>
     *         <p>
     *         Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages
     *         and libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    public List<String> ns() {
        return ns;
    }

    /**
     * <p>
     * An attribute of type Binary Set. For example:
     * </p>
     * <p>
     * <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An attribute of type Binary Set. For example:</p>
     *         <p>
     *         <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
     */
    public List<ByteBuffer> bs() {
        return bs;
    }

    /**
     * <p>
     * An attribute of type Map. For example:
     * </p>
     * <p>
     * <code>"M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An attribute of type Map. For example:</p>
     *         <p>
     *         <code>"M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}</code>
     */
    public Map<String, AttributeValue> m() {
        return m;
    }

    /**
     * <p>
     * An attribute of type List. For example:
     * </p>
     * <p>
     * <code>"L": ["Cookies", "Coffee", 3.14159]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An attribute of type List. For example:</p>
     *         <p>
     *         <code>"L": ["Cookies", "Coffee", 3.14159]</code>
     */
    public List<AttributeValue> l() {
        return l;
    }

    /**
     * <p>
     * An attribute of type Boolean. For example:
     * </p>
     * <p>
     * <code>"BOOL": true</code>
     * </p>
     * 
     * @return An attribute of type Boolean. For example:</p>
     *         <p>
     *         <code>"BOOL": true</code>
     */
    public Boolean bool() {
        return bool;
    }

    /**
     * <p>
     * An attribute of type Null. For example:
     * </p>
     * <p>
     * <code>"NULL": true</code>
     * </p>
     * 
     * @return An attribute of type Null. For example:</p>
     *         <p>
     *         <code>"NULL": true</code>
     */
    public Boolean nul() {
        return nul;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s());
        hashCode = 31 * hashCode + Objects.hashCode(n());
        hashCode = 31 * hashCode + Objects.hashCode(b());
        hashCode = 31 * hashCode + Objects.hashCode(ss());
        hashCode = 31 * hashCode + Objects.hashCode(ns());
        hashCode = 31 * hashCode + Objects.hashCode(bs());
        hashCode = 31 * hashCode + Objects.hashCode(m());
        hashCode = 31 * hashCode + Objects.hashCode(l());
        hashCode = 31 * hashCode + Objects.hashCode(bool());
        hashCode = 31 * hashCode + Objects.hashCode(nul());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AttributeValue)) {
            return false;
        }
        AttributeValue other = (AttributeValue) obj;
        return Objects.equals(s(), other.s()) && Objects.equals(n(), other.n()) && Objects.equals(b(), other.b())
                && Objects.equals(ss(), other.ss()) && Objects.equals(ns(), other.ns()) && Objects.equals(bs(), other.bs())
                && Objects.equals(m(), other.m()) && Objects.equals(l(), other.l()) && Objects.equals(bool(), other.bool())
                && Objects.equals(nul(), other.nul());
    }

    @Override
    public String toString() {
        return ToString.builder("AttributeValue").add("S", s()).add("N", n()).add("B", b()).add("SS", ss()).add("NS", ns())
                .add("BS", bs()).add("M", m()).add("L", l()).add("BOOL", bool()).add("NUL", nul()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S":
            return Optional.of(clazz.cast(s()));
        case "N":
            return Optional.of(clazz.cast(n()));
        case "B":
            return Optional.of(clazz.cast(b()));
        case "SS":
            return Optional.of(clazz.cast(ss()));
        case "NS":
            return Optional.of(clazz.cast(ns()));
        case "BS":
            return Optional.of(clazz.cast(bs()));
        case "M":
            return Optional.of(clazz.cast(m()));
        case "L":
            return Optional.of(clazz.cast(l()));
        case "BOOL":
            return Optional.of(clazz.cast(bool()));
        case "NUL":
            return Optional.of(clazz.cast(nul()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AttributeValueMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, AttributeValue> {
        /**
         * <p>
         * An attribute of type String. For example:
         * </p>
         * <p>
         * <code>"S": "Hello"</code>
         * </p>
         * 
         * @param s
         *        An attribute of type String. For example:</p>
         *        <p>
         *        <code>"S": "Hello"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s(String s);

        /**
         * <p>
         * An attribute of type Number. For example:
         * </p>
         * <p>
         * <code>"N": "123.45"</code>
         * </p>
         * <p>
         * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
         * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * </p>
         * 
         * @param n
         *        An attribute of type Number. For example:</p>
         *        <p>
         *        <code>"N": "123.45"</code>
         *        </p>
         *        <p>
         *        Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages
         *        and libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder n(String n);

        /**
         * <p>
         * An attribute of type Binary. For example:
         * </p>
         * <p>
         * <code>"B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"</code>
         * </p>
         * <p>
         * To preserve immutability, the remaining bytes in the provided buffer will be copied into a new buffer when
         * set.
         * </p>
         *
         * @param b
         *        An attribute of type Binary. For example:</p>
         *        <p>
         *        <code>"B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder b(ByteBuffer b);

        /**
         * <p>
         * An attribute of type String Set. For example:
         * </p>
         * <p>
         * <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
         * </p>
         * 
         * @param ss
         *        An attribute of type String Set. For example:</p>
         *        <p>
         *        <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ss(Collection<String> ss);

        /**
         * <p>
         * An attribute of type String Set. For example:
         * </p>
         * <p>
         * <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
         * </p>
         * 
         * @param ss
         *        An attribute of type String Set. For example:</p>
         *        <p>
         *        <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ss(String... ss);

        /**
         * <p>
         * An attribute of type Number Set. For example:
         * </p>
         * <p>
         * <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
         * </p>
         * <p>
         * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
         * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * </p>
         * 
         * @param ns
         *        An attribute of type Number Set. For example:</p>
         *        <p>
         *        <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
         *        </p>
         *        <p>
         *        Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages
         *        and libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ns(Collection<String> ns);

        /**
         * <p>
         * An attribute of type Number Set. For example:
         * </p>
         * <p>
         * <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
         * </p>
         * <p>
         * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
         * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * </p>
         * 
         * @param ns
         *        An attribute of type Number Set. For example:</p>
         *        <p>
         *        <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
         *        </p>
         *        <p>
         *        Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages
         *        and libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ns(String... ns);

        /**
         * <p>
         * An attribute of type Binary Set. For example:
         * </p>
         * <p>
         * <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
         * </p>
         * 
         * @param bs
         *        An attribute of type Binary Set. For example:</p>
         *        <p>
         *        <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bs(Collection<ByteBuffer> bs);

        /**
         * <p>
         * An attribute of type Binary Set. For example:
         * </p>
         * <p>
         * <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
         * </p>
         * 
         * @param bs
         *        An attribute of type Binary Set. For example:</p>
         *        <p>
         *        <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bs(ByteBuffer... bs);

        /**
         * <p>
         * An attribute of type Map. For example:
         * </p>
         * <p>
         * <code>"M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}</code>
         * </p>
         * 
         * @param m
         *        An attribute of type Map. For example:</p>
         *        <p>
         *        <code>"M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder m(Map<String, AttributeValue> m);

        /**
         * <p>
         * An attribute of type List. For example:
         * </p>
         * <p>
         * <code>"L": ["Cookies", "Coffee", 3.14159]</code>
         * </p>
         * 
         * @param l
         *        An attribute of type List. For example:</p>
         *        <p>
         *        <code>"L": ["Cookies", "Coffee", 3.14159]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder l(Collection<AttributeValue> l);

        /**
         * <p>
         * An attribute of type List. For example:
         * </p>
         * <p>
         * <code>"L": ["Cookies", "Coffee", 3.14159]</code>
         * </p>
         * 
         * @param l
         *        An attribute of type List. For example:</p>
         *        <p>
         *        <code>"L": ["Cookies", "Coffee", 3.14159]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder l(AttributeValue... l);

        /**
         * <p>
         * An attribute of type Boolean. For example:
         * </p>
         * <p>
         * <code>"BOOL": true</code>
         * </p>
         * 
         * @param bool
         *        An attribute of type Boolean. For example:</p>
         *        <p>
         *        <code>"BOOL": true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bool(Boolean bool);

        /**
         * <p>
         * An attribute of type Null. For example:
         * </p>
         * <p>
         * <code>"NULL": true</code>
         * </p>
         * 
         * @param nul
         *        An attribute of type Null. For example:</p>
         *        <p>
         *        <code>"NULL": true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nul(Boolean nul);
    }

    static final class BuilderImpl implements Builder {
        private String s;

        private String n;

        private ByteBuffer b;

        private List<String> ss;

        private List<String> ns;

        private List<ByteBuffer> bs;

        private Map<String, AttributeValue> m;

        private List<AttributeValue> l;

        private Boolean bool;

        private Boolean nul;

        private BuilderImpl() {
        }

        private BuilderImpl(AttributeValue model) {
            s(model.s);
            n(model.n);
            b(model.b);
            ss(model.ss);
            ns(model.ns);
            bs(model.bs);
            m(model.m);
            l(model.l);
            bool(model.bool);
            nul(model.nul);
        }

        public final String getS() {
            return s;
        }

        @Override
        public final Builder s(String s) {
            this.s = s;
            return this;
        }

        public final void setS(String s) {
            this.s = s;
        }

        public final String getN() {
            return n;
        }

        @Override
        public final Builder n(String n) {
            this.n = n;
            return this;
        }

        public final void setN(String n) {
            this.n = n;
        }

        public final ByteBuffer getB() {
            return b;
        }

        @Override
        public final Builder b(ByteBuffer b) {
            this.b = StandardMemberCopier.copy(b);
            return this;
        }

        public final void setB(ByteBuffer b) {
            this.b = StandardMemberCopier.copy(b);
        }

        public final Collection<String> getSS() {
            return ss;
        }

        @Override
        public final Builder ss(Collection<String> ss) {
            this.ss = StringSetAttributeValueCopier.copy(ss);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ss(String... ss) {
            ss(Arrays.asList(ss));
            return this;
        }

        public final void setSS(Collection<String> ss) {
            this.ss = StringSetAttributeValueCopier.copy(ss);
        }

        public final Collection<String> getNS() {
            return ns;
        }

        @Override
        public final Builder ns(Collection<String> ns) {
            this.ns = NumberSetAttributeValueCopier.copy(ns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ns(String... ns) {
            ns(Arrays.asList(ns));
            return this;
        }

        public final void setNS(Collection<String> ns) {
            this.ns = NumberSetAttributeValueCopier.copy(ns);
        }

        public final Collection<ByteBuffer> getBS() {
            return bs;
        }

        @Override
        public final Builder bs(Collection<ByteBuffer> bs) {
            this.bs = BinarySetAttributeValueCopier.copy(bs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder bs(ByteBuffer... bs) {
            bs(Arrays.asList(bs));
            return this;
        }

        public final void setBS(Collection<ByteBuffer> bs) {
            this.bs = BinarySetAttributeValueCopier.copy(bs);
        }

        public final Map<String, Builder> getM() {
            return m != null ? CollectionUtils.mapValues(m, AttributeValue::toBuilder) : null;
        }

        @Override
        public final Builder m(Map<String, AttributeValue> m) {
            this.m = MapAttributeValueCopier.copy(m);
            return this;
        }

        public final void setM(Map<String, BuilderImpl> m) {
            this.m = MapAttributeValueCopier.copyFromBuilder(m);
        }

        public final Collection<Builder> getL() {
            return l != null ? l.stream().map(AttributeValue::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder l(Collection<AttributeValue> l) {
            this.l = ListAttributeValueCopier.copy(l);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder l(AttributeValue... l) {
            l(Arrays.asList(l));
            return this;
        }

        public final void setL(Collection<BuilderImpl> l) {
            this.l = ListAttributeValueCopier.copyFromBuilder(l);
        }

        public final Boolean getBOOL() {
            return bool;
        }

        @Override
        public final Builder bool(Boolean bool) {
            this.bool = bool;
            return this;
        }

        public final void setBOOL(Boolean bool) {
            this.bool = bool;
        }

        public final Boolean getNUL() {
            return nul;
        }

        @Override
        public final Builder nul(Boolean nul) {
            this.nul = nul;
            return this;
        }

        public final void setNUL(Boolean nul) {
            this.nul = nul;
        }

        @Override
        public AttributeValue build() {
            return new AttributeValue(this);
        }
    }
}
