/*
 * Copyright 2010-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package software.amazon.awssdk.extensions.dynamodb.mappingclient.operations;

import static software.amazon.awssdk.extensions.dynamodb.mappingclient.core.Utils.readAndTransformSingleItem;

import java.util.Map;

import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.extensions.dynamodb.mappingclient.MappedTable;
import software.amazon.awssdk.extensions.dynamodb.mappingclient.OperationContext;
import software.amazon.awssdk.services.dynamodb.model.AttributeValue;

@SdkPublicApi
public class UnmappedItem {
    private final Map<String, AttributeValue> itemMap;

    private UnmappedItem(Map<String, AttributeValue> itemMap) {
        this.itemMap = itemMap;
    }

    public static UnmappedItem of(Map<String, AttributeValue> itemMap) {
        return new UnmappedItem(itemMap);
    }

    public <T> T getItem(MappedTable<T> mappedTable) {
        return readAndTransformSingleItem(itemMap,
                                          mappedTable.getTableSchema(),
                                          OperationContext.of(mappedTable.getTableName()),
                                          mappedTable.getMapperExtension());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        UnmappedItem that = (UnmappedItem) o;

        return itemMap != null ? itemMap.equals(that.itemMap) : that.itemMap == null;
    }

    @Override
    public int hashCode() {
        return itemMap != null ? itemMap.hashCode() : 0;
    }
}
