/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dsql;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.dsql.model.AccessDeniedException;
import software.amazon.awssdk.services.dsql.model.ConflictException;
import software.amazon.awssdk.services.dsql.model.CreateClusterRequest;
import software.amazon.awssdk.services.dsql.model.CreateClusterResponse;
import software.amazon.awssdk.services.dsql.model.CreateMultiRegionClustersRequest;
import software.amazon.awssdk.services.dsql.model.CreateMultiRegionClustersResponse;
import software.amazon.awssdk.services.dsql.model.DeleteClusterRequest;
import software.amazon.awssdk.services.dsql.model.DeleteClusterResponse;
import software.amazon.awssdk.services.dsql.model.DeleteMultiRegionClustersRequest;
import software.amazon.awssdk.services.dsql.model.DeleteMultiRegionClustersResponse;
import software.amazon.awssdk.services.dsql.model.DsqlException;
import software.amazon.awssdk.services.dsql.model.GetClusterRequest;
import software.amazon.awssdk.services.dsql.model.GetClusterResponse;
import software.amazon.awssdk.services.dsql.model.InternalServerException;
import software.amazon.awssdk.services.dsql.model.ListClustersRequest;
import software.amazon.awssdk.services.dsql.model.ListClustersResponse;
import software.amazon.awssdk.services.dsql.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.dsql.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.dsql.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dsql.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.dsql.model.TagResourceRequest;
import software.amazon.awssdk.services.dsql.model.TagResourceResponse;
import software.amazon.awssdk.services.dsql.model.ThrottlingException;
import software.amazon.awssdk.services.dsql.model.UntagResourceRequest;
import software.amazon.awssdk.services.dsql.model.UntagResourceResponse;
import software.amazon.awssdk.services.dsql.model.UpdateClusterRequest;
import software.amazon.awssdk.services.dsql.model.UpdateClusterResponse;
import software.amazon.awssdk.services.dsql.model.ValidationException;
import software.amazon.awssdk.services.dsql.paginators.ListClustersIterable;
import software.amazon.awssdk.services.dsql.waiters.DsqlWaiter;

/**
 * Service client for accessing Amazon Aurora DSQL. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * This is an interface reference for Amazon Aurora DSQL. It contains documentation for one of the programming or
 * command line interfaces you can use to manage Amazon Aurora DSQL.
 * </p>
 * <p>
 * Amazon Aurora DSQL is a serverless, distributed SQL database suitable for workloads of any size. Aurora DSQL is
 * available in both single-Region and multi-Region configurations, so your clusters and databases are always available
 * even if an Availability Zone or an Amazon Web Services Region are unavailable. Aurora DSQL lets you focus on using
 * your data to acquire new insights for your business and customers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface DsqlClient extends AwsClient {
    String SERVICE_NAME = "dsql";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "dsql";

    /**
     * <p>
     * Creates a cluster in Amazon Aurora DSQL.
     * </p>
     *
     * @param createClusterRequest
     * @return Result of the CreateCluster operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The service limit was exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateClusterResponse createCluster(CreateClusterRequest createClusterRequest) throws ServiceQuotaExceededException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a cluster in Amazon Aurora DSQL.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateClusterRequest.Builder} avoiding the need to
     * create one manually via {@link CreateClusterRequest#builder()}
     * </p>
     *
     * @param createClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.CreateClusterRequest.Builder} to create a request.
     * @return Result of the CreateCluster operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The service limit was exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateClusterResponse createCluster(Consumer<CreateClusterRequest.Builder> createClusterRequest)
            throws ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, ConflictException, AwsServiceException, SdkClientException, DsqlException {
        return createCluster(CreateClusterRequest.builder().applyMutation(createClusterRequest).build());
    }

    /**
     * <p>
     * Creates multi-Region clusters in Amazon Aurora DSQL. Multi-Region clusters require a linked Region list, which is
     * an array of the Regions in which you want to create linked clusters. Multi-Region clusters require a witness
     * Region, which participates in quorum in failure scenarios.
     * </p>
     *
     * @param createMultiRegionClustersRequest
     * @return Result of the CreateMultiRegionClusters operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The service limit was exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.CreateMultiRegionClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/CreateMultiRegionClusters"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateMultiRegionClustersResponse createMultiRegionClusters(
            CreateMultiRegionClustersRequest createMultiRegionClustersRequest) throws ServiceQuotaExceededException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates multi-Region clusters in Amazon Aurora DSQL. Multi-Region clusters require a linked Region list, which is
     * an array of the Regions in which you want to create linked clusters. Multi-Region clusters require a witness
     * Region, which participates in quorum in failure scenarios.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateMultiRegionClustersRequest.Builder} avoiding
     * the need to create one manually via {@link CreateMultiRegionClustersRequest#builder()}
     * </p>
     *
     * @param createMultiRegionClustersRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.CreateMultiRegionClustersRequest.Builder} to create a
     *        request.
     * @return Result of the CreateMultiRegionClusters operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The service limit was exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.CreateMultiRegionClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/CreateMultiRegionClusters"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateMultiRegionClustersResponse createMultiRegionClusters(
            Consumer<CreateMultiRegionClustersRequest.Builder> createMultiRegionClustersRequest)
            throws ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, ConflictException, AwsServiceException, SdkClientException, DsqlException {
        return createMultiRegionClusters(CreateMultiRegionClustersRequest.builder()
                .applyMutation(createMultiRegionClustersRequest).build());
    }

    /**
     * <p>
     * Deletes a cluster in Amazon Aurora DSQL.
     * </p>
     *
     * @param deleteClusterRequest
     * @return Result of the DeleteCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteClusterResponse deleteCluster(DeleteClusterRequest deleteClusterRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, ConflictException,
            AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a cluster in Amazon Aurora DSQL.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteClusterRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteClusterRequest#builder()}
     * </p>
     *
     * @param deleteClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.DeleteClusterRequest.Builder} to create a request.
     * @return Result of the DeleteCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteClusterResponse deleteCluster(Consumer<DeleteClusterRequest.Builder> deleteClusterRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, ConflictException, AwsServiceException, SdkClientException, DsqlException {
        return deleteCluster(DeleteClusterRequest.builder().applyMutation(deleteClusterRequest).build());
    }

    /**
     * <p>
     * Deletes a multi-Region cluster in Amazon Aurora DSQL.
     * </p>
     *
     * @param deleteMultiRegionClustersRequest
     * @return Result of the DeleteMultiRegionClusters operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.DeleteMultiRegionClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteMultiRegionClusters"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteMultiRegionClustersResponse deleteMultiRegionClusters(
            DeleteMultiRegionClustersRequest deleteMultiRegionClustersRequest) throws ThrottlingException, AccessDeniedException,
            ValidationException, InternalServerException, ResourceNotFoundException, ConflictException, AwsServiceException,
            SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a multi-Region cluster in Amazon Aurora DSQL.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteMultiRegionClustersRequest.Builder} avoiding
     * the need to create one manually via {@link DeleteMultiRegionClustersRequest#builder()}
     * </p>
     *
     * @param deleteMultiRegionClustersRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.DeleteMultiRegionClustersRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteMultiRegionClusters operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.DeleteMultiRegionClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteMultiRegionClusters"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteMultiRegionClustersResponse deleteMultiRegionClusters(
            Consumer<DeleteMultiRegionClustersRequest.Builder> deleteMultiRegionClustersRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, ConflictException,
            AwsServiceException, SdkClientException, DsqlException {
        return deleteMultiRegionClusters(DeleteMultiRegionClustersRequest.builder()
                .applyMutation(deleteMultiRegionClustersRequest).build());
    }

    /**
     * <p>
     * Retrieves information about a cluster.
     * </p>
     *
     * @param getClusterRequest
     * @return Result of the GetCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.GetCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/GetCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default GetClusterResponse getCluster(GetClusterRequest getClusterRequest) throws ThrottlingException, AccessDeniedException,
            ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves information about a cluster.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetClusterRequest.Builder} avoiding the need to
     * create one manually via {@link GetClusterRequest#builder()}
     * </p>
     *
     * @param getClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.GetClusterRequest.Builder} to create a request.
     * @return Result of the GetCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.GetCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/GetCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default GetClusterResponse getCluster(Consumer<GetClusterRequest.Builder> getClusterRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DsqlException {
        return getCluster(GetClusterRequest.builder().applyMutation(getClusterRequest).build());
    }

    /**
     * <p>
     * Retrieves information about a list of clusters.
     * </p>
     *
     * @param listClustersRequest
     * @return Result of the ListClusters operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default ListClustersResponse listClusters(ListClustersRequest listClustersRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves information about a list of clusters.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListClustersRequest.Builder} avoiding the need to
     * create one manually via {@link ListClustersRequest#builder()}
     * </p>
     *
     * @param listClustersRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.ListClustersRequest.Builder} to create a request.
     * @return Result of the ListClusters operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default ListClustersResponse listClusters(Consumer<ListClustersRequest.Builder> listClustersRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        return listClusters(ListClustersRequest.builder().applyMutation(listClustersRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     *     for (software.amazon.awssdk.services.dsql.model.ListClustersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)} operation.</b>
     * </p>
     *
     * @param listClustersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default ListClustersIterable listClustersPaginator(ListClustersRequest listClustersRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DsqlException {
        return new ListClustersIterable(this, listClustersRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     *     for (software.amazon.awssdk.services.dsql.model.ListClustersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListClustersRequest.Builder} avoiding the need to
     * create one manually via {@link ListClustersRequest#builder()}
     * </p>
     *
     * @param listClustersRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.ListClustersRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default ListClustersIterable listClustersPaginator(Consumer<ListClustersRequest.Builder> listClustersRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        return listClustersPaginator(ListClustersRequest.builder().applyMutation(listClustersRequest).build());
    }

    /**
     * <p>
     * Lists all of the tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all of the tags for a resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.ListTagsForResourceRequest.Builder} to create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DsqlException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Tags a resource with a map of key and value pairs.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The service limit was exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ServiceQuotaExceededException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Tags a resource with a map of key and value pairs.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The service limit was exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.UntagResourceRequest.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates a cluster.
     * </p>
     *
     * @param updateClusterRequest
     * @return Result of the UpdateCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.UpdateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateClusterResponse updateCluster(UpdateClusterRequest updateClusterRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, ConflictException,
            AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a cluster.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateClusterRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateClusterRequest#builder()}
     * </p>
     *
     * @param updateClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.UpdateClusterRequest.Builder} to create a request.
     * @return Result of the UpdateCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.UpdateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateClusterResponse updateCluster(Consumer<UpdateClusterRequest.Builder> updateClusterRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, ConflictException, AwsServiceException, SdkClientException, DsqlException {
        return updateCluster(UpdateClusterRequest.builder().applyMutation(updateClusterRequest).build());
    }

    /**
     * Creates an instance of {@link DsqlUtilities} object with the configuration set on this client.
     */
    default DsqlUtilities utilities() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create an instance of {@link DsqlWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link DsqlWaiter}
     */
    default DsqlWaiter waiter() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link DsqlClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static DsqlClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link DsqlClient}.
     */
    static DsqlClientBuilder builder() {
        return new DefaultDsqlClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default DsqlServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
