/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.drs;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.drs.model.AccessDeniedException;
import software.amazon.awssdk.services.drs.model.ConflictException;
import software.amazon.awssdk.services.drs.model.CreateExtendedSourceServerRequest;
import software.amazon.awssdk.services.drs.model.CreateExtendedSourceServerResponse;
import software.amazon.awssdk.services.drs.model.CreateReplicationConfigurationTemplateRequest;
import software.amazon.awssdk.services.drs.model.CreateReplicationConfigurationTemplateResponse;
import software.amazon.awssdk.services.drs.model.DeleteJobRequest;
import software.amazon.awssdk.services.drs.model.DeleteJobResponse;
import software.amazon.awssdk.services.drs.model.DeleteRecoveryInstanceRequest;
import software.amazon.awssdk.services.drs.model.DeleteRecoveryInstanceResponse;
import software.amazon.awssdk.services.drs.model.DeleteReplicationConfigurationTemplateRequest;
import software.amazon.awssdk.services.drs.model.DeleteReplicationConfigurationTemplateResponse;
import software.amazon.awssdk.services.drs.model.DeleteSourceServerRequest;
import software.amazon.awssdk.services.drs.model.DeleteSourceServerResponse;
import software.amazon.awssdk.services.drs.model.DescribeJobLogItemsRequest;
import software.amazon.awssdk.services.drs.model.DescribeJobLogItemsResponse;
import software.amazon.awssdk.services.drs.model.DescribeJobsRequest;
import software.amazon.awssdk.services.drs.model.DescribeJobsResponse;
import software.amazon.awssdk.services.drs.model.DescribeRecoveryInstancesRequest;
import software.amazon.awssdk.services.drs.model.DescribeRecoveryInstancesResponse;
import software.amazon.awssdk.services.drs.model.DescribeRecoverySnapshotsRequest;
import software.amazon.awssdk.services.drs.model.DescribeRecoverySnapshotsResponse;
import software.amazon.awssdk.services.drs.model.DescribeReplicationConfigurationTemplatesRequest;
import software.amazon.awssdk.services.drs.model.DescribeReplicationConfigurationTemplatesResponse;
import software.amazon.awssdk.services.drs.model.DescribeSourceServersRequest;
import software.amazon.awssdk.services.drs.model.DescribeSourceServersResponse;
import software.amazon.awssdk.services.drs.model.DisconnectRecoveryInstanceRequest;
import software.amazon.awssdk.services.drs.model.DisconnectRecoveryInstanceResponse;
import software.amazon.awssdk.services.drs.model.DisconnectSourceServerRequest;
import software.amazon.awssdk.services.drs.model.DisconnectSourceServerResponse;
import software.amazon.awssdk.services.drs.model.DrsException;
import software.amazon.awssdk.services.drs.model.DrsRequest;
import software.amazon.awssdk.services.drs.model.GetFailbackReplicationConfigurationRequest;
import software.amazon.awssdk.services.drs.model.GetFailbackReplicationConfigurationResponse;
import software.amazon.awssdk.services.drs.model.GetLaunchConfigurationRequest;
import software.amazon.awssdk.services.drs.model.GetLaunchConfigurationResponse;
import software.amazon.awssdk.services.drs.model.GetReplicationConfigurationRequest;
import software.amazon.awssdk.services.drs.model.GetReplicationConfigurationResponse;
import software.amazon.awssdk.services.drs.model.InitializeServiceRequest;
import software.amazon.awssdk.services.drs.model.InitializeServiceResponse;
import software.amazon.awssdk.services.drs.model.InternalServerException;
import software.amazon.awssdk.services.drs.model.ListExtensibleSourceServersRequest;
import software.amazon.awssdk.services.drs.model.ListExtensibleSourceServersResponse;
import software.amazon.awssdk.services.drs.model.ListStagingAccountsRequest;
import software.amazon.awssdk.services.drs.model.ListStagingAccountsResponse;
import software.amazon.awssdk.services.drs.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.drs.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.drs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.drs.model.RetryDataReplicationRequest;
import software.amazon.awssdk.services.drs.model.RetryDataReplicationResponse;
import software.amazon.awssdk.services.drs.model.ReverseReplicationRequest;
import software.amazon.awssdk.services.drs.model.ReverseReplicationResponse;
import software.amazon.awssdk.services.drs.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.drs.model.StartFailbackLaunchRequest;
import software.amazon.awssdk.services.drs.model.StartFailbackLaunchResponse;
import software.amazon.awssdk.services.drs.model.StartRecoveryRequest;
import software.amazon.awssdk.services.drs.model.StartRecoveryResponse;
import software.amazon.awssdk.services.drs.model.StartReplicationRequest;
import software.amazon.awssdk.services.drs.model.StartReplicationResponse;
import software.amazon.awssdk.services.drs.model.StopFailbackRequest;
import software.amazon.awssdk.services.drs.model.StopFailbackResponse;
import software.amazon.awssdk.services.drs.model.StopReplicationRequest;
import software.amazon.awssdk.services.drs.model.StopReplicationResponse;
import software.amazon.awssdk.services.drs.model.TagResourceRequest;
import software.amazon.awssdk.services.drs.model.TagResourceResponse;
import software.amazon.awssdk.services.drs.model.TerminateRecoveryInstancesRequest;
import software.amazon.awssdk.services.drs.model.TerminateRecoveryInstancesResponse;
import software.amazon.awssdk.services.drs.model.ThrottlingException;
import software.amazon.awssdk.services.drs.model.UninitializedAccountException;
import software.amazon.awssdk.services.drs.model.UntagResourceRequest;
import software.amazon.awssdk.services.drs.model.UntagResourceResponse;
import software.amazon.awssdk.services.drs.model.UpdateFailbackReplicationConfigurationRequest;
import software.amazon.awssdk.services.drs.model.UpdateFailbackReplicationConfigurationResponse;
import software.amazon.awssdk.services.drs.model.UpdateLaunchConfigurationRequest;
import software.amazon.awssdk.services.drs.model.UpdateLaunchConfigurationResponse;
import software.amazon.awssdk.services.drs.model.UpdateReplicationConfigurationRequest;
import software.amazon.awssdk.services.drs.model.UpdateReplicationConfigurationResponse;
import software.amazon.awssdk.services.drs.model.UpdateReplicationConfigurationTemplateRequest;
import software.amazon.awssdk.services.drs.model.UpdateReplicationConfigurationTemplateResponse;
import software.amazon.awssdk.services.drs.model.ValidationException;
import software.amazon.awssdk.services.drs.paginators.DescribeJobLogItemsIterable;
import software.amazon.awssdk.services.drs.paginators.DescribeJobsIterable;
import software.amazon.awssdk.services.drs.paginators.DescribeRecoveryInstancesIterable;
import software.amazon.awssdk.services.drs.paginators.DescribeRecoverySnapshotsIterable;
import software.amazon.awssdk.services.drs.paginators.DescribeReplicationConfigurationTemplatesIterable;
import software.amazon.awssdk.services.drs.paginators.DescribeSourceServersIterable;
import software.amazon.awssdk.services.drs.paginators.ListExtensibleSourceServersIterable;
import software.amazon.awssdk.services.drs.paginators.ListStagingAccountsIterable;
import software.amazon.awssdk.services.drs.transform.CreateExtendedSourceServerRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.CreateReplicationConfigurationTemplateRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DeleteJobRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DeleteRecoveryInstanceRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DeleteReplicationConfigurationTemplateRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DeleteSourceServerRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DescribeJobLogItemsRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DescribeJobsRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DescribeRecoveryInstancesRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DescribeRecoverySnapshotsRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DescribeReplicationConfigurationTemplatesRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DescribeSourceServersRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DisconnectRecoveryInstanceRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.DisconnectSourceServerRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.GetFailbackReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.GetLaunchConfigurationRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.GetReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.InitializeServiceRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.ListExtensibleSourceServersRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.ListStagingAccountsRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.RetryDataReplicationRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.ReverseReplicationRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.StartFailbackLaunchRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.StartRecoveryRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.StartReplicationRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.StopFailbackRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.StopReplicationRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.TerminateRecoveryInstancesRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.UpdateFailbackReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.UpdateLaunchConfigurationRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.UpdateReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.drs.transform.UpdateReplicationConfigurationTemplateRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link DrsClient}.
 *
 * @see DrsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDrsClient implements DrsClient {
    private static final Logger log = Logger.loggerFor(DefaultDrsClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final DrsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultDrsClient(DrsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Create an extended source server in the target Account based on the source server in staging account.
     * </p>
     *
     * @param createExtendedSourceServerRequest
     * @return Result of the CreateExtendedSourceServer operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ServiceQuotaExceededException
     *         The request could not be completed because its exceeded the service quota.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.CreateExtendedSourceServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/CreateExtendedSourceServer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateExtendedSourceServerResponse createExtendedSourceServer(
            CreateExtendedSourceServerRequest createExtendedSourceServerRequest) throws ResourceNotFoundException,
            InternalServerException, ServiceQuotaExceededException, ThrottlingException, AccessDeniedException,
            ValidationException, UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateExtendedSourceServerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateExtendedSourceServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExtendedSourceServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExtendedSourceServer");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateExtendedSourceServerRequest, CreateExtendedSourceServerResponse>()
                            .withOperationName("CreateExtendedSourceServer").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createExtendedSourceServerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateExtendedSourceServerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new ReplicationConfigurationTemplate.
     * </p>
     *
     * @param createReplicationConfigurationTemplateRequest
     * @return Result of the CreateReplicationConfigurationTemplate operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ServiceQuotaExceededException
     *         The request could not be completed because its exceeded the service quota.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.CreateReplicationConfigurationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/CreateReplicationConfigurationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateReplicationConfigurationTemplateResponse createReplicationConfigurationTemplate(
            CreateReplicationConfigurationTemplateRequest createReplicationConfigurationTemplateRequest)
            throws InternalServerException, ServiceQuotaExceededException, ThrottlingException, AccessDeniedException,
            ValidationException, UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateReplicationConfigurationTemplateResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, CreateReplicationConfigurationTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createReplicationConfigurationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateReplicationConfigurationTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateReplicationConfigurationTemplateRequest, CreateReplicationConfigurationTemplateResponse>()
                            .withOperationName("CreateReplicationConfigurationTemplate").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(createReplicationConfigurationTemplateRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateReplicationConfigurationTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a single Job by ID.
     * </p>
     *
     * @param deleteJobRequest
     * @return Result of the DeleteJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DeleteJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DeleteJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteJobResponse deleteJob(DeleteJobRequest deleteJobRequest) throws ResourceNotFoundException,
            InternalServerException, ConflictException, ThrottlingException, UninitializedAccountException, AwsServiceException,
            SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJob");

            return clientHandler.execute(new ClientExecutionParams<DeleteJobRequest, DeleteJobResponse>()
                    .withOperationName("DeleteJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new DeleteJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a single Recovery Instance by ID. This deletes the Recovery Instance resource from Elastic Disaster
     * Recovery. The Recovery Instance must be disconnected first in order to delete it.
     * </p>
     *
     * @param deleteRecoveryInstanceRequest
     * @return Result of the DeleteRecoveryInstance operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DeleteRecoveryInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DeleteRecoveryInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteRecoveryInstanceResponse deleteRecoveryInstance(DeleteRecoveryInstanceRequest deleteRecoveryInstanceRequest)
            throws InternalServerException, ConflictException, ThrottlingException, AccessDeniedException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRecoveryInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRecoveryInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRecoveryInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRecoveryInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteRecoveryInstanceRequest, DeleteRecoveryInstanceResponse>()
                            .withOperationName("DeleteRecoveryInstance").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteRecoveryInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteRecoveryInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a single Replication Configuration Template by ID
     * </p>
     *
     * @param deleteReplicationConfigurationTemplateRequest
     * @return Result of the DeleteReplicationConfigurationTemplate operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DeleteReplicationConfigurationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DeleteReplicationConfigurationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteReplicationConfigurationTemplateResponse deleteReplicationConfigurationTemplate(
            DeleteReplicationConfigurationTemplateRequest deleteReplicationConfigurationTemplateRequest)
            throws ResourceNotFoundException, InternalServerException, ConflictException, ThrottlingException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteReplicationConfigurationTemplateResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DeleteReplicationConfigurationTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteReplicationConfigurationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteReplicationConfigurationTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteReplicationConfigurationTemplateRequest, DeleteReplicationConfigurationTemplateResponse>()
                            .withOperationName("DeleteReplicationConfigurationTemplate").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteReplicationConfigurationTemplateRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteReplicationConfigurationTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a single Source Server by ID. The Source Server must be disconnected first.
     * </p>
     *
     * @param deleteSourceServerRequest
     * @return Result of the DeleteSourceServer operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DeleteSourceServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DeleteSourceServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSourceServerResponse deleteSourceServer(DeleteSourceServerRequest deleteSourceServerRequest)
            throws ResourceNotFoundException, InternalServerException, ConflictException, ThrottlingException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSourceServerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSourceServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSourceServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSourceServer");

            return clientHandler.execute(new ClientExecutionParams<DeleteSourceServerRequest, DeleteSourceServerResponse>()
                    .withOperationName("DeleteSourceServer").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSourceServerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSourceServerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a detailed Job log with pagination.
     * </p>
     *
     * @param describeJobLogItemsRequest
     * @return Result of the DescribeJobLogItems operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeJobLogItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeJobLogItems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeJobLogItemsResponse describeJobLogItems(DescribeJobLogItemsRequest describeJobLogItemsRequest)
            throws InternalServerException, ThrottlingException, ValidationException, UninitializedAccountException,
            AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJobLogItemsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeJobLogItemsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobLogItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobLogItems");

            return clientHandler.execute(new ClientExecutionParams<DescribeJobLogItemsRequest, DescribeJobLogItemsResponse>()
                    .withOperationName("DescribeJobLogItems").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeJobLogItemsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeJobLogItemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a detailed Job log with pagination.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobLogItems(software.amazon.awssdk.services.drs.model.DescribeJobLogItemsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeJobLogItemsIterable responses = client.describeJobLogItemsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.drs.paginators.DescribeJobLogItemsIterable responses = client
     *             .describeJobLogItemsPaginator(request);
     *     for (software.amazon.awssdk.services.drs.model.DescribeJobLogItemsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeJobLogItemsIterable responses = client.describeJobLogItemsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobLogItems(software.amazon.awssdk.services.drs.model.DescribeJobLogItemsRequest)} operation.</b>
     * </p>
     *
     * @param describeJobLogItemsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeJobLogItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeJobLogItems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeJobLogItemsIterable describeJobLogItemsPaginator(DescribeJobLogItemsRequest describeJobLogItemsRequest)
            throws InternalServerException, ThrottlingException, ValidationException, UninitializedAccountException,
            AwsServiceException, SdkClientException, DrsException {
        return new DescribeJobLogItemsIterable(this, applyPaginatorUserAgent(describeJobLogItemsRequest));
    }

    /**
     * <p>
     * Returns a list of Jobs. Use the JobsID and fromDate and toDate filters to limit which jobs are returned. The
     * response is sorted by creationDataTime - latest date first. Jobs are created by the StartRecovery,
     * TerminateRecoveryInstances and StartFailbackLaunch APIs. Jobs are also created by DiagnosticLaunch and
     * TerminateDiagnosticInstances, which are APIs available only to *Support* and only used in response to relevant
     * support tickets.
     * </p>
     *
     * @param describeJobsRequest
     * @return Result of the DescribeJobs operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeJobsResponse describeJobs(DescribeJobsRequest describeJobsRequest) throws InternalServerException,
            ThrottlingException, ValidationException, UninitializedAccountException, AwsServiceException, SdkClientException,
            DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobs");

            return clientHandler.execute(new ClientExecutionParams<DescribeJobsRequest, DescribeJobsResponse>()
                    .withOperationName("DescribeJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of Jobs. Use the JobsID and fromDate and toDate filters to limit which jobs are returned. The
     * response is sorted by creationDataTime - latest date first. Jobs are created by the StartRecovery,
     * TerminateRecoveryInstances and StartFailbackLaunch APIs. Jobs are also created by DiagnosticLaunch and
     * TerminateDiagnosticInstances, which are APIs available only to *Support* and only used in response to relevant
     * support tickets.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #describeJobs(software.amazon.awssdk.services.drs.model.DescribeJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeJobsIterable responses = client.describeJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.drs.paginators.DescribeJobsIterable responses = client.describeJobsPaginator(request);
     *     for (software.amazon.awssdk.services.drs.model.DescribeJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeJobsIterable responses = client.describeJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobs(software.amazon.awssdk.services.drs.model.DescribeJobsRequest)} operation.</b>
     * </p>
     *
     * @param describeJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeJobsIterable describeJobsPaginator(DescribeJobsRequest describeJobsRequest) throws InternalServerException,
            ThrottlingException, ValidationException, UninitializedAccountException, AwsServiceException, SdkClientException,
            DrsException {
        return new DescribeJobsIterable(this, applyPaginatorUserAgent(describeJobsRequest));
    }

    /**
     * <p>
     * Lists all Recovery Instances or multiple Recovery Instances by ID.
     * </p>
     *
     * @param describeRecoveryInstancesRequest
     * @return Result of the DescribeRecoveryInstances operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeRecoveryInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeRecoveryInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeRecoveryInstancesResponse describeRecoveryInstances(
            DescribeRecoveryInstancesRequest describeRecoveryInstancesRequest) throws InternalServerException,
            ThrottlingException, AccessDeniedException, UninitializedAccountException, AwsServiceException, SdkClientException,
            DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRecoveryInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRecoveryInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRecoveryInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRecoveryInstances");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeRecoveryInstancesRequest, DescribeRecoveryInstancesResponse>()
                            .withOperationName("DescribeRecoveryInstances").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeRecoveryInstancesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeRecoveryInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all Recovery Instances or multiple Recovery Instances by ID.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeRecoveryInstances(software.amazon.awssdk.services.drs.model.DescribeRecoveryInstancesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeRecoveryInstancesIterable responses = client.describeRecoveryInstancesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.drs.paginators.DescribeRecoveryInstancesIterable responses = client
     *             .describeRecoveryInstancesPaginator(request);
     *     for (software.amazon.awssdk.services.drs.model.DescribeRecoveryInstancesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeRecoveryInstancesIterable responses = client.describeRecoveryInstancesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeRecoveryInstances(software.amazon.awssdk.services.drs.model.DescribeRecoveryInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeRecoveryInstancesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeRecoveryInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeRecoveryInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeRecoveryInstancesIterable describeRecoveryInstancesPaginator(
            DescribeRecoveryInstancesRequest describeRecoveryInstancesRequest) throws InternalServerException,
            ThrottlingException, AccessDeniedException, UninitializedAccountException, AwsServiceException, SdkClientException,
            DrsException {
        return new DescribeRecoveryInstancesIterable(this, applyPaginatorUserAgent(describeRecoveryInstancesRequest));
    }

    /**
     * <p>
     * Lists all Recovery Snapshots for a single Source Server.
     * </p>
     *
     * @param describeRecoverySnapshotsRequest
     * @return Result of the DescribeRecoverySnapshots operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeRecoverySnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeRecoverySnapshots" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeRecoverySnapshotsResponse describeRecoverySnapshots(
            DescribeRecoverySnapshotsRequest describeRecoverySnapshotsRequest) throws InternalServerException,
            ThrottlingException, AccessDeniedException, ValidationException, UninitializedAccountException, AwsServiceException,
            SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRecoverySnapshotsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRecoverySnapshotsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRecoverySnapshotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRecoverySnapshots");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeRecoverySnapshotsRequest, DescribeRecoverySnapshotsResponse>()
                            .withOperationName("DescribeRecoverySnapshots").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeRecoverySnapshotsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeRecoverySnapshotsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all Recovery Snapshots for a single Source Server.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeRecoverySnapshots(software.amazon.awssdk.services.drs.model.DescribeRecoverySnapshotsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeRecoverySnapshotsIterable responses = client.describeRecoverySnapshotsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.drs.paginators.DescribeRecoverySnapshotsIterable responses = client
     *             .describeRecoverySnapshotsPaginator(request);
     *     for (software.amazon.awssdk.services.drs.model.DescribeRecoverySnapshotsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeRecoverySnapshotsIterable responses = client.describeRecoverySnapshotsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeRecoverySnapshots(software.amazon.awssdk.services.drs.model.DescribeRecoverySnapshotsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeRecoverySnapshotsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeRecoverySnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeRecoverySnapshots" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeRecoverySnapshotsIterable describeRecoverySnapshotsPaginator(
            DescribeRecoverySnapshotsRequest describeRecoverySnapshotsRequest) throws InternalServerException,
            ThrottlingException, AccessDeniedException, ValidationException, UninitializedAccountException, AwsServiceException,
            SdkClientException, DrsException {
        return new DescribeRecoverySnapshotsIterable(this, applyPaginatorUserAgent(describeRecoverySnapshotsRequest));
    }

    /**
     * <p>
     * Lists all ReplicationConfigurationTemplates, filtered by Source Server IDs.
     * </p>
     *
     * @param describeReplicationConfigurationTemplatesRequest
     * @return Result of the DescribeReplicationConfigurationTemplates operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeReplicationConfigurationTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeReplicationConfigurationTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeReplicationConfigurationTemplatesResponse describeReplicationConfigurationTemplates(
            DescribeReplicationConfigurationTemplatesRequest describeReplicationConfigurationTemplatesRequest)
            throws ResourceNotFoundException, InternalServerException, ThrottlingException, ValidationException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeReplicationConfigurationTemplatesResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeReplicationConfigurationTemplatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeReplicationConfigurationTemplatesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReplicationConfigurationTemplates");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeReplicationConfigurationTemplatesRequest, DescribeReplicationConfigurationTemplatesResponse>()
                            .withOperationName("DescribeReplicationConfigurationTemplates").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeReplicationConfigurationTemplatesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeReplicationConfigurationTemplatesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all ReplicationConfigurationTemplates, filtered by Source Server IDs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeReplicationConfigurationTemplates(software.amazon.awssdk.services.drs.model.DescribeReplicationConfigurationTemplatesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeReplicationConfigurationTemplatesIterable responses = client.describeReplicationConfigurationTemplatesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.drs.paginators.DescribeReplicationConfigurationTemplatesIterable responses = client
     *             .describeReplicationConfigurationTemplatesPaginator(request);
     *     for (software.amazon.awssdk.services.drs.model.DescribeReplicationConfigurationTemplatesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeReplicationConfigurationTemplatesIterable responses = client.describeReplicationConfigurationTemplatesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeReplicationConfigurationTemplates(software.amazon.awssdk.services.drs.model.DescribeReplicationConfigurationTemplatesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeReplicationConfigurationTemplatesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeReplicationConfigurationTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeReplicationConfigurationTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeReplicationConfigurationTemplatesIterable describeReplicationConfigurationTemplatesPaginator(
            DescribeReplicationConfigurationTemplatesRequest describeReplicationConfigurationTemplatesRequest)
            throws ResourceNotFoundException, InternalServerException, ThrottlingException, ValidationException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        return new DescribeReplicationConfigurationTemplatesIterable(this,
                applyPaginatorUserAgent(describeReplicationConfigurationTemplatesRequest));
    }

    /**
     * <p>
     * Lists all Source Servers or multiple Source Servers filtered by ID.
     * </p>
     *
     * @param describeSourceServersRequest
     * @return Result of the DescribeSourceServers operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeSourceServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeSourceServers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeSourceServersResponse describeSourceServers(DescribeSourceServersRequest describeSourceServersRequest)
            throws InternalServerException, ThrottlingException, ValidationException, UninitializedAccountException,
            AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSourceServersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSourceServersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSourceServersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSourceServers");

            return clientHandler.execute(new ClientExecutionParams<DescribeSourceServersRequest, DescribeSourceServersResponse>()
                    .withOperationName("DescribeSourceServers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeSourceServersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeSourceServersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all Source Servers or multiple Source Servers filtered by ID.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeSourceServers(software.amazon.awssdk.services.drs.model.DescribeSourceServersRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeSourceServersIterable responses = client.describeSourceServersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.drs.paginators.DescribeSourceServersIterable responses = client
     *             .describeSourceServersPaginator(request);
     *     for (software.amazon.awssdk.services.drs.model.DescribeSourceServersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.DescribeSourceServersIterable responses = client.describeSourceServersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeSourceServers(software.amazon.awssdk.services.drs.model.DescribeSourceServersRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeSourceServersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DescribeSourceServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DescribeSourceServers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeSourceServersIterable describeSourceServersPaginator(DescribeSourceServersRequest describeSourceServersRequest)
            throws InternalServerException, ThrottlingException, ValidationException, UninitializedAccountException,
            AwsServiceException, SdkClientException, DrsException {
        return new DescribeSourceServersIterable(this, applyPaginatorUserAgent(describeSourceServersRequest));
    }

    /**
     * <p>
     * Disconnect a Recovery Instance from Elastic Disaster Recovery. Data replication is stopped immediately. All AWS
     * resources created by Elastic Disaster Recovery for enabling the replication of the Recovery Instance will be
     * terminated / deleted within 90 minutes. If the agent on the Recovery Instance has not been prevented from
     * communicating with the Elastic Disaster Recovery service, then it will receive a command to uninstall itself
     * (within approximately 10 minutes). The following properties of the Recovery Instance will be changed immediately:
     * dataReplicationInfo.dataReplicationState will be set to DISCONNECTED; The totalStorageBytes property for each of
     * dataReplicationInfo.replicatedDisks will be set to zero; dataReplicationInfo.lagDuration and
     * dataReplicationInfo.lagDuration will be nullified.
     * </p>
     *
     * @param disconnectRecoveryInstanceRequest
     * @return Result of the DisconnectRecoveryInstance operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DisconnectRecoveryInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DisconnectRecoveryInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisconnectRecoveryInstanceResponse disconnectRecoveryInstance(
            DisconnectRecoveryInstanceRequest disconnectRecoveryInstanceRequest) throws ResourceNotFoundException,
            InternalServerException, ConflictException, ThrottlingException, AccessDeniedException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisconnectRecoveryInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisconnectRecoveryInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disconnectRecoveryInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisconnectRecoveryInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<DisconnectRecoveryInstanceRequest, DisconnectRecoveryInstanceResponse>()
                            .withOperationName("DisconnectRecoveryInstance").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disconnectRecoveryInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisconnectRecoveryInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disconnects a specific Source Server from Elastic Disaster Recovery. Data replication is stopped immediately. All
     * AWS resources created by Elastic Disaster Recovery for enabling the replication of the Source Server will be
     * terminated / deleted within 90 minutes. You cannot disconnect a Source Server if it has a Recovery Instance. If
     * the agent on the Source Server has not been prevented from communicating with the Elastic Disaster Recovery
     * service, then it will receive a command to uninstall itself (within approximately 10 minutes). The following
     * properties of the SourceServer will be changed immediately: dataReplicationInfo.dataReplicationState will be set
     * to DISCONNECTED; The totalStorageBytes property for each of dataReplicationInfo.replicatedDisks will be set to
     * zero; dataReplicationInfo.lagDuration and dataReplicationInfo.lagDuration will be nullified.
     * </p>
     *
     * @param disconnectSourceServerRequest
     * @return Result of the DisconnectSourceServer operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.DisconnectSourceServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/DisconnectSourceServer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisconnectSourceServerResponse disconnectSourceServer(DisconnectSourceServerRequest disconnectSourceServerRequest)
            throws ResourceNotFoundException, InternalServerException, ConflictException, ThrottlingException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisconnectSourceServerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisconnectSourceServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disconnectSourceServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisconnectSourceServer");

            return clientHandler
                    .execute(new ClientExecutionParams<DisconnectSourceServerRequest, DisconnectSourceServerResponse>()
                            .withOperationName("DisconnectSourceServer").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disconnectSourceServerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisconnectSourceServerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all Failback ReplicationConfigurations, filtered by Recovery Instance ID.
     * </p>
     *
     * @param getFailbackReplicationConfigurationRequest
     * @return Result of the GetFailbackReplicationConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.GetFailbackReplicationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/GetFailbackReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetFailbackReplicationConfigurationResponse getFailbackReplicationConfiguration(
            GetFailbackReplicationConfigurationRequest getFailbackReplicationConfigurationRequest)
            throws ResourceNotFoundException, InternalServerException, ThrottlingException, UninitializedAccountException,
            AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFailbackReplicationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetFailbackReplicationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getFailbackReplicationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFailbackReplicationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetFailbackReplicationConfigurationRequest, GetFailbackReplicationConfigurationResponse>()
                            .withOperationName("GetFailbackReplicationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getFailbackReplicationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetFailbackReplicationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a LaunchConfiguration, filtered by Source Server IDs.
     * </p>
     *
     * @param getLaunchConfigurationRequest
     * @return Result of the GetLaunchConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.GetLaunchConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/GetLaunchConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetLaunchConfigurationResponse getLaunchConfiguration(GetLaunchConfigurationRequest getLaunchConfigurationRequest)
            throws ResourceNotFoundException, InternalServerException, ThrottlingException, UninitializedAccountException,
            AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLaunchConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLaunchConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLaunchConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLaunchConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetLaunchConfigurationRequest, GetLaunchConfigurationResponse>()
                            .withOperationName("GetLaunchConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getLaunchConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetLaunchConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a ReplicationConfiguration, filtered by Source Server ID.
     * </p>
     *
     * @param getReplicationConfigurationRequest
     * @return Result of the GetReplicationConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.GetReplicationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/GetReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReplicationConfigurationResponse getReplicationConfiguration(
            GetReplicationConfigurationRequest getReplicationConfigurationRequest) throws ResourceNotFoundException,
            InternalServerException, ThrottlingException, AccessDeniedException, UninitializedAccountException,
            AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReplicationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetReplicationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getReplicationConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReplicationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetReplicationConfigurationRequest, GetReplicationConfigurationResponse>()
                            .withOperationName("GetReplicationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getReplicationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetReplicationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Initialize Elastic Disaster Recovery.
     * </p>
     *
     * @param initializeServiceRequest
     * @return Result of the InitializeService operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.InitializeService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/InitializeService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public InitializeServiceResponse initializeService(InitializeServiceRequest initializeServiceRequest)
            throws InternalServerException, ThrottlingException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InitializeServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                InitializeServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initializeServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitializeService");

            return clientHandler.execute(new ClientExecutionParams<InitializeServiceRequest, InitializeServiceResponse>()
                    .withOperationName("InitializeService").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(initializeServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new InitializeServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of source servers on a staging account that are extensible, which means that: a. The source server
     * is not already extended into this Account. b. The source server on the Account we’re reading from is not an
     * extension of another source server.
     * </p>
     *
     * @param listExtensibleSourceServersRequest
     * @return Result of the ListExtensibleSourceServers operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.ListExtensibleSourceServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/ListExtensibleSourceServers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListExtensibleSourceServersResponse listExtensibleSourceServers(
            ListExtensibleSourceServersRequest listExtensibleSourceServersRequest) throws InternalServerException,
            ThrottlingException, AccessDeniedException, ValidationException, UninitializedAccountException, AwsServiceException,
            SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExtensibleSourceServersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListExtensibleSourceServersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExtensibleSourceServersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExtensibleSourceServers");

            return clientHandler
                    .execute(new ClientExecutionParams<ListExtensibleSourceServersRequest, ListExtensibleSourceServersResponse>()
                            .withOperationName("ListExtensibleSourceServers").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listExtensibleSourceServersRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListExtensibleSourceServersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of source servers on a staging account that are extensible, which means that: a. The source server
     * is not already extended into this Account. b. The source server on the Account we’re reading from is not an
     * extension of another source server.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listExtensibleSourceServers(software.amazon.awssdk.services.drs.model.ListExtensibleSourceServersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.ListExtensibleSourceServersIterable responses = client.listExtensibleSourceServersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.drs.paginators.ListExtensibleSourceServersIterable responses = client
     *             .listExtensibleSourceServersPaginator(request);
     *     for (software.amazon.awssdk.services.drs.model.ListExtensibleSourceServersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.ListExtensibleSourceServersIterable responses = client.listExtensibleSourceServersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listExtensibleSourceServers(software.amazon.awssdk.services.drs.model.ListExtensibleSourceServersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listExtensibleSourceServersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.ListExtensibleSourceServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/ListExtensibleSourceServers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListExtensibleSourceServersIterable listExtensibleSourceServersPaginator(
            ListExtensibleSourceServersRequest listExtensibleSourceServersRequest) throws InternalServerException,
            ThrottlingException, AccessDeniedException, ValidationException, UninitializedAccountException, AwsServiceException,
            SdkClientException, DrsException {
        return new ListExtensibleSourceServersIterable(this, applyPaginatorUserAgent(listExtensibleSourceServersRequest));
    }

    /**
     * <p>
     * Returns an array of staging accounts for existing extended source servers.
     * </p>
     *
     * @param listStagingAccountsRequest
     * @return Result of the ListStagingAccounts operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.ListStagingAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/ListStagingAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStagingAccountsResponse listStagingAccounts(ListStagingAccountsRequest listStagingAccountsRequest)
            throws InternalServerException, ThrottlingException, AccessDeniedException, ValidationException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStagingAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListStagingAccountsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStagingAccountsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStagingAccounts");

            return clientHandler.execute(new ClientExecutionParams<ListStagingAccountsRequest, ListStagingAccountsResponse>()
                    .withOperationName("ListStagingAccounts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listStagingAccountsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStagingAccountsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of staging accounts for existing extended source servers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listStagingAccounts(software.amazon.awssdk.services.drs.model.ListStagingAccountsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.ListStagingAccountsIterable responses = client.listStagingAccountsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.drs.paginators.ListStagingAccountsIterable responses = client
     *             .listStagingAccountsPaginator(request);
     *     for (software.amazon.awssdk.services.drs.model.ListStagingAccountsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.drs.paginators.ListStagingAccountsIterable responses = client.listStagingAccountsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStagingAccounts(software.amazon.awssdk.services.drs.model.ListStagingAccountsRequest)} operation.</b>
     * </p>
     *
     * @param listStagingAccountsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.ListStagingAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/ListStagingAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStagingAccountsIterable listStagingAccountsPaginator(ListStagingAccountsRequest listStagingAccountsRequest)
            throws InternalServerException, ThrottlingException, AccessDeniedException, ValidationException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        return new ListStagingAccountsIterable(this, applyPaginatorUserAgent(listStagingAccountsRequest));
    }

    /**
     * <p>
     * List all tags for your Elastic Disaster Recovery resources.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ThrottlingException, AccessDeniedException,
            ValidationException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * WARNING: RetryDataReplication is deprecated. Causes the data replication initiation sequence to begin immediately
     * upon next Handshake for the specified Source Server ID, regardless of when the previous initiation started. This
     * command will work only if the Source Server is stalled or is in a DISCONNECTED or STOPPED state.
     * </p>
     *
     * @param retryDataReplicationRequest
     * @return Result of the RetryDataReplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.RetryDataReplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/RetryDataReplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RetryDataReplicationResponse retryDataReplication(RetryDataReplicationRequest retryDataReplicationRequest)
            throws ResourceNotFoundException, InternalServerException, ThrottlingException, ValidationException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RetryDataReplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RetryDataReplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retryDataReplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RetryDataReplication");

            return clientHandler.execute(new ClientExecutionParams<RetryDataReplicationRequest, RetryDataReplicationResponse>()
                    .withOperationName("RetryDataReplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(retryDataReplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RetryDataReplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Start replication to origin / target region - applies only to protected instances that originated in EC2. For
     * recovery instances on target region - starts replication back to origin region. For failback instances on origin
     * region - starts replication to target region to re-protect them.
     * </p>
     *
     * @param reverseReplicationRequest
     * @return Result of the ReverseReplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.ReverseReplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/ReverseReplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ReverseReplicationResponse reverseReplication(ReverseReplicationRequest reverseReplicationRequest)
            throws ResourceNotFoundException, InternalServerException, ConflictException, ThrottlingException,
            AccessDeniedException, ValidationException, UninitializedAccountException, AwsServiceException, SdkClientException,
            DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReverseReplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ReverseReplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, reverseReplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReverseReplication");

            return clientHandler.execute(new ClientExecutionParams<ReverseReplicationRequest, ReverseReplicationResponse>()
                    .withOperationName("ReverseReplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(reverseReplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ReverseReplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Initiates a Job for launching the machine that is being failed back to from the specified Recovery Instance. This
     * will run conversion on the failback client and will reboot your machine, thus completing the failback process.
     * </p>
     *
     * @param startFailbackLaunchRequest
     * @return Result of the StartFailbackLaunch operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ServiceQuotaExceededException
     *         The request could not be completed because its exceeded the service quota.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.StartFailbackLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/StartFailbackLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartFailbackLaunchResponse startFailbackLaunch(StartFailbackLaunchRequest startFailbackLaunchRequest)
            throws InternalServerException, ConflictException, ServiceQuotaExceededException, ThrottlingException,
            ValidationException, UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartFailbackLaunchResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartFailbackLaunchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFailbackLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFailbackLaunch");

            return clientHandler.execute(new ClientExecutionParams<StartFailbackLaunchRequest, StartFailbackLaunchResponse>()
                    .withOperationName("StartFailbackLaunch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startFailbackLaunchRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartFailbackLaunchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Launches Recovery Instances for the specified Source Servers. For each Source Server you may choose a point in
     * time snapshot to launch from, or use an on demand snapshot.
     * </p>
     *
     * @param startRecoveryRequest
     * @return Result of the StartRecovery operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ServiceQuotaExceededException
     *         The request could not be completed because its exceeded the service quota.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.StartRecovery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/StartRecovery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartRecoveryResponse startRecovery(StartRecoveryRequest startRecoveryRequest) throws InternalServerException,
            ConflictException, ServiceQuotaExceededException, ThrottlingException, UninitializedAccountException,
            AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartRecoveryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartRecoveryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startRecoveryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartRecovery");

            return clientHandler.execute(new ClientExecutionParams<StartRecoveryRequest, StartRecoveryResponse>()
                    .withOperationName("StartRecovery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startRecoveryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartRecoveryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts replication for a stopped Source Server. This action would make the Source Server protected again and
     * restart billing for it.
     * </p>
     *
     * @param startReplicationRequest
     * @return Result of the StartReplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.StartReplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/StartReplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartReplicationResponse startReplication(StartReplicationRequest startReplicationRequest)
            throws ResourceNotFoundException, InternalServerException, ConflictException, ThrottlingException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartReplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartReplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startReplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartReplication");

            return clientHandler.execute(new ClientExecutionParams<StartReplicationRequest, StartReplicationResponse>()
                    .withOperationName("StartReplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startReplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartReplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops the failback process for a specified Recovery Instance. This changes the Failback State of the Recovery
     * Instance back to FAILBACK_NOT_STARTED.
     * </p>
     *
     * @param stopFailbackRequest
     * @return Result of the StopFailback operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.StopFailback
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/StopFailback" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopFailbackResponse stopFailback(StopFailbackRequest stopFailbackRequest) throws ResourceNotFoundException,
            InternalServerException, ThrottlingException, UninitializedAccountException, AwsServiceException, SdkClientException,
            DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopFailbackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopFailbackResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopFailbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopFailback");

            return clientHandler.execute(new ClientExecutionParams<StopFailbackRequest, StopFailbackResponse>()
                    .withOperationName("StopFailback").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopFailbackRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopFailbackRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops replication for a Source Server. This action would make the Source Server unprotected, delete its existing
     * snapshots and stop billing for it.
     * </p>
     *
     * @param stopReplicationRequest
     * @return Result of the StopReplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.StopReplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/StopReplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopReplicationResponse stopReplication(StopReplicationRequest stopReplicationRequest)
            throws ResourceNotFoundException, InternalServerException, ConflictException, ThrottlingException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopReplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopReplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopReplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopReplication");

            return clientHandler.execute(new ClientExecutionParams<StopReplicationRequest, StopReplicationResponse>()
                    .withOperationName("StopReplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopReplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopReplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or overwrites only the specified tags for the specified Elastic Disaster Recovery resource or resources.
     * When you specify an existing tag key, the value is overwritten with the new value. Each resource can have a
     * maximum of 50 tags. Each tag consists of a key and optional value.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ThrottlingException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Initiates a Job for terminating the EC2 resources associated with the specified Recovery Instances, and then will
     * delete the Recovery Instances from the Elastic Disaster Recovery service.
     * </p>
     *
     * @param terminateRecoveryInstancesRequest
     * @return Result of the TerminateRecoveryInstances operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ServiceQuotaExceededException
     *         The request could not be completed because its exceeded the service quota.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.TerminateRecoveryInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/TerminateRecoveryInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TerminateRecoveryInstancesResponse terminateRecoveryInstances(
            TerminateRecoveryInstancesRequest terminateRecoveryInstancesRequest) throws InternalServerException,
            ConflictException, ServiceQuotaExceededException, ThrottlingException, UninitializedAccountException,
            AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TerminateRecoveryInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, TerminateRecoveryInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, terminateRecoveryInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TerminateRecoveryInstances");

            return clientHandler
                    .execute(new ClientExecutionParams<TerminateRecoveryInstancesRequest, TerminateRecoveryInstancesResponse>()
                            .withOperationName("TerminateRecoveryInstances").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(terminateRecoveryInstancesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new TerminateRecoveryInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified set of tags from the specified set of Elastic Disaster Recovery resources.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ThrottlingException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Allows you to update the failback replication configuration of a Recovery Instance by ID.
     * </p>
     *
     * @param updateFailbackReplicationConfigurationRequest
     * @return Result of the UpdateFailbackReplicationConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.UpdateFailbackReplicationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/UpdateFailbackReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateFailbackReplicationConfigurationResponse updateFailbackReplicationConfiguration(
            UpdateFailbackReplicationConfigurationRequest updateFailbackReplicationConfigurationRequest)
            throws ResourceNotFoundException, InternalServerException, ThrottlingException, AccessDeniedException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFailbackReplicationConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, UpdateFailbackReplicationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateFailbackReplicationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFailbackReplicationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateFailbackReplicationConfigurationRequest, UpdateFailbackReplicationConfigurationResponse>()
                            .withOperationName("UpdateFailbackReplicationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateFailbackReplicationConfigurationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateFailbackReplicationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a LaunchConfiguration by Source Server ID.
     * </p>
     *
     * @param updateLaunchConfigurationRequest
     * @return Result of the UpdateLaunchConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.UpdateLaunchConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/UpdateLaunchConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateLaunchConfigurationResponse updateLaunchConfiguration(
            UpdateLaunchConfigurationRequest updateLaunchConfigurationRequest) throws ResourceNotFoundException,
            InternalServerException, ConflictException, ThrottlingException, ValidationException, UninitializedAccountException,
            AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLaunchConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateLaunchConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLaunchConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLaunchConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateLaunchConfigurationRequest, UpdateLaunchConfigurationResponse>()
                            .withOperationName("UpdateLaunchConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateLaunchConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateLaunchConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Allows you to update a ReplicationConfiguration by Source Server ID.
     * </p>
     *
     * @param updateReplicationConfigurationRequest
     * @return Result of the UpdateReplicationConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.UpdateReplicationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/UpdateReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateReplicationConfigurationResponse updateReplicationConfiguration(
            UpdateReplicationConfigurationRequest updateReplicationConfigurationRequest) throws ResourceNotFoundException,
            InternalServerException, ConflictException, ThrottlingException, AccessDeniedException, ValidationException,
            UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateReplicationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateReplicationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateReplicationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateReplicationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateReplicationConfigurationRequest, UpdateReplicationConfigurationResponse>()
                            .withOperationName("UpdateReplicationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateReplicationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateReplicationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a ReplicationConfigurationTemplate by ID.
     * </p>
     *
     * @param updateReplicationConfigurationTemplateRequest
     * @return Result of the UpdateReplicationConfigurationTemplate operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource for this operation was not found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by the AWS service.
     * @throws UninitializedAccountException
     *         The account performing the request has not been initialized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DrsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DrsClient.UpdateReplicationConfigurationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/drs-2020-02-26/UpdateReplicationConfigurationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateReplicationConfigurationTemplateResponse updateReplicationConfigurationTemplate(
            UpdateReplicationConfigurationTemplateRequest updateReplicationConfigurationTemplateRequest)
            throws ResourceNotFoundException, InternalServerException, ThrottlingException, AccessDeniedException,
            ValidationException, UninitializedAccountException, AwsServiceException, SdkClientException, DrsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateReplicationConfigurationTemplateResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, UpdateReplicationConfigurationTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateReplicationConfigurationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "drs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateReplicationConfigurationTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateReplicationConfigurationTemplateRequest, UpdateReplicationConfigurationTemplateResponse>()
                            .withOperationName("UpdateReplicationConfigurationTemplate").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateReplicationConfigurationTemplateRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateReplicationConfigurationTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends DrsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DrsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UninitializedAccountException")
                                .exceptionBuilderSupplier(UninitializedAccountException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final DrsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
