/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dlm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>[Custom snapshot and AMI policies only]</b> Specifies a retention rule for snapshots created by snapshot policies,
 * or for AMIs created by AMI policies.
 * </p>
 * <note>
 * <p>
 * For snapshot policies that have an <a
 * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>, this retention rule
 * applies to standard tier retention. When the retention threshold is met, snapshots are moved from the standard to the
 * archive tier.
 * </p>
 * <p>
 * For snapshot policies that do not have an <b>ArchiveRule</b>, snapshots are permanently deleted when this retention
 * threshold is met.
 * </p>
 * </note>
 * <p>
 * You can retain snapshots based on either a count or a time interval.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Count-based retention</b>
 * </p>
 * <p>
 * You must specify <b>Count</b>. If you specify an <a
 * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a> for the schedule,
 * then you can specify a retention count of <code>0</code> to archive snapshots immediately after creation. If you
 * specify a <a href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_FastRestoreRule.html">FastRestoreRule</a>,
 * <a href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ShareRule.html">ShareRule</a>, or a <a
 * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_CrossRegionCopyRule.html">CrossRegionCopyRule</a>, then
 * you must specify a retention count of <code>1</code> or more.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Age-based retention</b>
 * </p>
 * <p>
 * You must specify <b>Interval</b> and <b>IntervalUnit</b>. If you specify an <a
 * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a> for the schedule,
 * then you can specify a retention interval of <code>0</code> days to archive snapshots immediately after creation. If
 * you specify a <a
 * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_FastRestoreRule.html">FastRestoreRule</a>, <a
 * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ShareRule.html">ShareRule</a>, or a <a
 * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_CrossRegionCopyRule.html">CrossRegionCopyRule</a>, then
 * you must specify a retention interval of <code>1</code> day or more.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RetainRule implements SdkPojo, Serializable, ToCopyableBuilder<RetainRule.Builder, RetainRule> {
    private static final SdkField<Integer> COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Count")
            .getter(getter(RetainRule::count)).setter(setter(Builder::count))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Count").build()).build();

    private static final SdkField<Integer> INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Interval").getter(getter(RetainRule::interval)).setter(setter(Builder::interval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Interval").build()).build();

    private static final SdkField<String> INTERVAL_UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IntervalUnit").getter(getter(RetainRule::intervalUnitAsString)).setter(setter(Builder::intervalUnit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IntervalUnit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COUNT_FIELD, INTERVAL_FIELD,
            INTERVAL_UNIT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer count;

    private final Integer interval;

    private final String intervalUnit;

    private RetainRule(BuilderImpl builder) {
        this.count = builder.count;
        this.interval = builder.interval;
        this.intervalUnit = builder.intervalUnit;
    }

    /**
     * <p>
     * The number of snapshots to retain for each volume, up to a maximum of 1000. For example if you want to retain a
     * maximum of three snapshots, specify <code>3</code>. When the fourth snapshot is created, the oldest retained
     * snapshot is deleted, or it is moved to the archive tier if you have specified an <a
     * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
     * </p>
     * 
     * @return The number of snapshots to retain for each volume, up to a maximum of 1000. For example if you want to
     *         retain a maximum of three snapshots, specify <code>3</code>. When the fourth snapshot is created, the
     *         oldest retained snapshot is deleted, or it is moved to the archive tier if you have specified an <a
     *         href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
     */
    public final Integer count() {
        return count;
    }

    /**
     * <p>
     * The amount of time to retain each snapshot. The maximum is 100 years. This is equivalent to 1200 months, 5200
     * weeks, or 36500 days.
     * </p>
     * 
     * @return The amount of time to retain each snapshot. The maximum is 100 years. This is equivalent to 1200 months,
     *         5200 weeks, or 36500 days.
     */
    public final Integer interval() {
        return interval;
    }

    /**
     * <p>
     * The unit of time for time-based retention. For example, to retain snapshots for 3 months, specify
     * <code>Interval=3</code> and <code>IntervalUnit=MONTHS</code>. Once the snapshot has been retained for 3 months,
     * it is deleted, or it is moved to the archive tier if you have specified an <a
     * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #intervalUnit} will
     * return {@link RetentionIntervalUnitValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #intervalUnitAsString}.
     * </p>
     * 
     * @return The unit of time for time-based retention. For example, to retain snapshots for 3 months, specify
     *         <code>Interval=3</code> and <code>IntervalUnit=MONTHS</code>. Once the snapshot has been retained for 3
     *         months, it is deleted, or it is moved to the archive tier if you have specified an <a
     *         href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
     * @see RetentionIntervalUnitValues
     */
    public final RetentionIntervalUnitValues intervalUnit() {
        return RetentionIntervalUnitValues.fromValue(intervalUnit);
    }

    /**
     * <p>
     * The unit of time for time-based retention. For example, to retain snapshots for 3 months, specify
     * <code>Interval=3</code> and <code>IntervalUnit=MONTHS</code>. Once the snapshot has been retained for 3 months,
     * it is deleted, or it is moved to the archive tier if you have specified an <a
     * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #intervalUnit} will
     * return {@link RetentionIntervalUnitValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #intervalUnitAsString}.
     * </p>
     * 
     * @return The unit of time for time-based retention. For example, to retain snapshots for 3 months, specify
     *         <code>Interval=3</code> and <code>IntervalUnit=MONTHS</code>. Once the snapshot has been retained for 3
     *         months, it is deleted, or it is moved to the archive tier if you have specified an <a
     *         href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
     * @see RetentionIntervalUnitValues
     */
    public final String intervalUnitAsString() {
        return intervalUnit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(count());
        hashCode = 31 * hashCode + Objects.hashCode(interval());
        hashCode = 31 * hashCode + Objects.hashCode(intervalUnitAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RetainRule)) {
            return false;
        }
        RetainRule other = (RetainRule) obj;
        return Objects.equals(count(), other.count()) && Objects.equals(interval(), other.interval())
                && Objects.equals(intervalUnitAsString(), other.intervalUnitAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RetainRule").add("Count", count()).add("Interval", interval())
                .add("IntervalUnit", intervalUnitAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Count":
            return Optional.ofNullable(clazz.cast(count()));
        case "Interval":
            return Optional.ofNullable(clazz.cast(interval()));
        case "IntervalUnit":
            return Optional.ofNullable(clazz.cast(intervalUnitAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Count", COUNT_FIELD);
        map.put("Interval", INTERVAL_FIELD);
        map.put("IntervalUnit", INTERVAL_UNIT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RetainRule, T> g) {
        return obj -> g.apply((RetainRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RetainRule> {
        /**
         * <p>
         * The number of snapshots to retain for each volume, up to a maximum of 1000. For example if you want to retain
         * a maximum of three snapshots, specify <code>3</code>. When the fourth snapshot is created, the oldest
         * retained snapshot is deleted, or it is moved to the archive tier if you have specified an <a
         * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
         * </p>
         * 
         * @param count
         *        The number of snapshots to retain for each volume, up to a maximum of 1000. For example if you want to
         *        retain a maximum of three snapshots, specify <code>3</code>. When the fourth snapshot is created, the
         *        oldest retained snapshot is deleted, or it is moved to the archive tier if you have specified an <a
         *        href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder count(Integer count);

        /**
         * <p>
         * The amount of time to retain each snapshot. The maximum is 100 years. This is equivalent to 1200 months, 5200
         * weeks, or 36500 days.
         * </p>
         * 
         * @param interval
         *        The amount of time to retain each snapshot. The maximum is 100 years. This is equivalent to 1200
         *        months, 5200 weeks, or 36500 days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interval(Integer interval);

        /**
         * <p>
         * The unit of time for time-based retention. For example, to retain snapshots for 3 months, specify
         * <code>Interval=3</code> and <code>IntervalUnit=MONTHS</code>. Once the snapshot has been retained for 3
         * months, it is deleted, or it is moved to the archive tier if you have specified an <a
         * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
         * </p>
         * 
         * @param intervalUnit
         *        The unit of time for time-based retention. For example, to retain snapshots for 3 months, specify
         *        <code>Interval=3</code> and <code>IntervalUnit=MONTHS</code>. Once the snapshot has been retained for
         *        3 months, it is deleted, or it is moved to the archive tier if you have specified an <a
         *        href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
         * @see RetentionIntervalUnitValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RetentionIntervalUnitValues
         */
        Builder intervalUnit(String intervalUnit);

        /**
         * <p>
         * The unit of time for time-based retention. For example, to retain snapshots for 3 months, specify
         * <code>Interval=3</code> and <code>IntervalUnit=MONTHS</code>. Once the snapshot has been retained for 3
         * months, it is deleted, or it is moved to the archive tier if you have specified an <a
         * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
         * </p>
         * 
         * @param intervalUnit
         *        The unit of time for time-based retention. For example, to retain snapshots for 3 months, specify
         *        <code>Interval=3</code> and <code>IntervalUnit=MONTHS</code>. Once the snapshot has been retained for
         *        3 months, it is deleted, or it is moved to the archive tier if you have specified an <a
         *        href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a>.
         * @see RetentionIntervalUnitValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RetentionIntervalUnitValues
         */
        Builder intervalUnit(RetentionIntervalUnitValues intervalUnit);
    }

    static final class BuilderImpl implements Builder {
        private Integer count;

        private Integer interval;

        private String intervalUnit;

        private BuilderImpl() {
        }

        private BuilderImpl(RetainRule model) {
            count(model.count);
            interval(model.interval);
            intervalUnit(model.intervalUnit);
        }

        public final Integer getCount() {
            return count;
        }

        public final void setCount(Integer count) {
            this.count = count;
        }

        @Override
        public final Builder count(Integer count) {
            this.count = count;
            return this;
        }

        public final Integer getInterval() {
            return interval;
        }

        public final void setInterval(Integer interval) {
            this.interval = interval;
        }

        @Override
        public final Builder interval(Integer interval) {
            this.interval = interval;
            return this;
        }

        public final String getIntervalUnit() {
            return intervalUnit;
        }

        public final void setIntervalUnit(String intervalUnit) {
            this.intervalUnit = intervalUnit;
        }

        @Override
        public final Builder intervalUnit(String intervalUnit) {
            this.intervalUnit = intervalUnit;
            return this;
        }

        @Override
        public final Builder intervalUnit(RetentionIntervalUnitValues intervalUnit) {
            this.intervalUnit(intervalUnit == null ? null : intervalUnit.toString());
            return this;
        }

        @Override
        public RetainRule build() {
            return new RetainRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
