/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dax.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the subnet associated with a DAX cluster. This parameter refers to subnets defined in Amazon Virtual
 * Private Cloud (Amazon VPC) and used with DAX.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Subnet implements SdkPojo, Serializable, ToCopyableBuilder<Subnet.Builder, Subnet> {
    private static final SdkField<String> SUBNET_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubnetIdentifier").getter(getter(Subnet::subnetIdentifier)).setter(setter(Builder::subnetIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetIdentifier").build()).build();

    private static final SdkField<String> SUBNET_AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubnetAvailabilityZone").getter(getter(Subnet::subnetAvailabilityZone))
            .setter(setter(Builder::subnetAvailabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetAvailabilityZone").build())
            .build();

    private static final SdkField<List<String>> SUPPORTED_NETWORK_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SupportedNetworkTypes")
            .getter(getter(Subnet::supportedNetworkTypesAsStrings))
            .setter(setter(Builder::supportedNetworkTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedNetworkTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUBNET_IDENTIFIER_FIELD,
            SUBNET_AVAILABILITY_ZONE_FIELD, SUPPORTED_NETWORK_TYPES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String subnetIdentifier;

    private final String subnetAvailabilityZone;

    private final List<String> supportedNetworkTypes;

    private Subnet(BuilderImpl builder) {
        this.subnetIdentifier = builder.subnetIdentifier;
        this.subnetAvailabilityZone = builder.subnetAvailabilityZone;
        this.supportedNetworkTypes = builder.supportedNetworkTypes;
    }

    /**
     * <p>
     * The system-assigned identifier for the subnet.
     * </p>
     * 
     * @return The system-assigned identifier for the subnet.
     */
    public final String subnetIdentifier() {
        return subnetIdentifier;
    }

    /**
     * <p>
     * The Availability Zone (AZ) for the subnet.
     * </p>
     * 
     * @return The Availability Zone (AZ) for the subnet.
     */
    public final String subnetAvailabilityZone() {
        return subnetAvailabilityZone;
    }

    /**
     * <p>
     * The network types supported by this subnet. Returns an array of strings that can include <code>ipv4</code>,
     * <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6 only, or dual-stack
     * deployments.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedNetworkTypes} method.
     * </p>
     * 
     * @return The network types supported by this subnet. Returns an array of strings that can include
     *         <code>ipv4</code>, <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6
     *         only, or dual-stack deployments.
     */
    public final List<NetworkType> supportedNetworkTypes() {
        return NetworkTypeListCopier.copyStringToEnum(supportedNetworkTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedNetworkTypes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupportedNetworkTypes() {
        return supportedNetworkTypes != null && !(supportedNetworkTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The network types supported by this subnet. Returns an array of strings that can include <code>ipv4</code>,
     * <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6 only, or dual-stack
     * deployments.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedNetworkTypes} method.
     * </p>
     * 
     * @return The network types supported by this subnet. Returns an array of strings that can include
     *         <code>ipv4</code>, <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6
     *         only, or dual-stack deployments.
     */
    public final List<String> supportedNetworkTypesAsStrings() {
        return supportedNetworkTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(subnetIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(subnetAvailabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedNetworkTypes() ? supportedNetworkTypesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Subnet)) {
            return false;
        }
        Subnet other = (Subnet) obj;
        return Objects.equals(subnetIdentifier(), other.subnetIdentifier())
                && Objects.equals(subnetAvailabilityZone(), other.subnetAvailabilityZone())
                && hasSupportedNetworkTypes() == other.hasSupportedNetworkTypes()
                && Objects.equals(supportedNetworkTypesAsStrings(), other.supportedNetworkTypesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Subnet").add("SubnetIdentifier", subnetIdentifier())
                .add("SubnetAvailabilityZone", subnetAvailabilityZone())
                .add("SupportedNetworkTypes", hasSupportedNetworkTypes() ? supportedNetworkTypesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SubnetIdentifier":
            return Optional.ofNullable(clazz.cast(subnetIdentifier()));
        case "SubnetAvailabilityZone":
            return Optional.ofNullable(clazz.cast(subnetAvailabilityZone()));
        case "SupportedNetworkTypes":
            return Optional.ofNullable(clazz.cast(supportedNetworkTypesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SubnetIdentifier", SUBNET_IDENTIFIER_FIELD);
        map.put("SubnetAvailabilityZone", SUBNET_AVAILABILITY_ZONE_FIELD);
        map.put("SupportedNetworkTypes", SUPPORTED_NETWORK_TYPES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Subnet, T> g) {
        return obj -> g.apply((Subnet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Subnet> {
        /**
         * <p>
         * The system-assigned identifier for the subnet.
         * </p>
         * 
         * @param subnetIdentifier
         *        The system-assigned identifier for the subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIdentifier(String subnetIdentifier);

        /**
         * <p>
         * The Availability Zone (AZ) for the subnet.
         * </p>
         * 
         * @param subnetAvailabilityZone
         *        The Availability Zone (AZ) for the subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetAvailabilityZone(String subnetAvailabilityZone);

        /**
         * <p>
         * The network types supported by this subnet. Returns an array of strings that can include <code>ipv4</code>,
         * <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6 only, or dual-stack
         * deployments.
         * </p>
         * 
         * @param supportedNetworkTypes
         *        The network types supported by this subnet. Returns an array of strings that can include
         *        <code>ipv4</code>, <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6
         *        only, or dual-stack deployments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedNetworkTypesWithStrings(Collection<String> supportedNetworkTypes);

        /**
         * <p>
         * The network types supported by this subnet. Returns an array of strings that can include <code>ipv4</code>,
         * <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6 only, or dual-stack
         * deployments.
         * </p>
         * 
         * @param supportedNetworkTypes
         *        The network types supported by this subnet. Returns an array of strings that can include
         *        <code>ipv4</code>, <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6
         *        only, or dual-stack deployments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedNetworkTypesWithStrings(String... supportedNetworkTypes);

        /**
         * <p>
         * The network types supported by this subnet. Returns an array of strings that can include <code>ipv4</code>,
         * <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6 only, or dual-stack
         * deployments.
         * </p>
         * 
         * @param supportedNetworkTypes
         *        The network types supported by this subnet. Returns an array of strings that can include
         *        <code>ipv4</code>, <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6
         *        only, or dual-stack deployments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedNetworkTypes(Collection<NetworkType> supportedNetworkTypes);

        /**
         * <p>
         * The network types supported by this subnet. Returns an array of strings that can include <code>ipv4</code>,
         * <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6 only, or dual-stack
         * deployments.
         * </p>
         * 
         * @param supportedNetworkTypes
         *        The network types supported by this subnet. Returns an array of strings that can include
         *        <code>ipv4</code>, <code>ipv6</code>, or both, indicating whether the subnet supports IPv4 only, IPv6
         *        only, or dual-stack deployments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedNetworkTypes(NetworkType... supportedNetworkTypes);
    }

    static final class BuilderImpl implements Builder {
        private String subnetIdentifier;

        private String subnetAvailabilityZone;

        private List<String> supportedNetworkTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Subnet model) {
            subnetIdentifier(model.subnetIdentifier);
            subnetAvailabilityZone(model.subnetAvailabilityZone);
            supportedNetworkTypesWithStrings(model.supportedNetworkTypes);
        }

        public final String getSubnetIdentifier() {
            return subnetIdentifier;
        }

        public final void setSubnetIdentifier(String subnetIdentifier) {
            this.subnetIdentifier = subnetIdentifier;
        }

        @Override
        public final Builder subnetIdentifier(String subnetIdentifier) {
            this.subnetIdentifier = subnetIdentifier;
            return this;
        }

        public final String getSubnetAvailabilityZone() {
            return subnetAvailabilityZone;
        }

        public final void setSubnetAvailabilityZone(String subnetAvailabilityZone) {
            this.subnetAvailabilityZone = subnetAvailabilityZone;
        }

        @Override
        public final Builder subnetAvailabilityZone(String subnetAvailabilityZone) {
            this.subnetAvailabilityZone = subnetAvailabilityZone;
            return this;
        }

        public final Collection<String> getSupportedNetworkTypes() {
            if (supportedNetworkTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedNetworkTypes;
        }

        public final void setSupportedNetworkTypes(Collection<String> supportedNetworkTypes) {
            this.supportedNetworkTypes = NetworkTypeListCopier.copy(supportedNetworkTypes);
        }

        @Override
        public final Builder supportedNetworkTypesWithStrings(Collection<String> supportedNetworkTypes) {
            this.supportedNetworkTypes = NetworkTypeListCopier.copy(supportedNetworkTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedNetworkTypesWithStrings(String... supportedNetworkTypes) {
            supportedNetworkTypesWithStrings(Arrays.asList(supportedNetworkTypes));
            return this;
        }

        @Override
        public final Builder supportedNetworkTypes(Collection<NetworkType> supportedNetworkTypes) {
            this.supportedNetworkTypes = NetworkTypeListCopier.copyEnumToString(supportedNetworkTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedNetworkTypes(NetworkType... supportedNetworkTypes) {
            supportedNetworkTypes(Arrays.asList(supportedNetworkTypes));
            return this;
        }

        @Override
        public Subnet build() {
            return new Subnet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
