/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentity;

import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.config.AwsSyncClientConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocol;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolMetadata;
import software.amazon.awssdk.core.client.ClientExecutionParams;
import software.amazon.awssdk.core.client.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cognitoidentity.model.CognitoIdentityException;
import software.amazon.awssdk.services.cognitoidentity.model.ConcurrentModificationException;
import software.amazon.awssdk.services.cognitoidentity.model.CreateIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.CreateIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentitiesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentitiesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DeveloperUserAlreadyRegisteredException;
import software.amazon.awssdk.services.cognitoidentity.model.ExternalServiceException;
import software.amazon.awssdk.services.cognitoidentity.model.GetCredentialsForIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetCredentialsForIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdentityPoolRolesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdentityPoolRolesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenForDeveloperIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenForDeveloperIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenResponse;
import software.amazon.awssdk.services.cognitoidentity.model.InternalErrorException;
import software.amazon.awssdk.services.cognitoidentity.model.InvalidIdentityPoolConfigurationException;
import software.amazon.awssdk.services.cognitoidentity.model.InvalidParameterException;
import software.amazon.awssdk.services.cognitoidentity.model.LimitExceededException;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentitiesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentitiesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentityPoolsRequest;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentityPoolsResponse;
import software.amazon.awssdk.services.cognitoidentity.model.LookupDeveloperIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.LookupDeveloperIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.MergeDeveloperIdentitiesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.MergeDeveloperIdentitiesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.NotAuthorizedException;
import software.amazon.awssdk.services.cognitoidentity.model.ResourceConflictException;
import software.amazon.awssdk.services.cognitoidentity.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cognitoidentity.model.SetIdentityPoolRolesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.SetIdentityPoolRolesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.TooManyRequestsException;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkDeveloperIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkDeveloperIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.UpdateIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.UpdateIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.transform.CreateIdentityPoolRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.CreateIdentityPoolResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DeleteIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DeleteIdentitiesResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DeleteIdentityPoolRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DeleteIdentityPoolResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DescribeIdentityPoolRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DescribeIdentityPoolResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DescribeIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DescribeIdentityResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetCredentialsForIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetCredentialsForIdentityResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetIdRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetIdResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetIdentityPoolRolesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetIdentityPoolRolesResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetOpenIdTokenForDeveloperIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetOpenIdTokenForDeveloperIdentityResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetOpenIdTokenRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetOpenIdTokenResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.ListIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.ListIdentitiesResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.ListIdentityPoolsRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.ListIdentityPoolsResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.LookupDeveloperIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.LookupDeveloperIdentityResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.MergeDeveloperIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.MergeDeveloperIdentitiesResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.SetIdentityPoolRolesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.SetIdentityPoolRolesResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UnlinkDeveloperIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UnlinkDeveloperIdentityResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UnlinkIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UnlinkIdentityResponseUnmarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UpdateIdentityPoolRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UpdateIdentityPoolResponseUnmarshaller;

/**
 * Internal implementation of {@link CognitoIdentityClient}.
 *
 * @see CognitoIdentityClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCognitoIdentityClient implements CognitoIdentityClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final AwsSyncClientConfiguration clientConfiguration;

    protected DefaultCognitoIdentityClient(AwsSyncClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration, null);
        this.protocolFactory = init();
        this.clientConfiguration = clientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new identity pool. The identity pool is a store of user identity information that is specific to your
     * AWS account. The limit on identity pools is 60 per account. The keys for <code>SupportedLoginProviders</code> are
     * as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Facebook: <code>graph.facebook.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Google: <code>accounts.google.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon: <code>www.amazon.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Twitter: <code>api.twitter.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Digits: <code>www.digits.com</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param createIdentityPoolRequest
     *        Input to the CreateIdentityPool action.
     * @return Result of the CreateIdentityPool operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws LimitExceededException
     *         Thrown when the total number of user pools has exceeded a preset limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.CreateIdentityPool
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/CreateIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateIdentityPoolResponse createIdentityPool(CreateIdentityPoolRequest createIdentityPoolRequest)
            throws InvalidParameterException, NotAuthorizedException, ResourceConflictException, TooManyRequestsException,
            InternalErrorException, LimitExceededException, AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<CreateIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateIdentityPoolResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateIdentityPoolRequest, CreateIdentityPoolResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(createIdentityPoolRequest).withMarshaller(new CreateIdentityPoolRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes identities from an identity pool. You can specify a list of 1-60 identities that you want to delete.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param deleteIdentitiesRequest
     *        Input to the <code>DeleteIdentities</code> action.
     * @return Result of the DeleteIdentities operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.DeleteIdentities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DeleteIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteIdentitiesResponse deleteIdentities(DeleteIdentitiesRequest deleteIdentitiesRequest)
            throws InvalidParameterException, TooManyRequestsException, InternalErrorException, AwsServiceException,
            SdkClientException, CognitoIdentityException {

        HttpResponseHandler<DeleteIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteIdentitiesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteIdentitiesRequest, DeleteIdentitiesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteIdentitiesRequest).withMarshaller(new DeleteIdentitiesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a user pool. Once a pool is deleted, users will not be able to authenticate with the pool.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param deleteIdentityPoolRequest
     *        Input to the DeleteIdentityPool action.
     * @return Result of the DeleteIdentityPool operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.DeleteIdentityPool
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DeleteIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteIdentityPoolResponse deleteIdentityPool(DeleteIdentityPoolRequest deleteIdentityPoolRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException,
            InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<DeleteIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteIdentityPoolResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteIdentityPoolRequest, DeleteIdentityPoolResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteIdentityPoolRequest).withMarshaller(new DeleteIdentityPoolRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns metadata related to the given identity, including when the identity was created and any associated linked
     * logins.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param describeIdentityRequest
     *        Input to the <code>DescribeIdentity</code> action.
     * @return Result of the DescribeIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.DescribeIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DescribeIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeIdentityResponse describeIdentity(DescribeIdentityRequest describeIdentityRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException,
            InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<DescribeIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeIdentityResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeIdentityRequest, DescribeIdentityResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeIdentityRequest).withMarshaller(new DescribeIdentityRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets details about a particular identity pool, including the pool name, ID description, creation date, and
     * current number of users.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param describeIdentityPoolRequest
     *        Input to the DescribeIdentityPool action.
     * @return Result of the DescribeIdentityPool operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.DescribeIdentityPool
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DescribeIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeIdentityPoolResponse describeIdentityPool(DescribeIdentityPoolRequest describeIdentityPoolRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException,
            InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<DescribeIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeIdentityPoolResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeIdentityPoolRequest, DescribeIdentityPoolResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeIdentityPoolRequest)
                .withMarshaller(new DescribeIdentityPoolRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns credentials for the provided identity ID. Any provided logins will be validated against supported login
     * providers. If the token is for cognito-identity.amazonaws.com, it will be passed through to AWS Security Token
     * Service with the appropriate role for the token.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param getCredentialsForIdentityRequest
     *        Input to the <code>GetCredentialsForIdentity</code> action.
     * @return Result of the GetCredentialsForIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InvalidIdentityPoolConfigurationException
     *         Thrown if the identity pool has no role associated for the given auth type (auth/unauth) or if the
     *         AssumeRole fails.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws ExternalServiceException
     *         An exception thrown when a dependent service such as Facebook or Twitter is not responding
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.GetCredentialsForIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetCredentialsForIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCredentialsForIdentityResponse getCredentialsForIdentity(
            GetCredentialsForIdentityRequest getCredentialsForIdentityRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, ResourceConflictException, TooManyRequestsException,
            InvalidIdentityPoolConfigurationException, InternalErrorException, ExternalServiceException, AwsServiceException,
            SdkClientException, CognitoIdentityException {

        HttpResponseHandler<GetCredentialsForIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetCredentialsForIdentityResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<GetCredentialsForIdentityRequest, GetCredentialsForIdentityResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(getCredentialsForIdentityRequest)
                        .withMarshaller(new GetCredentialsForIdentityRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Generates (or retrieves) a Cognito ID. Supplying multiple logins will create an implicit linked account.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param getIdRequest
     *        Input to the GetId action.
     * @return Result of the GetId operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws LimitExceededException
     *         Thrown when the total number of user pools has exceeded a preset limit.
     * @throws ExternalServiceException
     *         An exception thrown when a dependent service such as Facebook or Twitter is not responding
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.GetId
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetId" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetIdResponse getId(GetIdRequest getIdRequest) throws InvalidParameterException, ResourceNotFoundException,
            NotAuthorizedException, ResourceConflictException, TooManyRequestsException, InternalErrorException,
            LimitExceededException, ExternalServiceException, AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<GetIdResponse> responseHandler = protocolFactory.createResponseHandler(new JsonOperationMetadata()
                .withPayloadJson(true).withHasStreamingSuccessResponse(false), new GetIdResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetIdRequest, GetIdResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getIdRequest)
                .withMarshaller(new GetIdRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the roles for an identity pool.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param getIdentityPoolRolesRequest
     *        Input to the <code>GetIdentityPoolRoles</code> action.
     * @return Result of the GetIdentityPoolRoles operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.GetIdentityPoolRoles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetIdentityPoolRoles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetIdentityPoolRolesResponse getIdentityPoolRoles(GetIdentityPoolRolesRequest getIdentityPoolRolesRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<GetIdentityPoolRolesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetIdentityPoolRolesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetIdentityPoolRolesRequest, GetIdentityPoolRolesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getIdentityPoolRolesRequest)
                .withMarshaller(new GetIdentityPoolRolesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets an OpenID token, using a known Cognito ID. This known Cognito ID is returned by <a>GetId</a>. You can
     * optionally add additional logins for the identity. Supplying multiple logins creates an implicit link.
     * </p>
     * <p>
     * The OpenId token is valid for 15 minutes.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param getOpenIdTokenRequest
     *        Input to the GetOpenIdToken action.
     * @return Result of the GetOpenIdToken operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws ExternalServiceException
     *         An exception thrown when a dependent service such as Facebook or Twitter is not responding
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.GetOpenIdToken
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetOpenIdToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOpenIdTokenResponse getOpenIdToken(GetOpenIdTokenRequest getOpenIdTokenRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, ResourceConflictException, TooManyRequestsException,
            InternalErrorException, ExternalServiceException, AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<GetOpenIdTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetOpenIdTokenResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetOpenIdTokenRequest, GetOpenIdTokenResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getOpenIdTokenRequest).withMarshaller(new GetOpenIdTokenRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Registers (or retrieves) a Cognito <code>IdentityId</code> and an OpenID Connect token for a user authenticated
     * by your backend authentication process. Supplying multiple logins will create an implicit linked account. You can
     * only specify one developer provider as part of the <code>Logins</code> map, which is linked to the identity pool.
     * The developer provider is the "domain" by which Cognito will refer to your users.
     * </p>
     * <p>
     * You can use <code>GetOpenIdTokenForDeveloperIdentity</code> to create a new identity and to link new logins (that
     * is, user credentials issued by a public provider or developer provider) to an existing identity. When you want to
     * create a new identity, the <code>IdentityId</code> should be null. When you want to associate a new login with an
     * existing authenticated/unauthenticated identity, you can do so by providing the existing <code>IdentityId</code>.
     * This API will create the identity in the specified <code>IdentityPoolId</code>.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param getOpenIdTokenForDeveloperIdentityRequest
     *        Input to the <code>GetOpenIdTokenForDeveloperIdentity</code> action.
     * @return Result of the GetOpenIdTokenForDeveloperIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws DeveloperUserAlreadyRegisteredException
     *         The provided developer user identifier is already registered with Cognito under a different identity ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.GetOpenIdTokenForDeveloperIdentity
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetOpenIdTokenForDeveloperIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOpenIdTokenForDeveloperIdentityResponse getOpenIdTokenForDeveloperIdentity(
            GetOpenIdTokenForDeveloperIdentityRequest getOpenIdTokenForDeveloperIdentityRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, DeveloperUserAlreadyRegisteredException, AwsServiceException,
            SdkClientException, CognitoIdentityException {

        HttpResponseHandler<GetOpenIdTokenForDeveloperIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetOpenIdTokenForDeveloperIdentityResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<GetOpenIdTokenForDeveloperIdentityRequest, GetOpenIdTokenForDeveloperIdentityResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(getOpenIdTokenForDeveloperIdentityRequest)
                        .withMarshaller(new GetOpenIdTokenForDeveloperIdentityRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the identities in a pool.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param listIdentitiesRequest
     *        Input to the ListIdentities action.
     * @return Result of the ListIdentities operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.ListIdentities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/ListIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListIdentitiesResponse listIdentities(ListIdentitiesRequest listIdentitiesRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException, InternalErrorException,
            AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<ListIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListIdentitiesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListIdentitiesRequest, ListIdentitiesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listIdentitiesRequest).withMarshaller(new ListIdentitiesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all of the Cognito identity pools registered for your account.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param listIdentityPoolsRequest
     *        Input to the ListIdentityPools action.
     * @return Result of the ListIdentityPools operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.ListIdentityPools
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/ListIdentityPools"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListIdentityPoolsResponse listIdentityPools(ListIdentityPoolsRequest listIdentityPoolsRequest)
            throws InvalidParameterException, NotAuthorizedException, TooManyRequestsException, InternalErrorException,
            AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<ListIdentityPoolsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListIdentityPoolsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListIdentityPoolsRequest, ListIdentityPoolsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listIdentityPoolsRequest).withMarshaller(new ListIdentityPoolsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the <code>IdentityID</code> associated with a <code>DeveloperUserIdentifier</code> or the list of
     * <code>DeveloperUserIdentifier</code>s associated with an <code>IdentityId</code> for an existing identity. Either
     * <code>IdentityID</code> or <code>DeveloperUserIdentifier</code> must not be null. If you supply only one of these
     * values, the other value will be searched in the database and returned as a part of the response. If you supply
     * both, <code>DeveloperUserIdentifier</code> will be matched against <code>IdentityID</code>. If the values are
     * verified against the database, the response returns both values and is the same as the request. Otherwise a
     * <code>ResourceConflictException</code> is thrown.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param lookupDeveloperIdentityRequest
     *        Input to the <code>LookupDeveloperIdentityInput</code> action.
     * @return Result of the LookupDeveloperIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.LookupDeveloperIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/LookupDeveloperIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public LookupDeveloperIdentityResponse lookupDeveloperIdentity(LookupDeveloperIdentityRequest lookupDeveloperIdentityRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<LookupDeveloperIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new LookupDeveloperIdentityResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<LookupDeveloperIdentityRequest, LookupDeveloperIdentityResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(lookupDeveloperIdentityRequest)
                .withMarshaller(new LookupDeveloperIdentityRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Merges two users having different <code>IdentityId</code>s, existing in the same identity pool, and identified by
     * the same developer provider. You can use this action to request that discrete users be merged and identified as a
     * single user in the Cognito environment. Cognito associates the given source user (
     * <code>SourceUserIdentifier</code>) with the <code>IdentityId</code> of the <code>DestinationUserIdentifier</code>
     * . Only developer-authenticated users can be merged. If the users to be merged are associated with the same public
     * provider, but as two different users, an exception will be thrown.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param mergeDeveloperIdentitiesRequest
     *        Input to the <code>MergeDeveloperIdentities</code> action.
     * @return Result of the MergeDeveloperIdentities operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.MergeDeveloperIdentities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/MergeDeveloperIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public MergeDeveloperIdentitiesResponse mergeDeveloperIdentities(
            MergeDeveloperIdentitiesRequest mergeDeveloperIdentitiesRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, ResourceConflictException, TooManyRequestsException,
            InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<MergeDeveloperIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new MergeDeveloperIdentitiesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<MergeDeveloperIdentitiesRequest, MergeDeveloperIdentitiesResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(mergeDeveloperIdentitiesRequest)
                        .withMarshaller(new MergeDeveloperIdentitiesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sets the roles for an identity pool. These roles are used when making calls to <a>GetCredentialsForIdentity</a>
     * action.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param setIdentityPoolRolesRequest
     *        Input to the <code>SetIdentityPoolRoles</code> action.
     * @return Result of the SetIdentityPoolRoles operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws ConcurrentModificationException
     *         Thrown if there are parallel requests to modify a resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.SetIdentityPoolRoles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/SetIdentityPoolRoles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetIdentityPoolRolesResponse setIdentityPoolRoles(SetIdentityPoolRolesRequest setIdentityPoolRolesRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CognitoIdentityException {

        HttpResponseHandler<SetIdentityPoolRolesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new SetIdentityPoolRolesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<SetIdentityPoolRolesRequest, SetIdentityPoolRolesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(setIdentityPoolRolesRequest)
                .withMarshaller(new SetIdentityPoolRolesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Unlinks a <code>DeveloperUserIdentifier</code> from an existing identity. Unlinked developer users will be
     * considered new identities next time they are seen. If, for a given Cognito identity, you remove all federated
     * identities as well as the developer user identifier, the Cognito identity becomes inaccessible.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param unlinkDeveloperIdentityRequest
     *        Input to the <code>UnlinkDeveloperIdentity</code> action.
     * @return Result of the UnlinkDeveloperIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.UnlinkDeveloperIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/UnlinkDeveloperIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UnlinkDeveloperIdentityResponse unlinkDeveloperIdentity(UnlinkDeveloperIdentityRequest unlinkDeveloperIdentityRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<UnlinkDeveloperIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new UnlinkDeveloperIdentityResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<UnlinkDeveloperIdentityRequest, UnlinkDeveloperIdentityResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(unlinkDeveloperIdentityRequest)
                .withMarshaller(new UnlinkDeveloperIdentityRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Unlinks a federated identity from an existing account. Unlinked logins will be considered new identities next
     * time they are seen. Removing the last linked login will make this identity inaccessible.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param unlinkIdentityRequest
     *        Input to the UnlinkIdentity action.
     * @return Result of the UnlinkIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws ExternalServiceException
     *         An exception thrown when a dependent service such as Facebook or Twitter is not responding
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.UnlinkIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/UnlinkIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UnlinkIdentityResponse unlinkIdentity(UnlinkIdentityRequest unlinkIdentityRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, ResourceConflictException, TooManyRequestsException,
            InternalErrorException, ExternalServiceException, AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<UnlinkIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new UnlinkIdentityResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<UnlinkIdentityRequest, UnlinkIdentityResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(unlinkIdentityRequest).withMarshaller(new UnlinkIdentityRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates a user pool.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param updateIdentityPoolRequest
     *        An object representing an Amazon Cognito identity pool.
     * @return Result of the UpdateIdentityPool operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws ConcurrentModificationException
     *         Thrown if there are parallel requests to modify a resource.
     * @throws LimitExceededException
     *         Thrown when the total number of user pools has exceeded a preset limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.UpdateIdentityPool
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/UpdateIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateIdentityPoolResponse updateIdentityPool(UpdateIdentityPoolRequest updateIdentityPoolRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, ConcurrentModificationException, LimitExceededException,
            AwsServiceException, SdkClientException, CognitoIdentityException {

        HttpResponseHandler<UpdateIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new UpdateIdentityPoolResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<UpdateIdentityPoolRequest, UpdateIdentityPoolResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(updateIdentityPoolRequest).withMarshaller(new UpdateIdentityPoolRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler() {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }

    private software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory init() {
        return new AwsJsonProtocolFactory(new JsonClientMetadata()
                .withSupportsCbor(false)
                .withSupportsIon(false)
                .withBaseServiceExceptionClass(
                        software.amazon.awssdk.services.cognitoidentity.model.CognitoIdentityException.class)
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ConcurrentModificationException").withModeledClass(
                                ConcurrentModificationException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidParameterException").withModeledClass(
                                InvalidParameterException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceNotFoundException").withModeledClass(
                                ResourceNotFoundException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("DeveloperUserAlreadyRegisteredException").withModeledClass(
                                DeveloperUserAlreadyRegisteredException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceConflictException").withModeledClass(
                                ResourceConflictException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ExternalServiceException").withModeledClass(
                                ExternalServiceException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("NotAuthorizedException").withModeledClass(
                                NotAuthorizedException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidIdentityPoolConfigurationException").withModeledClass(
                                InvalidIdentityPoolConfigurationException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InternalErrorException").withModeledClass(
                                InternalErrorException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("TooManyRequestsException").withModeledClass(
                                TooManyRequestsException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("LimitExceededException").withModeledClass(
                                LimitExceededException.class)), AwsJsonProtocolMetadata.builder().protocolVersion("1.1")
                .protocol(AwsJsonProtocol.AWS_JSON).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
