/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codestarnotifications;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.codestarnotifications.internal.CodestarNotificationsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.codestarnotifications.model.AccessDeniedException;
import software.amazon.awssdk.services.codestarnotifications.model.CodestarNotificationsException;
import software.amazon.awssdk.services.codestarnotifications.model.ConcurrentModificationException;
import software.amazon.awssdk.services.codestarnotifications.model.ConfigurationException;
import software.amazon.awssdk.services.codestarnotifications.model.CreateNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.CreateNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteTargetRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteTargetResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DescribeNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DescribeNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.InvalidNextTokenException;
import software.amazon.awssdk.services.codestarnotifications.model.LimitExceededException;
import software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListTargetsResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.codestarnotifications.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codestarnotifications.model.SubscribeRequest;
import software.amazon.awssdk.services.codestarnotifications.model.SubscribeResponse;
import software.amazon.awssdk.services.codestarnotifications.model.TagResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.TagResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UnsubscribeRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UnsubscribeResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UntagResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UntagResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UpdateNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UpdateNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ValidationException;
import software.amazon.awssdk.services.codestarnotifications.transform.CreateNotificationRuleRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.DeleteNotificationRuleRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.DeleteTargetRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.DescribeNotificationRuleRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListEventTypesRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListNotificationRulesRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListTargetsRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.SubscribeRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.UnsubscribeRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.UpdateNotificationRuleRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CodestarNotificationsClient}.
 *
 * @see CodestarNotificationsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodestarNotificationsClient implements CodestarNotificationsClient {
    private static final Logger log = Logger.loggerFor(DefaultCodestarNotificationsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodestarNotificationsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a notification rule for a resource. The rule specifies the events you want notifications about and the
     * targets (such as Chatbot topics or Chatbot clients configured for Slack) where you want to receive them.
     * </p>
     *
     * @param createNotificationRuleRequest
     * @return Result of the CreateNotificationRule operation returned by the service.
     * @throws ResourceAlreadyExistsException
     *         A resource with the same name or ID already exists. Notification rule names must be unique in your Amazon
     *         Web Services account.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws LimitExceededException
     *         One of the AWS CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification
     *         rules, notifications, resources, and targets. For more information, see Limits.
     * @throws ConfigurationException
     *         Some or all of the configuration is incomplete, missing, or not valid.
     * @throws ConcurrentModificationException
     *         AWS CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws AccessDeniedException
     *         AWS CodeStar Notifications can't create the notification rule because you do not have sufficient
     *         permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.CreateNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/CreateNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateNotificationRuleResponse createNotificationRule(CreateNotificationRuleRequest createNotificationRuleRequest)
            throws ResourceAlreadyExistsException, ValidationException, LimitExceededException, ConfigurationException,
            ConcurrentModificationException, AccessDeniedException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateNotificationRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createNotificationRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNotificationRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNotificationRule");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateNotificationRuleRequest, CreateNotificationRuleResponse>()
                            .withOperationName("CreateNotificationRule").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createNotificationRuleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateNotificationRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a notification rule for a resource.
     * </p>
     *
     * @param deleteNotificationRuleRequest
     * @return Result of the DeleteNotificationRule operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws LimitExceededException
     *         One of the AWS CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification
     *         rules, notifications, resources, and targets. For more information, see Limits.
     * @throws ConcurrentModificationException
     *         AWS CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DeleteNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DeleteNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteNotificationRuleResponse deleteNotificationRule(DeleteNotificationRuleRequest deleteNotificationRuleRequest)
            throws ValidationException, LimitExceededException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteNotificationRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteNotificationRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNotificationRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNotificationRule");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteNotificationRuleRequest, DeleteNotificationRuleResponse>()
                            .withOperationName("DeleteNotificationRule").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteNotificationRuleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteNotificationRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specified target for notifications.
     * </p>
     *
     * @param deleteTargetRequest
     * @return Result of the DeleteTarget operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DeleteTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DeleteTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteTargetResponse deleteTarget(DeleteTargetRequest deleteTargetRequest) throws ValidationException,
            AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTargetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTargetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTargetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTargetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTarget");

            return clientHandler.execute(new ClientExecutionParams<DeleteTargetRequest, DeleteTargetResponse>()
                    .withOperationName("DeleteTarget").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteTargetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTargetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a specified notification rule.
     * </p>
     *
     * @param describeNotificationRuleRequest
     * @return Result of the DescribeNotificationRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DescribeNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DescribeNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeNotificationRuleResponse describeNotificationRule(
            DescribeNotificationRuleRequest describeNotificationRuleRequest) throws ResourceNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeNotificationRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeNotificationRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeNotificationRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNotificationRule");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeNotificationRuleRequest, DescribeNotificationRuleResponse>()
                            .withOperationName("DescribeNotificationRule").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeNotificationRuleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeNotificationRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the event types available for configuring notifications.
     * </p>
     *
     * @param listEventTypesRequest
     * @return Result of the ListEventTypes operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListEventTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListEventTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEventTypesResponse listEventTypes(ListEventTypesRequest listEventTypesRequest) throws InvalidNextTokenException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEventTypesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEventTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEventTypesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEventTypes");

            return clientHandler.execute(new ClientExecutionParams<ListEventTypesRequest, ListEventTypesResponse>()
                    .withOperationName("ListEventTypes").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listEventTypesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEventTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the notification rules for an Amazon Web Services account.
     * </p>
     *
     * @param listNotificationRulesRequest
     * @return Result of the ListNotificationRules operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListNotificationRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListNotificationRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotificationRulesResponse listNotificationRules(ListNotificationRulesRequest listNotificationRulesRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNotificationRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListNotificationRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNotificationRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNotificationRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNotificationRules");

            return clientHandler.execute(new ClientExecutionParams<ListNotificationRulesRequest, ListNotificationRulesResponse>()
                    .withOperationName("ListNotificationRules").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listNotificationRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListNotificationRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the tags associated with a notification rule.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the notification rule targets for an Amazon Web Services account.
     * </p>
     *
     * @param listTargetsRequest
     * @return Result of the ListTargets operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTargetsResponse listTargets(ListTargetsRequest listTargetsRequest) throws InvalidNextTokenException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTargetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTargetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTargetsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTargetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTargets");

            return clientHandler.execute(new ClientExecutionParams<ListTargetsRequest, ListTargetsResponse>()
                    .withOperationName("ListTargets").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listTargetsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTargetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an association between a notification rule and an Chatbot topic or Chatbot client so that the associated
     * target can receive notifications when the events described in the rule are triggered.
     * </p>
     *
     * @param subscribeRequest
     * @return Result of the Subscribe operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ConfigurationException
     *         Some or all of the configuration is incomplete, missing, or not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.Subscribe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/Subscribe"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SubscribeResponse subscribe(SubscribeRequest subscribeRequest) throws ValidationException, ResourceNotFoundException,
            ConfigurationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SubscribeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SubscribeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(subscribeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, subscribeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Subscribe");

            return clientHandler.execute(new ClientExecutionParams<SubscribeRequest, SubscribeResponse>()
                    .withOperationName("Subscribe").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(subscribeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SubscribeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates a set of provided tags with a notification rule.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws LimitExceededException
     *         One of the AWS CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification
     *         rules, notifications, resources, and targets. For more information, see Limits.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ConcurrentModificationException
     *         AWS CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            LimitExceededException, ValidationException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes an association between a notification rule and an Chatbot topic so that subscribers to that topic stop
     * receiving notifications when the events described in the rule are triggered.
     * </p>
     *
     * @param unsubscribeRequest
     * @return Result of the Unsubscribe operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.Unsubscribe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/Unsubscribe"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UnsubscribeResponse unsubscribe(UnsubscribeRequest unsubscribeRequest) throws ValidationException,
            AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UnsubscribeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UnsubscribeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(unsubscribeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unsubscribeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Unsubscribe");

            return clientHandler.execute(new ClientExecutionParams<UnsubscribeRequest, UnsubscribeResponse>()
                    .withOperationName("Unsubscribe").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(unsubscribeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UnsubscribeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the association between one or more provided tags and a notification rule.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws LimitExceededException
     *         One of the AWS CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification
     *         rules, notifications, resources, and targets. For more information, see Limits.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ConcurrentModificationException
     *         AWS CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            LimitExceededException, ValidationException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a notification rule for a resource. You can change the events that trigger the notification rule, the
     * status of the rule, and the targets that receive the notifications.
     * </p>
     * <note>
     * <p>
     * To add or remove tags for a notification rule, you must use <a>TagResource</a> and <a>UntagResource</a>.
     * </p>
     * </note>
     *
     * @param updateNotificationRuleRequest
     * @return Result of the UpdateNotificationRule operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ConfigurationException
     *         Some or all of the configuration is incomplete, missing, or not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.UpdateNotificationRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/UpdateNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateNotificationRuleResponse updateNotificationRule(UpdateNotificationRuleRequest updateNotificationRuleRequest)
            throws ValidationException, ResourceNotFoundException, ConfigurationException, AwsServiceException,
            SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateNotificationRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateNotificationRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNotificationRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codestar notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNotificationRule");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateNotificationRuleRequest, UpdateNotificationRuleResponse>()
                            .withOperationName("UpdateNotificationRule").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateNotificationRuleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateNotificationRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        CodestarNotificationsServiceClientConfigurationBuilder serviceConfigBuilder = new CodestarNotificationsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodestarNotificationsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConfigurationException")
                                .exceptionBuilderSupplier(ConfigurationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    @Override
    public final CodestarNotificationsServiceClientConfiguration serviceClientConfiguration() {
        return new CodestarNotificationsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
