/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurureviewer.waiters;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.codegurureviewer.CodeGuruReviewerClient;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeCodeReviewRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeCodeReviewResponse;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeRepositoryAssociationRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeRepositoryAssociationResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface CodeGuruReviewerWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link CodeGuruReviewerClient#describeCodeReview} API until the desired condition
     * {@code CodeReviewCompleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeCodeReviewRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCodeReviewResponse> waitUntilCodeReviewCompleted(
            DescribeCodeReviewRequest describeCodeReviewRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link CodeGuruReviewerClient#describeCodeReview} API until the desired condition
     * {@code CodeReviewCompleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeCodeReviewRequest#builder()}
     *
     * @param describeCodeReviewRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCodeReviewResponse> waitUntilCodeReviewCompleted(
            Consumer<DescribeCodeReviewRequest.Builder> describeCodeReviewRequest) {
        return waitUntilCodeReviewCompleted(DescribeCodeReviewRequest.builder().applyMutation(describeCodeReviewRequest).build());
    }

    /**
     * Polls {@link CodeGuruReviewerClient#describeCodeReview} API until the desired condition
     * {@code CodeReviewCompleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeCodeReviewRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCodeReviewResponse> waitUntilCodeReviewCompleted(
            DescribeCodeReviewRequest describeCodeReviewRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link CodeGuruReviewerClient#describeCodeReview} API until the desired condition
     * {@code CodeReviewCompleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeCodeReviewRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCodeReviewResponse> waitUntilCodeReviewCompleted(
            Consumer<DescribeCodeReviewRequest.Builder> describeCodeReviewRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilCodeReviewCompleted(DescribeCodeReviewRequest.builder().applyMutation(describeCodeReviewRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link CodeGuruReviewerClient#describeRepositoryAssociation} API until the desired condition
     * {@code RepositoryAssociationSucceeded} is met, or until it is determined that the resource will never enter into
     * the desired state
     *
     * @param describeRepositoryAssociationRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeRepositoryAssociationResponse> waitUntilRepositoryAssociationSucceeded(
            DescribeRepositoryAssociationRequest describeRepositoryAssociationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link CodeGuruReviewerClient#describeRepositoryAssociation} API until the desired condition
     * {@code RepositoryAssociationSucceeded} is met, or until it is determined that the resource will never enter into
     * the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeRepositoryAssociationRequest#builder()}
     *
     * @param describeRepositoryAssociationRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeRepositoryAssociationResponse> waitUntilRepositoryAssociationSucceeded(
            Consumer<DescribeRepositoryAssociationRequest.Builder> describeRepositoryAssociationRequest) {
        return waitUntilRepositoryAssociationSucceeded(DescribeRepositoryAssociationRequest.builder()
                .applyMutation(describeRepositoryAssociationRequest).build());
    }

    /**
     * Polls {@link CodeGuruReviewerClient#describeRepositoryAssociation} API until the desired condition
     * {@code RepositoryAssociationSucceeded} is met, or until it is determined that the resource will never enter into
     * the desired state
     *
     * @param describeRepositoryAssociationRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeRepositoryAssociationResponse> waitUntilRepositoryAssociationSucceeded(
            DescribeRepositoryAssociationRequest describeRepositoryAssociationRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link CodeGuruReviewerClient#describeRepositoryAssociation} API until the desired condition
     * {@code RepositoryAssociationSucceeded} is met, or until it is determined that the resource will never enter into
     * the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeRepositoryAssociationRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeRepositoryAssociationResponse> waitUntilRepositoryAssociationSucceeded(
            Consumer<DescribeRepositoryAssociationRequest.Builder> describeRepositoryAssociationRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilRepositoryAssociationSucceeded(
                DescribeRepositoryAssociationRequest.builder().applyMutation(describeRepositoryAssociationRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link CodeGuruReviewerWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultCodeGuruReviewerWaiter.builder();
    }

    /**
     * Create an instance of {@link CodeGuruReviewerWaiter} with the default configuration.
     * <p>
     * <b>A default {@link CodeGuruReviewerClient} will be created to poll resources. It is recommended to share a
     * single instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link CodeGuruReviewerWaiter}
     */
    static CodeGuruReviewerWaiter create() {
        return DefaultCodeGuruReviewerWaiter.builder().build();
    }

    interface Builder {
        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link CodeGuruReviewerClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(CodeGuruReviewerClient client);

        /**
         * Builds an instance of {@link CodeGuruReviewerWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link CodeGuruReviewerWaiter}
         */
        CodeGuruReviewerWaiter build();
    }
}
