/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the target AWS Lambda function during an AWS Lambda deployment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LambdaTarget implements SdkPojo, Serializable, ToCopyableBuilder<LambdaTarget.Builder, LambdaTarget> {
    private static final SdkField<String> DEPLOYMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LambdaTarget::deploymentId)).setter(setter(Builder::deploymentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentId").build()).build();

    private static final SdkField<String> TARGET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LambdaTarget::targetId)).setter(setter(Builder::targetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetId").build()).build();

    private static final SdkField<String> TARGET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LambdaTarget::targetArn)).setter(setter(Builder::targetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LambdaTarget::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(LambdaTarget::lastUpdatedAt)).setter(setter(Builder::lastUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedAt").build()).build();

    private static final SdkField<List<LifecycleEvent>> LIFECYCLE_EVENTS_FIELD = SdkField
            .<List<LifecycleEvent>> builder(MarshallingType.LIST)
            .getter(getter(LambdaTarget::lifecycleEvents))
            .setter(setter(Builder::lifecycleEvents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lifecycleEvents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LifecycleEvent> builder(MarshallingType.SDK_POJO)
                                            .constructor(LifecycleEvent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LambdaFunctionInfo> LAMBDA_FUNCTION_INFO_FIELD = SdkField
            .<LambdaFunctionInfo> builder(MarshallingType.SDK_POJO).getter(getter(LambdaTarget::lambdaFunctionInfo))
            .setter(setter(Builder::lambdaFunctionInfo)).constructor(LambdaFunctionInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lambdaFunctionInfo").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DEPLOYMENT_ID_FIELD,
            TARGET_ID_FIELD, TARGET_ARN_FIELD, STATUS_FIELD, LAST_UPDATED_AT_FIELD, LIFECYCLE_EVENTS_FIELD,
            LAMBDA_FUNCTION_INFO_FIELD));

    private static final long serialVersionUID = 1L;

    private final String deploymentId;

    private final String targetId;

    private final String targetArn;

    private final String status;

    private final Instant lastUpdatedAt;

    private final List<LifecycleEvent> lifecycleEvents;

    private final LambdaFunctionInfo lambdaFunctionInfo;

    private LambdaTarget(BuilderImpl builder) {
        this.deploymentId = builder.deploymentId;
        this.targetId = builder.targetId;
        this.targetArn = builder.targetArn;
        this.status = builder.status;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.lifecycleEvents = builder.lifecycleEvents;
        this.lambdaFunctionInfo = builder.lambdaFunctionInfo;
    }

    /**
     * <p>
     * The unique ID of a deployment.
     * </p>
     * 
     * @return The unique ID of a deployment.
     */
    public String deploymentId() {
        return deploymentId;
    }

    /**
     * <p>
     * The unique ID of a deployment target that has a type of <code>lambdaTarget</code>.
     * </p>
     * 
     * @return The unique ID of a deployment target that has a type of <code>lambdaTarget</code>.
     */
    public String targetId() {
        return targetId;
    }

    /**
     * <p>
     * The ARN of the target.
     * </p>
     * 
     * @return The ARN of the target.
     */
    public String targetArn() {
        return targetArn;
    }

    /**
     * <p>
     * The status an AWS Lambda deployment's target Lambda function.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TargetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status an AWS Lambda deployment's target Lambda function.
     * @see TargetStatus
     */
    public TargetStatus status() {
        return TargetStatus.fromValue(status);
    }

    /**
     * <p>
     * The status an AWS Lambda deployment's target Lambda function.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TargetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status an AWS Lambda deployment's target Lambda function.
     * @see TargetStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The date and time when the target Lambda function was updated by a deployment.
     * </p>
     * 
     * @return The date and time when the target Lambda function was updated by a deployment.
     */
    public Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * Returns true if the LifecycleEvents property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasLifecycleEvents() {
        return lifecycleEvents != null && !(lifecycleEvents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The lifecycle events of the deployment to this target Lambda function.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLifecycleEvents()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The lifecycle events of the deployment to this target Lambda function.
     */
    public List<LifecycleEvent> lifecycleEvents() {
        return lifecycleEvents;
    }

    /**
     * <p>
     * A <code>LambdaFunctionInfo</code> object that describes a target Lambda function.
     * </p>
     * 
     * @return A <code>LambdaFunctionInfo</code> object that describes a target Lambda function.
     */
    public LambdaFunctionInfo lambdaFunctionInfo() {
        return lambdaFunctionInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deploymentId());
        hashCode = 31 * hashCode + Objects.hashCode(targetId());
        hashCode = 31 * hashCode + Objects.hashCode(targetArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleEvents());
        hashCode = 31 * hashCode + Objects.hashCode(lambdaFunctionInfo());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LambdaTarget)) {
            return false;
        }
        LambdaTarget other = (LambdaTarget) obj;
        return Objects.equals(deploymentId(), other.deploymentId()) && Objects.equals(targetId(), other.targetId())
                && Objects.equals(targetArn(), other.targetArn()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt())
                && Objects.equals(lifecycleEvents(), other.lifecycleEvents())
                && Objects.equals(lambdaFunctionInfo(), other.lambdaFunctionInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LambdaTarget").add("DeploymentId", deploymentId()).add("TargetId", targetId())
                .add("TargetArn", targetArn()).add("Status", statusAsString()).add("LastUpdatedAt", lastUpdatedAt())
                .add("LifecycleEvents", lifecycleEvents()).add("LambdaFunctionInfo", lambdaFunctionInfo()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "deploymentId":
            return Optional.ofNullable(clazz.cast(deploymentId()));
        case "targetId":
            return Optional.ofNullable(clazz.cast(targetId()));
        case "targetArn":
            return Optional.ofNullable(clazz.cast(targetArn()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "lastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        case "lifecycleEvents":
            return Optional.ofNullable(clazz.cast(lifecycleEvents()));
        case "lambdaFunctionInfo":
            return Optional.ofNullable(clazz.cast(lambdaFunctionInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LambdaTarget, T> g) {
        return obj -> g.apply((LambdaTarget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LambdaTarget> {
        /**
         * <p>
         * The unique ID of a deployment.
         * </p>
         * 
         * @param deploymentId
         *        The unique ID of a deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentId(String deploymentId);

        /**
         * <p>
         * The unique ID of a deployment target that has a type of <code>lambdaTarget</code>.
         * </p>
         * 
         * @param targetId
         *        The unique ID of a deployment target that has a type of <code>lambdaTarget</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetId(String targetId);

        /**
         * <p>
         * The ARN of the target.
         * </p>
         * 
         * @param targetArn
         *        The ARN of the target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArn(String targetArn);

        /**
         * <p>
         * The status an AWS Lambda deployment's target Lambda function.
         * </p>
         * 
         * @param status
         *        The status an AWS Lambda deployment's target Lambda function.
         * @see TargetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status an AWS Lambda deployment's target Lambda function.
         * </p>
         * 
         * @param status
         *        The status an AWS Lambda deployment's target Lambda function.
         * @see TargetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetStatus
         */
        Builder status(TargetStatus status);

        /**
         * <p>
         * The date and time when the target Lambda function was updated by a deployment.
         * </p>
         * 
         * @param lastUpdatedAt
         *        The date and time when the target Lambda function was updated by a deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * The lifecycle events of the deployment to this target Lambda function.
         * </p>
         * 
         * @param lifecycleEvents
         *        The lifecycle events of the deployment to this target Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleEvents(Collection<LifecycleEvent> lifecycleEvents);

        /**
         * <p>
         * The lifecycle events of the deployment to this target Lambda function.
         * </p>
         * 
         * @param lifecycleEvents
         *        The lifecycle events of the deployment to this target Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleEvents(LifecycleEvent... lifecycleEvents);

        /**
         * <p>
         * The lifecycle events of the deployment to this target Lambda function.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LifecycleEvent>.Builder} avoiding the need
         * to create one manually via {@link List<LifecycleEvent>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LifecycleEvent>.Builder#build()} is called immediately and
         * its result is passed to {@link #lifecycleEvents(List<LifecycleEvent>)}.
         * 
         * @param lifecycleEvents
         *        a consumer that will call methods on {@link List<LifecycleEvent>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lifecycleEvents(List<LifecycleEvent>)
         */
        Builder lifecycleEvents(Consumer<LifecycleEvent.Builder>... lifecycleEvents);

        /**
         * <p>
         * A <code>LambdaFunctionInfo</code> object that describes a target Lambda function.
         * </p>
         * 
         * @param lambdaFunctionInfo
         *        A <code>LambdaFunctionInfo</code> object that describes a target Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lambdaFunctionInfo(LambdaFunctionInfo lambdaFunctionInfo);

        /**
         * <p>
         * A <code>LambdaFunctionInfo</code> object that describes a target Lambda function.
         * </p>
         * This is a convenience that creates an instance of the {@link LambdaFunctionInfo.Builder} avoiding the need to
         * create one manually via {@link LambdaFunctionInfo#builder()}.
         *
         * When the {@link Consumer} completes, {@link LambdaFunctionInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #lambdaFunctionInfo(LambdaFunctionInfo)}.
         * 
         * @param lambdaFunctionInfo
         *        a consumer that will call methods on {@link LambdaFunctionInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lambdaFunctionInfo(LambdaFunctionInfo)
         */
        default Builder lambdaFunctionInfo(Consumer<LambdaFunctionInfo.Builder> lambdaFunctionInfo) {
            return lambdaFunctionInfo(LambdaFunctionInfo.builder().applyMutation(lambdaFunctionInfo).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String deploymentId;

        private String targetId;

        private String targetArn;

        private String status;

        private Instant lastUpdatedAt;

        private List<LifecycleEvent> lifecycleEvents = DefaultSdkAutoConstructList.getInstance();

        private LambdaFunctionInfo lambdaFunctionInfo;

        private BuilderImpl() {
        }

        private BuilderImpl(LambdaTarget model) {
            deploymentId(model.deploymentId);
            targetId(model.targetId);
            targetArn(model.targetArn);
            status(model.status);
            lastUpdatedAt(model.lastUpdatedAt);
            lifecycleEvents(model.lifecycleEvents);
            lambdaFunctionInfo(model.lambdaFunctionInfo);
        }

        public final String getDeploymentId() {
            return deploymentId;
        }

        @Override
        public final Builder deploymentId(String deploymentId) {
            this.deploymentId = deploymentId;
            return this;
        }

        public final void setDeploymentId(String deploymentId) {
            this.deploymentId = deploymentId;
        }

        public final String getTargetId() {
            return targetId;
        }

        @Override
        public final Builder targetId(String targetId) {
            this.targetId = targetId;
            return this;
        }

        public final void setTargetId(String targetId) {
            this.targetId = targetId;
        }

        public final String getTargetArn() {
            return targetArn;
        }

        @Override
        public final Builder targetArn(String targetArn) {
            this.targetArn = targetArn;
            return this;
        }

        public final void setTargetArn(String targetArn) {
            this.targetArn = targetArn;
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(TargetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        public final Collection<LifecycleEvent.Builder> getLifecycleEvents() {
            return lifecycleEvents != null ? lifecycleEvents.stream().map(LifecycleEvent::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder lifecycleEvents(Collection<LifecycleEvent> lifecycleEvents) {
            this.lifecycleEvents = LifecycleEventListCopier.copy(lifecycleEvents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleEvents(LifecycleEvent... lifecycleEvents) {
            lifecycleEvents(Arrays.asList(lifecycleEvents));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleEvents(Consumer<LifecycleEvent.Builder>... lifecycleEvents) {
            lifecycleEvents(Stream.of(lifecycleEvents).map(c -> LifecycleEvent.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setLifecycleEvents(Collection<LifecycleEvent.BuilderImpl> lifecycleEvents) {
            this.lifecycleEvents = LifecycleEventListCopier.copyFromBuilder(lifecycleEvents);
        }

        public final LambdaFunctionInfo.Builder getLambdaFunctionInfo() {
            return lambdaFunctionInfo != null ? lambdaFunctionInfo.toBuilder() : null;
        }

        @Override
        public final Builder lambdaFunctionInfo(LambdaFunctionInfo lambdaFunctionInfo) {
            this.lambdaFunctionInfo = lambdaFunctionInfo;
            return this;
        }

        public final void setLambdaFunctionInfo(LambdaFunctionInfo.BuilderImpl lambdaFunctionInfo) {
            this.lambdaFunctionInfo = lambdaFunctionInfo != null ? lambdaFunctionInfo.build() : null;
        }

        @Override
        public LambdaTarget build() {
            return new LambdaTarget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
