/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchlogs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Indicates how to transform ingested log events to metric data in a CloudWatch metric.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricTransformation implements SdkPojo, Serializable,
        ToCopyableBuilder<MetricTransformation.Builder, MetricTransformation> {
    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("metricName").getter(getter(MetricTransformation::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metricName").build()).build();

    private static final SdkField<String> METRIC_NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("metricNamespace").getter(getter(MetricTransformation::metricNamespace))
            .setter(setter(Builder::metricNamespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metricNamespace").build()).build();

    private static final SdkField<String> METRIC_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("metricValue").getter(getter(MetricTransformation::metricValue)).setter(setter(Builder::metricValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metricValue").build()).build();

    private static final SdkField<Double> DEFAULT_VALUE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("defaultValue").getter(getter(MetricTransformation::defaultValue)).setter(setter(Builder::defaultValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultValue").build()).build();

    private static final SdkField<Map<String, String>> DIMENSIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("dimensions")
            .getter(getter(MetricTransformation::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dimensions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("unit")
            .getter(getter(MetricTransformation::unitAsString)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("unit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_NAME_FIELD,
            METRIC_NAMESPACE_FIELD, METRIC_VALUE_FIELD, DEFAULT_VALUE_FIELD, DIMENSIONS_FIELD, UNIT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("metricName", METRIC_NAME_FIELD);
                    put("metricNamespace", METRIC_NAMESPACE_FIELD);
                    put("metricValue", METRIC_VALUE_FIELD);
                    put("defaultValue", DEFAULT_VALUE_FIELD);
                    put("dimensions", DIMENSIONS_FIELD);
                    put("unit", UNIT_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String metricName;

    private final String metricNamespace;

    private final String metricValue;

    private final Double defaultValue;

    private final Map<String, String> dimensions;

    private final String unit;

    private MetricTransformation(BuilderImpl builder) {
        this.metricName = builder.metricName;
        this.metricNamespace = builder.metricNamespace;
        this.metricValue = builder.metricValue;
        this.defaultValue = builder.defaultValue;
        this.dimensions = builder.dimensions;
        this.unit = builder.unit;
    }

    /**
     * <p>
     * The name of the CloudWatch metric.
     * </p>
     * 
     * @return The name of the CloudWatch metric.
     */
    public final String metricName() {
        return metricName;
    }

    /**
     * <p>
     * A custom namespace to contain your metric in CloudWatch. Use namespaces to group together metrics that are
     * similar. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Namespace"
     * >Namespaces</a>.
     * </p>
     * 
     * @return A custom namespace to contain your metric in CloudWatch. Use namespaces to group together metrics that
     *         are similar. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Namespace"
     *         >Namespaces</a>.
     */
    public final String metricNamespace() {
        return metricNamespace;
    }

    /**
     * <p>
     * The value to publish to the CloudWatch metric when a filter pattern matches a log event.
     * </p>
     * 
     * @return The value to publish to the CloudWatch metric when a filter pattern matches a log event.
     */
    public final String metricValue() {
        return metricValue;
    }

    /**
     * <p>
     * (Optional) The value to emit when a filter pattern does not match a log event. This value can be null.
     * </p>
     * 
     * @return (Optional) The value to emit when a filter pattern does not match a log event. This value can be null.
     */
    public final Double defaultValue() {
        return defaultValue;
    }

    /**
     * For responses, this returns true if the service returned a value for the Dimensions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The fields to use as dimensions for the metric. One metric filter can include as many as three dimensions.
     * </p>
     * <important>
     * <p>
     * Metrics extracted from log events are charged as custom metrics. To prevent unexpected high charges, do not
     * specify high-cardinality fields such as <code>IPAddress</code> or <code>requestID</code> as dimensions. Each
     * different value found for a dimension is treated as a separate metric and accrues charges as a separate custom
     * metric.
     * </p>
     * <p>
     * CloudWatch Logs disables a metric filter if it generates 1000 different name/value pairs for your specified
     * dimensions within a certain amount of time. This helps to prevent accidental high charges.
     * </p>
     * <p>
     * You can also set up a billing alarm to alert you if your charges are higher than expected. For more information,
     * see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/monitor_estimated_charges_with_cloudwatch.html">
     * Creating a Billing Alarm to Monitor Your Estimated Amazon Web Services Charges</a>.
     * </p>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensions} method.
     * </p>
     * 
     * @return The fields to use as dimensions for the metric. One metric filter can include as many as three
     *         dimensions.</p> <important>
     *         <p>
     *         Metrics extracted from log events are charged as custom metrics. To prevent unexpected high charges, do
     *         not specify high-cardinality fields such as <code>IPAddress</code> or <code>requestID</code> as
     *         dimensions. Each different value found for a dimension is treated as a separate metric and accrues
     *         charges as a separate custom metric.
     *         </p>
     *         <p>
     *         CloudWatch Logs disables a metric filter if it generates 1000 different name/value pairs for your
     *         specified dimensions within a certain amount of time. This helps to prevent accidental high charges.
     *         </p>
     *         <p>
     *         You can also set up a billing alarm to alert you if your charges are higher than expected. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/monitor_estimated_charges_with_cloudwatch.html"
     *         > Creating a Billing Alarm to Monitor Your Estimated Amazon Web Services Charges</a>.
     *         </p>
     */
    public final Map<String, String> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The unit to assign to the metric. If you omit this, the unit is set as <code>None</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit to assign to the metric. If you omit this, the unit is set as <code>None</code>.
     * @see StandardUnit
     */
    public final StandardUnit unit() {
        return StandardUnit.fromValue(unit);
    }

    /**
     * <p>
     * The unit to assign to the metric. If you omit this, the unit is set as <code>None</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit to assign to the metric. If you omit this, the unit is set as <code>None</code>.
     * @see StandardUnit
     */
    public final String unitAsString() {
        return unit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(metricNamespace());
        hashCode = 31 * hashCode + Objects.hashCode(metricValue());
        hashCode = 31 * hashCode + Objects.hashCode(defaultValue());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(unitAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricTransformation)) {
            return false;
        }
        MetricTransformation other = (MetricTransformation) obj;
        return Objects.equals(metricName(), other.metricName()) && Objects.equals(metricNamespace(), other.metricNamespace())
                && Objects.equals(metricValue(), other.metricValue()) && Objects.equals(defaultValue(), other.defaultValue())
                && hasDimensions() == other.hasDimensions() && Objects.equals(dimensions(), other.dimensions())
                && Objects.equals(unitAsString(), other.unitAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MetricTransformation").add("MetricName", metricName()).add("MetricNamespace", metricNamespace())
                .add("MetricValue", metricValue()).add("DefaultValue", defaultValue())
                .add("Dimensions", hasDimensions() ? dimensions() : null).add("Unit", unitAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "metricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "metricNamespace":
            return Optional.ofNullable(clazz.cast(metricNamespace()));
        case "metricValue":
            return Optional.ofNullable(clazz.cast(metricValue()));
        case "defaultValue":
            return Optional.ofNullable(clazz.cast(defaultValue()));
        case "dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "unit":
            return Optional.ofNullable(clazz.cast(unitAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<MetricTransformation, T> g) {
        return obj -> g.apply((MetricTransformation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricTransformation> {
        /**
         * <p>
         * The name of the CloudWatch metric.
         * </p>
         * 
         * @param metricName
         *        The name of the CloudWatch metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * A custom namespace to contain your metric in CloudWatch. Use namespaces to group together metrics that are
         * similar. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Namespace"
         * >Namespaces</a>.
         * </p>
         * 
         * @param metricNamespace
         *        A custom namespace to contain your metric in CloudWatch. Use namespaces to group together metrics that
         *        are similar. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Namespace"
         *        >Namespaces</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricNamespace(String metricNamespace);

        /**
         * <p>
         * The value to publish to the CloudWatch metric when a filter pattern matches a log event.
         * </p>
         * 
         * @param metricValue
         *        The value to publish to the CloudWatch metric when a filter pattern matches a log event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricValue(String metricValue);

        /**
         * <p>
         * (Optional) The value to emit when a filter pattern does not match a log event. This value can be null.
         * </p>
         * 
         * @param defaultValue
         *        (Optional) The value to emit when a filter pattern does not match a log event. This value can be null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultValue(Double defaultValue);

        /**
         * <p>
         * The fields to use as dimensions for the metric. One metric filter can include as many as three dimensions.
         * </p>
         * <important>
         * <p>
         * Metrics extracted from log events are charged as custom metrics. To prevent unexpected high charges, do not
         * specify high-cardinality fields such as <code>IPAddress</code> or <code>requestID</code> as dimensions. Each
         * different value found for a dimension is treated as a separate metric and accrues charges as a separate
         * custom metric.
         * </p>
         * <p>
         * CloudWatch Logs disables a metric filter if it generates 1000 different name/value pairs for your specified
         * dimensions within a certain amount of time. This helps to prevent accidental high charges.
         * </p>
         * <p>
         * You can also set up a billing alarm to alert you if your charges are higher than expected. For more
         * information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/monitor_estimated_charges_with_cloudwatch.html"
         * > Creating a Billing Alarm to Monitor Your Estimated Amazon Web Services Charges</a>.
         * </p>
         * </important>
         * 
         * @param dimensions
         *        The fields to use as dimensions for the metric. One metric filter can include as many as three
         *        dimensions.</p> <important>
         *        <p>
         *        Metrics extracted from log events are charged as custom metrics. To prevent unexpected high charges,
         *        do not specify high-cardinality fields such as <code>IPAddress</code> or <code>requestID</code> as
         *        dimensions. Each different value found for a dimension is treated as a separate metric and accrues
         *        charges as a separate custom metric.
         *        </p>
         *        <p>
         *        CloudWatch Logs disables a metric filter if it generates 1000 different name/value pairs for your
         *        specified dimensions within a certain amount of time. This helps to prevent accidental high charges.
         *        </p>
         *        <p>
         *        You can also set up a billing alarm to alert you if your charges are higher than expected. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/monitor_estimated_charges_with_cloudwatch.html"
         *        > Creating a Billing Alarm to Monitor Your Estimated Amazon Web Services Charges</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Map<String, String> dimensions);

        /**
         * <p>
         * The unit to assign to the metric. If you omit this, the unit is set as <code>None</code>.
         * </p>
         * 
         * @param unit
         *        The unit to assign to the metric. If you omit this, the unit is set as <code>None</code>.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The unit to assign to the metric. If you omit this, the unit is set as <code>None</code>.
         * </p>
         * 
         * @param unit
         *        The unit to assign to the metric. If you omit this, the unit is set as <code>None</code>.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(StandardUnit unit);
    }

    static final class BuilderImpl implements Builder {
        private String metricName;

        private String metricNamespace;

        private String metricValue;

        private Double defaultValue;

        private Map<String, String> dimensions = DefaultSdkAutoConstructMap.getInstance();

        private String unit;

        private BuilderImpl() {
        }

        private BuilderImpl(MetricTransformation model) {
            metricName(model.metricName);
            metricNamespace(model.metricNamespace);
            metricValue(model.metricValue);
            defaultValue(model.defaultValue);
            dimensions(model.dimensions);
            unit(model.unit);
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final String getMetricNamespace() {
            return metricNamespace;
        }

        public final void setMetricNamespace(String metricNamespace) {
            this.metricNamespace = metricNamespace;
        }

        @Override
        public final Builder metricNamespace(String metricNamespace) {
            this.metricNamespace = metricNamespace;
            return this;
        }

        public final String getMetricValue() {
            return metricValue;
        }

        public final void setMetricValue(String metricValue) {
            this.metricValue = metricValue;
        }

        @Override
        public final Builder metricValue(String metricValue) {
            this.metricValue = metricValue;
            return this;
        }

        public final Double getDefaultValue() {
            return defaultValue;
        }

        public final void setDefaultValue(Double defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public final Builder defaultValue(Double defaultValue) {
            this.defaultValue = defaultValue;
            return this;
        }

        public final Map<String, String> getDimensions() {
            if (dimensions instanceof SdkAutoConstructMap) {
                return null;
            }
            return dimensions;
        }

        public final void setDimensions(Map<String, String> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
        }

        @Override
        public final Builder dimensions(Map<String, String> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
            return this;
        }

        public final String getUnit() {
            return unit;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        public final Builder unit(StandardUnit unit) {
            this.unit(unit == null ? null : unit.toString());
            return this;
        }

        @Override
        public MetricTransformation build() {
            return new MetricTransformation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
