/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchlogs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains delivery configurations that apply only when the delivery destination resource is an S3
 * bucket.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3DeliveryConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<S3DeliveryConfiguration.Builder, S3DeliveryConfiguration> {
    private static final SdkField<String> SUFFIX_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("suffixPath").getter(getter(S3DeliveryConfiguration::suffixPath)).setter(setter(Builder::suffixPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("suffixPath").build()).build();

    private static final SdkField<Boolean> ENABLE_HIVE_COMPATIBLE_PATH_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("enableHiveCompatiblePath")
            .getter(getter(S3DeliveryConfiguration::enableHiveCompatiblePath)).setter(setter(Builder::enableHiveCompatiblePath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enableHiveCompatiblePath").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUFFIX_PATH_FIELD,
            ENABLE_HIVE_COMPATIBLE_PATH_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("suffixPath", SUFFIX_PATH_FIELD);
                    put("enableHiveCompatiblePath", ENABLE_HIVE_COMPATIBLE_PATH_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String suffixPath;

    private final Boolean enableHiveCompatiblePath;

    private S3DeliveryConfiguration(BuilderImpl builder) {
        this.suffixPath = builder.suffixPath;
        this.enableHiveCompatiblePath = builder.enableHiveCompatiblePath;
    }

    /**
     * <p>
     * This string allows re-configuring the S3 object prefix to contain either static or variable sections. The valid
     * variables to use in the suffix path will vary by each log source. See
     * ConfigurationTemplate$allowedSuffixPathFields for more info on what values are supported in the suffix path for
     * each log source.
     * </p>
     * 
     * @return This string allows re-configuring the S3 object prefix to contain either static or variable sections. The
     *         valid variables to use in the suffix path will vary by each log source. See
     *         ConfigurationTemplate$allowedSuffixPathFields for more info on what values are supported in the suffix
     *         path for each log source.
     */
    public final String suffixPath() {
        return suffixPath;
    }

    /**
     * <p>
     * This parameter causes the S3 objects that contain delivered logs to use a prefix structure that allows for
     * integration with Apache Hive.
     * </p>
     * 
     * @return This parameter causes the S3 objects that contain delivered logs to use a prefix structure that allows
     *         for integration with Apache Hive.
     */
    public final Boolean enableHiveCompatiblePath() {
        return enableHiveCompatiblePath;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(suffixPath());
        hashCode = 31 * hashCode + Objects.hashCode(enableHiveCompatiblePath());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3DeliveryConfiguration)) {
            return false;
        }
        S3DeliveryConfiguration other = (S3DeliveryConfiguration) obj;
        return Objects.equals(suffixPath(), other.suffixPath())
                && Objects.equals(enableHiveCompatiblePath(), other.enableHiveCompatiblePath());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3DeliveryConfiguration").add("SuffixPath", suffixPath())
                .add("EnableHiveCompatiblePath", enableHiveCompatiblePath()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "suffixPath":
            return Optional.ofNullable(clazz.cast(suffixPath()));
        case "enableHiveCompatiblePath":
            return Optional.ofNullable(clazz.cast(enableHiveCompatiblePath()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<S3DeliveryConfiguration, T> g) {
        return obj -> g.apply((S3DeliveryConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3DeliveryConfiguration> {
        /**
         * <p>
         * This string allows re-configuring the S3 object prefix to contain either static or variable sections. The
         * valid variables to use in the suffix path will vary by each log source. See
         * ConfigurationTemplate$allowedSuffixPathFields for more info on what values are supported in the suffix path
         * for each log source.
         * </p>
         * 
         * @param suffixPath
         *        This string allows re-configuring the S3 object prefix to contain either static or variable sections.
         *        The valid variables to use in the suffix path will vary by each log source. See
         *        ConfigurationTemplate$allowedSuffixPathFields for more info on what values are supported in the suffix
         *        path for each log source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suffixPath(String suffixPath);

        /**
         * <p>
         * This parameter causes the S3 objects that contain delivered logs to use a prefix structure that allows for
         * integration with Apache Hive.
         * </p>
         * 
         * @param enableHiveCompatiblePath
         *        This parameter causes the S3 objects that contain delivered logs to use a prefix structure that allows
         *        for integration with Apache Hive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableHiveCompatiblePath(Boolean enableHiveCompatiblePath);
    }

    static final class BuilderImpl implements Builder {
        private String suffixPath;

        private Boolean enableHiveCompatiblePath;

        private BuilderImpl() {
        }

        private BuilderImpl(S3DeliveryConfiguration model) {
            suffixPath(model.suffixPath);
            enableHiveCompatiblePath(model.enableHiveCompatiblePath);
        }

        public final String getSuffixPath() {
            return suffixPath;
        }

        public final void setSuffixPath(String suffixPath) {
            this.suffixPath = suffixPath;
        }

        @Override
        public final Builder suffixPath(String suffixPath) {
            this.suffixPath = suffixPath;
            return this;
        }

        public final Boolean getEnableHiveCompatiblePath() {
            return enableHiveCompatiblePath;
        }

        public final void setEnableHiveCompatiblePath(Boolean enableHiveCompatiblePath) {
            this.enableHiveCompatiblePath = enableHiveCompatiblePath;
        }

        @Override
        public final Builder enableHiveCompatiblePath(Boolean enableHiveCompatiblePath) {
            this.enableHiveCompatiblePath = enableHiveCompatiblePath;
            return this;
        }

        @Override
        public S3DeliveryConfiguration build() {
            return new S3DeliveryConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
